-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Formation_Profile_Recorder.Container;
with Ada.Streams.Stream_Io;
with Ada.Characters.Latin_1;
with System;
with Interfaces.C;
with Jpats_Formation_Types;
with Jpats_Simulated_Aircraft;
with Jpats_Secondary_Flight_Controls;
with Jpats_Landing_Gear;
with Jpats_DCLS;

--with Log;

package body Jpats_Formation_Profile_Recorder.Controller is

   use type Interfaces.C.Int;
   use type Jpats_Formation_Types.Segment_Type;

   function Copy ( From_Name : System.Address;
                   To_Name : System.Address ) return Interfaces.C.Int;
   pragma Import ( C, Copy, "copy" );

   -- Implementation note -- need to decide how to make segment offset
   -- table a part of the file.  My thought right now is to combine
   -- the segment offset array with the snapshot data when I copy the
   -- data from the RAM file to the disk file.
   --
   -- The disk file would have one word with the number of segments,
   -- one word with the number of snapshots, then the segment array,
   -- then the snapshots.  Should be able to use stream_io to write
   -- the file.

   procedure Copy_File ( From_Name : in String;
                         To_Name   : in String ) is
      From_Name_C : Interfaces.C.Char_Array ( 0 .. From_Name'Length );
      To_Name_C   : Interfaces.C.Char_Array ( 0 .. To_Name'Length );
      Result : Interfaces.C.Int;
   begin
      From_Name_C := Interfaces.C.To_C ( From_Name );
      To_Name_C   := Interfaces.C.To_C ( To_Name );
      Result := Copy ( From_Name_C(0)'Address, To_Name_C(0)'Address );
      if Result < 0 then
         -- error handling
         null;
      end if;
   end Copy_File;

   procedure Close_File ( File : in out Ada.Streams.Stream_Io.File_Type ) is
   begin
      Ada.Streams.Stream_Io.Close ( File => File );

   exception
      when others =>
         -- log message (can't close file)
         raise;
   end Close_File;


   procedure Open_File
     ( File : in out Ada.Streams.Stream_Io.File_Type;
       Name : in String ) is
   begin
      Ada.Streams.Stream_Io.Create
        ( File => File,
          Mode => Ada.Streams.Stream_Io.Out_File,
          Name => Name );
   exception
      when others =>
         -- log message (can't open file)
         raise;
   end Open_File;

   procedure Rewind_To ( File : in out Ada.Streams.Stream_Io.File_Type;
                         Index : in Ada.Streams.Stream_Io.Positive_Count  )  is
   begin
      Ada.Streams.Stream_Io.Set_Index
        ( File => File,
          To => Index );
   exception
      when others =>
         -- log message (can't set file index)
         raise;
   end Rewind_To;

   procedure Take_Snapshot
     ( Stream : in out Ada.Streams.Stream_Io.Stream_Access  ) is
      The_Snapshot : Jpats_Formation_Types.Prerecorded_Profile_Snapshot;
   begin
      The_Snapshot :=
        ( The_Pitch =>
            Jpats_Simulated_Aircraft.Get_Pitch_Angle,
          The_Roll =>
            Jpats_Simulated_Aircraft.Get_Roll_Angle,
          The_Heading =>
            Jpats_Simulated_Aircraft.Get_Hdg_Angle,
          The_Latitude =>
            Jpats_Simulated_Aircraft.Get_North,
          The_Longitude =>
            Jpats_Simulated_Aircraft.Get_East,
          The_Altitude =>
            Jpats_Simulated_Aircraft.Get_Aircraft_Geometric_Altitude,
          The_Mean_Flap_Position =>
            Jpats_Secondary_Flight_Controls.Mean_Flap_Position,
          The_Speedbrake_Position =>
            Jpats_Secondary_Flight_Controls.Speedbrake_Position,
          The_Mean_Landing_Gear_Position =>
            Jpats_Landing_Gear.Mean_Gear_Position,
          The_Elevator_Position =>
            Jpats_DCLS.Get_Elev_Pos,
          The_Rudder_Position =>
            Jpats_DCLS.Get_Rudder_Position,
          The_Right_Aileron_Position =>
            Jpats_DCLS.Get_Right_Aileron_Position,
          The_Left_Aileron_Position =>
            Jpats_DCLS.Get_Left_Aileron_Position,
          The_Indicated_Airspeed => JPATS_Simulated_Aircraft.Get_Calibrated_Airspeed * 1.687809);
                                                                 -- 1.687809 = ft-per-sec/knots

      Jpats_Formation_Types.Prerecorded_Profile_Snapshot'Write
        ( Stream,
          The_Snapshot );

   exception
      when others =>
         raise;
   end Take_Snapshot;

   procedure Initialize is
   begin

      -- initialize container

      Container.This_Subsystem.The_State := Jpats_Formation_Types.Idle;
      Container.This_Subsystem.Recording_Start_Requested := False;
      Container.This_Subsystem.Segment_End_Requested := False;
      Container.This_Subsystem.Segment_Rewind_Requested := False;
      Container.This_Subsystem.Profile_Done_Requested := False;
      Container.This_Subsystem.The_Header.The_Segment_Count := 0;
      Container.This_Subsystem.The_Header.The_Snapshot_Count := 0;
      Container.This_Subsystem.The_Header.The_Segment_Offset_Array := ( others => 0 );

      -- The following should probably be done in the VxWorks startup script
      --
      -- create RAM disk
      --
      -- BLK_DEV *ramDevCreate
      --           (char *ramAddr, int bytesPerBlk, int blksPerTrack,
      --            int nBlocks, int blkOffset)
      --
      --
      -- Test for null result
      --
      -- Pass return value as parameter to dosFsMkfs
      --
      -- DOS_VOL_DESC *dosFsMkfs
      --   (
      --   char     *volName,  /* volume name to use             */
      --   BLK_DEV  *pBlkDev   /* pointer to block device struct */
      --   }
      --
      -- Test for null result
      --

   end Initialize;

   -- The update procedure implements a simple state machine with five
   -- states -- Idle, Initializing, Recording, Paused, Finishing and
   -- four possible events, all of which are requests to change state.
   -- The events are requests to start recording, end the current
   -- segment (pausing recording and marking the snapshot), rewind to
   -- the beginning of the current segment (allowing the instructor to
   -- "record over" a mistake), and end the profile.

   -- The recorded profile is written to a file on a RAM disk and when
   -- the profile is complete the file is copied to a known location
   -- the hard drive.  The IOS can then move the file where the
   -- instructor pleases.

   -- This procedure only controls the collection and storage of a
   -- profile.  Another software subsystem will handle coordination of
   -- this recorder with the rest of simulation -- for example,
   -- dealing with a flight freeze during recording.

   procedure Update
     (Iconst : in Float) is

--      Standard_Ram_File_Name : constant String := "/ata0/formflt/record.tmp";
--      Standard_Ram_File_Name : constant String := "ramdisk:formrcrd.tmp";
--      Standard_Temp_File_Name : constant String := "/ata0/formflt/record.dat";
      Standard_Ram_File_Name : constant String := "/ramdisk/record.tmp";
      Standard_Temp_File_Name : constant String := "/ramdisk/record.dat";

   begin

      -- evaluate state

      case Container.This_Subsystem.The_State is

         when Jpats_Formation_Types.Idle =>

            -- look for signal to start recording
            if Container.This_Subsystem.Recording_Start_Requested then

               Container.This_Subsystem.The_State :=
                 Jpats_Formation_Types.Initializing;

            end if;

         when Jpats_Formation_Types.Initializing =>
            -- single-pass state

            -- open a file on the RAM disk
            Open_File ( Container.This_Subsystem.The_Temporary_File,
                        Standard_Ram_File_Name );

            -- get a stream pointer for use in T'Read and T'Write
            Container.This_Subsystem.The_File_Stream :=
              Ada.Streams.Stream_Io.Stream
              ( Container.This_Subsystem.The_Temporary_File );

            -- Write a zeroed header and segment offset array to make
            -- space at the beginning of the file.  Before we close
            -- the file we'll overwrite this with actual data for the
            -- current profile

            Jpats_Formation_Types.Profile_Header'Write
              ( Container.This_Subsystem.The_File_Stream,
                Container.This_Subsystem.The_Header );

            -- reset segment array and segment number
            Container.This_Subsystem.The_Header.The_Segment_Count := 1;
            Container.This_Subsystem.The_Header.The_Segment_Offset_Array
              ( Container.This_Subsystem.The_Header.The_Segment_Count )
              := 1;
            Container.This_Subsystem.The_Header.The_Snapshot_Count := 1;
            Container.This_Subsystem.The_State :=
              Jpats_Formation_Types.Recording;
            Container.This_Subsystem.The_Current_Segment_Start_Index
              := Ada.Streams.Stream_Io.Index
                 ( Container.This_Subsystem.The_Temporary_File  );

         when Jpats_Formation_Types.Recording =>

            -- record snapshot
            Take_Snapshot ( Container.This_Subsystem.The_File_Stream );
            Container.This_Subsystem.The_Header.The_Snapshot_Count :=
              Container.This_Subsystem.The_Header.The_Snapshot_Count + 1;

            if Container.This_Subsystem.Profile_Done_Requested then

               Container.This_Subsystem.The_State
                 := Jpats_Formation_Types.Finishing;

            elsif Container.This_Subsystem.Segment_Rewind_Requested then
               -- go back to beginning of current segment
               -- using fseek and the offset from the beginning

               Container.This_Subsystem.The_Header.The_Snapshot_Count :=
                 Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                 ( Container.This_Subsystem.The_Header.The_Segment_Count );
               Rewind_To ( Container.This_Subsystem.The_Temporary_File,
                           Container.This_Subsystem.The_Current_Segment_Start_Index );
               Container.This_Subsystem.The_State
                 := Jpats_Formation_Types.Paused;

            elsif Container.This_Subsystem.Segment_End_Requested then
               -- set segment number and start of next segment

               Container.This_Subsystem.The_Header.The_Segment_Count :=
                 Container.This_Subsystem.The_Header.The_Segment_Count + 1;

--                Ada.Text_Io.Put_Line("Segment_End_Requested the_segment_count = " &
--                                     Jpats_Formation_Types.Segment_Type'Image
--                                     (Container.This_Subsystem.The_Header.The_Segment_Count));


--                Ada.Text_Io.Put_Line("and the_snapshot_count = " &
--                                     Natural'Image
--                                     (Container.This_Subsystem.The_Header.The_Snapshot_Count));



               if Container.This_Subsystem.The_Header.The_Segment_Count
                 > Jpats_Formation_Types.Max_Segments then

                  -- if maxed out on segments, finish profile next pass
                  Container.This_Subsystem.The_State
                    := Jpats_Formation_Types.Finishing;

               else

                  -- get offset for new segment
                  Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                    ( Container.This_Subsystem.The_Header.The_Segment_Count )
                    := Container.This_Subsystem.The_Header.The_Snapshot_Count;

               end if;

               -- bookmark the start of the segment, in case we want
               -- to rewind
               Container.This_Subsystem.The_Current_Segment_Start_Index
                 := Ada.Streams.Stream_Io.Index
                    ( Container.This_Subsystem.The_Temporary_File  );

               -- ????? Do we want the system always to pause when a
               -- segment end is selected ?????

               Container.This_Subsystem.The_State
                 := Jpats_Formation_Types.Paused;

            end if;

         when Jpats_Formation_Types.Paused =>
            -- wait for flag to resume

            -- look for signal to start recording
            if Container.This_Subsystem.Recording_Start_Requested then
               Container.This_Subsystem.The_State :=
                 Jpats_Formation_Types.Recording;
            end if;

            --***************************************************************
            if Container.This_Subsystem.Profile_Done_Requested then
               Container.This_Subsystem.The_State
                 := Jpats_Formation_Types.Finishing;
            end if;
            --***************************************************************

         when Jpats_Formation_Types.Finishing =>
            -- single-pass state

            -- rewind to the start of the file and write header to the profile
            Ada.Streams.Stream_Io.Reset
              ( Container.This_Subsystem.The_Temporary_File );

            --***************************************************************
            --***************************************************************
--             Ada.Text_Io.Put("The_Segment_Count = " &
--                             Jpats_Formation_Types.Segment_Type'Image
--                             (Container.This_Subsystem.The_Header.The_Segment_Count));
--             Ada.Text_Io.Put_Line("The_Snapshot_Count = " &
--                                  Natural'Image
--                                  (Container.This_Subsystem.The_Header.The_Snapshot_Count));
--             for I in 1..50
--             loop
--                Ada.Text_Io.Put(
--                                Natural'Image
--        (Container.This_Subsystem.The_Header.The_Segment_Offset_Array(Jpats_Formation_Types.Valid_Segment_Type(I))));
--             end loop;
--             Ada.Text_Io.New_Line;
            --***************************************************************
            --***************************************************************


            Jpats_Formation_Types.Profile_Header'Write
              ( Container.This_Subsystem.The_File_Stream,
                Container.This_Subsystem.The_Header );

            -- close out current profile by closing the file and
            -- copying it from RAM disk to the hard drive
            Close_File ( Container.This_Subsystem.The_Temporary_File );
            Copy_File ( Standard_Ram_File_Name,
                        Standard_Temp_File_Name );

            -- reset counters
            Container.This_Subsystem.The_Header.The_Snapshot_Count := 0;
            Container.This_Subsystem.The_Header.The_Segment_Count := 0;
            Container.This_Subsystem.The_Header.The_Segment_Offset_Array
              := ( others => 0 );

            Container.This_Subsystem.The_State
              := Jpats_Formation_Types.Idle;

      end case;

      -- clear all signal flags for next pass
      Container.This_Subsystem.Recording_Start_Requested
        := False;
      Container.This_Subsystem.Segment_End_Requested
        := False;
      Container.This_Subsystem.Segment_Rewind_Requested
        := False;
      Container.This_Subsystem.Profile_Done_Requested
        := False;

   end Update;

end Jpats_Formation_Profile_Recorder.Controller;

