-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Formation_Profile_Playback.Container;
with Jpats_Formation_Profile_Playback.Controller;
with Log;

package body Jpats_Formation_Profile_Playback is

   -- This package implements a simple state machine with five
   -- states -- Idle, Initializing, Playing, Paused, Finishing and
   -- five possible events, all of which are requests to change state.
   -- The events are requests to start playing, rewind to
   -- the beginning of the current segment, go directly to a specific segment,
   -- pause playback, and stop and reset playback.

   -- The playback file is copied from a known location on the hard
   -- drive to a file on a RAM disk, for speed of access during
   -- playback.

   -- This package only controls the reading and playback of a
   -- profile.  Another software subsystem will handle coordination of
   -- this playback system with the rest of simulation -- for example,
   -- dealing with a flight freeze during playback.

   function State
     return Jpats_Formation_Types.Formation_Profile_Playback_State is
   begin
      return Container.This_Subsystem.The_State;
   end State;

   procedure Start_Playback is
      -- instruct the player to start a new profile, or to resume
      -- recording after a pause
   begin
      Container.This_Subsystem.Playback_Start_Requested := True;
   end Start_Playback;

   procedure Rewind_To_Start_Of_Current_Segment is
      -- instruct the player to go back to the beginning of the current
      -- segment
   begin
      Container.This_Subsystem.Segment_Rewind_Requested := True;
   end Rewind_To_Start_Of_Current_Segment;

   procedure Go_To_Segment ( A_Segment : in Natural ) is
      -- instruct the recorder to go to the beginning of the specified
      -- segment.
   begin
      Container.This_Subsystem.Segment_Goto_Requested := True;
      Container.This_Subsystem.The_Selected_Segment :=
        Jpats_Formation_Types.Segment_Type ( A_Segment );
   end Go_To_Segment;

   procedure Pause_Playback is
      -- instruct the recorder to pause playback at current snapshot
   begin
      Container.This_Subsystem.Playback_Pause_Requested := True;
   end Pause_Playback;

   procedure Stop_Playback is
      -- instruct the recorder to stop playback and reset
   begin
      Container.This_Subsystem.Profile_Done_Requested := True;
   end Stop_Playback;

   function Pitch return Angle_Types.Radians is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Pitch;
   end Pitch;

   function Roll return Angle_Types.Radians is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Roll;
   end Roll;

   function Heading return Angle_Types.Radians is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Heading;
   end Heading;

   function Latitude return Lat_Long_Types.Latitude_Degrees is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Latitude;
   end Latitude;

   function Longitude return Lat_Long_Types.Longitude_Degrees is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Longitude;
   end Longitude;

   function Altitude return Length_Types.Feet is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Altitude;
   end Altitude;

   function Mean_Flap_Position
     return Jpats_Secondary_Flight_Controls_Types.Flap_Position_Type is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Mean_Flap_Position;
   end Mean_Flap_Position;

   function Speedbrake_Position
     return Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Speedbrake_Position;
   end Speedbrake_Position;

   function Elevator_Position
     return Angle_Types.Degrees is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Elevator_Position;
   end Elevator_Position;

   function Rudder_Position
     return Angle_Types.Degrees is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Rudder_Position;
   end Rudder_Position;

   function Right_Aileron_Position
     return Angle_Types.Degrees is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Right_Aileron_Position;
   end Right_Aileron_Position;

   function Left_Aileron_Position
     return Angle_Types.Degrees is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Left_Aileron_Position;
   end Left_Aileron_Position;

   function Mean_Landing_Gear_Position
     return Jpats_Landing_Gear_Types.Gear_Position_Type is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Mean_Landing_Gear_Position;
   end Mean_Landing_Gear_Position;

   function Indicated_Airspeed
     return Length_Types.Knots
   is
   begin
      return Container.This_Subsystem.The_Current_Snapshot.The_Indicated_Airspeed * 0.5924838;
   end Indicated_Airspeed;

   function Current_Track_Number
     return Integer
   is
   begin
      return Controller.Current_Track;
   end Current_Track_Number;

   function Playback_Is_Active
     return Boolean
   is
   begin
      return Container.This_Subsystem.Playback_Is_Active;
   exception
      when others =>
         Log.Report("Jpats_Formation_Profile_Playback.Playback_Is_Active()");
         raise;
   end Playback_Is_Active;

   function Current_Playback_File_Index
     return Ada.Streams.Stream_IO.Positive_Count
   is
   begin
      return Container.This_Subsystem.Current_Playback_File_Index;
   exception
      when others =>
         Log.Report("Jpats_Formation_Profile_Playback.Current_Playback_File_Index()");
         raise;
   end Current_Playback_File_Index;

   function The_Temporary_File
     return File_Access
   is
   begin
      return Container.This_Subsystem.The_Temporary_File'Access;
   exception
      when others =>
         Log.Report("Jpats_Formation_Profile_Playback.The_Temporary_File()");
         raise;
   end The_Temporary_File;

end Jpats_Formation_Profile_Playback;
