-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Jpats_Formation_Profile_Playback.Container;
with Ada.Streams.Stream_Io;
with Ada.Characters.Latin_1;
with System;
with Interfaces.C;
with Jpats_Formation_Types;
with JPATS_Simulated_Aircraft;
with Log;

package body Jpats_Formation_Profile_Playback.Controller is

   use type Interfaces.C.Int;
   use type Jpats_Formation_Types.Segment_Type;
   package Sim_Ac renames Jpats_Simulated_Aircraft;
   Freeze_Lp : Boolean renames Container.This_Subsystem.FPPC_Freeze_Lp;

   FILE_DNE_ERROR : exception;

   function Copy ( From_Name : System.Address;
                   To_Name : System.Address ) return Interfaces.C.Int;
   pragma Import ( C, Copy, "copy" );

   procedure Copy_File ( From_Name : in String;
                         To_Name   : in String ) is
      From_Name_C : Interfaces.C.Char_Array ( 0 .. From_Name'Length );
      To_Name_C   : Interfaces.C.Char_Array ( 0 .. To_Name'Length );
      Result : Interfaces.C.Int;
   begin
      From_Name_C := Interfaces.C.To_C ( From_Name );
      To_Name_C   := Interfaces.C.To_C ( To_Name );
      Result := Copy ( From_Name_C(0)'Address, To_Name_C(0)'Address );
      if Result < 0 then
         -- error handling
         Log.Report("copy_file in Jpats_Formation_Profile_Playback.Controller FILE_DNE_ERROR exception.");
         raise FILE_DNE_ERROR;
         --         null;
      end if;
   end Copy_File;

   procedure Close_File ( File : in out Ada.Streams.Stream_Io.File_Type ) is
   begin
      Ada.Streams.Stream_Io.Close ( File => File );

   exception
      when others =>
         Log.report("Jpats_Formation_Profile_Playback.Controller.Close_File() can't close file");
         raise;
   end Close_File;


   procedure Open_File
     ( File : in out Ada.Streams.Stream_Io.File_Type;
       Name : in String ) is
   begin
      Ada.Streams.Stream_Io.Create
        ( File => File,
          Mode => Ada.Streams.Stream_Io.in_File,
          Name => Name );
   exception
      when others =>
         Log.report ("Jpats_Formation_Profile_Playback.Controller.Open_File() can't open file");
         raise;
   end Open_File;

   procedure Rewind_To ( File : in out Ada.Streams.Stream_Io.File_Type;
                         Index : in Ada.Streams.Stream_Io.Positive_Count  )  is
   begin
      Ada.Streams.Stream_Io.Set_Index
        ( File => File,
          To => Index );
   exception
      when others =>
         Log.report("Jpats_Formation_Profile_Playback.Controller.Rewind_To() can't set file index");
         raise;
   end Rewind_To;

   procedure Rewind_To ( File : in out Ada.Streams.Stream_Io.File_Type;
                         Snapshot : in Positive )  is

      Index : Positive;

   begin

      -- calculate byte position in file.  1 is the index of the first
      -- byte in the file, and 1 is the index of the first snapshot

      Index :=
        Jpats_Formation_Types.Profile_Header_Size
        + ( Jpats_Formation_Types.Profile_Snapshot_Size
            * (Snapshot - 1))
        + 1;


--       Log.Report("index :=" & Positive'Image(Index));
--       Log.Report("header size " & Natural'Image(Jpats_Formation_Types.Profile_Header_Size));
--       Log.Report("snapshot size  = " & natural'Image(Jpats_Formation_Types.Profile_Snapshot_Size));
--       Log.Report("snapshot = " & Positive'Image(Snapshot));


      Ada.Streams.Stream_Io.Set_Index
        ( File => File,
          To => Ada.Streams.Stream_Io.Positive_Count ( Index ) );

   exception
      when others =>
         Log.report("Jpats_Formation_Profile_Playback.Controller.Rewind_To() can't set file index");
         raise;
   end Rewind_To;

   procedure Get_Snapshot
     ( Stream : in out Ada.Streams.Stream_Io.Stream_Access  ) is
   begin

      Jpats_Formation_Types.Prerecorded_Profile_Snapshot'Read
        ( Stream,
          Container.This_Subsystem.The_Current_Snapshot );

   exception
      when others =>
         Log.Report("GET_SNAPSHOT CRASHED *** and the gear value is " &
                              Jpats_Landing_Gear_Types.Gear_Position_Type'Image
                              (Container.This_Subsystem.The_Current_Snapshot.The_Mean_Landing_Gear_Position) );
         raise;
   end Get_Snapshot;

   procedure Initialize is
   begin

      -- initialize container

      Container.This_Subsystem.The_State := Jpats_Formation_Types.Idle;
      Container.This_Subsystem.Playback_Start_Requested := False;
      Container.This_Subsystem.Playback_Pause_Requested := False;
      Container.This_Subsystem.Segment_Goto_Requested := False;
      Container.This_Subsystem.The_Selected_Segment := 0;
      Container.This_Subsystem.Segment_Rewind_Requested := False;
      Container.This_Subsystem.Profile_Done_Requested := False;
      Container.This_Subsystem.The_Header.The_Segment_Count := 0;
      Container.This_Subsystem.The_Header.The_Snapshot_Count := 0;
      Container.This_Subsystem.The_Header.The_Segment_Offset_Array := ( others => 0 );

      -- The following should probably be done in the VxWorks startup script
      --
      -- create RAM disk
      --
      -- BLK_DEV *ramDevCreate
      --           (char *ramAddr, int bytesPerBlk, int blksPerTrack,
      --            int nBlocks, int blkOffset)
      --
      --
      -- Test for null result
      --
      -- Pass return value as parameter to dosFsMkfs
      --
      -- DOS_VOL_DESC *dosFsMkfs
      --   (
      --   char     *volName,  /* volume name to use             */
      --   BLK_DEV  *pBlkDev   /* pointer to block device struct */
      --   }
      --
      -- Test for null result
      --

   end Initialize;

   -- This package implements a simple state machine with five
   -- states -- Idle, Initializing, Playback, Paused, Finishing and
   -- five possible events, all of which are requests to change state.
   -- The events are requests to start playback, rewind to
   -- the beginning of the current segment, go directly to a specific segment,
   -- pause playback, and stop and reset playback.

   -- The playback file is copied from a known location on the hard
   -- drive to a file on a RAM disk, for speed of access during
   -- playback.

   -- This package only controls the reading and playback of a
   -- profile.  Another software subsystem will handle coordination of
   -- this playback system with the rest of simulation -- for example,
   -- dealing with a flight freeze during playback.

   State_Lp :
     Jpats_Formation_Types.Formation_Profile_Playback_State
     renames Container.This_Subsystem.FPPC_State_Lp;

   procedure Update
     (Iconst : in Float) is

      Standard_Ram_File_Name : constant String := "/ramdisk/playback.tmp";
      Standard_Temp_File_Name : constant String := "/ramdisk/playback.dat";

      use type Jpats_Formation_Types.Formation_Profile_Playback_State;
      use type Jpats_Formation_Types.Segment_Type;
      use type Jpats_Formation_Types.Valid_Segment_Type;

      Freeze :constant Boolean := Sim_Ac.Get_Flight_Freeze;
      Freeze_Toggled_On : constant Boolean := Freeze and not Freeze_Lp;
      Freeze_Toggled_Off : constant Boolean := not Freeze and Freeze_Lp;

   begin

      State_Lp := Container.This_Subsystem.The_State;

      case Container.This_Subsystem.The_State is

         when Jpats_Formation_Types.Idle =>


            -- look for signal to start playback
            if Container.This_Subsystem.Playback_Start_Requested then

               Container.This_Subsystem.The_State :=
                 Jpats_Formation_Types.Initializing;

            end if;

         when Jpats_Formation_Types.Initializing =>
            -- single-pass state

            begin

               -- copy the file from the temp location to the RAM disk
               -- copy_file() raise FILE_DNE_ERROR if can't open file to copy
               Copy_File ( Standard_Temp_File_Name,
                           Standard_Ram_File_Name );

               -- open a file on the RAM disk
               Open_File ( Container.This_Subsystem.The_Temporary_File,
                           Standard_Ram_File_Name );

               -- get a stream pointer for use in T'Read and T'Write
               Container.This_Subsystem.The_File_Stream :=
                 Ada.Streams.Stream_Io.Stream
                 ( Container.This_Subsystem.The_Temporary_File );

               -- Read the header from the beginning of the file.

               Jpats_Formation_Types.Profile_Header'Read
                 ( Container.This_Subsystem.The_File_Stream,
                   Container.This_Subsystem.The_Header );


               --  Since playback doesn't automatically freeze upon init any longer
               --  this playback machine should start playing immediately unless the
               --  simulator is in freeze.
               if Freeze then
                  Container.This_Subsystem.The_State :=
                    Jpats_Formation_Types.Paused;
               else
                  Container.This_Subsystem.The_State :=
                    Jpats_Formation_Types.Playback;
               end if;

               Container.This_Subsystem.The_Segment_Index := 1;
               Container.This_Subsystem.The_Snapshot_Index := 0;

               -- bookmark the start of the segment, in case we want
               -- to rewind
               Container.This_Subsystem.The_Current_Segment_Start_Index
                 := Ada.Streams.Stream_Io.Index
                 ( Container.This_Subsystem.The_Temporary_File  );


               begin -- retrieve snapshot
                  Get_Snapshot ( Container.This_Subsystem.The_File_Stream );
               exception
                  when others =>
                     Log.Report("get_snapshot() call in initialize state exception.");
                     raise;
               end;
               -- increment snapshot counter
               Container.This_Subsystem.The_Snapshot_Index :=
                 Container.This_Subsystem.The_Snapshot_Index + 1;


            exception
               when FILE_DNE_ERROR =>
                  Container.This_Subsystem.The_State := Jpats_Formation_Types.Idle;
                  ---                  raise FILE_DNE_ERROR;
            end;

         when Jpats_Formation_Types.Playback =>



            if Freeze_Toggled_On then

               Container.This_Subsystem.The_State := Jpats_Formation_Types.Paused;

            elsif Container.This_Subsystem.The_Snapshot_Index + 1 >=
              Container.This_Subsystem.The_Header.The_Snapshot_Count then

               -- are we done?
               -- just pause at the end in case the instructor wants
               -- to repeat a segment before closing this profile
               Container.This_Subsystem.The_State
                 := Jpats_Formation_Types.Paused;

            else
               -- still going

               begin


                  if Container.This_Subsystem.The_Segment_Index < Container.This_Subsystem.The_Header.The_Segment_Count then

                     if Container.This_Subsystem.The_Snapshot_Index >
                       Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                       ( Jpats_Formation_Types.Valid_Segment_Type(Container.This_Subsystem.The_Segment_Index + 1) ) then
                        -- are we in a new segment?
                        begin

                           Container.This_Subsystem.The_Segment_Index :=
                             Container.This_Subsystem.The_Segment_Index + 1;

                           -- bookmark the start of the segment, in case we want
                           -- to rewind
                           Container.This_Subsystem.The_Current_Segment_Start_Index
                             := Ada.Streams.Stream_Io.Index
                             ( Container.This_Subsystem.The_Temporary_File  );
                        exception
                           when others =>
                              Log.Report(" -- a2 --");
                              raise;
                        end;

                     end if;

                  end if;

               exception
                  when others =>
                     raise;
               end;

               begin -- retrieve snapshot
                  Get_Snapshot ( Container.This_Subsystem.The_File_Stream );
               exception
                  when others =>
                     Log.Report("get_snapshot() call in playback state exception.");
                     raise;
               end;

               -- increment snapshot counter
               Container.This_Subsystem.The_Snapshot_Index :=
                 Container.This_Subsystem.The_Snapshot_Index + 1;

               if Container.This_Subsystem.Playback_Pause_Requested then

                  Container.This_Subsystem.The_State := Jpats_Formation_Types.Paused;

               elsif Container.This_Subsystem.Profile_Done_Requested then

                  Container.This_Subsystem.The_State
                    := Jpats_Formation_Types.Finishing;

               elsif Container.This_Subsystem.Segment_Rewind_Requested then
                  -- go back to beginning of current segment

                  Rewind_To ( Container.This_Subsystem.The_Temporary_File,
                              Container.This_Subsystem.The_Current_Segment_Start_Index );
                  Container.This_Subsystem.The_State
                    := Jpats_Formation_Types.Paused;

               elsif Container.This_Subsystem.Segment_Goto_Requested then
                  -- set segment number and go to first snapshot of that segment

                  if Container.This_Subsystem.The_Selected_Segment
                    <= Container.This_Subsystem.The_Header.The_Segment_Count then

                     Rewind_To
                       ( Container.This_Subsystem.The_Temporary_File,
                         Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                         ( Container.This_Subsystem.The_Selected_Segment ) );

                  end if;

                  -- bookmark the start of the segment, in case we want
                  -- to rewind
                  Container.This_Subsystem.The_Current_Segment_Start_Index
                    := Ada.Streams.Stream_Io.Index
                    ( Container.This_Subsystem.The_Temporary_File  );


                  Container.This_Subsystem.The_Snapshot_Index := Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                    (Container.This_Subsystem.The_Selected_Segment);
                  Container.This_Subsystem.The_Segment_Index := Container.This_Subsystem.The_Selected_Segment;

                  begin -- retrieve snapshot
                     Get_Snapshot ( Container.This_Subsystem.The_File_Stream );
                  exception
                     when others =>
                        Log.Report("get_snapshot() call in playback & goto state exception.");
                        raise;
                  end;

                  -- increment snapshot counter
                  Container.This_Subsystem.The_Snapshot_Index :=
                    Container.This_Subsystem.The_Snapshot_Index + 1;

                  -- pause until reactivated
                  if Freeze then
                     Container.This_Subsystem.The_State
                       := Jpats_Formation_Types.Paused;
                  else
                     Container.This_Subsystem.The_State
                       := Jpats_Formation_Types.Playback;
                  end if;

               end if;

            end if;

         when Jpats_Formation_Types.Paused =>
            -- wait for flag to resume

            -- look for signal to start playback
            if Freeze_Toggled_Off then

               if not (Container.This_Subsystem.The_Snapshot_Index + 1 >=
                       Container.This_Subsystem.The_Header.The_Snapshot_Count) then

                  begin -- retrieve snapshot
                     Get_Snapshot ( Container.This_Subsystem.The_File_Stream );
                  exception
                     when others =>
                        Log.Report("get_snapshot() call in paused and toggled off state exception.");
                        raise;
                  end;

                  -- increment snapshot counter
                  Container.This_Subsystem.The_Snapshot_Index :=
                    Container.This_Subsystem.The_Snapshot_Index + 1;
               end if;

               Container.This_Subsystem.The_State :=
                 Jpats_Formation_Types.Playback;

            end if;

            -- look for signal to start playback
            if Container.This_Subsystem.Playback_Start_Requested and not Freeze_Toggled_Off then

               if not (Container.This_Subsystem.The_Snapshot_Index + 1 >=
                       Container.This_Subsystem.The_Header.The_Snapshot_Count) then

                  begin -- retrieve snapshot
                     Get_Snapshot ( Container.This_Subsystem.The_File_Stream );
                  exception
                     when others =>
                        Log.Report("get_snapshot() call in paused & start request state exception.");
                        raise;
                  end;

                  -- increment snapshot counter
                  Container.This_Subsystem.The_Snapshot_Index :=
                    Container.This_Subsystem.The_Snapshot_Index + 1;

               end if;

               Container.This_Subsystem.The_State :=
                 Jpats_Formation_Types.Playback;

            end if;

            if Container.This_Subsystem.Profile_Done_Requested then
               Container.This_Subsystem.The_State :=
                 Jpats_Formation_Types.Finishing;
            end if;


            if Container.This_Subsystem.Segment_Goto_Requested then

                  if Container.This_Subsystem.The_Selected_Segment
                    <= Container.This_Subsystem.The_Header.The_Segment_Count then

                     Rewind_To
                       ( Container.This_Subsystem.The_Temporary_File,
                         Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                         ( Container.This_Subsystem.The_Selected_Segment ) );

                  end if;

                  -- bookmark the start of the segment, in case we want
                  -- to rewind
                  Container.This_Subsystem.The_Current_Segment_Start_Index
                    := Ada.Streams.Stream_Io.Index
                    ( Container.This_Subsystem.The_Temporary_File  );

                  Container.This_Subsystem.The_Snapshot_Index := Container.This_Subsystem.The_Header.The_Segment_Offset_Array
                    (Container.This_Subsystem.The_Selected_Segment);
                  Container.This_Subsystem.The_Segment_Index := Container.This_Subsystem.The_Selected_Segment;

                  begin -- retrieve snapshot
                     Get_Snapshot ( Container.This_Subsystem.The_File_Stream );
                  exception
                     when others =>
                        Log.Report("get_snapshot() call in paused & goto req exception.");
                        raise;
                  end;

                  -- increment snapshot counter
                  Container.This_Subsystem.The_Snapshot_Index :=
                    Container.This_Subsystem.The_Snapshot_Index + 1;


                  -- pause until reactivated
                  if Freeze then
                     Container.This_Subsystem.The_State
                       := Jpats_Formation_Types.Paused;
                  else
                     Container.This_Subsystem.The_State
                       := Jpats_Formation_Types.Playback;
                  end if;

            end if;


         when Jpats_Formation_Types.Finishing =>
            -- single-pass state

            -- close out current profile by closing the file and
            Close_File ( Container.This_Subsystem.The_Temporary_File );

            Container.This_Subsystem.The_State
              := Jpats_Formation_Types.Idle;

      end case;


      Container.This_Subsystem.Playback_Is_Active
        := not (Container.This_Subsystem.The_State = Jpats_Formation_Types.Idle);


      if Ada.Streams.Stream_Io.Is_Open(Container.This_Subsystem.The_Temporary_File) then
         Container.This_Subsystem.Current_Playback_File_Index
           := Ada.Streams.Stream_Io.Index(Container.This_Subsystem.The_Temporary_File);
      end if;


      -- clear all signal flags for next pass
      Container.This_Subsystem.Playback_Start_Requested
        := False;
      Container.This_Subsystem.Playback_Pause_Requested
        := False;
      Container.This_Subsystem.Segment_Goto_Requested
        := False;
      Container.This_Subsystem.The_Selected_Segment := 0;
      Container.This_Subsystem.Segment_Rewind_Requested
        := False;
      Container.This_Subsystem.Profile_Done_Requested
        := False;

      Freeze_Lp := Freeze;

   exception
      when others =>
         Log.Report ("Jpats_Formation_Profile_Playback.Controller.Update()");
         raise;
   end Update;


   function Current_Track
     return Integer
   is
   begin
      return Integer(Container.This_Subsystem.The_Segment_Index);
   exception
      when others =>
         Log.Report ("Jpats_Formation_Profile_Playback.Controller.Current_Track()");
         raise;
   end Current_Track;

end Jpats_Formation_Profile_Playback.Controller;




