-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Text_Io;
with Log;
with Jpats_Formation_Manager.Container;
with Jpats_Formation_Manager.Ios_Interface;
with Jpats_Formation_Types;
with Formation_Lead_Kinematic;
with Jpats_Simulated_Aircraft;
with Jpats_Atmosphere;
with Jpats_Landing_Gear;
with Jpats_Secondary_Flight_Controls;
with Jpats_Formation_Profile_Playback;
with Jpats_Formation_Profile_Recorder;
with Ada.Numerics; use Ada.Numerics;
with Ada.Numerics.Elementary_Functions; --use Ada.Numerics.Elementary_Functions;
with Ada.Numerics.Generic_Elementary_Functions;
with Jpats_Ios_Interface;
with Ada.Strings.Bounded;
with Ada.Characters.Latin_1;
with Ada.Streams.Stream_Io;
with Save_Restore.IOS_Interface;
with Save_Restore;
with Simulation_Dictionary;

package body Jpats_Formation_Manager.Controller is

   package Ctnr renames Container;
   package Playback renames Jpats_Formation_Profile_Playback;
   package Recorder renames Jpats_Formation_Profile_Recorder;
   package Sim_Ac renames Jpats_Simulated_Aircraft;
   package Gear renames Jpats_Landing_Gear;
   package Sec_Flt renames Jpats_Secondary_Flight_Controls;
   package Lead_Kinematic renames Formation_Lead_Kinematic;
   package JII renames Jpats_Ios_Interface;
   package Ios_If renames Jpats_Formation_Manager.Ios_Interface;
   package Sr_Ios_If renames Save_Restore.IOS_Interface;
   package JF_T renames Jpats_Formation_Types;
   package Bnd_16 renames JF_T.Bounded_16;
   Ios : Ctnr.Ios_Interface_Instance renames Ctnr.This_Ios_Interface;
   The_Lead : Ctnr.Formation_Lead_Instance renames Ctnr.The_Lead;
   use type JF_T.Formation_Profile_Playback_State;

   EOL : constant String := Ada.Characters.Latin_1.LF & Ada.Characters.Latin_1.CR;
   FREEZE_DURATION : constant Integer := 3; -- Unfreeze duration in computing cycles

   Freeze_Count : Integer := 0; -- Computing cycles since formation demo unfreeze initiated
   Formation_Demo_Initialized : Boolean := True;

   package Lf is new Ada.Numerics.Generic_Elementary_Functions (Long_Float);
   use Lf;

   procedure Reset_JPATS_Lead_Rand_Gen
   is
   begin
      Lead_Kinematic.Reset_Rrd_Gen_From_State(The_Lead.Kinematic_Driver);
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Reset_JPATS_Lead_Rand_Gen()");
         raise;
   end Reset_JPATS_Lead_Rand_Gen;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
   is
   begin
      --************** playback vars ******************

      JII.Register( Name               => "Formation_Track_1_Name",
                    Var_Array          => Ios.Formation_Track_1_Name'Address,
                    Set_Routine        => Ios_If.Set_Formation_Track_1_Name'Access,
                    Dimension_1_Length => 16
                    );
      JII.Register( Name               => "Formation_Track_2_Name",
                    Var_Array          => Ios.Formation_Track_2_Name'Address,
                    Set_Routine        => Ios_If.Set_Formation_Track_2_Name'Access,
                    Dimension_1_Length => 16
                    );
      JII.Register( Name               => "Formation_Track_3_Name",
                    Var_Array          => Ios.Formation_Track_3_Name'Address,
                    Set_Routine        => Ios_If.Set_Formation_Track_3_Name'Access,
                    Dimension_1_Length => 16
                    );
      JII.Register( Name               => "Formation_Track_4_Name",
                    Var_Array          => Ios.Formation_Track_4_Name'Address,
                    Set_Routine        => Ios_If.Set_Formation_Track_4_Name'Access,
                    Dimension_1_Length => 16
                    );
      JII.Register( Name               => "Formation_Track_5_Name",
                    Var_Array          => Ios.Formation_Track_5_Name'Address,
                    Set_Routine        => Ios_If.Set_Formation_Track_5_Name'Access,
                    Dimension_1_Length => 16
                    );
      JII.Register( Name               => "Formation_Track_6_Name",
                    Var_Array          => Ios.Formation_Track_6_Name'Address,
                    Set_Routine        => Ios_If.Set_Formation_Track_6_Name'Access,
                    Dimension_1_Length => 16
                    );
      JII.Register ( Name               => "Formation_Track_7_Name",
                     Var_Array          => Ios.Formation_Track_7_Name'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_7_Name'Access,
                     Dimension_1_Length => 16
                     );

      JII.Register ( Name               => "Formation_Track_8_Name",
                     Var_Array          => Ios.Formation_Track_8_Name'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_8_Name'Access,
                     Dimension_1_Length => 16
                     );
      JII.Register ( Name               => "Formation_Track_9_Name",
                     Var_Array          => Ios.Formation_Track_9_Name'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_9_Name'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_10_Name",
                     Var_Array          => Ios.Formation_Track_10_Name'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_10_Name'Access,
                     Dimension_1_Length => 16);


      -- **************************** record vars *******************************

      JII.Register ( Name               => "Formation_Track_1_Name_R",
                     Var_Array          => Ios.Formation_Track_1_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_1_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_2_Name_R",
                     Var_Array          => Ios.Formation_Track_2_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_2_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_3_Name_R",
                     Var_Array          => Ios.Formation_Track_3_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_3_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_4_Name_R",
                     Var_Array          => Ios.Formation_Track_4_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_4_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_5_Name_R",
                     Var_Array          => Ios.Formation_Track_5_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_5_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_6_Name_R",
                     Var_Array          => Ios.Formation_Track_6_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_6_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_7_Name_R",
                     Var_Array          => Ios.Formation_Track_7_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_7_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_8_Name_R",
                     Var_Array          => Ios.Formation_Track_8_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_8_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_9_Name_R",
                     Var_Array          => Ios.Formation_Track_9_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_9_Name_R'Access,
                     Dimension_1_Length => 16);
      JII.Register ( Name               => "Formation_Track_10_Name_R",
                     Var_Array          => Ios.Formation_Track_10_Name_R'Address,
                     Set_Routine        => Ios_If.Set_Formation_Track_10_Name_R'Access,
                     Dimension_1_Length => 16);

      Lead_Kinematic.Reset_RRD_Gen;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Initialize()");
         raise;
   end Initialize;


   --------------------------------
   -- Update_Lead_From_Kinematic --
   --------------------------------

   procedure Update_Lead_From_Kinematic
   is
   begin
      The_Lead.Pitch_Angle            := Lead_Kinematic.Pitch_Angle(The_Lead.Kinematic_Driver);
      The_Lead.Bank_Angle             := Lead_Kinematic.Bank_Angle(The_Lead.Kinematic_Driver);
      The_Lead.Heading                := Lead_Kinematic.Heading(The_Lead.Kinematic_Driver);
      The_Lead.Latitude               := Lead_Kinematic.Lat(The_Lead.Kinematic_Driver);
      The_Lead.Longitude              := Lead_Kinematic.Long(The_Lead.Kinematic_Driver);
      The_Lead.Geometric_Altitude     := Lead_Kinematic.Geometric_Altitude(The_Lead.Kinematic_Driver);
      The_Lead.Gear_Position          := Lead_Kinematic.Gear_Position(The_Lead.Kinematic_Driver);
      The_Lead.Speedbrake_Position    := Lead_Kinematic.Speedbrake_Position(The_Lead.Kinematic_Driver);
      The_Lead.Flap_Position          := Lead_Kinematic.Flap_Position(The_Lead.Kinematic_Driver);
      The_Lead.Elevator_Position      := Lead_Kinematic.Elevator_Position(The_Lead.Kinematic_Driver);
      The_Lead.Rudder_Position        := Lead_Kinematic.Rudder_Position(The_Lead.Kinematic_Driver);
      The_Lead.Right_Aileron_Position := Lead_Kinematic.Right_Aileron_Position(The_Lead.Kinematic_Driver);
      The_Lead.Left_Aileron_Position  := Lead_Kinematic.Left_Aileron_Position(The_Lead.Kinematic_Driver);
      The_Lead.V_True_Airspeed        := Lead_Kinematic.V_True_Airspeed(The_Lead.Kinematic_Driver);
      The_Lead.V_Indicated_Airspeed   := Lead_Kinematic.V_Indicated_Airspeed(The_Lead.Kinematic_Driver);
   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Lead_From_Kinematic()");
         raise;
   end Update_Lead_From_Kinematic;

   ------------------------------
   -- Update_Lead_From_Profile --
   ------------------------------

   procedure Update_Lead_From_Profile
   is

      P_A  : constant Float := Float(Playback.Pitch)*(180.0/Pi);
      B_A  : constant Float := Float(Playback.Roll*180.0/Pi);
      H    : constant Float := Float(Playback.Heading*180.0/Pi);
      Lat  : constant Long_Float := Long_Float(Playback.Latitude);
      Lon  : constant Long_Float := Long_Float(Playback.Longitude);
      G_A  : constant Float := Float(Playback.Altitude);
      VIAS : constant Float := Float(Playback.Indicated_Airspeed);
      G_P  : constant Float := Float(Playback.Mean_Landing_Gear_Position);
      S_P  : constant Float := Float(Playback.Speedbrake_Position);
      F_P  : constant Float := Float(Playback.Mean_Flap_Position);
      E_P  : constant Float := Float(Playback.Elevator_Position);
      R_P  : constant Float := Float(Playback.Rudder_Position);
      RA_P : constant Float := Float(Playback.Right_Aileron_Position);
      LA_P : constant Float := Float(Playback.Left_Aileron_Position);

      Knots_To_Ft_Per_Sec : constant Float := 1.6878098571;
      Ft_Per_Sec_To_knots : constant Float := 0.59248410217;

   begin

      Lead_Kinematic.Initialize
        ( An_Instance                 => The_Lead.Kinematic_Driver,
          Pitch_Angle_Init            => P_A,
          Bank_Angle_Init             => B_A,
          Heading_Init                => H,
          Lat_Init                    => Lat,
          Long_Init                   => Lon,
          Geometric_Altitude_Init     => G_A,
          V_Indicated_Airspeed_Init   => VIAS,
          Gear_Position_Init          => G_P,
          Speedbrake_Position_Init    => S_P,
          Flap_Position_Init          => F_P,
          Elevator_Position_Init      => E_P,
          Rudder_Position_Init        => R_P,
          Right_Aileron_Position_Init => RA_P,
          Left_Aileron_Position_Init  => LA_P
          );

      The_Lead.Pitch_Angle            := P_A;
      The_Lead.Bank_Angle             := B_A;
      The_Lead.Heading                := H;
      if Playback.State /= JF_T.Playback and Playback.State /= JF_T.Paused then
         The_Lead.Latitude            := 0.0;
         The_Lead.Longitude           := 0.0;
      else
         The_Lead.Latitude            := Lat;
         The_Lead.Longitude           := Lon;
      end if;
      The_Lead.Geometric_Altitude     := G_A;
      The_Lead.Gear_Position          := G_P;
      The_Lead.Speedbrake_Position    := S_P;
      The_Lead.Flap_Position          := F_P;
      The_Lead.Elevator_Position      := E_P;
      The_Lead.Rudder_Position        := R_P;
      The_Lead.Right_Aileron_Position := RA_P;
      The_Lead.Left_Aileron_Position  := LA_P;
      The_Lead.V_Indicated_Airspeed   := VIAS*Knots_To_Ft_Per_Sec;
      The_Lead.V_True_Airspeed        := Lead_Kinematic.V_True_Airspeed(The_Lead.Kinematic_Driver);

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Lead_From_Profile()");
         raise;
   end Update_Lead_From_Profile;


   ----------------------------
   -- Init_Lead_From_Ownship --
   ----------------------------
   -- Calculates where to put lead and initializes there.

   procedure Init_Lead_From_Ownship
   is

      --------------
      -- Quadrant --
      --------------
      -- Mapping world around ownship into usual quadrants
      subtype Quadrant_Type is Integer range 1..4;
      function Quadrant
        (Angle : Float)
        return Quadrant_Type
      is
         Q : Quadrant_Type;
         Bad_Angle : exception;
      begin
         if Angle >= 0.0 and Angle < 90.0 then
            Q := 1;
         elsif Angle >= 90.0 and Angle < 180.0 then
            Q := 2;
         elsif Angle >= 180.0 and Angle < 270.0 then
            Q := 3;
         elsif Angle >= 270.0 and Angle <= 360.0 then
            Q := 4;
         else
            raise Bad_Angle;
         end if;
         return Q;
      exception
         when Bad_Angle =>
            Log.Report("Bad_Angle exception in Quadrant()");
            raise;
         when others =>
            Log.Report("Jpats_Formation_Manager.Controller.Init_Lead_From_Ownship.Quadrant()");
            raise;
      end Quadrant;

      ---------------------------
      -- lead_relative_ownship --
      ---------------------------
      -- calculates feet in x and y from ownship with
      -- positive y towards ownship heading

      procedure Lead_Relative_Ownship
        (Heading_Angle_C           : in      Float;
         Positive_X1               : in      Boolean;
         X2_With_Addition          : in      Boolean;
         Lead_Lat_Relative_Ownship :    out  Long_Float;
         Lead_Lon_Relative_Ownship :    out  Long_Float)

      is

         M : constant Long_Float := Tan(Long_Float(Heading_Angle_C),360.0);
         A : constant Long_Float := 1.0 + M**2;
         X1,Y1,X2,Y2,Y_Intercept,B,C : Long_Float := 0.0;

      begin

         if Positive_X1 then
            X1 := Sqrt((20.0**2)/(1.0+M**2));
         else
            X1 := -Sqrt((20.0**2)/(1.0+M**2));
         end if;
         Y1 := -M*X1;
         Y_Intercept := Y1 - M*X1;
         B := 2.0*M*Y_Intercept - 2.0*Y1*M - 2.0*X1;
         C := Y_Intercept**2 - 2.0*Y1*Y_Intercept - 100.0**2 + X1**2 + Y1**2;
         if X2_With_Addition then
            X2 := (-B + Sqrt(B**2 - 4.0*A*C)) / (2.0*A);
         else
            X2 := (-B - Sqrt(B**2 - 4.0*A*C)) / (2.0*A);
         end if;
         Y2 := M*X2 + Y_Intercept;

         Lead_Lat_Relative_Ownship := Y2;
         Lead_Lon_Relative_Ownship := X2;

      exception
         when others =>
            Log.Report("Jpats_Formation_Manager.Controller.Init_Lead_From_Ownship.Lead_Relative_Ownship()");
            raise;
      end Lead_Relative_Ownship;

      Ownship_Heading   : constant Float := Float(Jpats_Simulated_Aircraft.Get_Hdg_Angle*(180.0/Pi));
      Ownship_Altitude  : constant Float := Float(Jpats_Simulated_Aircraft.Get_Aircraft_Geometric_Altitude);
      Ownship_Airspeed  : constant Float := Float(Jpats_Simulated_Aircraft.Get_Calibrated_Airspeed);
      Ownship_Latitude  : constant Long_Float := Long_Float(Jpats_Simulated_Aircraft.Get_North);
      Ownship_Longitude : constant Long_Float := Long_Float(Jpats_Simulated_Aircraft.Get_East);

      Lead_Latitude_Cartesian  : Long_Float;
      Lead_Longitude_Cartesian : Long_Float;
      Lead_Latitude            : Long_Float;
      Lead_Longitude           : Long_Float;

      -- Both planes have same heading.
      -- This heading is in usual xy plane (0 degrees along positive x axis)
      Heading_Angle_Cartesian : Float := 90.0 - Ownship_Heading;

   begin

      if Heading_Angle_Cartesian < 0.0 then -- keep heading in [0,360]
           Heading_Angle_Cartesian := Heading_Angle_Cartesian + 360.0;
      end if;

      if Heading_Angle_Cartesian = 0.0 or Heading_Angle_Cartesian = 360.0 then

         Lead_Latitude_Cartesian  := -20.0;
         Lead_Longitude_Cartesian := 100.0;

      elsif Heading_Angle_Cartesian = 90.0 then

         Lead_Latitude_Cartesian  := 100.0;
         Lead_Longitude_Cartesian := 20.0;

      elsif Heading_Angle_Cartesian = 180.0 then

         Lead_Latitude_Cartesian  := 20.0;
         Lead_Longitude_Cartesian := -100.0;

      elsif Heading_Angle_Cartesian = 270.0 then

         Lead_Latitude_Cartesian  := -100.0;
         Lead_Longitude_Cartesian := -20.0;

      else

         case Quadrant(Heading_Angle_Cartesian) is

            when 1 =>

               Lead_Relative_Ownship
                 ( Heading_Angle_C           =>  Heading_Angle_Cartesian,
                   Positive_X1               =>  True,
                   X2_With_Addition          =>  True,
                   Lead_Lat_Relative_Ownship =>  Lead_Latitude_Cartesian,
                   Lead_Lon_Relative_Ownship =>  Lead_Longitude_Cartesian
                   );

            when 2 =>

               Lead_Relative_Ownship
                 ( Heading_Angle_C           =>  Heading_Angle_Cartesian,
                   Positive_X1               =>  True,
                   X2_With_Addition          =>  False,
                   Lead_Lat_Relative_Ownship =>  Lead_Latitude_Cartesian,
                   Lead_Lon_Relative_Ownship =>  Lead_Longitude_Cartesian
                   );

            when 3 =>

               Lead_Relative_Ownship
                 ( Heading_Angle_C           =>  Heading_Angle_Cartesian,
                   Positive_X1               =>  False,
                   X2_With_Addition          =>  False,
                   Lead_Lat_Relative_Ownship =>  Lead_Latitude_Cartesian,
                   Lead_Lon_Relative_Ownship =>  Lead_Longitude_Cartesian
                   );

            when 4 =>

               Lead_Relative_Ownship
                 ( Heading_Angle_C           =>  Heading_Angle_Cartesian,
                   Positive_X1               =>  False,
                   X2_With_Addition          =>  True,
                   Lead_Lat_Relative_Ownship =>  Lead_Latitude_Cartesian,
                   Lead_Lon_Relative_Ownship =>  Lead_Longitude_Cartesian
                   );

         end case;

      end if;

      Lead_Latitude  := Ownship_Latitude + Lead_Latitude_Cartesian / 364566.0;
      Lead_Longitude := Ownship_Longitude + Lead_Longitude_Cartesian / (364566.0 * Cos(Lead_Latitude,360.0));

      Lead_Kinematic.Initialize
        ( An_Instance                 => The_Lead.Kinematic_Driver,
          Pitch_Angle_Init            => 0.0,
          Bank_Angle_Init             => 0.0,
          Heading_Init                => Ownship_Heading,
          Lat_Init                    => Lead_Latitude,
          Long_Init                   => Lead_Longitude,
          Geometric_Altitude_Init     => Ownship_Altitude,
          V_Indicated_Airspeed_Init   => Ownship_Airspeed,
          Gear_Position_Init          => 0.0,
          Speedbrake_Position_Init    => 0.0,
          Flap_Position_Init          => 0.0,
          Elevator_Position_Init      => 0.0,
          Rudder_Position_Init        => 0.0,
          Right_Aileron_Position_Init => 0.0,
          Left_Aileron_Position_Init  => 0.0
          );

      Update_Lead_From_Kinematic;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Init_Lead_From_Ownship()");
         raise;
   end Init_Lead_From_Ownship;

   ------------
   -- Update --
   ------------

   Kinematic_Driver_Lp   : Boolean renames Ctnr.This_Subsystem.FMC_Kinematic_Driver_Lp;
   Profile_Driver_Lp     : Boolean renames Ctnr.This_Subsystem.FMC_Profile_Driver_Lp;
   VIAS_Cmd_Lp           : Float   renames Ctnr.This_Subsystem.FMC_VIAS_Cmd_Lp;
   Accel_Vias_Lp         : Float   renames Ctnr.This_Subsystem.FMC_Accel_Vias_Lp;
   Pitch_Angle_Cmd_Lp    : Float   renames Ctnr.This_Subsystem.FMC_Pitch_Angle_Cmd_Lp;
   Pitch_Rate_Lp         : Float   renames Ctnr.This_Subsystem.FMC_Pitch_Rate_Lp;
   Profile_Playback_Lp   : Boolean renames Ctnr.This_Subsystem.FMC_Profile_Playback_Lp;
   Formation_Save_Cmd_Lp : Boolean renames Ctnr.This_Subsystem.FMC_Formation_Save_Cmd_Lp;
   Formation_Load_Cmd_Lp : Boolean renames Ctnr.This_Subsystem.FMC_Formation_Load_Cmd_Lp;
   Track_Counter_R       : Integer renames Ctnr.This_Subsystem.FMC_Track_Counter_R;
   Recorder_On_Lp        : Boolean renames Ctnr.This_Subsystem.FMC_Recorder_On_Lp;
   Freeze_Rq             : Boolean renames Ctnr.This_Subsystem.FMC_Freeze_Rq;
   Profile_Choosen_Lp    : Boolean renames Ctnr.This_Subsystem.FMC_Profile_Choosen_Lp;
   Profile_Choosen       : Boolean renames Ctnr.This_Subsystem.FMC_Profile_Choosen;
   Flight_Freeze_Lp      : Boolean renames Ctnr.This_Subsystem.FMC_Flight_Freeze_Lp;
   Track_Number_Lp       : Integer renames Ctnr.This_Subsystem.FMC_Track_Number_Lp;
   Recorder_Initialized  : Boolean renames Ctnr.This_Subsystem.FMC_Recorder_Initialized;

   Playback_Mode_Timer   : Ctnr.T1 renames Ctnr.This_Subsystem.FMC_Playback_Mode_Timer;

   The_Name : Bnd_16.Bounded_String renames Ctnr.This_Subsystem.FMC_The_Name;
   The_Rest : Bnd_16.Bounded_String renames Ctnr.This_Subsystem.FMC_The_Rest;
   Temp_String : String renames Ctnr.This_Subsystem.FMC_Temp_String;

   Track_Names   : Ctnr.TrackNamesType  renames Ctnr.This_Subsystem.FMC_Track_Names;
   Track_Names_R : Ctnr.TrackNamesTypeR renames Ctnr.This_Subsystem.FMC_Track_Names_R;

   Next_Track_Lp      : Boolean renames Ctnr.This_Subsystem.FMC_Next_Track_Lp;
   Next_Track_Timer   : Float   renames Ctnr.This_Subsystem.FMC_Next_Track_Timer;
   Next_Track_Timer_2 : Float   renames Ctnr.This_Subsystem.FMC_Next_Track_Timer_2;

   Previous_Track_Lp      : Boolean renames Ctnr.This_Subsystem.FMC_Previous_Track_Lp;
   Previous_Track_Timer   : Float   renames Ctnr.This_Subsystem.FMC_Previous_Track_Timer;
   Previous_Track_Timer_2 : Float   renames Ctnr.This_Subsystem.FMC_Previous_Track_Timer_2;

   Current_Track_Number_Lp : Integer renames Ctnr.This_Subsystem.FMC_Current_Track_Number_Lp;

   Pause_Request : Boolean renames Ctnr.This_Subsystem.FMC_Pause_Request;
   Start_Playback_Request : boolean renames Ctnr.This_Subsystem.FMC_Start_Playback_Request;
   Goto_Segment_Request : boolean renames Ctnr.This_Subsystem.FMC_Goto_Segment_Request;
   Goto_Segment_Request_Track_Number : Integer renames Ctnr.This_Subsystem.FMC_Goto_Segment_Request_Track_Number;

   Just_Selected_Play : Boolean renames Ctnr.This_Subsystem.FMC_Just_Selected_Play;

   function Freeze_Request
     return Boolean
   is
   begin
      return Freeze_Rq;
   end Freeze_Request;


   procedure Initialize_Tracknames
   is
   begin

      Ios.Formation_Track_1_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_2_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_3_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_4_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_5_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_6_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_7_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_8_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_9_Name_R  := Bnd_16.To_String(Ctnr.Not_Available);
      Ios.Formation_Track_10_Name_R := Bnd_16.To_String(Ctnr.Not_Available);

      for I in 1..10
      loop
         Track_Names_R(I) := Ctnr.Not_Available;
      end loop;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Initialize_Tracknames()");
         raise;
   end Initialize_Tracknames;

   procedure Update_Tracknames
   is
   begin

      The_Name := Bnd_16.To_Bounded_String("Track" & Integer'Image(Track_Counter_R));
      Track_Names_R(Track_Counter_R) := Bnd_16.Head(The_Name,Bnd_16.Max_Length);

      if Ios.Formation_Track_1_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_1_Name_R  := Bnd_16.To_String(Track_Names_R(1)); end if;
      if Ios.Formation_Track_2_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_2_Name_R  := Bnd_16.To_String(Track_Names_R(2)); end if;
      if Ios.Formation_Track_3_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_3_Name_R  := Bnd_16.To_String(Track_Names_R(3)); end if;
      if Ios.Formation_Track_4_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_4_Name_R  := Bnd_16.To_String(Track_Names_R(4)); end if;
      if Ios.Formation_Track_5_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_5_Name_R  := Bnd_16.To_String(Track_Names_R(5)); end if;
      if Ios.Formation_Track_6_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_6_Name_R  := Bnd_16.To_String(Track_Names_R(6)); end if;
      if Ios.Formation_Track_7_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_7_Name_R  := Bnd_16.To_String(Track_Names_R(7)); end if;
      if Ios.Formation_Track_8_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_8_Name_R  := Bnd_16.To_String(Track_Names_R(8)); end if;
      if Ios.Formation_Track_9_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_9_Name_R  := Bnd_16.To_String(Track_Names_R(9)); end if;
      if Ios.Formation_Track_10_Name_R = Bnd_16.To_String(Ctnr.Not_Available) then
         Ios.Formation_Track_10_Name_R := Bnd_16.To_String(Track_Names_R(10)); end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Tracknames()");
         raise;
   end Update_Tracknames;

   procedure Create_Index_File
   is

      P_File : Ada.Text_Io.File_Type;

   begin

      Ada.Text_Io.Create(P_File,Ada.Text_Io.Out_File,"/ramdisk/record.idx");
      Ada.Text_Io.Set_Output(P_File);
      Ada.Text_Io.Put(Ios.Formation_Track_1_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_2_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_3_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_4_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_5_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_6_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_7_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_8_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_9_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Put(Ios.Formation_Track_10_Name_R); Ada.Text_Io.New_Line;
      Ada.Text_Io.Set_Output(Ada.Text_Io.Standard_Output);
      Ada.Text_Io.Close(P_File);

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Create_Index_File()");
         raise;
   end Create_Index_File;



   procedure Update_Recorder
   is

      Recorder_On               : constant Boolean := Ios.Recorder_On;
      Recorder_Off              : constant Boolean := not Recorder_On;
      Recorder_Toggled_On       : constant Boolean := Recorder_On and not Recorder_On_LP;
      Recorder_Toggled_Off      : constant Boolean := not Recorder_On and Recorder_On_LP;
      Flight_Freeze             : constant Boolean := Sim_Ac.Get_Flight_Freeze;
      Flight_Freeze_Toggled_On  : constant Boolean := Flight_Freeze and not Flight_Freeze_LP;
      Flight_Freeze_Toggled_Off : constant Boolean := not Flight_Freeze and Flight_Freeze_LP;
      Save_Command_Selected     : constant Boolean := Ios.Formation_Save_Cmd and not Formation_Save_Cmd_Lp;


   begin

      if Recorder_Toggled_On then

         Freeze_Rq := True;                  -- request freeze
         Track_Counter_R := 1;               -- will begin recording track 1
         Recorder_Initialized := True;       -- latch to insure time necessary to init recorder
         Initialize_Tracknames;              -- set all track names to "not available"

      elsif Recorder_On then

         if Flight_Freeze_Toggled_Off then -- begin recording

              Recorder.Start_Recording;

         else -- recording

            if Flight_Freeze_Toggled_On and not Recorder_Initialized then

               Recorder.End_Profile_Segment;              -- end track
               Update_Tracknames;                         -- name current track "track i" i=Track_Counter_R
               Track_Counter_R := Track_Counter_R + 1;    -- number of track to be recorded next

            else

               Recorder_Initialized := False;  -- one pass flag

            end if;

         end if;

      elsif Recorder_Toggled_Off then

         if not Flight_Freeze then

            Recorder.End_Profile_Segment;   -- end track
            Update_Tracknames;              -- name track last recorded

         end if;

         Recorder.Complete_Profile;         -- end recording session

      elsif Recorder_Off then

         if Save_Command_Selected then

            Create_Index_File;                   -- save names of tracks to index file

         else
            Ios.Formation_Save_Cmd := False;     -- inform ios index file has been created
         end if;

      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Recorder()");
         raise;
   end Update_Recorder;


   -------------------------------------
   -- Read_Tracknames_From_Index_File --
   -------------------------------------

   procedure Read_Tracknames_From_Index_File
   is

      R_File : Ada.Text_Io.File_Type;
      Counter : Natural;
      use type Bnd_16.Bounded_String;

   begin

      begin   -- protect from nonexistent index file
         Ada.Text_Io.Open(R_File, Ada.Text_Io.In_File, "/ramdisk/playback.idx");
      exception
         when Ada.Text_Io.Name_Error =>
            Ada.Text_Io.Create(R_File, Ada.Text_Io.In_File, "/ramdisk/playback.idx");
      end;

      for I in 1..10    -- read tracknames into array
      loop
         if not Ada.Text_Io.End_Of_File(R_File) then
            Ada.Text_Io.Get_Line(R_File, Temp_String, Counter);
            Track_Names(I) := Bnd_16.To_Bounded_String(Temp_String);
            Ada.Text_Io.Skip_Line(R_File);
         else
            Track_Names(I) := Ctnr.Not_Available;
         end if;
      end loop;

      Ada.Text_Io.Close(R_File); -- close index file

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Read_Tracknames_From_Index_File()");
         raise;
   end Read_Tracknames_From_Index_File;


   ------------------------
   -- Set_Ios_Tracknames --
   ------------------------

   procedure Set_Ios_Tracknames
   is

      Temp_Int : Integer := 0;
      use type Bnd_16.Bounded_String;

   begin

      for I in 1..10 -- count non "not available" tracknames for ios pulldown box
      loop
         if Track_Names(I) /= Ctnr.Not_Available then
            Temp_Int := Temp_Int + 1;
         end if;
      end loop;

      Ios.Formation_Number_Of_Tracks := Temp_Int;

      Ios.Formation_Track_1_Name  := Bnd_16.To_String(Track_Names(1));
      Ios.Formation_Track_2_Name  := Bnd_16.To_String(Track_Names(2));
      Ios.Formation_Track_3_Name  := Bnd_16.To_String(Track_Names(3));
      Ios.Formation_Track_4_Name  := Bnd_16.To_String(Track_Names(4));
      Ios.Formation_Track_5_Name  := Bnd_16.To_String(Track_Names(5));
      Ios.Formation_Track_6_Name  := Bnd_16.To_String(Track_Names(6));
      Ios.Formation_Track_7_Name  := Bnd_16.To_String(Track_Names(7));
      Ios.Formation_Track_8_Name  := Bnd_16.To_String(Track_Names(8));
      Ios.Formation_Track_9_Name  := Bnd_16.To_String(Track_Names(9));
      Ios.Formation_Track_10_Name := Bnd_16.To_String(Track_Names(10));

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Set_Ios_Tracknames()");
         raise;
   end Set_Ios_Tracknames;


   -----------------------
   -- Choose_Next_Track --
   -----------------------

   procedure Choose_Next_Track
     (Iconst : in   Float)
   is
   begin

      Next_Track_Timer := Next_Track_Timer + Iconst;
      if Next_Track_Timer >= 0.5 then
         if Ios.Formation_Track_Number < Ios.Formation_Number_Of_Tracks then
            Ios.Formation_Track_Number := Ios.Formation_Track_Number + 1;
         end if;
         Ios.Formation_Next_Track := False;
         Next_Track_Timer := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Choose_Next_Track()");
         raise;
   end Choose_Next_Track;

   ---------------------------
   -- Choose_Previous_Track --
   ---------------------------

   procedure Choose_Previous_Track
     (Iconst : in   Float)
   is
   begin

      Previous_Track_Timer := Previous_Track_Timer + Iconst;
      if Previous_Track_Timer >= 0.5 then
         if Ios.Formation_Track_Number > 1 then
            Ios.Formation_Track_Number := Ios.Formation_Track_Number - 1;
         end if;
         Ios.Formation_Previous_Track := False;
         Previous_Track_Timer := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Choose_Previous_Track()");
         raise;
   end Choose_Previous_Track;


   -------------------------
   -- Update_Track_choice --
   -------------------------

   procedure Update_Track_Choice
     (Iconst : in   Float)
   is
      Choose_Next_Track_Cmd : constant Boolean
        := ((Ios.Formation_Next_Track and not Next_Track_Lp)
            or Next_Track_Timer /= 0.0);  -- latched for visual effect
      Choose_Previous_Track_Cmd : constant Boolean
        := ((Ios.Formation_Previous_Track and not Previous_Track_Lp)
            or Previous_Track_Timer /= 0.0); -- latched for visual effect
   begin

      if Choose_Next_Track_Cmd and not Choose_Previous_Track_Cmd then
         Choose_Next_Track(Iconst);
      elsif Choose_Previous_Track_Cmd and not Choose_Next_Track_Cmd then
         Choose_Previous_Track(Iconst);
      else
         Ios.Formation_Next_Track := False;
         Ios.Formation_Previous_Track := False;
      end if;

      Next_Track_Lp := Ios.Formation_Next_Track;
      Previous_Track_Lp := Ios.Formation_Previous_Track;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Track_choice()");
         raise;
   end Update_Track_Choice;


   --------------------
   -- Profile_Exists --
   --------------------

   function Profile_Exists
     return Boolean
   is

      File : Ada.Streams.Stream_Io.File_Type;
      Name : constant String := "/ramdisk/playback.dat";

   begin

      Ada.Streams.Stream_IO.Open
        ( File => File,
          Mode => Ada.Streams.Stream_Io.in_File,
          Name => Name
          );

      Ada.Streams.Stream_Io.Close ( File => File );

      return True;

   exception
      when others =>
         return False;
   end Profile_Exists;


   -------------------
   -- Init_Playback --
   -------------------

   procedure Init_Playback
   is
   begin

      if Profile_Exists then
         Just_Selected_Play := True;
         Playback.Start_Playback;
      else
         Ios.Profile_Driver := False;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Ensure_Profile_Exists()");
         raise;
   end Init_Playback;


   -----------------------
   -- Init_Track_Choice --
   -----------------------

   procedure Init_Track_Choice
   is
   begin

      if Ios.Formation_Track_Number > 0
        and  Ios.Formation_Track_Number <= Ios.Formation_Number_Of_Tracks then
         Goto_Segment_Request := True;
         Goto_Segment_Request_Track_Number := Ios.Formation_Track_Number;
      else
         Goto_Segment_Request := True;
         Goto_Segment_Request_Track_Number := 1;
      end if;
      Just_Selected_Play := False;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Init_Track_Choice()");
         raise;
   end Init_Track_Choice;


   ---------------------
   -- Goto_Next_Track --
   ---------------------
   procedure Goto_Next_Track
     (Iconst : in   Float)
   is
   begin
      Next_Track_Timer_2 := Next_Track_Timer_2 + Iconst;
      if Next_Track_Timer_2 > 0.5 then
         --Freeze_Rq := True;
         if Ios.Formation_Track_Number < Ios.Formation_Number_Of_Tracks then
            Goto_Segment_Request := True;
            Goto_Segment_Request_Track_Number := Ios.Formation_Track_Number+1;
         end if;
         Ios.Formation_Next_Track := False;
         Next_Track_Timer_2 := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Goto_Next_Track()");
         raise;
   end Goto_Next_Track;

   -------------------------
   -- Goto_Previous_Track --
   -------------------------
   procedure Goto_Previous_Track
     (Iconst : in   Float)
   is
   begin
      Previous_Track_Timer_2 := Previous_Track_Timer_2 + Iconst;
      if Previous_Track_Timer_2 >= 0.5 then

         if Ios.Formation_Track_Number > 1 then
            --Freeze_Rq := True;
            Goto_Segment_Request := True;
            Goto_Segment_Request_Track_Number := Ios.Formation_Track_Number-1;
         end if;
         Ios.Formation_Previous_Track := False;
         Previous_Track_Timer_2 := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Goto_Previous_Track()");
         raise;
   end Goto_Previous_Track;

   ----------------------
   -- Goto_Other_Track --
   ----------------------
   procedure Goto_Other_Track
   is

   begin

      if Ios.Formation_Track_Number > 0 and  Ios.Formation_Track_Number <= 10 then
         Goto_Segment_Request := True;
         Goto_Segment_Request_Track_Number := Ios.Formation_Track_Number;
      else
         Goto_Segment_Request := True;
         Goto_Segment_Request_Track_Number := 1;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Goto_Other_Track()");
         raise;
   end Goto_Other_Track;

   ------------------
   -- Update_Track --
   ------------------

   procedure Update_Track
     (Iconst : in   Float)
   is
      Next_Track_Cmd : constant Boolean
        := (Ios.Formation_Next_Track and not Next_Track_Lp) or Next_Track_Timer_2 /= 0.0;

      Previous_Track_Cmd : constant Boolean
        := (Ios.Formation_Previous_Track and not Previous_Track_Lp) or Previous_Track_Timer_2 /= 0.0;

      Jump_To_Other_Track_Cmd :constant Boolean
        := (Ios.Formation_Track_Number /= Track_Number_Lp
            and Playback.Current_Track_Number = Current_Track_Number_Lp);

   begin

      if Next_Track_Cmd and not Previous_Track_Cmd and not Jump_To_Other_Track_Cmd then
         Goto_Next_Track(Iconst);
      elsif Previous_Track_Cmd and not Next_Track_Cmd and not Jump_To_Other_Track_Cmd then
         Goto_Previous_Track(Iconst);
      elsif Jump_To_Other_Track_Cmd and not Next_Track_Cmd and not Previous_Track_Cmd then
         Goto_Other_Track;
      else
         Ios.Formation_Previous_Track := False;
         Ios.Formation_Next_Track := False;
      end if;

      Next_Track_Lp := Ios.Formation_Next_Track;
      Previous_Track_Lp := Ios.Formation_Previous_Track;


   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Track()");
         raise;
   end Update_Track;

   -------------------
   -- Update_Visual --
   -------------------
   procedure Update_Visual
   is
   begin
      if Playback.State = JF_T.Playback or -- set visual on/off
        Playback.State =
        JF_T.Formation_Profile_Playback_State'(JF_T.Paused)
      then
         The_Lead.Is_Active := True;
      else
         The_Lead.Is_Active := False;
      end if;
   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Visual()");
         raise;
   end Update_Visual;


   -----------------------------
   -- Update_Profile_Playback --
   -----------------------------

   procedure Update_Profile_Playback
     (Iconst : in   Float)
   is

      Profile_Driver_On          : constant Boolean := Ios.Profile_Driver;
      Profile_Driver_Off         : constant Boolean := not Profile_Driver_On;
      Load_Command_Toggled_On    : constant Boolean := Ios.Formation_Load_Cmd and not Formation_Load_Cmd_Lp;
      Profile_Driver_Toggled_On  : Boolean := Profile_Driver_On and not Profile_Driver_Lp;
      Profile_Driver_Toggled_Off : constant Boolean := not Profile_Driver_On and Profile_Driver_Lp;

      Flight_Freeze              : constant Boolean := Sim_Ac.Get_Flight_Freeze or
                                  (Sr_Ios_If.Play_Formation_Demo and Save_Restore.Replay_Trimming) or
                                  (Sr_Ios_If.Play_Formation_Demo and not Save_Restore.Replay_In_Progress and
                                  Init_Formation_Demo_Lp and Formation_Demo_Initialized);

      Flight_Freeze_Toggled_On   : constant Boolean := (Flight_Freeze and not Flight_Freeze_Lp) or
                                  (Sr_Ios_If.Play_Formation_Demo and Save_Restore.Replay_Trimming and not Flight_Freeze_Lp) or
                                  (Sr_Ios_If.Play_Formation_Demo and not Save_Restore.Replay_In_Progress and
                                  Init_Formation_Demo_Lp and Formation_Demo_Initialized and not Flight_Freeze_Lp);

      Flight_Freeze_Toggled_Off  : Boolean := (not Flight_Freeze and Flight_Freeze_Lp) or
                                  (Sr_Ios_If.Play_Formation_Demo and not Save_Restore.Replay_Trimming and Flight_Freeze_Lp) or
                                  (Sr_Ios_If.Play_Formation_Demo and Save_Restore.Replay_In_Progress and
                                  Init_Formation_Demo_Lp and Formation_Demo_Initialized and Flight_Freeze_Lp);


   begin

      if Load_Command_Toggled_On then
         Read_Tracknames_From_Index_File;
         Set_Ios_Tracknames;
         Ios.Formation_Track_Number := 1;       -- current track number to profile beginning
         Ios.Formation_Load_Cmd := False;       -- tell IOS tracknames are loaded
         if Sr_Ios_If.Play_Formation_Demo then
            Sr_Ios_If.New_Formation_Demo_Loaded := False;
            Sr_Ios_If.Formation_Demo_Leadship_Loaded := True;
            Formation_Demo_Initialized := False; -- Initiate the Formation Demo Playback
            Init_Formation_Demo_Lp := True;
            Ios.Formation_Track_Number := Sr_Ios_If.Formation_Demo_Track_Number;
            Goto_Other_Track;
         end if;
      end if;


      if Profile_Driver_Toggled_On then

         Init_Playback;

      elsif Profile_Driver_On then

         if Just_Selected_Play then -- one pass
              Init_Track_Choice;
         end if;

         Update_Track(Iconst);

         Start_Playback_Request := Flight_Freeze_Toggled_Off;

         Pause_Request := Flight_Freeze_Toggled_On;

         Ios.Formation_Track_Number := Playback.Current_Track_Number;

      elsif Profile_Driver_Toggled_Off then
         if Sr_Ios_If.Play_Formation_Demo then
            Ios.Profile_Driver := True;
            Log.Report("Profile driver toggled off..ignore");
         else
            Log.Report("Profile driver toggled off");
            Playback.Stop_Playback;
            Formation_Demo_Initialized := True;
         end if;

      elsif Profile_Driver_Off then

         Update_Track_Choice(Iconst);
         Playback.Stop_Playback;

      end if;

      Update_Visual; -- visual of lead on/off

      if not Flight_Freeze then
         Update_Lead_From_Profile;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update_Profile_Playback()");
         raise;
   end Update_Profile_Playback;


   ------------
   -- Update --
   ------------
   procedure Update
     (Iconst : in Float)
   is

      Airspeed_Driver_Selected       : Boolean := False;
      Pitch_Attitude_Driver_Selected : Boolean := False;


      R_File : Ada.Text_Io.File_Type;

      Temp_Int : Integer := 0;

      use type Bnd_16.Bounded_String;

   begin

      if Sr_Ios_If.Play_Formation_Demo and
              not Init_Formation_Demo_Lp and
              not Sr_Ios_If.New_Formation_Demo_Loaded then
         if Formation_Demo_Initialized then
            if Freeze_Count < 1 then
               Sim_Ac.Un_Freeze;
            end if;
            Freeze_Count := Freeze_Count + 1;
            if Freeze_Count > FREEZE_DURATION then
               Sr_Ios_If.Allow_Offset_Adjustment := True;
               Freeze_Count := 0;
               Sim_Ac.Set_System_Freeze;
               Init_Formation_Demo_Lp := True;
               Sr_Ios_If.Init_Formation_Demo_Replay_Delta;
            end if;
         else
            Formation_Demo_Initialized := True;
         end if;
      end if;
      
      if Formation_Demo_Leadship_Only_Lp and Ios_If.End_Formation_Demo_Leadship_Cmd then
         Sr_Ios_If.Formation_Demo_Leadship_Loaded := False;
         Sr_Ios_If.Formation_Demo_Leadship_Only   := False;
         Ios_If.End_Formation_Demo_Leadship_Cmd   := False;
         Init_Formation_Demo_Lp                   := False;
         Sr_Ios_If.New_Formation_Demo_Loaded      := False;
         Sr_Ios_If.Allow_Offset_Adjustment        := False;
         Formation_Demo_Initialized               := True;
         Sr_Ios_If.Allow_Demo_Select              := True;
         Ios.Profile_Driver                       := False;
         Sr_Ios_If.Formation_Demo_Profile_Name    := Sr_Ios_If.Bounded_64.to_string(Sr_Ios_If.Spaces);
         Sr_Ios_If.Formation_Demo_Track_Number    := 1;
         Ios.Formation_Track_Number               := 1;
         Initialize_Tracknames;
         Set_Ios_Tracknames;
      end if;

      if Sim_Ac.Get_Flight_Freeze then
         Freeze_Rq := False;
      end if;

      if Playback.State = JF_T.Playback and Start_Playback_Request then
         Start_Playback_Request := False;
      elsif Start_Playback_Request then
         Playback.Start_Playback;
      end if;

      if Playback.State = JF_T.Formation_Profile_Playback_State'(JF_T.Paused) and Pause_Request then
         Pause_Request := False;
      elsif Pause_Request then
         Playback.Pause_playback;
      end if;

      if Ios.Profile_Driver
        and Goto_Segment_Request
        and Playback.Current_Track_Number /= Goto_Segment_Request_Track_Number then

         if Goto_Segment_Request_Track_Number > 0 and Goto_Segment_Request_Track_Number < 11 then
            Playback.Go_To_Segment(Natural(Goto_Segment_Request_Track_Number));
         else
            Playback.Go_To_Segment(Natural(1));
         end if;
         Goto_Segment_Request := False;

      elsif Ios.Profile_Driver
        and Goto_Segment_Request
        and Playback.Current_Track_Number = Goto_Segment_Request_Track_Number then

         Goto_Segment_Request := False;

      elsif not Ios.Profile_Driver then

         Goto_Segment_Request := False;

      end if;

      if Ios.Kinematic_Driver then --kinematic driver choosen

           Ios.Profile_Driver := False;

           if not Kinematic_Driver_Lp then -- if not choosen last pass init

                if Profile_Playback_Lp then -- init from profile

                 Init_From_Profile :
                     declare
                     begin
                        Update_Lead_From_Profile;
                     exception
                        when others =>
                           Log.Report("Jpats_Formation_Manager.Controller.Update().Init_From_Profile -- (block) ");
                           raise;
                     end Init_From_Profile;

                else -- init from ownship

               Init_From_Ownship :
                   declare

                      Ownship_Airspeed  : constant Float := Float(Sim_Ac.Get_Calibrated_Airspeed);
                      Init_Ias_Ios : Float := Ownship_Airspeed;

                   begin

                      if Ownship_Airspeed > 300.0 then
                         Init_Ias_Ios := 300.0;
                      elsif Ownship_Airspeed < 100.0 then
                         Init_Ias_Ios := 100.0;
                      end if;

                      Ios.Pitch_Angle              := 0.0;
                      Ios.Pitch_Rate               := 15.0;
                      Ios.Bank_Angle               := 0.0;
                      Ios.Roll_Rate                := 15.0;
                      Ios.Indicated_Airspeed       := Init_Ias_Ios;
                      Ios.Airspeed_Acceleration    := 50.0;

                      Ios.Wing_Dip_Cmd             := False;
                      Ios.Porpoise_Cmd             := False;
                      Ios.Big_Wing_Rock_Cmd        := False;
                      Ios.Small_Wing_Rock_Cmd      := False;
                      Ios.Flaps_Up                 := True;
                      Ios.Flaps_Ldg                := False;
                      Ios.Speedbrake_Deployed      := False;
                      Ios.Gear_Up                  := False;

                      Init_Lead_From_Ownship;

                   exception
                      when others =>
                         Log.Report("Jpats_Formation_Manager.Controller.Update().Init_From_Ownship -- (block)");
                         raise;
                   end Init_From_Ownship;

                end if;

           else --| normal kinematic update , read ios...drive kinematic

              The_Lead.Is_Active := True;

              if (Ios.Indicated_Airspeed /= VIAS_Cmd_Lp) or (Ios.Airspeed_Acceleration /= Accel_Vias_Lp) then
                 Airspeed_Driver_Selected := True;
              elsif (Ios.Pitch_Angle /= Pitch_Angle_Cmd_Lp) or (Ios.Pitch_Rate /= Pitch_Rate_Lp) then
                 Pitch_Attitude_Driver_Selected := True;
              end if;

              if not Sim_Ac.Get_Flight_Freeze then
                 Lead_Kinematic.Update
                   ( An_Instance                    => The_Lead.Kinematic_Driver,
                     Iconst                         => Iconst,
                     Pitch_Attitude_Driver_Selected => Pitch_Attitude_Driver_Selected,
                     Airspeed_Driver_Selected       => Airspeed_Driver_Selected,
                     Pitch_Angle_Commanded          => Ios.Pitch_Angle,
                     Pitch_Rate                     => Ios.Pitch_Rate,
                     V_Indicated_Airspeed_Commanded => Ios.Indicated_Airspeed,
                     Velocity_Gain                  => Ios.Airspeed_Acceleration,
                     Air_Density_At_Altitude        => Jpats_Atmosphere.Density,
                     Air_Density_At_Sea_Level       => 0.002378,
                     Bank_Angle_Commanded           => Ios.Bank_Angle,
                     Roll_Rate                      => Ios.Roll_Rate,
                     Wing_Dip_Commanded             => Ios.Wing_Dip_Cmd,
                     Porpoise_Commanded             => Ios.Porpoise_Cmd,
                     Big_Wing_Rock_Commanded        => Ios.Big_Wing_Rock_Cmd,
                     Small_Wing_Rock_Commanded      => Ios.Small_Wing_Rock_Cmd,
                     Flaps_Up                       => True,
                     Flaps_Ldg                      => False,
                     Speedbrake_Deployed            => False,
                     Gear_Up                        => True,
                     Wingman_Lat                    => Long_Float(Jpats_Simulated_Aircraft.Get_North),
                     Wingman_Long                   => Long_Float(Jpats_Simulated_Aircraft.Get_East)
                     );

                 Update_Lead_From_Kinematic;

              else
                 null;
              end if;

           end if;

           Playback.Stop_Playback;

      else  -- use playback machine (kinematic driver is not checked)

         Update_Profile_Playback(Iconst);

      end if; -- totally out of all kinematic and playback logic


      -- *******************************************************************
      -- *************************** RECORDING *****************************
      -- *******************************************************************

      Update_Recorder;


      --********************************************************************
      --***************************** LAST PASS ****************************
      --********************************************************************
      Track_Number_Lp := Ios.Formation_Track_Number;

      Formation_Save_Cmd_Lp := Ios.Formation_Save_Cmd;

      Profile_Playback_Lp := Ios.Profile_Driver;
      Profile_Driver_Lp   := Ios.Profile_Driver;
      Kinematic_Driver_Lp := Ios.Kinematic_Driver;

      VIAS_Cmd_Lp        := Ios.Indicated_Airspeed;
      Accel_Vias_Lp      := Ios.Airspeed_Acceleration;
      Pitch_Angle_Cmd_Lp := Ios.Pitch_Angle;
      Pitch_Rate_Lp      := Ios.Pitch_Rate;

      Formation_Load_Cmd_Lp := Ios.Formation_Load_Cmd;
      Recorder_On_Lp        := Ios.Recorder_On;
      Profile_Choosen_Lp    := Profile_Choosen;
      Flight_Freeze_Lp      := Sim_Ac.Get_Flight_Freeze or
                               (Sr_Ios_If.Play_Formation_Demo and Save_Restore.Replay_Trimming) or
                               (Sr_Ios_If.Play_Formation_Demo and not Save_Restore.Replay_In_Progress and
                               Init_Formation_Demo_Lp and Formation_Demo_Initialized);

      Current_Track_Number_Lp := Playback.Current_Track_Number;
      Formation_Demo_Leadship_Only_LP := Sr_Ios_If.Formation_Demo_Leadship_Only;

      declare

         Olat  : constant Long_Float := Long_Float(Sim_Ac.Get_North);
         Olon  : constant Long_Float := Long_Float(Sim_Ac.Get_East);
         Olatf : constant Long_Float := Olat - Long_Float'Floor(Olat);
         Llat  : constant Long_Float := The_Lead.Latitude;
         Llon  : constant Long_Float := The_Lead.Longitude;
         Llatf : constant Long_Float := Llat - Long_Float'Floor(Llat);

         Ft_Per_Deg : constant Long_Float := 364566.0;
         D          : constant Long_Float := Sqrt(((Olon-Llon)*Ft_Per_Deg*Cos(Llat,360.0))**2
                                                  +((Olat-Llat)*Ft_Per_Deg)**2);

      begin
         Ios.Ownship_Lat_Fract        := 100000.0*Olatf;
         Ios.Lead_Lat_Fract           := 100000.0*Llatf;
         Ios.Lead_To_Ownship_Distance := D;
      exception
         when others =>
            Log.Report("fractional()");
            raise;
      end;


      begin

         ios.Fm_Lead_Ias   := The_Lead.V_Indicated_Airspeed;
         ios.Fm_Float_1    := Ios.Airspeed_Acceleration;
         ios.Fm_Float_2    := Ios.Indicated_Airspeed;
         ios.Fm_Float_3    := The_Lead.V_Indicated_Airspeed * 0.5924838;
         ios.Fm_Float_4    := Sim_Ac.Get_Calibrated_Airspeed;
         ios.Fm_Float_5    := 0.0;

      exception
         when others =>
            Log.Report("formation manager...");
            raise;
      end;

      if Sr_Ios_If.Play_Formation_Demo and
         not Formation_Demo_Initialized then -- If a Formation Demo just loaded, start profile driver
            IOS.Profile_Driver := True;
            Init_Formation_Demo_Lp := False;
      end if;

   exception
      when others =>
         Log.Report("Jpats_Formation_Manager.Controller.Update()");
         raise;
   end Update;

end Jpats_Formation_Manager.Controller;
