-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Numerics; use Ada.Numerics;
with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;
with Log;
with Ada.Numerics.Generic_Elementary_Functions;
with Jpats_Atmosphere;

package body Formation_Lead_Kinematic is

   package Lf is new Ada.Numerics.Generic_Elementary_Functions (Long_Float);

   procedure Set_Bank_Angle( An_Instance : in out Instance;
                             Bank_Angle : in Float) is
   begin
     An_Instance.Bank_Angle := Bank_Angle; 
   end Set_Bank_Angle;

   procedure Reset_Rrd_Gen_From_State
     (An_Instance : in out Instance)
   is
   begin
      Random_Rock_Direction.Reset(RRD_Gen,An_Instance.RRD_Gen_State);
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Reset_Rrd_Gen_From_State()");
         raise;
   end Reset_Rrd_Gen_From_State;

   procedure Reset_Rrd_Gen
   is
   begin
      Random_Rock_Direction.Reset(RRD_Gen);
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Reset_Rrd_Gen()");
         raise;
   end Reset_Rrd_Gen;

   function V_Indicated_Airspeed
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.V_Indicated_Airspeed;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.V_Indicated_Airspeed()");
         raise;
   end V_Indicated_Airspeed;

   function V_True_Airspeed
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.V_True_Airspeed;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.V_True_Airspeed()");
         raise;
   end V_True_Airspeed;

   function Bank_Angle
     (An_Instance : in Instance)
      return Float
   is
   begin
      return An_Instance.Bank_Angle;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Bank_Angle()");
         raise;
   end Bank_Angle;

   function Lat
     (An_Instance : in Instance)
      return Long_Float
   is
   begin
      return An_Instance.Lat;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Lat()");
         raise;
   end Lat;

   function Geometric_Altitude
     (An_Instance : in Instance)
      return Float
   is
   begin
      return An_Instance.Geometric_Altitude;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Geometric_Altitude()");
         raise;
   end Geometric_Altitude;

   function Heading
     (An_Instance : in Instance)
      return Float
   is
   begin
      return An_Instance.Heading;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Heading()");
         raise;
   end Heading;

   function Long
     (An_Instance : in Instance)
      return Long_Float
   is
   begin
      return An_Instance.Long;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Long()");
         raise;
   end Long;

   function Pitch_Angle
     (An_Instance : in Instance)
      return Float
   is
   begin
      return An_Instance.Pitch_Angle;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Pitch_Angle()");
         raise;
   end Pitch_Angle;

   function Flap_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Flap_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Flap_Position()");
         raise;
   end Flap_Position;

   function Speedbrake_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Speedbrake_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Speedbrake_Position()");
         raise;
   end Speedbrake_Position;

   function Gear_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Gear_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Gear_Position()");
         raise;
   end Gear_Position;

   function Elevator_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Elevator_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Elevator_Position()");
         raise;
   end Elevator_Position;

   function Rudder_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Rudder_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Rudder_Position()");
         raise;
   end Rudder_Position;

   function Right_Aileron_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Right_Aileron_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Right_Aileron_Position()");
         raise;
   end Right_Aileron_Position;

   function Left_Aileron_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Left_Aileron_Position;
   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Left_Aileron_Position()");
         raise;
   end Left_Aileron_Position;

   procedure Update_Angle_Of_Attack
     (An_Instance : in out Instance)
   is

      Angle_Of_Attack : Float renames An_Instance.Angle_Of_Attack;
      Bank_Angle      : Float renames An_Instance.Bank_Angle;
      Wg : constant Float := 5500.0;
      V_Indicated_Airspeed    : Float renames An_Instance.V_Indicated_Airspeed;

      Cos_Phi : Float;

      V_I : Float;

      Ft_Per_Sec_To_Knots : constant Float := 0.5924838;

   begin

      Cos_Phi := Cos(Bank_Angle,360.0);
      if Cos_Phi < Cos(75.0,360.0) then Cos_Phi := Cos(75.0,360.0); end if;

      V_I := Float'Max(90.0, Ft_Per_Sec_To_Knots * V_Indicated_Airspeed);

      -- Angle_Of_Attack := ((1.6882*20.83*Wg)/(V_I**2*Cos_Phi)) - 4.035;
      Angle_Of_Attack := 550.0 / (V_I * Cos_Phi);

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Angle_Of_Attack()");
         raise;
   end Update_Angle_Of_Attack;

   procedure Update_True_Airspeed
     (An_Instance : in out Instance)
   is

      V_Indicated_Airspeed    : Float renames An_Instance.V_Indicated_Airspeed;
      V_True_Airspeed         : Float renames An_Instance.V_True_Airspeed;

      Air_Density_At_Altitude  : constant Float := Jpats_Atmosphere.Density;
      Air_Density_At_Sea_Level : constant Float := 0.002378;

   begin

      V_True_Airspeed := V_Indicated_Airspeed * Sqrt( Air_Density_At_Sea_Level / Air_Density_At_Altitude );

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_True_Airspeed()");
         raise;
   end Update_True_Airspeed;

   procedure Update_Bank_Angle
     (An_Instance          : in out Instance;
      Iconst               : in     Float;
      Roll_Rate            : in     Float;
      Bank_Angle_Commanded : in     Float)
   is

      Bank_Angle  : Float renames An_Instance.Bank_Angle;
      B_A_C       : Float := Bank_Angle_Commanded;
      R_B         : Float;
      Roll_Rate_L : Float := Roll_Rate;
      At_B_A_C    : Boolean;
      Sign        : Float;
      Epsilon     : Float;

   begin

      if Roll_Rate < 1.0 then
         Roll_Rate_L := 1.0;
      elsif Roll_Rate > 120.0 then
         Roll_Rate_L := 120.0;
      end if;

      if abs(B_A_C) < 1.0 then  -- deadband so hdg stabilizes when selected near zero
         B_A_C := 0.0;
      elsif B_A_C <= -89.8 then
         B_A_C := -89.8;
      elsif B_A_C > 89.8 then
         B_A_C := 89.8;
      end if;

      Epsilon := Roll_Rate_L * Iconst;

      At_B_A_C := abs(Bank_Angle - B_A_C) <= Epsilon;

      if not At_B_A_C then
         if abs(Bank_Angle - B_A_C) <= (3.0*Epsilon) then
            R_B := Float'Max(1.0,120.0 - Roll_Rate_L);
            Bank_Angle := -Iconst * (Bank_Angle - B_A_C) * Sqrt(1.0+(1.0/R_B)) + Bank_Angle;
         else
            Sign := -1.0*(Bank_Angle - B_A_C)/abs(Bank_Angle - B_A_C);
            Bank_Angle := Bank_Angle + Sign * Roll_Rate_L * Iconst;
         end if;
      else
         null;
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Bank_Angle()");
         raise;
   end Update_Bank_Angle;

   procedure Update_Heading
     (An_Instance : in out Instance;
      Iconst      : in     Float)
   is

      Bank_Angle              : Float renames An_Instance.Bank_Angle;
      Heading                 : Float renames An_Instance.Heading;
      V_True_Airspeed         : Float renames An_Instance.V_True_Airspeed;

      Heading_Dot             : Float := 0.0;

      G                       : constant Float := 32.1846480;

      K                       : constant Float := 1.0;
   begin

      Heading_Dot := (180.0/Pi) * K * G * Tan(Bank_Angle,360.0) / Float'Max(V_True_Airspeed, 0.1);
      Heading     := Heading_Dot * Iconst + Heading;

      if Heading < 0.0 then
         Heading := Heading + 360.0;
      elsif Heading > 360.0 then
         Heading := Heading - 360.0;
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Heading()");
         raise;
   end Update_Heading;

   procedure Update_Velocity_Components_In_Earth_Axis
     (An_Instance : in out Instance)
   is

      Pitch_Angle             : Float renames An_Instance.Pitch_Angle;
      Climb_Angle             : Float renames An_Instance.Climb_Angle;
      Bank_Angle              : Float renames An_Instance.Bank_Angle;
      Heading                 : Float renames An_Instance.Heading;
      U_Body                  : Float renames An_Instance.U_Body;
      V_Body                  : Float renames An_Instance.V_Body;
      W_Body                  : Float renames An_Instance.W_Body;
      U_Earth                 : Float renames An_Instance.U_Earth;
      V_Earth                 : Float renames An_Instance.V_Earth;
      W_Earth                 : Float renames An_Instance.W_Earth;
      W_Earth_Lp              : Float renames An_Instance.W_Earth_Lp;

      U_Earth_Wind : constant Float := 0.0;
      V_Earth_Wind : constant Float := 0.0;
      W_Earth_Wind : constant Float := 0.0;

      V_True :  Float renames An_Instance.V_True_Airspeed;

   begin

      W_Earth_Lp := W_Earth;

      U_Earth := V_True * Cos(Climb_Angle,360.0) * Cos(Heading,360.0) + U_Earth_Wind;
      V_Earth := V_True * Cos(Climb_Angle,360.0) * Sin(Heading,360.0) + V_Earth_Wind;
      W_Earth := -V_True * Sin(Climb_Angle,360.0) + W_Earth_Wind;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Velocity_Components_In_Earth_Axis()");
         raise;
   end Update_Velocity_Components_In_Earth_Axis;

   procedure Airspeed_Driver
     (An_Instance                    : in out Instance;
      Iconst                         : in     Float;
      Velocity_Gain                  : in     Float;
      V_Indicated_Airspeed_Commanded : in     Float)
   is

      Pitch_Angle             : Float renames An_Instance.Pitch_Angle;
      V_Indicated_Airspeed    : Float renames An_Instance.V_Indicated_Airspeed;
      W_Earth                 : Float renames An_Instance.W_Earth;
      W_Earth_Lp              : Float renames An_Instance.W_Earth_Lp;
      Climb_Angle             : Float renames An_Instance.Climb_Angle;
      Angle_Of_Attack         : Float renames An_Instance.Angle_Of_Attack;

      K_Roc                   : Float := 1.0;
      R_S_3                   : constant Float := 1.0;

      Velocity_Gain_L : Float := 0.0;
      V_I_A : Float := 0.0;

   begin

      Velocity_Gain_L := Sqrt(Velocity_Gain / 10.0);

      V_I_A := (V_Indicated_Airspeed_Commanded - V_Indicated_Airspeed);

      if V_I_A >= Velocity_Gain_L then
         V_I_A := Velocity_Gain_L;
      elsif V_I_A <= -Velocity_Gain_L then
         V_I_A := -Velocity_Gain_L;
      end if;

      V_Indicated_Airspeed := V_Indicated_Airspeed + Iconst * V_I_A;

      Update_Angle_Of_Attack(An_Instance);

      K_Roc := -Sqrt(1.0+(1.0/R_S_3))*Iconst;
      if V_Indicated_Airspeed /= 0.0 then
         Pitch_Angle := Pitch_Angle - (K_Roc/V_Indicated_Airspeed) * (W_Earth - W_Earth_Lp);
      else
         null;
      end if;

      Climb_Angle := Pitch_Angle - Angle_Of_Attack;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Airspeed_Driver()");
         raise;
   end Airspeed_Driver;

   procedure Pitch_Attitude_Driver
     (An_Instance           : in out Instance;
      Iconst                : in     Float;
      Pitch_Rate            : in     Float;
      Pitch_Angle_Commanded : in     Float)
   is

      Pitch_Angle             : Float renames An_Instance.Pitch_Angle;
      V_Indicated_Airspeed    : Float renames An_Instance.V_Indicated_Airspeed;
      W_Earth                 : Float renames An_Instance.W_Earth;
      W_Earth_Lp              : Float renames An_Instance.W_Earth_Lp;

      K1                      : constant Float := 1.0;

      Pitch_Rate_L : Float := Pitch_Rate;
      Sign : Float;
      R_P : Float;
      At_Target_Pitch_Angle : Boolean;
      Epsilon : Float;

   begin

      if Pitch_Rate > 30.0 then
         Pitch_Rate_L := 30.0;
      elsif Pitch_Rate < 1.0 then
         Pitch_Rate_L := 1.0;
      end if;

      Epsilon := Iconst * Pitch_Rate_L;

      At_Target_Pitch_Angle := abs(Pitch_Angle-Pitch_Angle_Commanded) <= Epsilon;

      if not At_Target_Pitch_Angle then
         if abs(Pitch_Angle-Pitch_Angle_Commanded) <= (3.0*Epsilon) then
            R_P := Float'Max(1.0,30.0 - Pitch_Rate_L);
            Pitch_Angle := -Iconst*(Sqrt(1.0+(1.0/R_P))*(Pitch_Angle-Pitch_Angle_Commanded)) + Pitch_Angle;
         else
            Sign := -1.0*(Pitch_Angle-Pitch_Angle_Commanded) / abs(Pitch_Angle-Pitch_Angle_Commanded);
            Pitch_Angle := Pitch_Angle + Sign * Iconst * Pitch_Rate_L;
         end if;
      else
         null;
      end if;

      V_Indicated_Airspeed := V_Indicated_Airspeed + K1 * (W_Earth - W_Earth_Lp);

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Pitch_Attitude_Driver()");
         raise;
   end Pitch_Attitude_Driver;

   procedure Climb_Angle_Driver
     (An_Instance           : in out Instance;
      Iconst                : in     Float;
      Climb_Angle_Rate      : in     Float;
      Climb_Angle_Commanded : in     Float)
   is

      Climb_Angle             : Float renames An_Instance.Climb_Angle;
      Angle_Of_Attack         : Float renames An_Instance.Angle_Of_Attack;
      Pitch_Angle             : Float renames An_Instance.Pitch_Angle;
      W_Earth                 : Float renames An_Instance.W_Earth;
      W_Earth_Lp              : Float renames An_Instance.W_Earth_Lp;
      V_Indicated_Airspeed    : Float renames An_Instance.V_Indicated_Airspeed;

      K1 : constant Float := 1.0;

      Epsilon : Float := 0.0;
      Climb_Angle_Rate_L : Float := 0.0;
      At_Target_Climb_Angle : Boolean;
      Sign : Float := 1.0;
      C_A_B : Float := 0.0;
      Ccc : Float := 0.0;

   begin

      Climb_Angle_Rate_L := Climb_Angle_Rate / 10.0;

      CCC := Climb_Angle_Commanded  - Climb_Angle;

      if CCC >= Climb_Angle_Rate_L then
         Ccc:=Climb_Angle_Rate_L;
      elsif Ccc <= -Climb_Angle_Rate_L then
         Ccc := -Climb_Angle_Rate_L;
      end if;

      Epsilon := Iconst * Climb_Angle_Rate_L;

      At_Target_Climb_Angle := abs(Climb_Angle - Climb_Angle_Commanded) <= Epsilon;

      Climb_Angle := Climb_Angle + Iconst * CCC;

      Update_Angle_Of_Attack(An_Instance);

      Pitch_Angle := Climb_Angle + Angle_Of_Attack;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Climb_Angle_Driver()");
         raise;
   end Climb_Angle_Driver;

   procedure Update_Position
     (An_Instance : in out Instance;
      Iconst      : in     Float)
   is

      use Lf; -- long float elementary functions

      Lat                    : Long_Float renames An_Instance.Lat;
      Long                   : Long_Float renames An_Instance.Long;
      Geometric_Altitude     : Float renames An_Instance.Geometric_Altitude;
      U_Earth                : Float renames An_Instance.U_Earth;
      V_Earth                : Float renames An_Instance.V_Earth;
      W_Earth                : Float renames An_Instance.W_Earth;

   begin

      Lat := Lat + Long_Float(Iconst * U_Earth) / 364566.0;
      if Lat < -90.0 then
         Lat := abs(Lat) - 180.0;
      elsif Lat > 90.0 then
         Lat := abs(Lat - 180.0);
      end if;

      Long := Long + Long_Float(Iconst * V_Earth) / (364566.0 * Long_Float'Max(0.00000001,Cos(Lat,360.0)));
      if Long < -180.0 then
         Long := Long - 360.0;
      elsif Long > 180.0 then
         Long := Long + 360.0;
      end if;

      Geometric_Altitude := Geometric_Altitude - Iconst * W_Earth;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Position()");
         raise;
   end Update_Position;

   procedure Update_Flaps
     (An_Instance  : in out Instance;
      Iconst       : in     Float;
      Flaps_Up     : in     Boolean;
      Flaps_Ldg    : in     Boolean)
   is

      Flap_Position : Float renames An_Instance.Flap_Position;

   begin

      if Flaps_Up then
         Flap_Position := Float'Max(0.0,Flap_Position - Iconst);
      elsif Flaps_Ldg then
         Flap_Position := Float'Min(50.0,Flap_Position + Iconst);
      else
         if Flap_Position > 23.0 then
            Flap_Position := Float'Max(23.0,Flap_Position - Iconst);
         elsif Flap_Position < 23.0 then
            Flap_Position := Float'Min(23.0,Flap_Position + Iconst);
         end if;
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Flaps()");
         raise;
   end Update_Flaps;

   procedure Update_Speedbrake
     (An_Instance         : in out Instance;
      Iconst              : in     Float;
      Speedbrake_Deployed : in     Boolean)
   is

      Speedbrake_Position : Float renames An_Instance.Speedbrake_Position;

   begin

      if Speedbrake_Deployed then
         Speedbrake_Position := Float'Min(1.0, Speedbrake_Position + Iconst);
      else
         Speedbrake_Position := Float'Max(0.0, Speedbrake_Position - Iconst);
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Speedbrake()");
         raise;
   end Update_Speedbrake;

   procedure Update_Gear
     (An_Instance : in out Instance;
      Iconst      : in     Float;
      Gear_Up     : in     Boolean)
   is

      Gear_Position : Float renames An_Instance.Gear_Position;

   begin

      if Gear_Up then
         Gear_Position := Float'Max(0.0, Gear_Position - Iconst);
      else
         Gear_Position := Float'Min(1.0, Gear_Position + Iconst);
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Gear()");
         raise;
   end Update_Gear;

   -- used in wing dip maneuvers
   --
   procedure Update_Bank_Angle
     (An_Instance           : in out Instance;
      Iconst                : in     Float;
      Left_Wing_Dip         : in     Boolean;
      Roll_Rate             : in     Float;
      Achieved_Target_Angle : in     Boolean;
      Achieved_Mark_Angle   : in     Boolean)
   is

      Bank_Angle : Float renames An_Instance.Bank_Angle;

      Roll_Rate_L : Float := Roll_Rate;
      procedure Roll_Left
        (Roll_Rate : Float)
      is
      begin

         Bank_Angle := Bank_Angle - Roll_Rate * Iconst;

      end Roll_Left;

      procedure Roll_Right
        (Roll_Rate : Float)
      is
      begin

         Bank_Angle := Bank_Angle + Roll_Rate * Iconst;

      end Roll_Right;

   begin

      if Roll_Rate > 120.0 then
         Roll_Rate_L := 120.0;
      elsif Roll_Rate_L < 1.0 then
         Roll_Rate_L := 1.0;
      end if;

      if Left_Wing_Dip  then
         if not Achieved_Target_Angle then
            Roll_Left(Roll_Rate_L);
         elsif not Achieved_Mark_Angle then
            Roll_Right(Roll_Rate_L);
         end if;
      else
         if not Achieved_Target_Angle then
            Roll_Right(Roll_Rate_L);
         elsif not Achieved_Mark_Angle then
            Roll_Left(Roll_Rate_L);
         end if;
      end if;

      if Bank_Angle < -180.0 then
         Bank_Angle := Bank_Angle + 360.0;
      elsif Bank_Angle > 180.0 then
         Bank_Angle := Bank_Angle - 360.0;
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Bank_Angle() -- Wing dip");
         raise;
   end Update_Bank_Angle;

   -- used in wing rock maneuvers
   --
   procedure Update_Bank_Angle
     (An_Instance           : in out Instance;
      Iconst                : in     Float;
      Rock_Start            : in     Rock_Direction_Type;
      Roll_Rate             : in     Float;
      Achieved_First_Angle  : in     Boolean;
      Achieved_Second_Angle : in     Boolean;
      Achieved_Third_Angle  : in     Boolean;
      Achieved_Fourth_Angle : in     Boolean;
      Achieved_Mark_Angle   : in     Boolean)
   is

      Bank_Angle : Float renames An_Instance.Bank_Angle;


      Roll_Rate_L : Float := Roll_Rate;
      procedure Roll_Left
        (Roll_Rate : Float)
      is
      begin

         Bank_Angle := Bank_Angle - Roll_Rate * Iconst;

      end Roll_Left;

      procedure Roll_Right
        (Roll_Rate : Float)
      is
      begin

         Bank_Angle := Bank_Angle + Roll_Rate * Iconst;

      end Roll_Right;

   begin

      if Roll_Rate > 120.0 then
         Roll_Rate_L := 120.0;
      elsif Roll_Rate < 1.0 then
         Roll_Rate_L := 1.0;
      end if;

      if Rock_Start = Up then

         if not Achieved_First_Angle then
            Roll_Right(Roll_Rate_L);
         elsif not Achieved_Second_Angle then
            Roll_Left(Roll_Rate_L);
         elsif not Achieved_Third_Angle then
            Roll_Right(Roll_Rate_L);
         elsif not Achieved_Fourth_Angle then
            Roll_Left(Roll_Rate_L);
         elsif not Achieved_Mark_Angle then
            Roll_Right(Roll_Rate_L);
         end if;

      else

         if not Achieved_First_Angle then
            Roll_Left(Roll_Rate_L);
         elsif not Achieved_Second_Angle then
            Roll_Right(Roll_Rate_L);
         elsif not Achieved_Third_Angle then
            Roll_Left(Roll_Rate_L);
         elsif not Achieved_Fourth_Angle then
            Roll_Right(Roll_Rate_L);
         elsif not Achieved_Mark_Angle then
            Roll_Left(Roll_Rate_L);
         end if;

      end if;

      if Bank_Angle < -180.0 then
         Bank_Angle := Bank_Angle + 360.0;
      elsif Bank_Angle > 180.0 then
         Bank_Angle := Bank_Angle - 360.0;
      end if;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Bank_Angle() -- wing rock");
         raise;
   end Update_Bank_Angle;

   -- used in porpoise maneuver
   --
   procedure Update_Pitch_Attitude
     (An_Instance           : in out Instance;
      Iconst                : in     Float;
      Pitch_Rate            : in     Float;
      Achieved_First_Angle  : in     Boolean;
      Achieved_Second_Angle : in     Boolean;
      Achieved_Third_Angle  : in     Boolean;
      Achieved_Fourth_Angle : in     Boolean;
      Achieved_Mark_Angle   : in     Boolean)
   is

      Pitch_Angle : Float renames An_Instance.Pitch_Angle;

      Pitch_Rate_L : Float := Pitch_Rate;
      procedure Pitch_Up
        (Pitch_Rate : Float)
      is
      begin

         Pitch_Angle := Pitch_Angle + Pitch_Rate * Iconst;

      end Pitch_Up;

      procedure Pitch_Down
        (Pitch_Rate : Float)
      is
      begin

         Pitch_Angle := Pitch_Angle - Pitch_Rate * Iconst;

      end Pitch_Down;

   begin

      if Pitch_Rate > 30.0 then
         Pitch_Rate_L := 30.0;
      elsif Pitch_Rate_L < 1.0 then
         Pitch_Rate_L := 1.0;
      end if;

      if not Achieved_First_Angle then
         Pitch_Up(Pitch_Rate_L);
      elsif not Achieved_Second_Angle then
         Pitch_Down(Pitch_Rate_L);
      elsif not Achieved_Third_Angle then
         Pitch_Up(Pitch_Rate_L);
      elsif not Achieved_Fourth_Angle then
         Pitch_Down(Pitch_Rate_L);
      elsif not Achieved_Mark_Angle then
         Pitch_Up(Pitch_Rate_L);
      end if;

      if Pitch_Angle < -180.0 then
         Pitch_Angle := Pitch_Angle + 360.0;
      elsif Pitch_Angle > 180.0 then
         Pitch_Angle := Pitch_Angle - 360.0;
      end if;

      -- want airspeed constant during maneuver so not updating
      -- from last airspeed prior to maneuver

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Pitch_Attitude()");
         raise;
   end Update_Pitch_Attitude;

   procedure Update_Elevator
     (An_Instance  : in out Instance)
   is
   begin

      An_Instance.Elevator_Position := 0.0;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Elevator()");
         raise;
   end Update_Elevator;

   procedure Update_Rudder
     (An_Instance  : in out Instance)
   is
   begin

      An_Instance.Rudder_Position := 0.0;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Rudder()");
         raise;
   end Update_Rudder;

   procedure Update_Right_Aileron
     (An_Instance  : in out Instance)
   is
   begin

      An_Instance.Right_Aileron_Position := 0.0;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Right_Aileron()");
         raise;
   end Update_Right_Aileron;

   procedure Update_Left_Aileron
     (An_Instance  : in out Instance)
   is
   begin

      An_Instance.Left_Aileron_Position := 0.0;

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.Update_Left_Aileron()");
         raise;
   end Update_Left_Aileron;

   procedure Initialize

     (An_Instance                 : in out Instance;
      Pitch_Angle_Init            : in     Float;
      Bank_Angle_Init             : in     Float;
      Heading_Init                : in     Float;
      Lat_Init                    : in     Long_Float;
      Long_Init                   : in     Long_Float;
      Geometric_Altitude_Init     : in     Float;
      V_Indicated_Airspeed_Init   : in     Float;
      Gear_Position_Init          : in     Float;
      Speedbrake_Position_Init    : in     Float;
      Flap_Position_Init          : in     Float;
      Elevator_Position_Init      : in     Float;
      Rudder_Position_Init        : in     Float;
      Right_Aileron_Position_Init : in     Float;
      Left_Aileron_Position_Init  : in     Float)

   is

      Pitch_Angle            : Float renames An_Instance.Pitch_Angle;
      Bank_Angle             : Float renames An_Instance.Bank_Angle;
      Heading                : Float renames An_Instance.Heading;
      Lat                    : Long_Float renames An_Instance.Lat;
      Long                   : Long_Float renames An_Instance.Long;
      Geometric_Altitude     : Float renames An_Instance.Geometric_Altitude;
      V_Indicated_Airspeed   : Float renames An_Instance.V_Indicated_Airspeed;
      Gear_Position          : Float renames An_Instance.Gear_Position;
      Speedbrake_Position    : Float renames An_Instance.Speedbrake_Position;
      Flap_Position          : Float renames An_Instance.Flap_Position;
      Elevator_Position      : Float renames An_Instance.Elevator_Position;
      Rudder_Position        : Float renames An_Instance.Rudder_Position;
      Right_Aileron_Position : Float renames An_Instance.Right_Aileron_Position;
      Left_Aileron_Position  : Float renames An_Instance.Left_Aileron_Position;
      U_Body                 : Float renames An_Instance.U_Body;
      V_Body                 : Float renames An_Instance.V_Body;
      W_Body                 : Float renames An_Instance.W_Body;
      W_Earth                : Float renames An_Instance.W_Earth;
      W_Earth_Lp             : Float renames An_Instance.W_Earth_Lp;


      Air_Density_At_Altitude        : constant Float := Jpats_Atmosphere.Density;
      Air_Density_At_Sea_Level       : constant Float := 0.002378;
      W_Earth_Wind                   : constant Float := 0.0;

      Knots_To_Ft_Per_Sec : constant Float := 1.6878098571;

   begin

      Pitch_Angle          := Pitch_Angle_Init;
      Bank_Angle           := Bank_Angle_Init;
      Heading              := Heading_Init;
      Lat                  := Lat_Init;
      Long                 := Long_Init;
      Geometric_Altitude   := Geometric_Altitude_Init;

      if V_Indicated_Airspeed_Init < 100.0 then
         V_Indicated_Airspeed := 100.0 * Knots_To_Ft_Per_Sec;
      else
         V_Indicated_Airspeed := V_Indicated_Airspeed_Init * Knots_To_Ft_Per_Sec;
      end if;

      Gear_Position          := Gear_Position_Init;
      Speedbrake_Position    := Speedbrake_Position_Init;
      Flap_Position          := Flap_Position_Init;
      Elevator_Position      := Elevator_Position_Init;
      Rudder_Position        := Rudder_Position_Init;
      Right_Aileron_Position := Right_Aileron_Position_Init;
      Left_Aileron_Position  := Left_Aileron_Position_Init;

      Update_True_Airspeed (An_Instance);

      Update_Angle_Of_Attack (An_Instance);

      -- Calculate W_Earth for setting W_Earth_Lp in Update_Velocity_Components_In_Earth_Axis()
      W_Earth := (-Sin(Pitch_Angle,360.0)*U_Body +
                  Sin(Bank_Angle,360.0)*Cos(Pitch_Angle,360.0)*V_Body +
                  Cos(Bank_Angle,360.0)*Cos(Pitch_Angle,360.0)*W_Body +
                  W_Earth_Wind);
      Update_Velocity_Components_In_Earth_Axis (An_Instance);


      Random_Rock_Direction.Reset(RRD_Gen);

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.initialize()");
         raise;
   end Initialize;

   procedure Update
     (An_Instance                    : in out Instance;
      Iconst                         : in     Float;
      Pitch_Attitude_Driver_Selected : in     Boolean := False;
      Airspeed_Driver_Selected       : in     Boolean := True;
      Pitch_Angle_Commanded          : in     Float := 0.0;
      Pitch_Rate                     : in     Float := 1.0;
      V_Indicated_Airspeed_Commanded : in     Float := 0.0;
      Velocity_Gain                  : in     Float := 0.0;
      Air_Density_At_Altitude        : in     Float := 0.0;
      Air_Density_At_Sea_Level       : in     Float := 0.0;
      Bank_Angle_Commanded           : in     Float := 0.0;
      Roll_Rate                      : in     Float := 1.0;
      Wing_Dip_Commanded             : in out Boolean;
      Porpoise_Commanded             : in out Boolean;
      Big_Wing_Rock_Commanded        : in out Boolean;
      Small_Wing_Rock_Commanded      : in out Boolean;
      Flaps_Up                       : in     Boolean := True;
      Flaps_Ldg                      : in     Boolean := False;
      Speedbrake_Deployed            : in     Boolean := False;
      Gear_Up                        : in     Boolean := False;
      Wingman_Lat                    : in     Long_Float := 0.0;
      Wingman_Long                   : in     Long_Float := 0.0)
   is

      use Lf;

      State       : State_Type renames An_Instance.State;
      State_Lp    : State_Type renames An_Instance.State_Lp;
      Bank_Angle  : Float      renames An_Instance.Bank_Angle;
      Pitch_Angle : Float      renames An_Instance.Pitch_Angle;
      Lat         : Long_Float renames An_Instance.Lat;
      Long        : Long_Float renames An_Instance.Long;
      Heading     : Float      renames An_Instance.Heading;

      -- knots -> ft/s
      Knots_To_Ft_Per_Sec : constant Float := 1.6878098571;
      VIAS_Commanded : constant Float := V_Indicated_Airspeed_Commanded * Knots_To_Ft_Per_Sec;

      Ft_Per_Deg : constant Long_Float := 364566.0;

      Wing_Dip_Complete        : Boolean    renames An_Instance.Wing_Dip_Complete;
      Porpoise_Complete        : Boolean    renames An_Instance.Porpoise_Complete;
      Big_Wing_Rock_Complete   : Boolean    renames An_Instance.Big_Wing_Rock_Complete;
      Small_Wing_Rock_Complete : Boolean    renames An_Instance.Small_Wing_Rock_Complete;
      Last_Driver              : State_Type renames An_Instance.Last_Driver;
      Bank_Angle_Mark          : Float      renames An_Instance.Bank_Angle_Mark;
      Pitch_Angle_Mark         : Float      renames An_Instance.Pitch_Angle_Mark;
      Left_Wing_Dip            : Boolean    renames An_Instance.Left_Wing_Dip;
      Wing_Lead_Distance       : Long_Float renames An_Instance.Wing_Lead_Distance;
      Wing_Dip_Angle           : Float      renames An_Instance.Wing_Dip_Angle;
      Target_Angle             : Float      renames An_Instance.Target_Angle;
      Achieved_Target_Angle    : Boolean    renames An_Instance.Achieved_Target_Angle;

      Achieved_Mark_Angle      : Boolean renames An_Instance.Achieved_Mark_Angle;

      Achieved_First_Angle     : Boolean   renames An_Instance.Achieved_First_Angle;
      Achieved_Second_Angle    : Boolean   renames An_Instance.Achieved_Second_Angle;
      Achieved_Third_Angle     : Boolean   renames An_Instance.Achieved_Third_Angle;
      Achieved_Fourth_Angle    : Boolean   renames An_Instance.Achieved_Fourth_Angle;
      Top_Angle                : Float     renames An_Instance.Top_Angle;
      Bottom_Angle             : Float     renames An_Instance.Bottom_Angle;
      Rock_Start               : Rock_Direction_Type renames An_Instance.Rock_Start;

      Lower_Nose_Angle         : Float renames An_Instance.Lower_Nose_Angle;
      Upper_Nose_Angle         : Float renames An_Instance.Upper_Nose_Angle;


      procedure Roll_Left
        (An_Instance : in out Instance;
         Iconst      : in     Float;
         Rr          : in     Float;
         Target      : in     Float)
      is
         RRt : Float := Rr;
      begin
         if RRt > 120.0 then
            RRt := 120.0;
         elsif RRt < 1.0 then
            RRt := 1.0;
         end if;
         Bank_Angle := Bank_Angle - Rrt * Iconst;
         if Bank_Angle < -180.0 then
            Bank_Angle := Bank_Angle + 360.0;
         end if;
         if abs(Target - Bank_Angle) < (Rrt * Iconst) then
            Bank_Angle := Target;
         end if;
      exception
         when others =>
            Log.Report("Formation_Lead_Kinematic.update.roll_left()");
            raise;
      end Roll_Left;

      procedure Roll_Right
        (An_Instance : in out Instance;
         Iconst      : in     Float;
         Rr          : in     Float;
         Target      : in     Float)
      is
         RRt : Float := Rr;
      begin
         if RRt > 120.0 then
            RRt := 120.0;
         elsif RRt < 1.0 then
            RRt := 1.0;
         end if;
         Bank_Angle := Bank_Angle + Rrt * Iconst;
         if Bank_Angle > 180.0 then
            Bank_Angle := Bank_Angle - 360.0;
         end if;
         if abs(Target - Bank_Angle) < (Rrt * Iconst) then
            Bank_Angle := Target;
         end if;
      exception
         when others =>
            Log.Report("Formation_Lead_Kinematic.update.roll_right()");
            raise;
      end Roll_Right;

   begin

      Update_Elevator (An_Instance);
      Update_Rudder (An_Instance);
      Update_Right_Aileron (An_Instance);
      Update_Left_Aileron (An_Instance);

      case State is

         when Initializing =>

            State_Lp := Initializing;
            if Wing_Dip_Commanded             then State := Wing_Dip;              end if;
            if Porpoise_Commanded             then State := Porpoise;              end if;
            if Big_Wing_Rock_Commanded        then State := Big_Wing_Rock;         end if;
            if Small_Wing_Rock_Commanded      then State := Small_Wing_Rock;       end if;
            if Airspeed_Driver_Selected       then State := Airspeed_Driver;       end if;
            if Pitch_Attitude_Driver_Selected then State := Pitch_Attitude_Driver; end if;

            State := Airspeed_Driver;

         when Wing_Dip =>

            if State /= State_Lp then

               Wing_Dip_Complete := False;

               Bank_Angle_Mark := Bank_Angle;

               declare

                  Lead_Hdg : Long_Float := Long_Float(90.0 - Heading);

                  Q1, Q2, Q3, Q4 : Boolean := False;

                  X1 : constant Long_Float := Wingman_Long - Long;
                  Y1 : constant Long_Float := Wingman_Lat - Lat;

                  Xroot : Long_Float := 0.0;

               begin

                  if Lead_Hdg < 0.0 then
                     Lead_Hdg := Lead_Hdg + 360.0;
                  elsif Lead_Hdg > 360.0 then
                     Lead_Hdg := Lead_Hdg - 360.0;
                  end if;

                  Q1 := Lead_Hdg < 90.0  and Lead_Hdg > 0.0;
                  Q2 := Lead_Hdg < 180.0 and Lead_Hdg > 90.0;
                  Q3 := Lead_Hdg < 270.0 and Lead_Hdg > 180.0;
                  Q4 := Lead_Hdg < 360.0 and Lead_Hdg > 270.0;

                  if Q1 then
                     Xroot := (Tan(Lead_Hdg,360.0)*X1 - Y1) / Tan(Lead_Hdg,360.0);
                     Left_Wing_Dip := Xroot >= 0.0;
                  elsif Q2 then
                     Xroot := (Tan(Lead_Hdg,360.0)*X1 - Y1) / Tan(Lead_Hdg,360.0);
                     Left_Wing_Dip := Xroot >= 0.0;
                  elsif Q3 then
                     Xroot := (Tan(Lead_Hdg,360.0)*X1 - Y1) / Tan(Lead_Hdg,360.0);
                     Left_Wing_Dip := Xroot <= 0.0;
                  elsif Q4 then
                     Xroot := (Tan(Lead_Hdg,360.0)*X1 - Y1) / Tan(Lead_Hdg,360.0);
                     Left_Wing_Dip := Xroot <= 0.0;
                  elsif Lead_Hdg = 0.0 then
                     Left_Wing_Dip := Wingman_Lat <= Lat;
                  elsif Lead_Hdg = 360.0 then
                     Left_Wing_Dip := Wingman_Lat <= Lat;
                  elsif Lead_Hdg = 90.0 then
                     Left_Wing_Dip := Wingman_Long >= Long;
                  elsif Lead_Hdg = 180.0 then
                     Left_Wing_Dip := Wingman_Lat >= Lat;
                  elsif Lead_Hdg = 270.0 then
                     Left_Wing_Dip := Wingman_Long <= Long;
                  else
                     Left_Wing_Dip := True;
                  end if;

               end;

               Wing_Lead_Distance := Sqrt (((Wingman_Long - Long)*Ft_Per_Deg*Cos(Lat,360.0))**2 +
                                           ((Wingman_Lat - Lat)*Ft_Per_Deg)**2);

               if Wing_Lead_Distance < 500.0 then
                  Wing_Dip_Angle := 15.0 + 30.0 * Float(Wing_Lead_Distance) / 500.0;
               elsif Wing_Lead_Distance < 5000.0 then
                  Wing_Dip_Angle := 45.0 + 45.0 * Float(Wing_Lead_Distance)  / 5000.0;
               else
                  Wing_Dip_Angle := 90.0;
               end if;

               if Left_Wing_Dip then
                  Target_Angle := Bank_Angle - Wing_Dip_Angle;
               else
                  Target_Angle := Bank_Angle + Wing_Dip_Angle;
               end if;

               if not Left_Wing_Dip and Target_Angle > 180.0 then
                  Target_Angle := Target_Angle - 360.0;
               elsif Left_Wing_Dip and Target_Angle < -180.0 then
                  Target_Angle := Target_Angle + 360.0;
               end if;

               Achieved_Target_Angle := False;
               Achieved_Mark_Angle := False;

            end if;

            if Last_Driver = Airspeed_Driver then
               Airspeed_Driver (An_Instance, Iconst, Velocity_Gain, VIAS_Commanded);
            else
               Climb_Angle_Driver  (An_Instance, Iconst, Pitch_Rate, Pitch_Angle_Commanded);
            end if;

            Update_True_Airspeed (An_Instance);

            if not Achieved_Target_Angle then

               if Left_Wing_Dip then
                  Roll_Left(An_Instance,Iconst,Roll_Rate,Target_Angle);
                  Achieved_Target_Angle := Bank_Angle <= Target_Angle; -- and Target_Angle/Bank_Angle >= 0.0;
               else
                  Roll_Right(An_Instance,Iconst,Roll_Rate,Target_Angle);
                  Achieved_Target_Angle := Bank_Angle >= Target_Angle; -- and Target_Angle/Bank_Angle >= 0.0;
               end if;

            elsif not Achieved_Mark_Angle then

               if Left_Wing_Dip then
                  Roll_Right(An_Instance,Iconst,Roll_Rate,Bank_Angle_Mark);
                  Achieved_Mark_Angle := Bank_Angle >= Bank_Angle_Mark; -- and Bank_Angle_Mark/Bank_Angle >= 0.0;
               else
                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bank_Angle_Mark);
                  Achieved_Mark_Angle := Bank_Angle <= Bank_Angle_Mark; -- and Bank_Angle_Mark/Bank_Angle >= 0.0;
               end if;

               Wing_Dip_Complete := Achieved_Mark_Angle;

            else
               Update_Bank_Angle (An_Instance, Iconst, Roll_Rate, Bank_Angle_Commanded);
            end if;

            Update_Velocity_Components_In_Earth_Axis (An_Instance);
            Update_Position (An_Instance, Iconst);
            Update_Gear (An_Instance, Iconst, Gear_Up);
            Update_Speedbrake (An_Instance, Iconst, Speedbrake_Deployed);
            Update_Flaps (An_Instance, Iconst, Flaps_Up, Flaps_Ldg);

            State_Lp := Wing_Dip;
            if Wing_Dip_Complete              then Wing_Dip_Commanded := False; State := Last_Driver;                                      end if;
            if Porpoise_Commanded             then Wing_Dip_Commanded := False; Wing_Dip_Complete := True; State := Porpoise;              end if;
            if Big_Wing_Rock_Commanded        then Wing_Dip_Commanded := False; Wing_Dip_Complete := True; State := Big_Wing_Rock;         end if;
            if Small_Wing_Rock_Commanded      then Wing_Dip_Commanded := False; Wing_Dip_Complete := True; State := Small_Wing_Rock;       end if;
            if Airspeed_Driver_Selected       then Wing_Dip_Commanded := False; Wing_Dip_Complete := True; State := Airspeed_Driver;       end if;
            if Pitch_Attitude_Driver_Selected then Wing_Dip_Commanded := False; Wing_Dip_Complete := True; State := Pitch_Attitude_Driver; end if;

         when Porpoise =>

            if State /= State_Lp then

               Porpoise_Complete := False;

               Pitch_Angle_Mark := Pitch_Angle;

               Wing_Lead_Distance := Sqrt (((Wingman_Long - Long)*Ft_Per_Deg*Cos(Lat,360.0))**2 +
                                           ((Wingman_Lat - Lat)*Ft_Per_Deg)**2);

               if Wing_Lead_Distance < 500.0 then
                  Lower_Nose_Angle := Pitch_Angle_Mark - 2.0 * Float(Wing_Lead_Distance) / 500.0;
                  Upper_Nose_Angle := Pitch_Angle_Mark + 5.0 + 2.5 * Float(Wing_Lead_Distance) / 500.0;
               elsif Wing_Lead_Distance < 5000.0 then
                  Lower_Nose_Angle := Pitch_Angle_Mark - 2.0 - 3.0 * Float(Wing_Lead_Distance) / 5000.0;
                  Upper_Nose_Angle := Pitch_Angle_Mark + 7.5 + 7.5 * Float(Wing_Lead_Distance) / 5000.0;
               else
                  Lower_Nose_Angle := Pitch_Angle_Mark - 5.0;
                  Upper_Nose_Angle := Pitch_Angle_Mark + 15.0;
               end if;

               if Lower_Nose_Angle < -180.0 then
                  Lower_Nose_Angle := Lower_Nose_Angle + 360.0;
               elsif Lower_Nose_Angle > 180.0 then
                  Lower_Nose_Angle := Lower_Nose_Angle - 360.0;
               end if;

               if Upper_Nose_Angle < -180.0 then
                  Upper_Nose_Angle := Upper_Nose_Angle + 360.0;
               elsif Upper_Nose_Angle > 180.0 then
                  Upper_Nose_Angle := Upper_Nose_Angle - 360.0;
               end if;

               Achieved_First_Angle  := False;
               Achieved_Second_Angle := False;
               Achieved_Third_Angle  := False;
               Achieved_Fourth_Angle := False;
               Achieved_Mark_Angle   := False;

            end if;

            if not Achieved_First_Angle then

               Update_Pitch_Attitude (An_Instance,
                                      Iconst,
                                      Pitch_Rate,
                                      Achieved_First_Angle,
                                      Achieved_Second_Angle,
                                      Achieved_Third_Angle,
                                      Achieved_Fourth_Angle,
                                      Achieved_Mark_Angle);

               Achieved_First_Angle := Pitch_Angle >= Upper_Nose_Angle and Pitch_Angle / Upper_Nose_Angle > 0.0;
            elsif not Achieved_Second_Angle then

               Update_Pitch_Attitude (An_Instance,
                                      Iconst,
                                      Pitch_Rate,
                                      Achieved_First_Angle,
                                      Achieved_Second_Angle,
                                      Achieved_Third_Angle,
                                      Achieved_Fourth_Angle,
                                      Achieved_Mark_Angle);

               Achieved_Second_Angle := Pitch_Angle <= Lower_Nose_Angle and Pitch_Angle / Lower_Nose_Angle > 0.0;
            elsif not Achieved_Third_Angle then

               Update_Pitch_Attitude (An_Instance,
                                      Iconst,
                                      Pitch_Rate,
                                      Achieved_First_Angle,
                                      Achieved_Second_Angle,
                                      Achieved_Third_Angle,
                                      Achieved_Fourth_Angle,
                                      Achieved_Mark_Angle);

               Achieved_Third_Angle :=  Pitch_Angle >= Upper_Nose_Angle and Pitch_Angle / Upper_Nose_Angle > 0.0;
            elsif not Achieved_Fourth_Angle then

               Update_Pitch_Attitude (An_Instance,
                                      Iconst,
                                      Pitch_Rate,
                                      Achieved_First_Angle,
                                      Achieved_Second_Angle,
                                      Achieved_Third_Angle,
                                      Achieved_Fourth_Angle,
                                      Achieved_Mark_Angle);

               Achieved_Fourth_Angle := Pitch_Angle <= Lower_Nose_Angle and Pitch_Angle / Lower_Nose_Angle > 0.0;
            elsif not Achieved_Mark_Angle then

               Update_Pitch_Attitude (An_Instance,
                                      Iconst,
                                      Pitch_Rate,
                                      Achieved_First_Angle,
                                      Achieved_Second_Angle,
                                      Achieved_Third_Angle,
                                      Achieved_Fourth_Angle,
                                      Achieved_Mark_Angle);

               Achieved_Mark_Angle := Pitch_Angle >= Pitch_Angle_Mark and Pitch_Angle / Pitch_Angle_Mark > 0.0;
               Porpoise_Complete := Achieved_Mark_Angle;
            else
               Climb_Angle_Driver  (An_Instance, Iconst, Pitch_Rate, Pitch_Angle_Commanded);
            end if;

            Update_True_Airspeed (An_Instance);
            Update_Bank_Angle (An_Instance, Iconst, Roll_Rate, Bank_Angle_Commanded);
            Update_Heading (An_Instance, Iconst);
            Update_Velocity_Components_In_Earth_Axis (An_Instance);
            Update_Position (An_Instance, Iconst);
            Update_Gear (An_Instance, Iconst, Gear_Up);
            Update_Speedbrake (An_Instance, Iconst, Speedbrake_Deployed);
            Update_Flaps (An_Instance, Iconst, Flaps_Up, Flaps_Ldg);

            State_Lp := Porpoise;
            if Porpoise_Complete              then Porpoise_Commanded := False; State := Last_Driver;                                      end if;
            if Wing_Dip_Commanded             then Porpoise_Commanded := False; Porpoise_Complete := True; State := Wing_Dip;              end if;
            if Big_Wing_Rock_Commanded        then Porpoise_Commanded := False; Porpoise_Complete := True; State := Big_Wing_Rock;         end if;
            if Small_Wing_Rock_Commanded      then Porpoise_Commanded := False; Porpoise_Complete := True; State := Small_Wing_Rock;       end if;
            if Airspeed_Driver_Selected       then Porpoise_Commanded := False; Porpoise_Complete := True; State := Airspeed_Driver;       end if;
            if Pitch_Attitude_Driver_Selected then Porpoise_Commanded := False; Porpoise_Complete := True; State := Pitch_Attitude_Driver; end if;

         when Big_Wing_Rock =>

            if State /= State_Lp then

               Big_Wing_Rock_Complete := False;

               Bank_Angle_Mark := Bank_Angle;

               Top_Angle    := Bank_Angle_Mark + 60.0;
               Bottom_Angle := Bank_Angle_Mark - 60.0;

               if Top_Angle > 180.0 then
                  Top_Angle := Top_Angle - 360.0;
               elsif Top_Angle < -180.0 then
                  Top_Angle := Top_Angle + 360.0;
               end if;

               if Bottom_Angle > 180.0 then
                  Bottom_Angle := Bottom_Angle - 360.0;
               elsif Bottom_Angle < -180.0 then
                  Bottom_Angle := Bottom_Angle + 360.0;
               end if;

               Rock_Start := Random_Rock_Direction.Random(RRD_Gen);

               Achieved_First_Angle  := False;
               Achieved_Second_Angle := False;
               Achieved_Third_Angle  := False;
               Achieved_Fourth_Angle := False;
               Achieved_Mark_Angle   := False;

            end if;

            if Last_Driver = Airspeed_Driver then
               Airspeed_Driver (An_Instance, Iconst, Velocity_Gain, VIAS_Commanded);
            else
               Climb_Angle_Driver  (An_Instance, Iconst, Pitch_Rate, Pitch_Angle_Commanded);
            end if;

            Update_True_Airspeed (An_Instance);

            if not Achieved_First_Angle then

               if Rock_Start = Up then

                  Roll_Right(An_Instance,Iconst,Roll_Rate,Top_Angle);
                  Achieved_First_Angle := Bank_Angle >= Top_Angle;

               else

                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bottom_Angle);
                  Achieved_First_Angle := Bank_Angle <= Bottom_Angle;

               end if;

            elsif not Achieved_Second_Angle then

               if Rock_Start = Up then

                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bottom_Angle);
                  Achieved_Second_Angle := Bank_Angle <= Bottom_Angle;

               else

                  Roll_Right(An_Instance,Iconst,Roll_Rate,Top_Angle);
                  Achieved_Second_Angle := Bank_Angle >= Top_Angle;

               end if;

            elsif not Achieved_Third_Angle then

               if Rock_Start = Up then

                  Roll_Right(An_Instance,Iconst,Roll_Rate,Top_Angle);
                  Achieved_Third_Angle := Bank_Angle >= Top_Angle;

               else

                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bottom_Angle);
                  Achieved_Third_Angle := Bank_Angle <= Bottom_Angle;

               end if;

               Big_Wing_Rock_Complete := Achieved_Third_Angle;

            else
               Update_Bank_Angle (An_Instance, Iconst, Roll_Rate, Bank_Angle_Commanded);
            end if;

            Update_Velocity_Components_In_Earth_Axis (An_Instance);
            Update_Position (An_Instance, Iconst);
            Update_Gear (An_Instance, Iconst, Gear_Up);
            Update_Speedbrake (An_Instance, Iconst, Speedbrake_Deployed);
            Update_Flaps (An_Instance, Iconst, Flaps_Up, Flaps_Ldg);

            State_Lp := Big_Wing_Rock;
            if Big_Wing_Rock_Complete         then Big_Wing_Rock_Commanded := False; State := Last_Driver;                                           end if;
            if Wing_Dip_Commanded             then Big_Wing_Rock_Commanded := False; Big_Wing_Rock_Complete := True; State := Wing_Dip;              end if;
            if Porpoise_Commanded             then Big_Wing_Rock_Commanded := False; Big_Wing_Rock_Complete := True; State := Porpoise;              end if;
            if Small_Wing_Rock_Commanded      then Big_Wing_Rock_Commanded := False; Big_Wing_Rock_Complete := True; State := Small_Wing_Rock;       end if;
            if Airspeed_Driver_Selected       then Big_Wing_Rock_Commanded := False; Big_Wing_Rock_Complete := True; State := Airspeed_Driver;       end if;
            if Pitch_Attitude_Driver_Selected then Big_Wing_Rock_Commanded := False; Big_Wing_Rock_Complete := True; State := Pitch_Attitude_Driver; end if;

         when Small_Wing_Rock =>

            if State /= State_Lp then

               Small_Wing_Rock_Complete := False;

               Bank_Angle_Mark := Bank_Angle;

               Top_Angle    := Bank_Angle_Mark + 15.0;
               Bottom_Angle := Bank_Angle_Mark - 15.0;

               if Top_Angle > 180.0 then
                  Top_Angle := Top_Angle - 360.0;
               elsif Top_Angle < -180.0 then
                  Top_Angle := Top_Angle + 360.0;
               end if;

               if Bottom_Angle > 180.0 then
                  Bottom_Angle := Bottom_Angle - 360.0;
               elsif Bottom_Angle < -180.0 then
                  Bottom_Angle := Bottom_Angle + 360.0;
               end if;

               Rock_Start := Random_Rock_Direction.Random(RRD_Gen);

               Achieved_First_Angle  := False;
               Achieved_Second_Angle := False;
               Achieved_Third_Angle  := False;
               Achieved_Fourth_Angle := False;
               Achieved_Mark_Angle   := False;

            end if;

            if Last_Driver = Airspeed_Driver then
               Airspeed_Driver (An_Instance, Iconst, Velocity_Gain, VIAS_Commanded);
            else
               Climb_Angle_Driver (An_Instance, Iconst, Pitch_Rate, Pitch_Angle_Commanded);
            end if;

            Update_True_Airspeed (An_Instance);

            if not Achieved_First_Angle then

               if Rock_Start = Up then

                  Roll_Right(An_Instance,Iconst,Roll_Rate,Top_Angle);
                  Achieved_First_Angle := Bank_Angle >= Top_Angle;

               else

                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bottom_Angle);
                  Achieved_First_Angle := Bank_Angle <= Bottom_Angle;

               end if;

            elsif not Achieved_Second_Angle then

               if Rock_Start = Up then

                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bottom_Angle);
                  Achieved_Second_Angle := Bank_Angle <= Bottom_Angle;

               else

                  Roll_Right(An_Instance,Iconst,Roll_Rate,Top_Angle);
                  Achieved_Second_Angle := Bank_Angle >= Top_Angle;

               end if;

            elsif not Achieved_Third_Angle then

               if Rock_Start = Up then

                  Roll_Right(An_Instance,Iconst,Roll_Rate,Bank_Angle_Mark);
                  Achieved_Third_Angle := Bank_Angle >= Bank_Angle_Mark;

               else

                  Roll_Left(An_Instance,Iconst,Roll_Rate,Bank_Angle_Mark);
                  Achieved_Third_Angle := Bank_Angle <= Bank_Angle_Mark;

               end if;

               Small_Wing_Rock_Complete := Achieved_Third_Angle;

            else
               Update_Bank_Angle (An_Instance, Iconst, Roll_Rate, Bank_Angle_Commanded);
            end if;

            Update_Velocity_Components_In_Earth_Axis (An_Instance);
            Update_Position (An_Instance, Iconst);
            Update_Gear (An_Instance, Iconst, Gear_Up);
            Update_Speedbrake (An_Instance, Iconst, Speedbrake_Deployed);
            Update_Flaps (An_Instance, Iconst, Flaps_Up, Flaps_Ldg);

            State_Lp := Small_Wing_Rock;
            if Small_Wing_Rock_Complete       then Small_Wing_Rock_Commanded := False; State := Last_Driver;                                             end if;
            if Wing_Dip_Commanded             then Small_Wing_Rock_Commanded := False; Small_Wing_Rock_Complete := True; State := Wing_Dip;              end if;
            if Porpoise_Commanded             then Small_Wing_Rock_Commanded := False; Small_Wing_Rock_Complete := True; State := Porpoise;              end if;
            if Big_Wing_Rock_Commanded        then Small_Wing_Rock_Commanded := False; Small_Wing_Rock_Complete := True; State := Big_Wing_Rock;         end if;
            if Airspeed_Driver_Selected       then Small_Wing_Rock_Commanded := False; Small_Wing_Rock_Complete := True; State := Airspeed_Driver;       end if;
            if Pitch_Attitude_Driver_Selected then Small_Wing_Rock_Commanded := False; Small_Wing_Rock_Complete := True; State := Pitch_Attitude_Driver; end if;

         when Pitch_Attitude_Driver =>

            Last_Driver := Pitch_Attitude_Driver;

            Climb_Angle_Driver(An_Instance, Iconst, Pitch_Rate, Pitch_Angle_Commanded);

            Update_True_Airspeed (An_Instance);
            Update_Bank_Angle (An_Instance, Iconst, Roll_Rate, Bank_Angle_Commanded);
            Update_Heading (An_Instance, Iconst);
            Update_Velocity_Components_In_Earth_Axis (An_Instance);
            Update_Position (An_Instance, Iconst);
            Update_Gear (An_Instance, Iconst, Gear_Up);
            Update_Speedbrake (An_Instance, Iconst, Speedbrake_Deployed);
            Update_Flaps (An_Instance, Iconst, Flaps_Up, Flaps_Ldg);

            State_Lp := Pitch_Attitude_Driver;
            if Wing_Dip_Commanded        then State := Wing_Dip;        end if;
            if Porpoise_Commanded        then State := Porpoise;        end if;
            if Big_Wing_Rock_Commanded   then State := Big_Wing_Rock;   end if;
            if Small_Wing_Rock_Commanded then State := Small_Wing_Rock; end if;
            if Airspeed_Driver_Selected  then State := Airspeed_Driver; end if;

         when Airspeed_Driver =>

            Last_Driver := Airspeed_Driver;

            Airspeed_Driver (An_Instance, Iconst, Velocity_Gain, VIAS_Commanded);
            Update_True_Airspeed (An_Instance);
            Update_Bank_Angle (An_Instance, Iconst, Roll_Rate, Bank_Angle_Commanded);
            Update_Heading (An_Instance, Iconst);
            Update_Velocity_Components_In_Earth_Axis (An_Instance);
            Update_Position (An_Instance, Iconst);
            Update_Gear (An_Instance, Iconst, Gear_Up);
            Update_Speedbrake (An_Instance, Iconst, Speedbrake_Deployed);
            Update_Flaps (An_Instance, Iconst, Flaps_Up, Flaps_Ldg);

            State_Lp := Airspeed_Driver;
            if Wing_Dip_Commanded             then State := Wing_Dip;              end if;
            if Porpoise_Commanded             then State := Porpoise;              end if;
            if Big_Wing_Rock_Commanded        then State := Big_Wing_Rock;         end if;
            if Small_Wing_Rock_Commanded      then State := Small_Wing_Rock;       end if;
            if Pitch_Attitude_Driver_Selected then State := Pitch_Attitude_Driver; end if;

         when others =>
            null;

      end case;

      Random_Rock_Direction.Save(RRD_Gen,An_Instance.RRD_Gen_State);

   exception
      when others =>
         Log.Report("Formation_Lead_Kinematic.update()");
         raise;
   end Update;

end Formation_Lead_Kinematic;
