-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
package body Airspeed is

   procedure Set_True_Airspeed
          (Velocity_Body_Axes :in     Coordinate_Types.Cartesian;
           An_Instance        :in out Instance) is
   begin
      An_Instance.The_True_Airspeed := Sqrt(Velocity_Body_Axes.x ** 2 +
                                            Velocity_Body_Axes.y ** 2 +
                                            Velocity_Body_Axes.z ** 2);
   end Set_True_Airspeed;

   procedure Set_True_Airspeed_based_on_Mach_Number
     (Ambient_Temperature :in     Temperature_Types.Rankine;
      An_Instance         :in out Instance) is
      Gamma            :constant Float := 1.4;
      Gas_Constant     :constant Float := 1716.16;
      Temp             :Float := 0.0;
   begin
      Temp := Gamma * Gas_Constant * Ambient_Temperature;
      An_Instance.Sonic_Airspeed    := Float'Max(400.0,Sqrt(temp));
      An_Instance.The_True_Airspeed := An_Instance.The_Mach_Number * An_Instance.Sonic_Airspeed;
   end Set_True_Airspeed_based_on_Mach_Number;

   --| This method to be used usually during formation flying an uncanned/dynamic/kinematic model.
   procedure Set_True_Airspeed_based_on_Indicated_Airspeed
     (Indicated_Airspeed    :in     Length_Types.Feet_per_Sec;
      SL_Density            :in     Mass_Types.Slugs_per_Cubic_Feet;
      Density_at_Altitude   :in     Mass_Types.Slugs_per_Cubic_Feet;
      An_Instance           :in out Instance) is
   begin
     -- An_Instance.The_True_Airspeed := Sqrt(SL_Density / Density_at_Altitude)
      An_Instance.The_True_Airspeed := Sqrt(0.002378 / Density_at_Altitude)
      * Indicated_Airspeed;
   end Set_True_Airspeed_based_on_Indicated_Airspeed;

   --| This procedure will set the True_Airspeed calculated during initalization, autotest
   --| or when True_Airspeed is defined by an approved other method.
   procedure Assign_True_Airspeed (True_Airspeed :in     Length_Types.Feet_per_Sec;
                                   An_Instance   :in out Instance) is
   begin
      An_Instance.The_True_Airspeed := True_Airspeed;
   end Assign_True_Airspeed;

   function Get_True_Airspeed(An_Instance :in Instance) return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_True_Airspeed;
   end Get_True_Airspeed;

   procedure Set_Mach_Number (Ambient_Temperature :in     Temperature_Types.Rankine;
                              An_Instance         :in out Instance) is
      Gamma          : constant Float := 1.4;
      Gas_Constant   : constant Float := 1716.16;
   begin
      An_Instance.Sonic_Airspeed            := Sqrt(Gamma * Gas_Constant * Ambient_Temperature);
      An_Instance.The_Mach_Number := An_Instance.The_True_Airspeed / An_Instance.Sonic_Airspeed;
   end Set_Mach_Number;

   procedure Set_Mach_Number_based_on_Calibrated_Airspeed
   --  (Ambient_Pressure_Ratio :in     Normalized_Types.Normalize;
      (Dens          :in     Mass_Types.Slugs_per_Cubic_Feet;
       An_Instance    :in out Instance) is
      Tas : Float := 0.0;
   begin
  --    An_Instance.The_Mach_Number := Sqrt(5.0 * (( 1.0 / Ambient_Pressure_Ratio
  --                  * (( 1.0 + 0.2 * (An_Instance.The_Calibrated_Airspeed / 661.5)**2) ** 3.5 - 1.0 )
  --                                                 + 1.0 ) ** 0.286 -1.0 ));
      Tas := An_Instance.The_Calibrated_Airspeed /Sqrt(Dens/0.002378);
      An_Instance.The_Mach_Number :=  Tas * 1.688/An_Instance.Sonic_Airspeed;


   end Set_Mach_Number_based_on_Calibrated_Airspeed;

   procedure Assign_Mach_Number (Mach_Number :in Length_Types.Mach;
                                 An_Instance :in out Instance) is
   begin
      An_Instance.The_Mach_Number := Mach_Number;
   end Assign_Mach_Number;

   function Get_Mach_Number(An_Instance :in Instance) return Length_Types.Mach is
   begin
      return An_Instance.The_Mach_Number;
   end Get_Mach_Number;

   procedure Set_Calibrated_Airspeed (Dens         :in     Float;
                                      An_Instance  :in out Instance) is
   begin
   --     An_Instance.The_Calibrated_Airspeed := 1479.1 * Sqrt((1.0 + Ambient_Pressure_Ratio
   --          * ((1.0 + 0.2 * An_Instance.The_Mach_Number**2)**3.5 - 1.0))**(1.0/3.5)- 1.0);
    An_Instance.The_Calibrated_Airspeed := An_Instance.The_True_Airspeed * Sqrt(Dens/0.002378)/1.688;
   end Set_Calibrated_Airspeed;

   procedure Assign_Calibrated_Airspeed (Calibrated_Airspeed :in     Length_Types.Knots;
                                         An_Instance         :in out Instance) is
   begin
      An_Instance.The_Calibrated_Airspeed := Calibrated_Airspeed;
   end Assign_Calibrated_Airspeed;

   function Get_Calibrated_Airspeed(An_Instance :in Instance) return Length_Types.Knots is
   begin
      return An_Instance.The_Calibrated_Airspeed;
   end Get_Calibrated_Airspeed;

   procedure Set_Rate_of_Climb
     (z_Vel                   :in     Length_Types.Feet_per_Sec;
      An_Instance             :in out Instance) is
   begin
      An_Instance.The_Rate_of_Climb := -z_Vel * 60.0;
   end Set_Rate_of_Climb;

--| Set_Rate_of_Climb_for_Altitude_Freeze method is to be used when altitude
--| freeze is commanded.  This method is used because the accelerations have
--| become unnaturally frozen and during altitude freeze acceleration effects
--| must be included.
   procedure Set_Rate_of_Climb_for_Altitude_Freeze
         (An_Instance :in out Instance) is
   begin
      An_Instance.The_Rate_of_Climb :=
        -0.014_055  * An_Instance.The_Mach_Number
        + 0.656_671 * An_Instance.The_Mach_Number **2
        -0.214_204  * An_Instance.The_Mach_Number**3;
   end Set_Rate_of_Climb_for_Altitude_Freeze;

   procedure Assign_Rate_Of_Climb
              (Rate_of_Climb      :in Length_Types.Feet_per_Min;
               An_Instance        :in out Instance) is
   begin
      An_Instance.The_Rate_of_Climb := Rate_of_Climb;
   end Assign_Rate_of_Climb;

   function Get_Rate_of_Climb(An_Instance :in Instance)
      return Length_Types.Feet_per_Min is
   begin
      return An_Instance.The_Rate_of_Climb;
   end Get_Rate_of_Climb;

   procedure Set_Ground_Speed
        (North_vel    :in     Length_Types.Feet_per_Sec;
         East_vel     :in     Length_Types.Feet_per_Sec;
         L_1          :in     Float;
         L_2          :in     Float;
         An_Instance  :in out Instance) is
   begin
      An_Instance.The_Ground_Speed :=
                 sqrt((L_1 * North_vel)**2
                      + (L_2 * East_vel)**2);
   end Set_Ground_Speed;

   procedure Assign_Ground_Speed (Ground_Speed :in     Length_Types.Feet_per_Sec;
                                  An_Instance  :in out Instance) is
   begin
      An_Instance.The_Ground_Speed := Ground_Speed;
   end Assign_Ground_Speed;

   function Get_Ground_Speed(An_Instance :in Instance)
      return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_Ground_Speed;
   end Get_Ground_Speed;

   procedure Calc_Dynamic_Pressure
              (Air_Density   :in     Mass_Types.Slugs_per_Cubic_Feet;
               True_Airspeed :in     Length_Types.Feet_per_Sec;
               An_Instance   :in out Instance) is
   begin
      An_Instance.The_Dynamic_Pressure := 0.5 * Air_Density *
                             True_Airspeed ** 2;
   end Calc_dynamic_pressure;

   function dynamic_pressure(An_Instance :in Instance)
      return Force_Types.Pounds_Per_Sq_Feet is
   begin
      return An_Instance.The_dynamic_pressure;
   end dynamic_pressure;

   procedure Set_Stall_Speed(GW           :in     Mass_Types.Lbm;
                             Flap_Position   :in     Angle_Types.Degrees;
                             z_Load_Factor   :in     Float;
                             SASSB_Table     :in out It.Singly_Indexed.Instance;
                             SASSGWE_Table   :in out It.Singly_Indexed.Instance;
                             An_Instance     :in out Instance) is
      Stall_Speed_Basic         :Float := 0.0;
      Stall_Speed_GW_Effect     :Float := 0.0;
      Intermidiate_Stall_Speed  :Float := 0.0;
   begin
      Stall_Speed_Basic := It.Singly_Indexed.Interpolate(Flap_Position,SASSB_Table'access);

      Stall_Speed_GW_Effect := It.Singly_Indexed.Interpolate(Flap_Position,SASSGWE_Table'access);

      Intermidiate_Stall_Speed  :=  Stall_Speed_GW_Effect * GW + Stall_Speed_Basic;
      --| Load Factor Effect
      An_Instance.The_Load_Factor_Effect :=  An_Instance.The_Load_Factor_Effect +
            An_Instance.The_Load_Factor_Effect
              * (z_Load_Factor - An_Instance.The_Load_Factor_Effect);

      An_Instance.The_Stall_Speed        :=  Intermidiate_Stall_Speed
        * Sqrt(abs(An_Instance.The_Load_Factor_Effect));

   end Set_Stall_Speed;

   procedure Assign_Stall_Speed (Init_Load_Factor_Effect :in Float;
                                 Init_Stall_Speed        :in Length_Types.Knots;
                                 An_Instance                     :in out Instance) is
   begin
      An_Instance.The_Load_Factor_Effect := Init_Load_Factor_Effect;
      An_Instance.The_Stall_Speed        := Init_Stall_Speed;
   end Assign_Stall_Speed;

   function Get_Stall_Speed(An_Instance :in Instance) return Length_Types.Knots is
   begin
      return An_Instance.The_Stall_Speed;
   end Get_Stall_Speed;

end Airspeed;
