-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Yogesh Tupe
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Coordinate_Transformation;

package body Acceleration is

  function Get_Body_Axis
    (An_Instance :in Instance)
    return Coordinate_Types.Cartesian is
  begin
    return An_Instance.The_Acc_Body;
  end Get_Body_Axis;

   function Load_Factor
     (An_Instance :in Instance)
     return Coordinate_Types.Cartesian is
   begin
     return An_Instance.The_load_factor;
   end Load_factor;

   procedure Assign_Load_Factor
     (An_Instance : in out Instance;
      Load_Factor : in Coordinate_Types.Cartesian) is
   begin
     An_Instance.The_Load_Factor := Load_Factor;
   end Assign_Load_Factor;

   procedure Calc_linear  --| linear Acceleration in Earth Axis & body axis
     (Force         :in     Coordinate_Types.Cartesian;
      Roll_Angle    :in     Angle_Types.Radians;
      Pitch_Angle   :in     Angle_Types.Radians;
      Yaw_Angle     :in     Angle_Types.Radians;
      Mass          :in     Mass_Types.Slugs_per_Cubic_Feet;
      Stopped       :in     Boolean;
      An_Instance   :in out Instance) is
      Input_Attribute : Coordinate_Types.Cartesian;
   begin
      An_Instance.The_Last_Pass_Acc_Earth := An_Instance.The_Acc_Earth;

      Input_attribute := (Force.x/Float'Max(Mass,0.1),
                          Force.y/Float'Max(Mass,0.1),
                          Force.z/Float'Max(Mass,0.1));

      An_Instance.The_Acc_Earth := Coordinate_Transformation.Body_to_Earth
          (Input_Attribute ,
           Roll_Angle      ,
           Pitch_Angle     ,
           Yaw_Angle       );

      An_Instance.The_Acc_Earth.z := An_Instance.The_Acc_Earth.z + 32.174;
      if Stopped then
         An_Instance.The_Acc_Earth.X := 0.0;
         An_Instance.The_Acc_Earth.y := 0.0;
      end if;

      An_Instance.The_Acc_body.x := Input_attribute.x;
      An_Instance.The_Acc_body.y := Input_attribute.y;
      An_Instance.The_Acc_body.z := Input_attribute.z;

      An_Instance.The_Load_Factor.x := An_Instance.The_Acc_body.x/32.174;
      An_Instance.The_Load_Factor.y := An_Instance.The_Acc_body.y/32.174;
      An_Instance.The_Load_Factor.z := An_Instance.The_Acc_body.z/32.174;
   end Calc_linear;

   function Get_Earth_Axis(An_Instance :in Instance)
         return Coordinate_Types.Cartesian is
   begin
         return An_Instance.The_Acc_Earth;
   end Get_Earth_Axis;

   function Get_Last_Pass_Earth_Axis(An_Instance :in Instance)
         return Coordinate_Types.Cartesian is
   begin
         return An_Instance.The_Last_Pass_Acc_Earth;
   end Get_Last_Pass_Earth_Axis;

 --|  Angular Acceleration in Body axis
   procedure Calc_Ang_Acc(Moment        :in     Coordinate_Types.Cartesian;
                          Inertia       :in     Coordinate_Types.Inertia_Axis;
                          Angular_Rate  :in     Coordinate_Types.Attitude;
                          An_Instance   :in out Instance) is
      Denominator_1 : Float := Inertia.xx * Inertia.zz - Inertia.xz**2;
      Denominator_2 : Float := Inertia.yy;
   begin
      An_Instance.The_Last_Pass_Ang := An_Instance.The_Ang;

      An_Instance.The_Ang.Roll :=
             (Moment.x * Inertia.zz + Moment.z * Inertia.xz
           +  Angular_Rate.Roll * Angular_Rate.Pitch * Inertia.xz
                      * (Inertia.xx - Inertia.yy + Inertia.zz)
           - Angular_Rate.Pitch * Angular_Rate.Yaw * (Inertia.zz * Inertia.zz
           + Inertia.xz * Inertia.xz - Inertia.yy * Inertia.zz))
                                       / Denominator_1;

      An_Instance.The_Ang.Pitch  :=   (Moment.Y
                                       + Angular_Rate.Roll * Angular_Rate.Yaw
                                       * (Inertia.zz - Inertia.xx)
                                       - (Angular_Rate.Roll * Angular_Rate.Roll
                                       - Angular_Rate.Yaw * Angular_Rate.Yaw)
                                       * Inertia.xz) / Denominator_2;
      An_Instance.The_Ang.Yaw   :=  (Moment.z * Inertia.xx
                 + Moment.x * Inertia.xz
                 - Angular_Rate.Pitch * Angular_Rate.Yaw * Inertia.xz
                 * (Inertia.xx - Inertia.yy + Inertia.zz)
                 + Angular_Rate.Roll * Angular_Rate.Pitch
                 * (Inertia.xx * Inertia.xx + Inertia.xz * Inertia.xz
                          - Inertia.xx * Inertia.yy)) / Denominator_1;
   end Calc_Ang_Acc;

   function Get_Angular(An_Instance :in Instance)
           return Coordinate_Types.Attitude is
   begin
          return An_Instance.The_Ang;
   end Get_Angular;

   function Get_Last_Pass_Angular(An_Instance :in Instance)
         return Coordinate_Types.Attitude is
   begin
         return An_Instance.The_Last_Pass_Ang;
   end Get_Last_Pass_Angular;

end Acceleration;



