-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Numerics; use Ada.Numerics;
with Log;
with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;

package body Tad is

   Tadmap_Table :  aliased Interpolation_Table.Doubly_Indexed.Instance;

   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("SFC_Dir") & "tadmap.ito",
         Table     => Tadmap_Table);
   exception
      when others =>
         Log.Report("Tad.Read_Tables()");
         raise;
   end Read_Tables;


   function Tad_Command
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Tad_Command;
   exception
      when others =>
         Log.Report("Tad.Tad_Command()");
         raise;
   end Tad_Command;

   function Tad_On
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Tad_On;
   exception
      when others =>
         Log.Report("Tad.Tad_On()");
         raise;
   end Tad_On;

   function Tad_Fail
     (An_Instance : in Instance)
     return Boolean
   is
   begin
      return An_Instance.Tad_Fail;
   exception
      when others =>
         Log.Report("Tad.Tad_Fail()");
         raise;
   end Tad_Fail;

   procedure update
     (An_Instance                   : in out Instance;
      Iconst                        : in     Float;
      Left_Wow_Relay                : in     Boolean;
      Right_Wow_Relay               : in     Boolean;
      Tad_Cb                        : in     Boolean;
      Kias_Adc                      : in     Length_Types.Knots;
      Pitch_Rate                    : in     Angle_Types.Radians_per_Sec;
      Pressure_Altitude             : in     Length_Types.Feet;
      Engine_Torque                 : in     Float;
      Rudder_Trim_Actuated_Position : in     Jpats_Secondary_Flight_Controls_Types.Rudder_Trim_Position_Type;
      Trim_Disconnect               : in     Boolean;
      Trim_Interrupt                : in     Boolean;
      Tad_Fault                     : in     Boolean;
      Rudder_Trim_Switch_Left       : in     Boolean;
      Rudder_Trim_Switch_Right      : in     Boolean;
      Tad_On_Switch                 : in     Boolean)
   is

      In_Air            : constant Boolean := (not Left_Wow_Relay and not Right_Wow_Relay) or (Kias_Adc > 80.0);
      Old_Pilot_Command : constant Float   := An_Instance.Pilot_Command;
      package Double renames Interpolation_Table.Doubly_Indexed;

   begin

--       if In_Air then Log.Report("In Air"); else
--         Log.Report("NOT In Air");      end if;

--      Ada.Text_Io.Put(Float'Image(Kias_Adc));
--      Ada.Text_Io.Put(Boolean'Image(Left_Wow_Relay));
--      Ada.Text_Io.Put(Boolean'Image(Right_Wow_Relay));
--      Ada.Text_Io.New_Line;



--       if not Tad_On_Switch then
--          An_Instance.Tad_Fail := False;
--       elsif not Tad_Cb or Trim_Interrupt or Trim_Disconnect or Tad_Fault then
--          An_Instance.Tad_Fail := True;
--       end if;

--       if Tad_On_Switch and not An_Instance.Tad_Fail then
--          An_Instance.Tad_On := True;
--       else
--          An_Instance.Tad_On := False;
--       end if;

      if not Tad_Cb or Tad_Fault then
         An_Instance.Tad_Fail := True;
      elsif not Tad_On_Switch then
         An_Instance.Tad_Fail := False;
      end if;

--       if not Tad_On_Switch then
--          An_Instance.Tad_Fail := False;
--       elsif not Tad_Cb or Tad_Fault then
--          An_Instance.Tad_Fail := True;
--       end if;

      if Tad_On_Switch and not An_Instance.Tad_Fail and not Trim_Interrupt and not Trim_Disconnect then
         An_Instance.Tad_On := True;
      else
         An_Instance.Tad_On := False;
      end if;


      if In_Air then
         An_Instance.Map_Base := Double.Interpolate (Float(Kias_Adc), Float(Engine_Torque),Tadmap_table'Access);
         An_Instance.Map_Command :=
           An_Instance.Map_Base - 2389.0 *
           ((Float'Max(-64.0, Float'Min(64.0, (Pitch_Rate * 180.0 / Pi)))) /
            (Float'Max(70.0, Float'Min(255.0, Kias_Adc)))**2) -
           0.000011 * Pressure_Altitude;
      else
         An_Instance.Map_Command := 5.0;
      end if;

      An_Instance.Map_Command := Float'Max(-5.3,Float'Min(9.2,An_Instance.Map_Command));

      if not An_Instance.Tad_On then
         if In_Air then
            An_Instance.Pilot_Command := Rudder_Trim_Actuated_Position - An_Instance.Map_Command;
         else
            An_Instance.Pilot_Command := 0.0;
         end if;
      else
         if Rudder_Trim_Switch_Left then
            An_Instance.Pilot_Command := Float'Max((-5.3 - An_Instance.Map_Command),
                                                   (Old_Pilot_Command - 0.97 * Iconst));
         elsif Rudder_Trim_Switch_Right then
            An_Instance.Pilot_Command := Float'Min((9.2 - An_Instance.Map_Command),
                                                   (Old_Pilot_Command + 0.97 * Iconst));
         else
            An_Instance.Pilot_Command := Float'Max((-5.3 - An_Instance.Map_Command),
                                                   (Float'Min((9.2 - An_Instance.Map_Command),
                                                              (Old_Pilot_Command))));
         end if;
      end if;

      An_Instance.Tad_Command := An_Instance.Map_Command + An_Instance.Pilot_Command;

   exception
      when others =>
         Log.Report("Tad.Update()");
         raise;
   end Update;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Is_Powered    := False; -- Boolean;
      An_Instance.In_Air        := False; -- Boolean;
      An_Instance.Map_Base      := 0.0; -- Float;
      An_Instance.Map_Command   := 0.0; -- Float;
      An_Instance.Pilot_Command := 0.0; -- Float;
      An_Instance.Tad_Command   := 0.0; -- Float;
      An_Instance.Tad_Fail      := False; -- Boolean;
      An_Instance.Tad_On        := False; -- Boolean;
   exception
      when others =>
         Log.Report("Tad.Initialize()");
         raise;
   end Initialize;

end Tad;

