-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Log;

package body Rudder_Trim is

   procedure Initialize
     (An_Instance : in out Instance) is
   begin
      An_Instance.Actuated_Position           := 0.0;
      An_Instance.Actuated_Position_Commanded := 0.0;
      An_Instance.Position                    := 0.0;
      An_Instance.Load                        := 0.0;
   exception
      when others =>
         Log.Report("Rudder_Trim.Initialize()");
         raise;
   end Initialize;

   function Position
     (An_Instance : in Instance)
      return Jpats_Secondary_Flight_Controls_Types.Rudder_Trim_Position_Type is
   begin
      return An_Instance.Position;
   exception
      when others =>
         Log.Report("Rudder_Trim.Position()");
         raise;
   end Position;

   function Load
     (An_Instance : in Instance)
      return Electrical_Units_Types.Amps is
   begin
      return An_Instance.Load;
   exception
      when others =>
         Log.Report("Rudder_Trim.Load()");
         raise;
   end Load;

   function Actuated_Position
     (An_Instance : in Instance)
      return Jpats_Secondary_Flight_Controls_Types.Rudder_Trim_Position_Type is
   begin
      return An_Instance.Actuated_Position;
   exception
      when others =>
         Log.Report("Rudder_Trim.Actuated_Position()");
         raise;
   end Actuated_Position;

   procedure Set_Position
     (An_Instance  : in out Instance;
      Tab_Position : in     Float;
      Rudder_Angle : in     Float)
   is
   begin
      An_Instance.Position := Tab_Position;
      An_Instance.Actuated_Position := Tab_Position - 0.2 * Rudder_Angle;
   exception
      when others =>
         Log.Report("Rudder_Trim.Set_Position()");
         raise;
   end Set_Position;


   procedure Update
     (An_Instance              : in out Instance;
      Iconst                   : in     Float;
      Rud_Trim_Cb              : in     Boolean;
      Tad_On                   : in     Boolean;
      Trim_Interrupt           : in     Boolean;
      Trim_Disconnect          : in     Boolean;
      Tad_Command              : in     Float;
      Rudder_Trim_Switch_Left  : in     Boolean;
      Rudder_Trim_Switch_Right : in     Boolean;
      Rudder_Angle             : in     Float;
      Trim_Fail_Malf           : in     Boolean;
      Full_Right_Malf          : in     Boolean)
   is


      package Sfc_T renames Jpats_Secondary_Flight_Controls_Types;

      Old_Actuated_Position           : constant Float := An_Instance.Actuated_Position;
      Old_Actuated_Position_Commanded : constant Float := An_Instance.Actuated_Position_Commanded;

      Rtrim : Float := 0.0;
      Rate  : Float := 0.0;

   begin

      if Rudder_Trim_Switch_Left then
         Rtrim := 1.0; -- sw left -> tab right
      elsif Rudder_Trim_Switch_Right then
         Rtrim := -1.0; -- sw right -> tab left
      end if;

      if not Rud_Trim_Cb or Trim_Interrupt or Trim_Disconnect or Trim_Fail_Malf then
         An_Instance.Actuated_Position_Commanded := Old_Actuated_Position_Commanded;
      elsif Full_Right_Malf then
         An_Instance.Actuated_Position_Commanded := (Sfc_T.Rudder_Trim_Position_Type'Max
                                                     (-6.0,Sfc_T.Rudder_Trim_Position_Type'Min
                                                      (11.0,Old_Actuated_Position + 0.74 * Iconst)));
      elsif Tad_On then
         An_Instance.Actuated_Position_Commanded := (Sfc_T.Rudder_Trim_Position_Type'Max
                                                     (-6.0,Sfc_T.Rudder_Trim_Position_Type'Min
                                                      (11.0,Tad_Command)));
      else -- tad off
         An_Instance.Actuated_Position_Commanded := (Sfc_T.Rudder_Trim_Position_Type'Max
                                                     (-6.0,Sfc_T.Rudder_Trim_Position_Type'Min
                                                      (11.0,Old_Actuated_Position - Rtrim * 1.02 * Iconst)));
      end if;

      Rate := Float'Max (-1.02, Float'Min (1.02, (An_Instance.Actuated_Position_Commanded - Old_Actuated_Position)/Iconst));

      An_Instance.Actuated_Position := (Sfc_T.Rudder_Trim_Position_Type'Max
                                        (-6.0, Sfc_T.Rudder_Trim_Position_Type'Min
                                         (11.0, Old_Actuated_Position + Rate * Iconst)));


      An_Instance.Position := An_Instance.Actuated_Position + 0.20 * Rudder_Angle;

      if abs (An_Instance.Actuated_Position - Old_Actuated_Position) /= 0.0 then
         An_Instance.Load := 0.665;
      else
         An_Instance.Load := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Rudder_Trim.Update()");
         raise;
   end Update;

end Rudder_Trim;

