-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Interpolation_Table.Doubly_Indexed;
with Simulation_Dictionary;
with Log;

package body Elevator_Trim is

   detabsup : aliased Interpolation_Table.Doubly_Indexed.Instance;
   detabsdn : aliased Interpolation_Table.Doubly_Indexed.Instance;


   procedure Read_Tables
   is
   begin
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("SFC_Dir") & "detabsup.ito",
         Table     => detabsup);
      Interpolation_Table.Read
        (File_Name => Simulation_Dictionary.Lookup ("SFC_Dir") & "detabsdn.ito",
         Table     => detabsdn);
   exception
      when others =>
         Log.Report("Elevator_Trim.Read_Tables()");
         raise;
   end Read_Tables;

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Position          := 0.0;
      An_Instance.Actuated_Position := 0.0;
      An_Instance.Load              := 0.0;
   exception
      when others =>
         Log.Report("Elevator_Trim.Initialize()");
         raise;
   end Initialize;

   function Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Position;
   exception
      when others =>
         Log.Report("Elevator_Trim.Position()");
         raise;
   end Position;

   function Actuated_Position
     (An_Instance : in Instance)
     return Float
   is
   begin
      return An_Instance.Actuated_Position;
   exception
      when others =>
         Log.Report("Elevator_Trim.Actuated_Position()");
         raise;
   end Actuated_Position;

   function Load
     (An_Instance : in Instance)
     return Electrical_Units_Types.Amps
   is
   begin
      return An_Instance.Load;
   exception
      when others =>
         Log.Report("Elevator_Trim.Load()");
         raise;
   end Load;


--   Sp_Sb_Pos_Lp         : Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type := 0.0;
--   Sp_Speedbrake_Effect_Up_Table_Lp : Boolean := False;

   procedure Set_Position
     (An_Instance         : in out Instance;
      Tab_Position        : in     Float;
      Speedbrake_Position : in     Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type;
      Dynamic_Pressure    : in     Force_Types.Pounds_Per_Sq_Feet)
   is
      package Double renames Interpolation_Table.Doubly_Indexed;
      Speedbrake_Effect_Up_Table : Boolean;
      Speedbrake_Effect : Float;
      Sp_Sb_Pos_Lp : Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type renames An_Instance.Sp_Sb_Pos_Lp;
      Sp_Speedbrake_Effect_Up_Table_Lp : Boolean renames An_Instance.Sp_Speedbrake_Effect_Up_Table_Lp;

   begin

--      An_Instance.Position := Tab_Position;
      An_Instance.Actuated_Position := Tab_Position;

      if Speedbrake_Position > Sp_Sb_Pos_Lp then
         Speedbrake_Effect_Up_Table := True;
      elsif Speedbrake_Position < Sp_Sb_Pos_Lp then
         Speedbrake_Effect_Up_Table := False;
      else
         Speedbrake_Effect_Up_Table := Sp_Speedbrake_Effect_Up_Table_Lp;
      end if;

      Sp_Sb_Pos_Lp := Speedbrake_Position;
      Sp_Speedbrake_Effect_Up_Table_Lp := Speedbrake_Effect_Up_Table;

      if Speedbrake_Effect_Up_Table then
         Speedbrake_Effect := Double.Interpolate (Float(Speedbrake_Position), Float(Dynamic_Pressure), Detabsdn'Access);
      else
         Speedbrake_Effect := Double.Interpolate (Float(Speedbrake_Position), Float(Dynamic_Pressure), Detabsup'Access);
      end if;

      An_Instance.Position := An_Instance.Actuated_Position + Speedbrake_Effect;

   exception
      when others =>
         Log.Report("Elevator_Trim.Set_Position()");
         raise;
   end Set_Position;


--   U_Sb_Pos_Lp         : Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type := 0.0;
--   U_Speedbrake_Effect_Up_Table_Lp : Boolean := False;

   procedure Update
     (An_Instance         : in out Instance;
      Iconst              : in     Float;
      Electrical_Power    : in     Boolean;
      Switch_Fwd          : in     Boolean;
      Switch_Aft          : in     Boolean;
      Speedbrake_Position : in     Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type;
      Trim_Interrupt      : in     Boolean;
      Trim_Disconnect     : in     Boolean;
      Dynamic_Pressure    : in     Force_Types.Pounds_Per_Sq_Feet;
      Fail_Malf           : in     Boolean;
      Run_Full_Up_Malf    : in     Boolean)
   is

      package Sfc_T renames Jpats_Secondary_Flight_Controls_Types;
      package Double renames Interpolation_Table.Doubly_Indexed;

      Old_Actuated_Position : constant Sfc_T.Elevator_Trim_Position_Type := An_Instance.Actuated_Position;

      Motion_Direction : Sfc_T.Elevator_Trim_Position_Type := 0.0;
      Rate             : Sfc_T.Elevator_Trim_Position_Type := 3.5;
      Speedbrake_Effect_Up_Table : Boolean;
      Speedbrake_Effect : Float;

      U_Sb_Pos_Lp : Jpats_Secondary_Flight_Controls_Types.Speedbrake_Position_Type renames An_Instance.U_Sb_Pos_Lp;
      U_Speedbrake_Effect_Up_Table_Lp : Boolean renames An_Instance.U_Speedbrake_Effect_Up_Table_Lp;

   begin

      if Old_Actuated_Position <= 0.0 then
         Rate := 3.0;
      end if;

      if not Trim_Interrupt and not Trim_Disconnect and Electrical_Power then
         if Fail_Malf then
            null;
         elsif Run_Full_Up_Malf then
            Motion_Direction := 1.0;
         else
            if Switch_Fwd then
               Motion_Direction := 1.0; -- want plane nose down -> trim tab up
            elsif Switch_Aft then
               Motion_Direction := -1.0; -- want plane nose up -> trim tab down
            end if;
         end if;
      end if;


      An_Instance.Actuated_Position := (Float'Max
                                        (-7.0, Float'Min
                                         (21.0, Old_Actuated_Position - Rate * Motion_Direction * Iconst)));

      if Speedbrake_Position > U_Sb_Pos_Lp then
         Speedbrake_Effect_Up_Table := True;
      elsif Speedbrake_Position < U_Sb_Pos_Lp then
         Speedbrake_Effect_Up_Table := False;
      else
         Speedbrake_Effect_Up_Table := U_Speedbrake_Effect_Up_Table_Lp;
      end if;

      U_Sb_Pos_Lp := Speedbrake_Position;
      U_Speedbrake_Effect_Up_Table_Lp := Speedbrake_Effect_Up_Table;


      if Speedbrake_Effect_Up_Table then
         Speedbrake_Effect := Double.Interpolate (Float(Speedbrake_Position), Float(Dynamic_Pressure), Detabsdn'Access);
      else
         Speedbrake_Effect := Double.Interpolate (Float(Speedbrake_Position), Float(Dynamic_Pressure), Detabsup'Access);
      end if;

      An_Instance.Position := An_Instance.Actuated_Position + Speedbrake_Effect;


      if abs (Old_Actuated_Position - An_Instance.Actuated_Position) /= 0.0 then
         An_Instance.Load := 0.665;
      else
         An_Instance.Load := 0.0;
      end if;

   exception
      when others =>
         Log.Report("Elevator_Trim.Update()");
         raise;
   end Update;


end Elevator_Trim;
