-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Log;

package body Aircraft_Flaps is

   procedure Initialize
     (An_Instance : in out Instance)
   is
   begin
      An_Instance.Deflection           := 0.0;
      An_Instance.Actuated_Deflection  := 0.0;
      An_Instance.Deflection_Commanded := 0.0;
      An_Instance.Asymmetry            := 0.0;
      An_Instance.Hydraulic_Load       := 0.0;
      An_Instance.Load                 := 0.0;

      An_Instance.Take_Off_Piston_Position := 0.0;
      An_Instance.Landing_Piston_Position  := 0.0;

   exception
      when others =>
         Log.Report("Aircraft_Flaps.Initialize()");
         raise;
   end Initialize;

   function Hydraulic_Load
     (An_Instance : in Instance)
     return Length_Types.Gpm
   is
   begin
      return An_Instance.Hydraulic_Load;
   exception
      when others =>
         Log.Report("Aircraft_Flaps.Hydraulic_Load()");
         raise;
   end Hydraulic_Load;

   function Asymmetry
     (An_Instance : in Instance)
     return Jpats_Secondary_Flight_Controls_Types.Flap_Asymmetry_Type
   is
   begin
      return An_Instance.Asymmetry;
   exception
      when others =>
         Log.Report("Aircraft_Flaps.Asymmetry()");
         raise;
   end Asymmetry;

   function Load
     (An_Instance : in Instance)
     return Electrical_Units_Types.Amps
   is
   begin
      return An_Instance.Load;
   exception
      when others =>
         Log.Report("Aircraft_Flaps.Load()");
         raise;
   end Load;

   function Deflection
     (An_Instance : in Instance)
     return Jpats_Secondary_Flight_Controls_Types.Flap_Position_Type
   is
   begin
      return An_Instance.Deflection;
   exception
      when others =>
         Log.Report("Aircraft_Flaps.Deflection()");
         raise;
   end Deflection;

   package Sfc_T renames Jpats_Secondary_Flight_Controls_Types;

   Actuated_Deflection_Before_Failure : Sfc_T.Flap_Position_Type := 0.0;
   Deflection_Delta : Sfc_T.Flap_Position_Type := 0.0;

   procedure Update
     (An_Instance              : in out Instance;
      Iconst                   : in     Float;
      Emerg_Gear_Handle_Extend : in     Boolean;
      Primary_Hyd_Fcn          : in     Jpats_Hydraulics_Types.Normalized_Pressure_Type;
      Emerg_Hyd_Fcn            : in     Jpats_Hydraulics_Types.Normalized_Pressure_Type;
      Flap_Selector_Up         : in     Boolean;
      Flap_Selector_Take_Off   : in     Boolean;
      Flap_Selector_Land       : in     Boolean;
      Flap_Cont_Cb             : in     Boolean;
      Mean_Gear_Position       : in     Jpats_Landing_Gear_Types.Gear_Position_Type;
      Dynamic_Pressure         : in     Force_Types.Pounds_Per_Sq_Feet;
      Fail_Left_Outboard_Flap  : in     Boolean;
      Fail_Right_Outboard_Flap : in     Boolean;
      Fail_Left_Wing_Flaps     : in     Boolean;
      Fail_Right_Wing_Flaps    : in     Boolean;
      Fail_All_Malf            : in     Boolean)
   is

      Deflection               : Sfc_T.Flap_Position_Type     renames An_Instance.Deflection;
      Actuated_Deflection      : Sfc_T.Flap_Position_Type     renames An_Instance.Actuated_Deflection;
      Deflection_Commanded     : Sfc_T.Flap_Position_Type     renames An_Instance.Deflection_Commanded;
      Asymmetry                : Sfc_T.Flap_Asymmetry_Type    renames An_Instance.Asymmetry;
      Hydraulic_Load           : Length_Types.Gpm             renames An_Instance.Hydraulic_Load;
      Load                     : Electrical_Units_Types.Amps  renames An_Instance.Load;
      Take_Off_Piston_Position : Float                        renames An_Instance.Take_Off_Piston_Position;
      Landing_Piston_Position  : Float                        renames An_Instance.Landing_Piston_Position;

      K_Hyd_Flap                   : Float;

      Old_Deflection_Commanded     : Float := Float (Deflection_Commanded);
      Old_Actuated_Deflection      : Float := Float (Actuated_Deflection);
      Old_Take_Off_Piston_Position : Float := Float (Take_Off_Piston_Position);
      Old_Landing_Piston_Position  : Float := Float (Landing_Piston_Position);


      Take_Off_Piston_Rate     : Float;
      Landing_Piston_Rate      : Float;
      Take_Off_Piston_Hyd_Load : Length_Types.Gpm := 0.0;
      Landing_Piston_Hyd_Load  : Length_Types.Gpm := 0.0;

   begin

      if Emerg_Gear_Handle_Extend then
         if Mean_Gear_Position >= 1.0 then
            K_Hyd_Flap := 1.14*Emerg_Hyd_Fcn;
         else
            K_Hyd_Flap := 0.0;
         end if;
      else
         if Flap_Cont_Cb then
            K_Hyd_Flap := Primary_Hyd_Fcn;
         else
            if Actuated_Deflection < 0.1 then
               K_Hyd_Flap := 0.0;
            else
               K_Hyd_Flap := Primary_Hyd_Fcn;
            end if;
         end if;
      end if;

      if Flap_Cont_Cb or Emerg_Gear_Handle_Extend then
         if Flap_Selector_Up then
            Deflection_Commanded := 0.0;
         elsif Flap_Selector_Take_Off then
            Deflection_Commanded := 23.0;
         elsif Flap_Selector_Land then
            Deflection_Commanded := 50.0;
         else
            Deflection_Commanded := Old_Deflection_Commanded;
         end if;
      else
         Deflection_Commanded := 0.0;
      end if;

      if Fail_All_Malf then
         null;
      else
         if Deflection_Commanded >= Old_Actuated_Deflection then

            if Old_Actuated_Deflection <= 23.0 then

               Take_Off_Piston_Rate     := Float (Float (K_Hyd_Flap) * (9.8 - 0.028 * Float (Dynamic_Pressure)));
               Take_Off_Piston_Position := Float (Float'Min(Deflection_Commanded, (Float(Old_Take_Off_Piston_Position) + Take_Off_Piston_Rate*Iconst)));
               Take_Off_Piston_Hyd_Load := Length_Types.Gpm(0.024 * Length_Types.Gpm(Take_Off_Piston_Position - Old_Take_Off_Piston_Position) / Length_Types.Gpm(Iconst));
               Landing_Piston_Rate      := Float (Float (K_Hyd_Flap) * (-6.9 - 0.011 * Float (Dynamic_Pressure)));
               Landing_Piston_Position  := Float (Float'Max(0.0, (Float (Old_Landing_Piston_Position) + Landing_Piston_Rate*Iconst)));
               Landing_Piston_Hyd_Load  := Length_Types.Gpm(-0.022* Length_Types.Gpm(Landing_Piston_Position - Old_Landing_Piston_Position) / Length_Types.Gpm(Iconst));

            else

               Take_Off_Piston_Rate     := Float (Float (K_Hyd_Flap) * (9.8 - 0.028 * Float (Dynamic_Pressure)));
               Take_Off_Piston_Position := float(Float'Min(23.0, (Float (Old_Take_Off_Piston_Position) + Take_Off_Piston_Rate*Iconst)));
               Take_Off_Piston_Hyd_Load := Length_Types.Gpm (0.024 * Length_Types.Gpm(Take_Off_Piston_Position - Old_Take_Off_Piston_Position) / Length_Types.Gpm(Iconst));
               Landing_Piston_Rate      := Float (Float(K_Hyd_Flap) * (7.3 - 0.044 * Float(Dynamic_Pressure)));
               Landing_Piston_Position  := float (Float'Min((Deflection_Commanded-23.0),(Float (Old_Landing_Piston_Position) + Landing_Piston_Rate*Iconst)));
               Landing_Piston_Hyd_Load  := Length_Types.Gpm(0.023 * Length_Types.Gpm(Landing_Piston_Position - Old_Landing_Piston_Position) / Length_Types.Gpm(Iconst));

            end if;

         elsif not Emerg_Gear_Handle_Extend then

            if Deflection_Commanded <  23.0 then

               Take_Off_Piston_Rate     := Float (Float (K_Hyd_Flap) * (-9.1 - 0.029 * Float(Dynamic_Pressure)));
               Take_Off_Piston_Position := float (Float'Max(Deflection_Commanded, (Float(Old_Take_Off_Piston_Position) + Take_Off_Piston_Rate*Iconst)));
               Take_Off_Piston_Hyd_Load := Length_Types.Gpm(-0.033 * Length_Types.Gpm(Take_Off_Piston_Position - Old_Take_Off_Piston_Position) / Length_Types.Gpm(Iconst));
               Landing_Piston_Rate      := Float (Float (K_Hyd_Flap) * (-6.9 - 0.011 * Float (Dynamic_Pressure)));
               Landing_Piston_Position  := float (Float'Max (0.0,( Float(Old_Landing_Piston_Position) + Landing_Piston_Rate*Iconst)));
               Landing_Piston_Hyd_Load  := Length_Types.Gpm(-0.022 * Length_Types.Gpm(Landing_Piston_Position - Old_Landing_Piston_Position) / Length_Types.Gpm(Iconst));

            else

               Take_Off_Piston_Rate     := Float (Float (K_Hyd_Flap) * (9.8 - 0.028 * Float (Dynamic_Pressure)));
               Take_Off_Piston_Position := float (Float'Min (23.0, ( Float(Old_Take_Off_Piston_Position) + Take_Off_Piston_Rate*Iconst)));
--               Take_Off_Piston_Hyd_Load := Length_Types.Gpm(-0.024 * Length_Types.Gpm(Take_Off_Piston_Position - Old_Take_Off_Piston_Position) / Length_Types.Gpm(Iconst));
               Take_Off_Piston_Hyd_Load := Length_Types.Gpm(0.024 * Length_Types.Gpm(Take_Off_Piston_Position - Old_Take_Off_Piston_Position) / Length_Types.Gpm(Iconst));
               Landing_Piston_Rate      := Float (Float (K_Hyd_Flap) * (-6.9 - 0.011 * Float (Dynamic_Pressure)));
               Landing_Piston_Position  := float (Float'Max ((Deflection_Commanded - 23.0),(Float(Old_Landing_Piston_Position) + Landing_Piston_Rate*Iconst)));
               Landing_Piston_Hyd_Load  := Length_Types.Gpm(-0.022 * Length_Types.Gpm(Landing_Piston_Position - Old_Landing_Piston_Position) / Length_Types.Gpm(Iconst));

            end if;

         end if;
      end if;

      if Take_Off_Piston_Position + Landing_Piston_Position > 50.0 then
         Actuated_Deflection := 50.0;
      elsif Take_Off_Piston_Position + Landing_Piston_Position < 0.0 then
         Actuated_Deflection := 0.0;
      else
         Actuated_Deflection := Take_Off_Piston_Position + Landing_Piston_Position;
      end if;

      Hydraulic_Load := 5.75 * (Take_Off_Piston_Hyd_Load + Landing_Piston_Hyd_Load);

      if Fail_Left_Outboard_Flap then             -- IOS Asymmetric Flap malfunction

         Deflection_Delta := Actuated_Deflection_Before_Failure - Actuated_Deflection;
         Deflection := Actuated_Deflection;
         Asymmetry  := 0.7 * Deflection_Delta;

      elsif Fail_Left_Wing_Flaps then             -- Not implemented
         Deflection := 0.5 * Actuated_Deflection;
         Asymmetry  := -Actuated_Deflection;
      elsif Fail_Right_Outboard_Flap then         -- Not implemented
         Deflection := 0.72 * Actuated_Deflection;
         Asymmetry  := 0.7 * Actuated_Deflection;
      elsif Fail_Right_Wing_Flaps  then           -- Not implemented
         Deflection := 0.5 * Actuated_Deflection;
         Asymmetry  := Actuated_Deflection;
      else
         Deflection := Actuated_Deflection;
         Asymmetry := 0.0;
         Actuated_Deflection_Before_Failure := Actuated_Deflection;
      end if;

   exception
      when others =>
         Log.Report("Aircraft_Flaps.Update()");
         raise;
   end Update;


end Aircraft_Flaps;





