-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Calendar;
with Ada.Strings;
with Ada.Strings.Fixed;

-----------------------------------------------------------------------------
-- Return with a string indicating the current time in the format:
-- Month/Day/Year Hour:Minute:Seconds.Fractional_Seconds
-----------------------------------------------------------------------------
function Time_Stamp (Time : in Ada.Calendar.Time := Ada.Calendar.Clock) return String is

   -- Change these to change the separators used in time strings
   Day_Separator  : constant STRING := "/";
   Time_Separator : constant STRING := ":";

   Seconds_Per_Minute : constant := 60;

   Seconds_Per_Hour   : constant := Seconds_Per_Minute * 60;

   -----------------------------------------------------------------------------
   -- Strip off any "file extension" from the given string. Specifically,
   -- remove everything up to the last '.' character. If the string has no
   -- '.' (or it's in the first position only) the given string will be returned
   -- unchanged.
   -----------------------------------------------------------------------------
   function Strip_Extension (From : in String) return String is
   begin

      -- Find the last '.' in the string (not counting the first character)
      for Dot_Location in reverse From'First+1 .. From'last loop
         if From(Dot_Location) = '.' then

            -- return the string up to the '.'
            return From(From'First..Dot_Location-1);

         end if;
      end loop;

      -- There wasn't an extension. Return the given string
      return From;

   end Strip_Extension;

   -----------------------------------------------------------------------------
   -- Return with a string indicating the current hour given the number of
   -- seconds past the hour.
   -----------------------------------------------------------------------------
   function Hour_Stamp (Seconds : in Ada.Calendar.Day_Duration) return String is
   begin

      return Ada.Strings.Fixed.Trim
               (Source => Integer'Image(Integer(Seconds) / Seconds_Per_Hour),
                Side => Ada.Strings.Left
                );
   end Hour_Stamp;

   -----------------------------------------------------------------------------
   -- Return with a string indicating the current minute given the number of
   -- seconds past the hour.
   -----------------------------------------------------------------------------
   function Minute_Stamp (Seconds : in Ada.Calendar.Day_Duration) return String is
   begin

      return Ada.Strings.Fixed.Tail
               (Source => Ada.Strings.Fixed.Trim
                            (Source => Integer'Image
                                         ((Integer(Seconds) mod Seconds_Per_Hour) /
                                          Seconds_Per_Minute
                                         ),
                             Side => Ada.Strings.Left
                            ),
                Count => 2,
                Pad => '0'
               );
   end Minute_Stamp;


   -----------------------------------------------------------------------------
   -- Return a "seconds" string with at least two digits before the decimal
   -- point. The input string MUST a string of the form returned by
   -- duration.image, but without the leading space or minus sign.
   -- (iaw: [(1-9)](0-9).(0.9)* )
   -----------------------------------------------------------------------------
   function Pad_Seconds (Seconds : in String) return String is
   begin
      if Seconds(Seconds'First + 1) = '.' then
         return "0" & Seconds;
      else
         return Seconds;
      end if;
   end Pad_Seconds;

   -----------------------------------------------------------------------------
   -- Return with a string indicating the current second given the number of
   -- seconds past the hour.
   -----------------------------------------------------------------------------
   function Second_Stamp (Seconds : in Ada.Calendar.Day_Duration) return String is
   begin

      return Pad_Seconds
               (Ada.Strings.Fixed.Trim
                  (Source => Ada.Calendar.Day_Duration'Image
                               (Seconds - Ada.Calendar.Day_Duration
                                            ((Integer(Seconds) / Seconds_Per_Minute ) *
                                             Seconds_Per_Minute
                                            )
                               ),
                   Side => Ada.Strings.Left
                  )
               );

   end Second_Stamp;

begin

   -- Get each component, convert it to a string, trim it, and tack them all together.
   return Ada.Strings.Fixed.Trim (Source => Integer'Image(Ada.Calendar.Month(Time)),
                                  Side => Ada.Strings.Left) & Day_Separator &
     Ada.Strings.Fixed.Trim (Source => Integer'Image (Ada.Calendar.Day(Time)),
                             Side => Ada.Strings.Left) & Day_Separator &
     Ada.Strings.Fixed.Trim (Source => Integer'Image (Ada.Calendar.Year(Time)),
                             Side => Ada.Strings.Left) & " " &
     Hour_Stamp(Ada.Calendar.Seconds(Time)) & Time_Separator &
     Minute_Stamp(Ada.Calendar.Seconds(Time)) & Time_Separator &
     Second_Stamp(Ada.Calendar.Seconds(Time));

end Time_Stamp;

