with Ada.Exceptions;       -- Custom exception messages
with Ada.Text_IO;
with Token;                -- Token analysis parent
with Token.Analyzer;
with Token.Line_Comment;   -- Recognizer for Line-oriented comments
with Token.End_Of_File;    -- Recognizer for the end of a file
with Token.Integer;        -- Recognizer for integer numbers
with Token.Real;           -- Recognizer for fixed and floating point real numbers
with Token.Keyword;        -- Recognizer for specified space-delimited words
with Token.Character_Set;  -- Recognizer for sequences of specified characters.
with Token.CSV_Field;      -- Recognizer for untyped CSV fields
with Ada.Characters.Latin_1;     -- CR and LF
with Ada.Characters.Handling;    -- Character handling
with Ada.Strings.Fixed;          -- String handling
with Ada.Strings.Maps.Constants;
with Log;                        -- Printing of warnings


--------------------------------------------------------------------------------
-- This package implements comma separated value (CSV) parsing
-- utilities for autotest time history data.
--
-- The reason this package has to be generic has to do with the way the Token
-- analysis packages work. They are generic packages, which declare a pointer to
-- token type. In order to use pointers without resorting to unchecked
-- programming, Ada dictates that the pointed-to object must not have a deeper
-- dynamic scope than the access type. But in order to make the parsing routines
-- task-safe, they must declare their own token and token analyzer locally. Thus
-- the token-analysis package must be instantiated locally as well. Since that
-- is done is the spec of this package, this package needs to be a generic so
-- that it may be instantiated locally.
--------------------------------------------------------------------------------

package body Time_History_G is

   Syntax        : exception;
   Invalid_Time_History : exception;

   -- The different errors that can occur.
   type Syntax_Error_ID is
     (No_Number_Of_Variables,
      Number_Of_Variables_Too_Large, Number_Of_Variables_Too_Small,
      Too_Many_Variables, Too_Few_Variables,
      Too_Many_Values, Too_Few_Values );

   -- The different tokens in an ASCII time_history file
   -- The Identifier token will match the same string as the Real token, so it
   -- needs to be listed after the real token.
   type CSV_Token_ID is (Real, Int, Comma, Comment, Whitespace, EOL, EOF);

   -- Instantiate a time_history file tokenizer
   package Time_History_Tokenizer is new Token.Analyzer(CSV_Token_ID);

   type Instance is
      record
         The_File : Ada.Text_IO.File_Type;
         The_Filename : String (1..256);
         The_Filename_Length : Natural := 0;
         The_Tokenizer : Time_History_Tokenizer.Instance;
         The_Variable_Count : Natural;
      end record;

   -- because of the way the the token.analyzer package is written,
   -- we can't pass the file object in to the "feeder" function, so
   -- we have to use a package global -- ick.
   File : Ada.Text_Io.File_Type;

   EOL_String : constant String (1..1) := (1 => Ada.Characters.Latin_1.CR);

   Whitespace_Set : constant Ada.Strings.Maps.Character_Set :=
     Ada.Strings.Maps.To_Set (Ada.Characters.Latin_1.HT &
                              Ada.Characters.Latin_1.Space);

   -- The tokens
   Comment_Token     : aliased Token.Class := Token.Line_Comment.Get("#", True);
   EOF_Token         : aliased Token.Class := Token.End_Of_File.Get;
   Real_Literal      : aliased Token.Class := Token.Real.Get;
   Integer_Literal   : aliased Token.Class := Token.Integer.Get;
   Comma_Token       : aliased Token.Class := Token.Keyword.Get(",");
   EOL_Token         : aliased Token.Class := Token.Keyword.Get
     (EOL_String);
   Whitespace_Token  : aliased Token.Class := Token.Character_Set.Get (Whitespace_Set);

   -- The specification of the syntax of an ASCII time_history
   Time_History_Syntax : Time_History_Tokenizer.Syntax :=
     (Real        => Real_Literal'Access,
      Int         => Integer_Literal'Access,
      Comma       => Comma_Token'Access,
      Comment     => Comment_Token'Access,
      Whitespace  => Whitespace_Token'Access,
      EOF         => EOF_Token'Access,
      EOL         => EOL_Token'Access);

   function Get_Text_Line return String;

   type Text_Feeder is access function return String;
   Feeder : constant Text_Feeder := Get_Text_Line'Access;

   ------------------------------------------------------------------------------------
   -- The routine used to feed data into the token analizer. This routine reads from
   -- the analizer file one line at a time until EOF is reached.
   ------------------------------------------------------------------------------------
   function Get_Text_Line return String is
      Buffer : String(1..512);
      Data_Length : Integer;
   begin

      Ada.Text_IO.Get_Line (File => File, Item => Buffer, Last => Data_Length);

      if Ada.Text_IO.End_Of_File(File) then
         return Buffer(1..Data_Length) & Token.EOF_Character;
      else
         return Buffer(1..Data_Length) & Token.EOL_Character;
      end if;
   exception
      when Ada.Text_IO.End_Error =>
         return (1 => Token.EOF_Character);

   end Get_Text_Line;

   procedure Syntax_Error (A_Handle : in Handle;
                           A_Token :  in Csv_Token_Id;
                           A_String : in String ) is

      File_Name : String ( 1 .. A_Handle.The_Filename_Length )
        := A_Handle.The_Filename ( 1 .. A_Handle.The_Filename_Length );

      Tokenizer : Time_History_Tokenizer.Instance := A_Handle.The_Tokenizer;

   begin

      Ada.Exceptions.Raise_Exception
        (Syntax'Identity, "in time_history file " &
         File_Name & " at line" &
         Integer'Image(Time_History_Tokenizer.Line(Tokenizer)) & ", column" &
         Integer'Image (Time_History_Tokenizer.Column(Tokenizer)) &
         ". Token " & Csv_Token_Id'Image (A_Token) & " received.  Expecting " &
         A_String & ".");

   end Syntax_Error;

   ------------------------------------------------------------------------------------
   -- Raise the given exception id, with an appropriate message based on the Syntax
   -- error's ID. Calling this routine causes an exception to be raised, so control
   -- will *not* return directly to the caller.
   ------------------------------------------------------------------------------------
   procedure Report_Error (A_Handle : in Handle;
                           ID        : in Syntax_Error_ID) is

      File_Name : String ( 1 .. A_Handle.The_Filename_Length )
        := A_Handle.The_Filename ( 1 .. A_Handle.The_Filename_Length );

      Tokenizer : Time_History_Tokenizer.Instance := A_Handle.The_Tokenizer;

   begin

      case ID is
      when No_Number_Of_Variables =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in time_history file" &
           File_Name & " first line must contain an integer number of variables." &
           " Read a " & CSV_Token_ID'Image(Time_History_Tokenizer.Token(Tokenizer)) &
           "instead");

      when Number_Of_Variables_Too_Small =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in time_history file" &
           File_Name & " number of variables specified must be greater than 0." &
           " Read " & Time_History_Tokenizer.Lexeme(Tokenizer) &
           ".");

      when Number_Of_Variables_Too_Large =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in time_history file" &
           File_Name & " number of variables specified must not exceed maximum of " &
           Integer'Image ( Max_Variables ) &
           " Read " & Time_History_Tokenizer.Lexeme(Tokenizer) &
           ".");

      when Too_Many_Values =>
         Ada.Exceptions.Raise_Exception (Invalid_Time_History'Identity, "in time_history file " &
           File_Name & " at line" &
           Integer'Image(Time_History_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Time_History_Tokenizer.Column(Tokenizer)) &
           ". More values provided than the number of variables specified");

      when Too_Few_Values =>
         Ada.Exceptions.Raise_Exception (Invalid_Time_History'Identity, "in time_history file " &
           File_Name & " at line" &
           Integer'Image(Time_History_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Time_History_Tokenizer.Column(Tokenizer)) &
           ". More values provided than the number of variables specified");
      when Too_Many_Variables =>
         Ada.Exceptions.Raise_Exception (Invalid_Time_History'Identity, "in time_history file " &
           File_Name & " at line" &
           Integer'Image(Time_History_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Time_History_Tokenizer.Column(Tokenizer)) &
           ". More variable indices provided than the number of variables specified");
      when Too_Few_Variables =>
         Ada.Exceptions.Raise_Exception (Invalid_Time_History'Identity, "in time_history file " &
           File_Name & " at line" &
           Integer'Image(Time_History_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Time_History_Tokenizer.Column(Tokenizer)) &
           ". Fewer variable indices provided than the number of variables specified");

      end case;

   end Report_Error;


   type Header_Parse_State is
     ( Start, Comment_Read, Count_Read, Ready_For_Variable, Variable_Read );

   procedure Open_Input_File
     ( A_Filename : in String;
       A_Handle : out Handle;
       A_Variable_Count : out Integer;
       A_Variable_Index_Array : out Variable_Index_Array ) is

      The_State : Header_Parse_State := Start;

      I : Integer := 0;
      -- the variable index count

      The_Token : Csv_Token_Id;

   begin
      A_Variable_Count := 0;
      A_Handle := new Instance;

      -- open the file, and put the resulting file object in a newly
      -- allocated instance
      if not Ada.Text_Io.Is_Open(File) then
        Ada.Text_Io.Open ( File => File,
                         Name => A_Filename,
                         Mode => Ada.Text_IO.In_File );
      end if;

      -- copy the name of the file into the instance

      A_Handle.The_Filename_Length := A_Filename'Length;
      A_Handle.The_Filename ( 1 .. A_Filename'Length ) := A_Filename;

      A_Handle.The_Tokenizer
        := Time_History_Tokenizer.Initialize
        (Language_Syntax => Time_History_Syntax,
         Feeder          => Get_Text_Line'access);

      -- read lines until a non-comment token is read
      loop
         Time_History_Tokenizer.Find_Next ( A_Handle.The_Tokenizer );

         The_Token := Time_History_Tokenizer.Token ( A_Handle.The_Tokenizer );
    --     Ada.Text_Io.Put_Line ( Csv_Token_Id'Image ( The_Token ) & " : " &
    --                            Header_Parse_State'Image ( The_State ) & " : " &
    --                            Time_History_Tokenizer.Lexeme ( A_Handle.The_Tokenizer ) );

         case The_State is
            when Start =>
               case The_Token is
                  when Comment =>
                     The_State := Comment_Read;
                  when Int =>
                     A_Variable_Count :=
                       Integer'Value
                       ( Time_History_Tokenizer.Lexeme ( A_Handle.The_Tokenizer ) );
                     if A_Variable_Count > Max_Variables then
                        Report_Error(A_Handle, Number_Of_Variables_Too_Large);
                     elsif A_Variable_Count < 1 then
                        Report_Error(A_Handle, Number_Of_Variables_Too_Small);
                     end if;
                     A_Handle.The_Variable_Count := A_Variable_Count;
                     The_State := Count_Read;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "comment or integer");
               end case;
            when Comment_Read =>
               case The_Token is
                  when Eol =>
                     The_State := Start;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "end of line");
               end case;

            when Count_Read =>
               case The_Token is
                  when Eol =>
                     The_State := Ready_For_Variable;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "end of line");
               end case;
            when Ready_For_Variable =>
               case The_Token is
                  when Int =>
                     I := I + 1;
                     if I > A_Variable_Count then
                        Report_Error ( A_Handle, Too_Many_Variables );
                     end if;
                     A_Variable_Index_Array(I) :=
                       Integer'Value
                       ( Time_History_Tokenizer.Lexeme ( A_Handle.The_Tokenizer ) );
                     The_State := Variable_Read;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "integer");
               end case;
            when Variable_Read =>
               case The_Token is
                  when Comma =>
                     The_State := Ready_For_Variable;
                  when Eol =>
                     if I < A_Variable_Count then
                        Report_Error ( A_Handle, Too_Few_Variables );
                     end if;
                     exit;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "comma or end of line");
               end case;
         end case;
      end loop;

   end Open_Input_File;

   type Value_Parse_State is
     ( Ready_For_Value, Value_Read );

   procedure Read_Input_Line
     ( A_Handle : in Handle;
       A_Value_Array : out Value_Array;
       No_More_Rows : out Boolean ) is
      The_State : Value_Parse_State := Ready_For_Value;
      I : Integer := 0;
      -- the variable index count
      The_Token : Csv_Token_Id;
   begin

      No_More_Rows := False;
  --    Ada.Text_Io.Put_Line ( "Reading input line" );

      loop
         Time_History_Tokenizer.Find_Next ( A_Handle.The_Tokenizer );

         The_Token := Time_History_Tokenizer.Token ( A_Handle.The_Tokenizer );

  --       Ada.Text_Io.Put_Line ( Csv_Token_Id'Image ( The_Token ) & " : " &
  --                              Value_Parse_State'Image ( The_State ) & " : " &
  --                              Time_History_Tokenizer.Lexeme ( A_Handle.The_Tokenizer ) );

         case The_State is
            when Ready_For_Value =>
               case The_Token is
                  when Real =>
                     I := I + 1;
                     if I > A_Handle.The_Variable_Count then
                        Report_Error ( A_Handle, Too_Many_Values );
                     end if;
                     A_Value_Array(I) :=
                       Float'Value
                       ( Time_History_Tokenizer.Lexeme ( A_Handle.The_Tokenizer ) );
                     The_State := Value_Read;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "real");
               end case;
            when Value_Read =>
               case The_Token is
                  when Comma =>
                     The_State := Ready_For_Value;
                  when Eol | Eof =>
                     if I < A_Handle.The_Variable_Count then
                        Report_Error ( A_Handle, Too_Few_Values );
                     end if;
                     if The_Token = Eof then
                        No_More_Rows := True;
                     end if;
                     exit;
                  when others =>
                     Syntax_Error (A_Handle, The_Token, "comma or end of line");
               end case;
         end case;
      end loop;

   end Read_Input_Line;

   procedure Close_Input_File
     ( A_Handle : in Handle ) is
   begin
      if Ada.Text_IO.Is_Open(File) then
         Ada.Text_Io.Close(File);
      end if;

   end Close_Input_File;

end Time_History_G;

