--
--            FlightSafety International Simulation Systems Division
--                     Broken Arrow, OK  USA  918-259-4000
--
--                  JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Ted Dennison
--
--  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--  technical or operational data or information from automatic dissemination
--  under the International Exchange Program or by other means.  This protection
--  covers information required solely for administrative or operational
--  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
--  WARNING:  This document contains technical data whose export is restricted
--  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--  Executive Order 12470.  Violation of these export control laws is subject
--  to severe criminal penalties.  Dissemination of this document is controlled
--  under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Interfaces.C;
with Interfaces.C.Strings;

use type Interfaces.C.Int;
use type Interfaces.C.Size_T;

-------------------------------------------------------------------------------
-- This package implements an Ada interface to the Stethescope real-time data
-- monitor. The current version supported by this package is 5.1
-------------------------------------------------------------------------------
package body Stethoscope is

   Next_Uninitialized_Index : constant := -1;

   type Boolean_Ptr is access Boolean;
   type Integer_Ptr is access Integer;

   -- As defined by a macro in rtitypes.h. This is the vxWorks value
   type C_Boolean is new Interfaces.C.Int;

   function ScopeSetCollectionMode
     (Mode        : in Collection_Mode;
      Scope_Index : in Interfaces.C.Int ) return C_Boolean;

   procedure ScopeShutdown (Scope_Index : in Interfaces.C.Int);

   function ScopeRegisterSignal( Name              : in Interfaces.C.Char_Array;
                                 Units             : in Interfaces.C.Char_Array;
                                 Ptr_To_Static_Var : in System.Address;
                                 Value_Type        : in Interfaces.C.Char_Array;
                                 Scope_Index       : in Interfaces.C.Int ) return C_Boolean;

   function ScopeRegisterSignalWithOffset( Name              : in Interfaces.C.Char_Array;
                                           Units             : in Interfaces.C.Char_Array;
                                           Ptr_To_Static_Var : in System.Address;
                                           Value_Type        : in Interfaces.C.Char_Array;
                                           Offset            : in Interfaces.C.Int;
                                           Scope_Index       : in Interfaces.C.Int ) return C_Boolean;

   function ScopeInstallSignalWithOffset( Name              : in Interfaces.C.Char_Array;
                                          Units             : in Interfaces.C.Char_Array;
                                          Ptr_To_Static_Var : in System.Address;
                                          Value_Type        : in Interfaces.C.Char_Array;
                                          Offset            : in Interfaces.C.Unsigned;
                                          Scope_Index       : in Interfaces.C.Int ) return C_Boolean;

   function ScopeActivateSignal( name       : in Interfaces.C.Char_Array;
                                 scopeIndex : in Interfaces.C.Int ) return C_boolean;

   function ScopeDeactivateSignal( name       : in Interfaces.C.Char_Array;
                                   scopeIndex : in Interfaces.C.Int ) return C_boolean;

   function ScopeMaxSamples( scopeIndex : in Interfaces.C.Int ) return Interfaces.C.Int;

   function ScopeRemoveSignal( Name_Ptr    : in Interfaces.C.Char_Array;
                               Scope_Index : in Interfaces.C.Int ) return C_boolean;


   function ScopeRemoveMultipleSignals( Name_Start   : in Interfaces.C.Char_Array;
                                        Scope_Index  : in Interfaces.C.Int
                                      ) return Interfaces.C.Int;

   procedure ScopeShowActiveSignals( Scope_Index : in Interfaces.C.Int );

   procedure ScopeShowSignals( Scope_Index : in Interfaces.C.Int );

   function ScopeRequestSamplingParameters( Sample_Divisor    : access Interfaces.C.Int;
                                            Number_Of_Samples : access Interfaces.C.Int;
                                            ScopeIndex        : in     Interfaces.C.Int
                                          ) return C_boolean;

   procedure ScopeCollectSignals( Scope_Index : in Interfaces.C.Int );

   procedure ScopeChangeSampleRate( newSampleRate : in Interfaces.C.Double;
                                    scopeIndex    : in Interfaces.C.Int );

   function ScopeSetTrigger( Source       : in Interfaces.C.Char_Array;
                             Level        : in Interfaces.C.Double;
                             Slope        : in Interfaces.C.int;
                             Delay_Amount : in Interfaces.C.C_Float;
                             Holdoff      : in Interfaces.C.C_Float;
                             ScopeIndex   : in Interfaces.C.int ) return C_boolean;



   function ScopeGetTrigger( Source       : in     Interfaces.C.Strings.Chars_Ptr;
                             Level        : access Interfaces.C.Double;
                             Slope        : access Interfaces.C.Int;
                             Delay_Amount : access Interfaces.C.C_Float;
                             Holdoff      : access Interfaces.C.C_Float;
                             Armed        : access Interfaces.C.Int;
                             ScopeIndex   : in     Interfaces.C.Int) return C_boolean;



   procedure ScopeSendScript( Script     : in Interfaces.C.Char_Array;
                              ScopeIndex : in Interfaces.C.Int );


   procedure ScopeClearScript( ScopeIndex : in Interfaces.C.Int );


   procedure ScopeSendAnnotation (CoordX     : in Interfaces.C.Double;
                                  coordY     : in Interfaces.C.Double;
                                  Text       : in Interfaces.C.Char_Array;
                                  ScopeIndex : in Interfaces.C.Int );


   procedure ScopeClearAnnotation( ScopeIndex : in Interfaces.C.Int );


   procedure ScopePrintVersion;


   function ScopeInitServerEx
     (DataBufSize         : in Interfaces.C.Int;
      SignalBufSize       : in Interfaces.C.Int;
      Debuglevel          : in Interfaces.C.Int;
      ProbeDaemonPriority : in Interfaces.C.Int;
      LinkDaemonPriority  : in Interfaces.C.Int;
      ScopeIndex          : in Interfaces.C.Int ) return Interfaces.C.Int;

   function ScopeVarExprAddrFindCallbackSet
     (Func       : in Variable_Address_Find;
      UserData   : in System.Address;
      ScopeIndex : in Interfaces.C.Int) return C_boolean;

   pragma import( c, ScopeInitServerEx, "ScopeInitServerEx" );
   pragma import( c, ScopeShutdown, "ScopeShutdown");
   pragma import( c, ScopePrintVersion, "ScopePrintVersion" );
   pragma import( c, ScopeSendScript, "ScopeSendScript" );
   pragma import( c, ScopeClearScript, "ScopeClearScript" );
   pragma import( c, ScopeSendAnnotation, "ScopeSendAnnotation" );
   pragma import( c, ScopeClearAnnotation, "ScopeClearAnnotation" );
   pragma import( c, ScopeChangeSampleRate, "ScopeChangeSampleRate" );
   pragma import( c, ScopeSetCollectionMode, "ScopeSetCollectionMode");
   pragma import( c, ScopeCollectSignals, "ScopeCollectSignals" );
   pragma import( c, ScopeMaxSamples,"ScopeMaxSamples");
   pragma import( c, ScopeRequestSamplingParameters, "ScopeRequestSamplingParameters");
   pragma import( c, ScopeGetTrigger, "ScopeGetTrigger" );
   pragma import( c, ScopeSetTrigger, "ScopeSetTrigger" );
   pragma import( c, ScopeRegisterSignal, "ScopeRegisterSignal" );
   pragma import( c, ScopeRegisterSignalWithOffset, "ScopeRegisterSignalWithOffset" );
   pragma import( c, ScopeRemoveSignal, "ScopeRemoveSignal" );
   pragma import( c, ScopeRemoveMultipleSignals, "ScopeRemoveMultipleSignals" );
   pragma import( c, ScopeActivateSignal, "ScopeActivateSignal" );
   pragma import( c, ScopeDeactivateSignal, "ScopeDeactivateSignal" );
   pragma import( c, ScopeInstallSignalWithOffset, "ScopeInstallSignalWithOffset" );
   pragma import( c, ScopeShowSignals, "ScopeShowSignals" );
   pragma import( c, ScopeShowActiveSignals, "ScopeShowActiveSignals" );
   pragma import( c, ScopeVarExprAddrFindCallbackSet, "ScopeVarExprAddrFindCallbackSet" );

   ----------------------------------------------------------------------------
   -- Register an object at the given address with the signal manager.
   ----------------------------------------------------------------------------
   procedure Register_Signal
     (Name           : in String;
      Units          : in String;
      Object_Address : in System.Address;
      Value_Type     : in String;
      Scope_Index    : in Stethoscope_Index := Default_Scope_Index) is

   begin
      if
        Scoperegistersignal
        (Name              => Interfaces.C.To_C (Name),
         Units             => Interfaces.C.To_C (Units),
         Ptr_To_Static_Var => Object_Address,
         Value_Type        => Interfaces.C.To_C (Value_Type),
         Scope_Index       => Interfaces.C.Int(Scope_Index)) = 0
      then
         raise Operation_Failed;
      end if;
   end Register_Signal;


   ----------------------------------------------------------------------------
   -- Register an object at the given offset from the given address with
   -- the signal manager.
   -- Note that the Object_Address must be the address of *the*address* of
   -- the object from which the ofset will be calculated.
   ----------------------------------------------------------------------------
   procedure Register_Signal_With_Offset
     (Name           : in String;
      Units          : in String;
      Object_Address : in System.Address;
      Value_Type     : in String;
      Offset         : in Integer := 0;
      Scope_Index    : in Stethoscope_Index := Default_Scope_Index) is

   begin
      if
        Scoperegistersignalwithoffset
        (Name              => Interfaces.C.To_C (Name),
         Units             => Interfaces.C.To_C (Units),
         Ptr_To_Static_Var => Object_Address,
         Value_Type        => Interfaces.C.To_C (Value_Type),
         Offset            => Interfaces.C.Int(Offset),
         Scope_Index       => Interfaces.C.Int(Scope_Index)) = 0
      then
         raise Operation_Failed;
      end if;
   end Register_Signal_With_Offset;


   ----------------------------------------------------------------------------
   -- Install an object at the given offset from the given address with
   -- Stethescope.
   ----------------------------------------------------------------------------
   procedure Install_Signal
     (Name           : in String;
      Units          : in String;
      Object_Address : in System.Address;
      Value_Type     : in String;
      Offset         : in Integer := 0;
      Scope_Index    : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        Scopeinstallsignalwithoffset
        (Name              => Interfaces.C.To_C (Name),
         Units             => Interfaces.C.To_C (Units),
         Ptr_To_Static_Var => Object_Address,
         Value_Type        => Interfaces.C.To_C (Value_Type),
         Offset            => Interfaces.C.Unsigned(Offset),
         Scope_Index       => Interfaces.C.Int(Scope_Index)) = 0
      then
         raise Operation_Failed;
      end if;
   end Install_Signal;

   ----------------------------------------------------------------------------
   -- Activate the given signal.
   ----------------------------------------------------------------------------
   procedure Activate_Signal
     (Name        : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        Scopeactivatesignal
        (name       => Interfaces.C.To_C (Name),
         scopeIndex => Interfaces.C.Int(Scope_Index)
         ) = 0
      then
         raise Operation_Failed;
      end if;
   end Activate_Signal;

   ----------------------------------------------------------------------------
   -- Deactivate the given signal.
   ----------------------------------------------------------------------------
   procedure Deactivate_Signal
     (Name        : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        Scopedeactivatesignal
        (name       => Interfaces.C.To_C (Name),
         scopeIndex => Interfaces.C.Int(Scope_Index)
         ) = 0
      then
         raise Operation_Failed;
      end if;
   end Deactivate_Signal;

   ----------------------------------------------------------------------------
   -- Return the scopes maximum samples.
   ----------------------------------------------------------------------------
   function Max_Samples (Scope_Index : in Stethoscope_Index := Default_Scope_Index)
     return Integer is
   begin
      return Integer(ScopeMaxSamples(Interfaces.C.Int(Scope_Index)));
   end Max_Samples;

   ----------------------------------------------------------------------------
   -- Remove the given signal.
   ----------------------------------------------------------------------------
   procedure Remove_Signal
     (Name        : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        Scoperemovesignal
        ( Name_Ptr    => Interfaces.C.To_C (Name),
          Scope_Index => Interfaces.C.Int (Scope_Index)
          ) = 0
      then
         raise Operation_Failed;
      end if;

   end Remove_Signal;

   ----------------------------------------------------------------------------
   -- Remove all signals whose names start with the given prefix
   ----------------------------------------------------------------------------
   procedure Remove_Signals
     (Name_Prefix : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        Scoperemovemultiplesignals
        (Name_Start   => Interfaces.C.To_C (Name_Prefix),
         Scope_Index  => Interfaces.C.Int (Scope_Index)
         ) > 0
      then
         raise Operation_Failed;
      end if;

   end Remove_Signals;

   ----------------------------------------------------------------------------
   -- Show all the signals.
   ----------------------------------------------------------------------------
   procedure Show_Signals (Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      ScopeShowSignals (Interfaces.C.Int (Scope_Index));
   end Show_Signals;


   ----------------------------------------------------------------------------
   -- Show all the signals that are being collected.
   ----------------------------------------------------------------------------
   procedure Show_Active_Signals (Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      ScopeShowActiveSignals (Interfaces.C.Int (Scope_Index));
   end Show_Active_Signals;


   ----------------------------------------------------------------------------
   -- Get the sampling rate parameters
   ----------------------------------------------------------------------------
   procedure Sampling_Rate_Parameters
     (Sample_Divisor    :    out Integer;
      Number_Of_Samples :    out Integer;
      Scope_Index       : in     Stethoscope_Index := Default_Scope_Index) is

      Divisor : aliased Interfaces.C.Int := -1;
      Number  : aliased Interfaces.C.Int := -1;
   begin
      if
        ScopeRequestSamplingParameters
        (Sample_Divisor    => Divisor'Access,
         Number_Of_Samples => Number'Access,
         ScopeIndex        => Interfaces.C.Int(Scope_Index)
         ) = 0
      then
         raise Operation_Failed;
      end if;

      Sample_Divisor := Integer(Divisor);
      Number_Of_Samples := Integer(Number);
   end Sampling_Rate_Parameters;

   ----------------------------------------------------------------------------
   -- Set the sampling rate parameters
   ----------------------------------------------------------------------------
   procedure Set_Sampling_Rate_Parameters
     (Sample_Divisor    : in Positive;
      Number_Of_Samples : in Positive;
      Scope_Index       : in Stethoscope_Index := Default_Scope_Index) is

      -- The routine modifies the arguments, so we need temp variables
      Divisor : aliased Interfaces.C.Int := Interfaces.C.Int (Sample_Divisor);
      Number  : aliased Interfaces.C.Int := Interfaces.C.Int (Number_Of_Samples);
   begin

      if
        ScopeRequestSamplingParameters
        (Sample_Divisor    => Divisor'access,
         Number_Of_Samples => Number'access,
         ScopeIndex        => Interfaces.C.Int(Scope_Index)
         ) = 0
      then
         raise Operation_Failed;
      end if;
   end Set_Sampling_Rate_Parameters;

   ----------------------------------------------------------------------------
   -- Collect Signals.
   ----------------------------------------------------------------------------
   procedure Collect_Signals (Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      ScopeCollectSignals (Interfaces.C.Int(Scope_Index));
   end Collect_Signals;

   ----------------------------------------------------------------------------
   -- Change the sample rate to the given (Hz) value
   ----------------------------------------------------------------------------
   procedure Change_Sample_Rate
     (New_Sample_Rate : in Long_Float;
      Scope_Index     : in Stethoscope_Index := Default_Scope_Index) is
   begin
      Scopechangesamplerate
        ( Newsamplerate => Interfaces.C.Double(New_Sample_Rate),
          Scopeindex    => Interfaces.C.Int (Scope_Index)
          );
   end Change_Sample_Rate;

   ----------------------------------------------------------------------------
   -- Set the trigger on a signal to the given level and slope.
   ----------------------------------------------------------------------------
   procedure Set_Trigger
     (Source       : in String;
      Level        : in Long_Float;
      Slope        : in Integer;
      Delay_Amount : in Float;
      Hold_Off     : in Float;
      Scope_Index  : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        ScopeSetTrigger( Source       => Interfaces.C.To_C (Source),
                         Level        => Interfaces.C.Double (Level),
                         Slope        => Interfaces.C.Int (Slope),
                         Delay_Amount => Interfaces.C.C_Float (Delay_Amount),
                         Holdoff      => Interfaces.C.C_Float (Hold_Off),
                         ScopeIndex   => Interfaces.C.Int (Scope_Index)
                         ) = 0
      then
         raise Operation_Failed;
      end if;
   end Set_Trigger;

   ----------------------------------------------------------------------------
   -- Get the current trigger arguments.
   ----------------------------------------------------------------------------
   procedure Get_Trigger
     (Source       :    out String;
      Source_Size  :    out Natural;
      Level        :    out Long_Float;
      Slope        :    out Integer;
      Delay_Amount :    out Float;
      Hold_Off     :    out Float;
      Armed        :    out Boolean;
      Scope_Index  : in     Stethoscope_Index := Default_Scope_Index) is

      C_Source : aliased Interfaces.C.Char_Array :=
        (1..Source'Length + 1 => Interfaces.C.To_C (' '));

      C_Level    : aliased Interfaces.C.Double;
      C_Slope    : aliased Interfaces.C.Int;
      C_Delay    : aliased Interfaces.C.C_Float;
      C_Hold_Off : aliased Interfaces.C.C_Float;
      C_Armed    : aliased Interfaces.C.Int;

   begin
      if
        ScopeGetTrigger( Source       => Interfaces.C.Strings.To_Chars_Ptr(C_Source'Unchecked_Access),
                         Level        => C_Level'Unchecked_Access,
                         Slope        => C_Slope'Unchecked_Access,
                         Delay_Amount => C_Delay'Unchecked_Access,
                         Holdoff      => C_Hold_Off'Unchecked_Access,
                         Armed        => C_Armed'Unchecked_Access,
                         ScopeIndex   => Interfaces.C.Int (Scope_Index)
                         ) = 0

      then
         raise Operation_Failed;
      end if;

      -- Assign the C values back to the Ada output variables
      Source_Size := Natural(Interfaces.C.Strings.Strlen
                             (Interfaces.C.Strings.To_Chars_Ptr
                              (C_Source'Unchecked_Access)));
      Source (Source'First..Source'First + Source_Size - 1) :=
        Interfaces.C.To_Ada (C_Source);
      Level        := Long_Float (C_Level);
      Slope        := Integer (C_Slope);
      Delay_Amount := Float (C_Delay);
      Hold_Off     := Float (C_Hold_Off);
      Armed        := C_Armed /= 0;
   end Get_Trigger;

   ----------------------------------------------------------------------------
   -- Send a script to stethescope
   ----------------------------------------------------------------------------
   procedure Send_Script
     (Script      : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      Scopesendscript (Script     => Interfaces.C.To_C (Script),
                       ScopeIndex => Interfaces.C.Int (Scope_Index)
                       );
   end Send_Script;

   ----------------------------------------------------------------------------
   -- Clear any scripts
   ----------------------------------------------------------------------------
   procedure Clear_Script (Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      Scopeclearscript (Interfaces.C.Int (Scope_Index));
   end Clear_Script;

   ----------------------------------------------------------------------------
   -- Put an annontation on the scope display at the given coordinates.
   ----------------------------------------------------------------------------
   procedure Send_Annotation
     (X_Coordinate : in Long_Float;
      Y_Coordinate : in Long_Float;
      Text         : in String;
      Scope_Index  : in Stethoscope_Index := Default_Scope_Index) is
   begin
      ScopeSendAnnotation (CoordX     => Interfaces.C.Double(X_Coordinate),
                           coordY     => Interfaces.C.Double(Y_Coordinate),
                           Text       => Interfaces.C.To_C (Text),
                           ScopeIndex => Interfaces.C.Int (Scope_Index)
                           );
   end Send_Annotation;

   ----------------------------------------------------------------------------
   -- Clear an annotation
   ----------------------------------------------------------------------------
   procedure Clear_Annotation (Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      Scopeclearannotation (Interfaces.C.Int (Scope_Index));
   end Clear_Annotation;

   ----------------------------------------------------------------------------
   -- Print the version string.
   ----------------------------------------------------------------------------
   procedure Print_Version is
   begin
      Scopeprintversion;
   end Print_Version;

   ----------------------------------------------------------------------------
   -- Initialize the stethescope server with the given buffer size and debug
   -- level.
   ----------------------------------------------------------------------------
   procedure Initialize_Server
     (Data_Buffer_Size   : in Natural := Default_Buffer_Size;
      Signal_Buffer_Size : in Natural := Default_Buffer_Size;
      Debug_Level        : in Debugging_Level := Error_Status;
      Probe_Priority     : in Natural := Default_Priority;
      Link_Priority      : in Natural := Default_Priority;
      Scope_Index        : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        ScopeInitServerEx (DataBufSize         => Interfaces.C.Int (Data_Buffer_Size),
                           SignalBufSize       => Interfaces.C.Int (Signal_Buffer_Size),
                           Debuglevel          => Interfaces.C.Int (Debugging_Level'Pos(Debug_Level)),
                           ProbeDaemonPriority => Interfaces.C.Int (Probe_Priority),
                           LinkDaemonPriority  => Interfaces.C.Int (Link_Priority),
                           Scopeindex          => Interfaces.C.Int (Scope_Index)
                         ) < 0
      then
         raise Operation_Failed;
      end if;
   end Initialize_Server;

   ----------------------------------------------------------------------------
   -- Start up the stethescope real-time daemons with the given buffer sizes,
   -- daemon priorities, and and debug level using any available Index.
   ----------------------------------------------------------------------------
   procedure Initialize_Any_Server
     (Data_Buffer_Size   : in      Natural := Default_Buffer_Size;
      Signal_Buffer_Size : in      Natural := Default_Buffer_Size;
      Debug_Level        : in      Debugging_Level := Error_Status;
      Probe_Priority     : in      Natural := Default_Priority;
      Link_Priority      : in      Natural := Default_Priority;
      Scope_Index        :     out Stethoscope_Index) is
      Result : Interfaces.C.Int;
   begin
      Result := ScopeInitServerEx
        (DataBufSize         => Interfaces.C.Int (Data_Buffer_Size),
         SignalBufSize       => Interfaces.C.Int (Signal_Buffer_Size),
         Debuglevel          => Interfaces.C.Int (Debugging_Level'Pos(Debug_Level)),
         ProbeDaemonPriority => Interfaces.C.Int (Probe_Priority),
         LinkDaemonPriority  => Interfaces.C.Int (Link_Priority),
         Scopeindex          => Next_Uninitialized_Index
         );
      if Result < 0 then
         raise Operation_Failed;
      else
         Scope_Index := Stethoscope_Index(Result);
      end if;
   end Initialize_Any_Server;


   ----------------------------------------------------------------------------
   -- Set the data collection mode.
   ----------------------------------------------------------------------------
   procedure Set_Collection_Mode
     (Mode        : in Collection_Mode;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        ScopeSetCollectionMode (Mode => Mode,
                                Scope_Index => Interfaces.C.Int (Scope_Index)
                                ) = 0
      then
         raise Operation_Failed;
      end if;
   end Set_Collection_Mode;

   ----------------------------------------------------------------------------
   -- Shut down the scope with the given index.
   ----------------------------------------------------------------------------
   procedure Shutdown (Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      ScopeShutdown (Interfaces.C.Int (Scope_Index));
   end Shutdown;

   ----------------------------------------------------------------------------
   -- Set the callback function for variable expression requests.
   ----------------------------------------------------------------------------
   procedure Set_Variable_Expression_Address_Find_Callback
     (Callback    : in Variable_Address_Find;
      User_Data   : in System.Address;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index) is
   begin
      if
        ScopeVarExprAddrFindCallbackSet
        (Func       => Callback,
         UserData   => User_Data,
         ScopeIndex => Interfaces.C.Int (Scope_Index)
         ) = 0
      then
         raise Operation_Failed;
      end if;
   end Set_Variable_Expression_Address_Find_Callback;

end Stethoscope;
