-------------------------------------------------------------------------------
--
--            FlightSafety International Simulation Systems Division
--                     Broken Arrow, OK  USA  918-259-4000
--
--                  JPATS T-6A Texan-II Flight Training Device
--
--
--   Engineer:  Ted Dennison
--
--  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--  technical or operational data or information from automatic dissemination
--  under the International Exchange Program or by other means.  This protection
--  covers information required solely for administrative or operational
--  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
--  WARNING:  This document contains technical data whose export is restricted
--  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--  Executive Order 12470.  Violation of these export control laws is subject
--  to severe criminal penalties.  Dissemination of this document is controlled
--  under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Unchecked_Conversion;
with System.Storage_Elements;

-------------------------------------------------------------------------------
-- This package implements an Ada interface to the ScopeProfile real-time
-- execution profiler. The current version supported by this package is 2.3
-------------------------------------------------------------------------------
package body ScopeProfile is

   function ProfileInit ( clock       : in Clock_Selection;
                          tid         : in Interfaces.C.Int;
                          granularity : in Granularity_Designation;
                          scopeIndex  : in Interfaces.C.Int;
                          maxSamples  : in Interfaces.C.Int;
                          maxStack    : in Interfaces.C.Int;
                          scopeBufferSize       : in Interfaces.C.Int;
                          scopeSignalBufferSize : in Interfaces.C.Int
                        ) return System.Address;
   pragma Import(C, ProfileInit, "ProfileInit" );

   procedure ProfileSample ( Profile_Info : in Instance);
   pragma Import(C, ProfileSample, "ProfileSample" );

   procedure ProfileFreeze( Freeze       : in Interfaces.C.Int;
                            Profile_Info : in Instance );
   pragma Import(C, ProfileFreeze, "ProfileFreeze" );

   type Boolean_Int_Map is array (Boolean) of Interfaces.C.Int;

   Int_Map : constant Boolean_Int_Map := (False => 0, True => 1);

   -----------------------------------------------------------------------------
   -- This routine initializes the profiler. It also optionally attaches the
   -- profilers to a clock and begins profile sampling.
   --
   -- This routine spawns an "analysis process" at a priority of 250. It also
   -- spawns a "sampler process" at priority 0. These priorities can be changed
   -- with taskPrioritySet if desired. It sets the sampling rate to 107.0 (or to
   -- the current system clock rate if there is no aux clock), and the analysis
   -- rate to 1.0 Hz. Both these can be changed as required.
   --
   -- This routine allocates and returns a data structure that indicates an
   -- active profile instance.
   -----------------------------------------------------------------------------
   function Initialize
     (Clock       : Clock_Selection         := No_Clock;
      Task_ID     : Interfaces.C.Int        := All_Tasks;
      Granularity : Granularity_Designation := Flat_Normal;
      Scope_Index : Index                   := Uninitialized_Scope;
      Max_Samples : Natural                 := Default_Size;
      Max_Stack   : Natural                 := Default_Size;
      Buffer_Size : Natural                 := Default_Size;
      Signal_Size : Natural                 := Default_Size
      ) return Instance is
      Result : Instance;
   begin
      Result :=  Instance
        (ProfileInit
         (clock       => Clock,
          tid         => Task_ID,
          granularity => Granularity,
          scopeIndex  => Interfaces.C.Int(Scope_Index),
          maxSamples  => Interfaces.C.Int(Max_Samples),
          maxStack    => Interfaces.C.Int(Max_Stack),
          scopeBufferSize       => Interfaces.C.Int(Buffer_Size),
          scopeSignalBufferSize => Interfaces.C.Int(Signal_Size)
          )
         );
      if Result = Instance(System.Null_Address) then
         raise Error;
      end if;
      return Result;
   end Initialize;

   -----------------------------------------------------------------------------
   -- This routine need only be called for implementations utilizing a user
   -- clock. Most implementations (that supplied a clock to Initialize) should
   -- ignore this routine.
   -- Sample may be called at interrupt priority.
   -----------------------------------------------------------------------------
   procedure Sample (Profile : in Instance) is
   begin
      ProfileSample (Profile);
   end Sample;

   -----------------------------------------------------------------------------
   -- Profile sampling is temporarily suspended. A specific execution period can
   -- be profilesd by calling "ProfileClear", executing the code, and then
   -- calling "ProfileFreeze". A specific section of code can be profiled by
   -- bracketing the code between calls to "ProfileFreeze".
   -- Freezing an already frozen profile or thawing an already thawed profile
   -- has no effect.
   -----------------------------------------------------------------------------
   procedure Freeze
     (Profile   : in Instance;
      Freeze_On : in Boolean := True
     ) is
   begin
      ProfileFreeze
        (Freeze       => Int_Map (Freeze_On),
         Profile_Info => Profile
         );
   end Freeze;

   -----------------------------------------------------------------------------
   -- Return a string representation of the given profile.
   -----------------------------------------------------------------------------
   function Image (Profile : Instance) return String is
   begin
      return System.Storage_Elements.Integer_Address'Image
        (System.Storage_Elements.To_Integer(System.Address(Profile)));
   end Image;

end ScopeProfile;
