--
--            FlightSafety International Simulation Systems Division
--                     Broken Arrow, OK  USA  918-259-4000
--
--                  JPATS T-6A Texan-II Flight Training Device
--
--
--   Engineer:  Ted Dennison
--
--  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--  technical or operational data or information from automatic dissemination
--  under the International Exchange Program or by other means.  This protection
--  covers information required solely for administrative or operational
--  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
--  WARNING:  This document contains technical data whose export is restricted
--  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--  Executive Order 12470.  Violation of these export control laws is subject
--  to severe criminal penalties.  Dissemination of this document is controlled
--  under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with System;
with Ada.Strings.Unbounded;
with Interfaces.C.Strings;

-------------------------------------------------------------------------------
-- This package implements an Ada interface to the Stethescope real-time data
-- monitor. The current version supported by this package is 5.1
-------------------------------------------------------------------------------
package Stethoscope is

   Operation_Failed : exception;

   Default_Priority    : constant := 0;
   Default_Buffer_Size : constant := 0;

   type Debugging_Level is (Off, Error_Status, Dump);
   type Collection_Mode is ( Disabled, One_Shot, Continuous, Single_Sweep, Normal );

   type Stethoscope_Index is range 0..15;
   Default_Scope_Index : Stethoscope_Index := 0;

   ----------------------------------------------------------------------------
   -- Set the data collection mode.
   -- If Mode is Disabled, and a collection cycle is active (i.e. date is
   -- currently being collected into the buffer), then the cycle will be
   -- completed. If Mode is One_Shot, the very next sample (call to
   -- Collect_Signals), is guaranteed to be collected, regardless of the
   -- "sample divisor" set in the main StethoScope window. If Mode is
   -- Single_Sweep or Normal, it is assumed that the triggering conditions
   -- have been previously set via Set_Trigger. In One_Shot and Single_Sweep
   -- modes, as opposed to Continuous and Normal modes, data collectino stops
   -- after one cycle.
   --
   -- If pretriggering is in effect, and the trigger event occurs before the
   -- requested number of "pre-collected" samples have been obtained, the
   -- trigger event will be accepted. See the documentation under "Data
   -- Collection" in the manual or from the on-line help utility for details.
   ----------------------------------------------------------------------------
   procedure Set_Collection_Mode
     (Mode        : in Collection_Mode;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Register an object at the given address with the signal manager.
   --
   -- Putting a '/' character in the Name will make the portion before the
   -- slash act like a directory on the Stethoscope display.
   --
   -- Value_Type must be one of:
   -- unsigned char     uchar        char
   -- unsigned short    ushort       short
   -- unsigned int      uint         int
   -- unsigned long     ulong        long
   -- float             double
   -- Also pointers may be used by following any of the above with one or more
   -- '*' characters.
   --
   -- If you use a local variable's address for Object_Address, very bad things
   -- will happen.
   ----------------------------------------------------------------------------
   procedure Register_Signal
     (Name           : in String;
      Units          : in String;
      Object_Address : in System.Address;
      Value_Type     : in String;
      Scope_Index    : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Register an object at the given offset from the given address with
   -- the signal manager.
   -- Note that the Object_Address must be the address of *the*address* of
   -- the object from which the ofset will be calculated.
   ----------------------------------------------------------------------------
   procedure Register_Signal_With_Offset
     (Name           : in String;
      Units          : in String;
      Object_Address : in System.Address;
      Value_Type     : in String;
      Offset         : in Integer := 0;
      Scope_Index    : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Register and activate an object at the given offset from the given
   -- address.
   ----------------------------------------------------------------------------
   procedure Install_Signal
     (Name           : in String;
      Units          : in String;
      Object_Address : in System.Address;
      Value_Type     : in String;
      Offset         : in Integer := 0;
      Scope_Index    : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Activate the given signal.
   ----------------------------------------------------------------------------
   procedure Activate_Signal
     (Name        : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Deactivate the given signal.
   ----------------------------------------------------------------------------
   procedure Deactivate_Signal
     (Name        : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Return maximum allowable number of samples.
   ----------------------------------------------------------------------------
   function Max_Samples (Scope_Index : in Stethoscope_Index := Default_Scope_Index) return Integer;

   ----------------------------------------------------------------------------
   -- Remove the given signal.
   ----------------------------------------------------------------------------
   procedure Remove_Signal
     (Name        : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Remove all signals whose names start with the given prefix.
   ----------------------------------------------------------------------------
   procedure Remove_Signals
     (Name_Prefix : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Show all the installed signals.
   ----------------------------------------------------------------------------
   procedure Show_Signals (Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Show all the signals that are being collected
   ----------------------------------------------------------------------------
   procedure Show_Active_Signals (Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Get the sampling rate parameters
   ----------------------------------------------------------------------------
   procedure Sampling_Rate_Parameters
     (Sample_Divisor    :    out Integer;
      Number_Of_Samples :    out Integer;
      Scope_Index       : in     Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Set the sampling rate parameters
   ----------------------------------------------------------------------------
   procedure Set_Sampling_Rate_Parameters
     (Sample_Divisor    : in Positive;
      Number_Of_Samples : in Positive;
      Scope_Index       : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Collect a sample from each active signal.
   ----------------------------------------------------------------------------
   procedure Collect_Signals (Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Change the sample rate to the given (Hz) value
   ----------------------------------------------------------------------------
   procedure Change_Sample_Rate
     (New_Sample_Rate : in Long_Float;
      Scope_Index     : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Set the trigger on a signal to the given level and slope.
   ----------------------------------------------------------------------------
   procedure Set_Trigger
     (Source       : in String;
      Level        : in Long_Float;
      Slope        : in Integer;
      Delay_Amount : in Float;
      Hold_Off     : in Float;
      Scope_Index  : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Get the current trigger arguments.
   ----------------------------------------------------------------------------
   procedure Get_Trigger
     (Source       :    out String;
      Source_Size  :    out Natural;
      Level        :    out Long_Float;
      Slope        :    out Integer;
      Delay_Amount :    out Float;
      Hold_Off     :    out Float;
      Armed        :    out Boolean;
      Scope_Index  : in     Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Send a script (eg: Matlab) line to stethescope
   ----------------------------------------------------------------------------
   procedure Send_Script
     (Script      : in String;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Clear the script window.
   ----------------------------------------------------------------------------
   procedure Clear_Script (Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Put an annontation on the scope display at the given coordinates.
   ----------------------------------------------------------------------------
   procedure Send_Annotation
     (X_Coordinate : in Long_Float;
      Y_Coordinate : in Long_Float;
      Text         : in String;
      Scope_Index  : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Clear the annotations.
   ----------------------------------------------------------------------------
   procedure Clear_Annotation
     (Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Print the version string for the scope.
   ----------------------------------------------------------------------------
   procedure Print_Version;

   ----------------------------------------------------------------------------
   -- Start up the stethescope real-time daemons with the given buffer sizes,
   -- daemon priorities, and and debug level.
   ----------------------------------------------------------------------------
   procedure Initialize_Server
     (Data_Buffer_Size   : in Natural := Default_Buffer_Size;
      Signal_Buffer_Size : in Natural := Default_Buffer_Size;
      Debug_Level        : in Debugging_Level := Error_Status;
      Probe_Priority     : in Natural := Default_Priority;
      Link_Priority      : in Natural := Default_Priority;
      Scope_Index        : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Start up the stethescope real-time daemons with the given buffer sizes,
   -- daemon priorities, and and debug level using any available Index.
   ----------------------------------------------------------------------------
   procedure Initialize_Any_Server
     (Data_Buffer_Size   : in      Natural := Default_Buffer_Size;
      Signal_Buffer_Size : in      Natural := Default_Buffer_Size;
      Debug_Level        : in      Debugging_Level := Error_Status;
      Probe_Priority     : in      Natural := Default_Priority;
      Link_Priority      : in      Natural := Default_Priority;
      Scope_Index        :     out Stethoscope_Index);

   ----------------------------------------------------------------------------
   -- Shut down the scope with the given index.
   ----------------------------------------------------------------------------
   procedure Shutdown (Scope_Index : in Stethoscope_Index := Default_Scope_Index);

   ----------------------------------------------------------------------------
   -- Callback function that will be called to obtain the address and type of a
   -- variable expression requested by the host.
   -- An implementation of this routine needs to return the address of the
   -- sampled variable in Variable_Address. Its offset from that address should
   -- be returned in Variable_Offset, or it can be set to -1. Its type needs to
   -- be given one of the valid types listed in the Stethoscope ScopeProbe
   -- documentation entry (with an ASCII null terminating the string).
   ----------------------------------------------------------------------------
   type Variable_Address_Find is access procedure
     (Variable_Expression : in     Interfaces.C.Strings.Chars_Ptr;
      Variable_Address    : in out System.Address;
      Variable_Offset     : in out Interfaces.C.Long;
      Variable_Type       : in out Interfaces.C.Char_Array;
      User_Data           : in     System.Address
      );
   pragma Convention (C, Variable_Address_Find);

   No_Variable_Address_Find : constant Variable_Address_Find := null;

   ----------------------------------------------------------------------------
   -- Set the callback function for variable expression requests.
   ----------------------------------------------------------------------------
   procedure Set_Variable_Expression_Address_Find_Callback
     (Callback    : in Variable_Address_Find;
      User_Data   : in System.Address;
      Scope_Index : in Stethoscope_Index := Default_Scope_Index);

end Stethoscope;
