-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Strings.Unbounded;
with Ada.Strings.Maps;
with Ada.Text_IO;
with Bc.Containers.Maps.Unbounded;
with Bc.Containers.Maps;
with BC.Containers;
with BC.Support.Unmanaged_Storage;
with Interfaces.C.Strings;
with Lock;
with Log;
with Token;
with Token.Analyzer;
with Token.Identifier;
with Token.Integer;
with Token.Keyword;
with Token.String;
with Token.Character_Set;
with Token.End_Of_File;
with Token.Nothing;
with Token.Line_Comment;

--------------------------------------------------------------------------------
-- This package provides a dictionary-style lookup of "definitions" of strings
-- based on lines at the end of the sim.cfg file.
--
-- The lines in the sim.cfg file should contain an identifer followed by an
-- equals sign and a quotation-delimited string or integer. eg:
--   Fred_File_Path = "master/subdir/"
--   Joe_File_Name = "joe.txt"
--   Foo_Size      = 3
--
-- Given the above entries, a lookup of the string "Joe_File_Name" should return
-- the string "joe.txt"
--------------------------------------------------------------------------------
package body Simulation_Dictionary is

   -----------------------------------------------------------------------------
   -- Lookup the given word in the dictionary, and return its definition.
   -- This routine is a C-wrapper for the Lookup routine. It dynamicly allocates
   -- memory for the result string. It is the caller's responsibility to
   -- deallocate this memory.
   -----------------------------------------------------------------------------
   function C_Lookup (Word : in Interfaces.C.Strings.Chars_Ptr) return
     Interfaces.C.Strings.Chars_Ptr;
   pragma Export (C, C_Lookup, "JPATS_Dictionary_Lookup");


   ---------------------------------------------
   -- OpenToken lexical analyzer declarations --
   ---------------------------------------------

   type Dictionary_Token_ID is (Thread, Identifier, Equals, String_Literal, Integer_Literal,
                                EOL, EOF, Whitespace, Comment, Default);

   package Tokenizer is new Token.Analyzer(Dictionary_Token_ID);

   Syntax : Tokenizer.Syntax :=
     (Thread          => new Token.Keyword.Instance'(Token.Keyword.Get("thread")),
      Identifier      => new Token.Identifier.Instance'(Token.Identifier.Get),
      Equals          => new Token.Keyword.Instance'(Token.Keyword.Get("=")),
      String_Literal  => new Token.String.Instance'(Token.String.Get(Escapeable => True)),
      Integer_Literal => new Token.Integer.Instance'(Token.Integer.Get),
      EOL             => new Token.Character_Set.Instance'
      (Token.Character_Set.Get
       (Set        => Ada.Strings.Maps.To_Set (Token.EOL_Character),
        Reportable => True
        )),
      EOF             => new Token.End_Of_File.Instance'(Token.End_Of_File.Get),
      Whitespace      => new Token.Character_Set.Instance'
      (Token.Character_Set.Get (Token.Character_Set.Standard_Whitespace)),
      Comment         => new Token.Line_Comment.Instance'(Token.Line_Comment.Get ("#")),
      Default         => new Token.Nothing.Instance'
      (Token.Nothing.Get (Reportable => True))
      );

   -- Routine to feed text to the analyzer
   function Get_Text_Line return String;

   Analyzer : Tokenizer.Instance := Tokenizer.Initialize
     (Language_Syntax => Syntax,
      Feeder          => Get_Text_Line'Access,
      Default         => Default);

   --------------------------------------
   -- Booch Map component declarations --
   --------------------------------------

   -- The number of hash buckets used in the hash algorithm
   Buckets : constant := 128;

   Unmanaged_Pool : BC.Support.Unmanaged_Storage.Pool;


   -- String hash routine. This routine will be used as the primary lookup for
   -- dictionary entries.
   function Hash (Word : in Ada.Strings.Unbounded.Unbounded_String) return Positive;

   package String_Containers is new BC.Containers
     (Item => Ada.Strings.Unbounded.Unbounded_String,
      "="  => Ada.Strings.Unbounded."="
      );

   package String_Maps is new String_Containers.Maps
     (Value => Ada.Strings.Unbounded.Unbounded_String,
      "="   => Ada.Strings.Unbounded."="
      );

   package String_Map is new String_Maps.Unbounded
     (Hash            => Hash,
      Buckets         => Buckets,
      Storage_Manager => BC.Support.Unmanaged_Storage.Pool,
      Storage         => Unmanaged_Pool
      );

   package Integer_Maps is new String_Containers.Maps
     (Value => Integer,
      "="   => Ada.Strings.Unbounded."="
      );

   package Integer_Map is new Integer_Maps.Unbounded
     (Hash            => Hash,
      Buckets         => Buckets,
      Storage_Manager => BC.Support.Unmanaged_Storage.Pool,
      Storage         => Unmanaged_Pool
      );

   --------------------------
   -- General Declarations --
   --------------------------

   Config_File_Name : constant String := "sim.cfg";
   Config_File      : Ada.Text_IO.File_Type;

   String_Dictionary  : String_Map.Unbounded_Map;
   Integer_Dictionary : Integer_Map.Unbounded_Map;

   -- Lock to synchronize dictionary accesses
   Dictionary_Lock : Lock.Instance;


   -------------------------
   -- "Callback" routines --
   -------------------------

   -----------------------------------------------------------------------------
   -- String hash routine. This routine will be used as the primary lookup for
   -- dictionary entries.
   -----------------------------------------------------------------------------
   function Hash (Word : in Ada.Strings.Unbounded.Unbounded_String) return Positive is

      Value : Natural := 0;
   begin

      for Char in 1..Ada.Strings.Unbounded.Length(Word) loop
         Value := Value + Character'Pos(Ada.Strings.Unbounded.Element
                                        (Source => Word,
                                         Index  => Char
                                         )) mod
           128;
      end loop;

      return Value + 1;

   end Hash;

   ------------------------------------------------------------------------------------
   -- The routine used to feed data into the token analizer. This routine reads from
   -- the analizer file one line at a time until EOF is reached.
   -- Some other routine must open and close the file.
   ------------------------------------------------------------------------------------
   function Get_Text_Line return String is
      Buffer : String(1..512);
      Data_Length : Integer;
   begin

      Ada.Text_IO.Get_Line (File => Config_File, Item => Buffer, Last => Data_Length);

      if Ada.Text_IO.End_Of_File(Config_File) then
         return Buffer(1..Data_Length) & Token.EOF_Character;
      else
         return Buffer(1..Data_Length) & Token.EOL_Character;
      end if;
   exception
      when Ada.Text_IO.End_Error =>
         return (1 => Token.EOF_Character);

   end Get_Text_Line;

   -----------------------
   -- Internal Routines --
   -----------------------

   -----------------------------------------------------------------------------
   -- Load the dictionary definitions in from disk
   -----------------------------------------------------------------------------
   procedure Initialize is
      Name : Ada.Strings.Unbounded.Unbounded_String;

      Duplicate_Entry : Boolean;
   begin
      String_Map.Clear  (String_Dictionary);
      Integer_Map.Clear (Integer_Dictionary);

      Ada.Text_IO.Open
        (Name => Config_File_Name,
         File => Config_File,
         Mode => Ada.Text_IO.In_File
         );

      Parse_File:
      loop

         Find_Mappings:
         loop

            -------------------------------------------------------------------------
            -- A valid sequence is an Identifier followed by an equals, followed by a
            -- string or integer literal
            Tokenizer.Find_Next (Analyzer);
            exit Find_Mappings when Tokenizer.Token (Analyzer) /= Identifier;
            Name := Ada.Strings.Unbounded.To_Unbounded_String (Tokenizer.Lexeme(Analyzer));

            Tokenizer.Find_Next (Analyzer);
            exit Find_Mappings when Tokenizer.Token (Analyzer) /= Equals;

            Tokenizer.Find_Next (Analyzer);
            case Tokenizer.Token (Analyzer) is
               when String_Literal =>
                  -------------------------------------------------------------------------
                  -- We found a valid mapping; Place it in the dictionary.
                  Duplicate_Entry := False;
                  begin
                     String_Map.Bind
                       (M => String_Dictionary,
                        I => Name,
                        V => Ada.Strings.Unbounded.To_Unbounded_String
                        (Token.String.Value(Token.String.Instance(Syntax(String_Literal).all)))
                        );
                  exception
                     when BC.Duplicate =>
                        Duplicate_Entry := True;
                  end;

                  if Duplicate_Entry then
                     Log.Report
                       (Event    => "Simulation dictionary entry " &
                        Ada.Strings.Unbounded.To_String (Name) & " duplicated in " &
                        Config_File_Name & "." & Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
                        "Value """ & Ada.Strings.Unbounded.To_String
                        (String_Map.Value_Of (M => String_Dictionary,
                                              I => Name
                                              )
                         ) & """ replaced by """ &
                        Token.String.Value(Token.String.Instance(Syntax(String_Literal).all)) &
                        """.",
                        Severity => Log.Warning
                        );

                     String_Map.Rebind
                       (M => String_Dictionary,
                        I => Name,
                        V => Ada.Strings.Unbounded.To_Unbounded_String
                        (Token.String.Value(Token.String.Instance(Syntax(String_Literal).all)))
                        );
                  end if;

               when Integer_Literal =>
                  -------------------------------------------------------------------------
                  -- We found a valid mapping; Place it in the dictionary.
                  Duplicate_Entry := False;
                  begin
                     Integer_Map.Bind
                       (M => Integer_Dictionary,
                        I => Name,
                        V => Integer'Value(Tokenizer.Lexeme(Analyzer))
                        );
                  exception
                     when BC.Duplicate =>
                        Duplicate_Entry := True;
                  end;

                  if Duplicate_Entry then
                     Log.Report
                       (Event    => "Simulation dictionary entry " &
                        Ada.Strings.Unbounded.To_String (Name) & " duplicated in " &
                        Config_File_Name & "." & Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
                        "Value """ & Integer'Image
                        (Integer_Map.Value_Of (M => Integer_Dictionary,
                                               I => Name
                                               )
                         ) & """ replaced by """ &
                        Tokenizer.Lexeme(Analyzer) &
                        """.",
                        Severity => Log.Warning
                        );

                     Integer_Map.Rebind
                       (M => Integer_Dictionary,
                        I => Name,
                        V => Integer'Value(Tokenizer.Lexeme(Analyzer))
                        );
                  end if;

               when others =>
                  exit Find_Mappings;
            end case;

         end loop Find_Mappings;

         ------------------------------------------------------------------------------
         -- We read a token that doesn't match a valid mapping sequence. If its an EOF,
         -- quit. Otherwise ignore it and try again.
         exit Parse_File when Tokenizer.Token (Analyzer) = EOF;

      end loop Parse_File;

      Ada.Text_IO.Close (Config_File);
   end Initialize;


   ---------------------
   -- Client Routines --
   ---------------------

   -----------------------------------------------------------------------------
   -- Lookup the given word in the dictionary, and return its definition.
   -- This routine is a C-wrapper for the Lookup routine. It dynamicly allocates
   -- memory for the result string. It is the caller's responsibility to
   -- deallocate this memory.
   -----------------------------------------------------------------------------
   function C_Lookup (Word : in Interfaces.C.Strings.Chars_Ptr) return
     Interfaces.C.Strings.Chars_Ptr is
      Result : Interfaces.C.Strings.Chars_Ptr := Interfaces.C.Strings.Null_Ptr;
   begin
      return Interfaces.C.Strings.New_String (Lookup (Interfaces.C.Strings.Value(Word)));
   exception
      when Undefined_Word =>
         return Interfaces.C.Strings.Null_Ptr;
   end C_Lookup;

   -----------------------------------------------------------------------------
   -- Lookup the given word in the dictionary, and return its definition. Note
   -- that the lookup is cached, so that subsequent lookups for the same word
   -- won't take as long.
   -----------------------------------------------------------------------------
   function Lookup (Word : in String) return String is

      Definition : Ada.Strings.Unbounded.Unbounded_String;
      Not_In_Dictionary : Boolean := False;
   begin
      Dictionary_Lock.Acquire;

      begin
         Definition :=
           String_Map.Value_Of (M => String_Dictionary,
                          I => Ada.Strings.Unbounded.To_Unbounded_String (Word)
                          );
      exception
         when BC.Not_Found =>
            Not_In_Dictionary := True;
         when others =>
            Dictionary_Lock.Release;
            raise;
      end;

      Dictionary_Lock.Release;

      if Not_In_Dictionary then
         -- Since this routine is often called during package elaborations,
         -- silently raising an exception is not sufficient.
         Log.Report
           (Event => "Word """ & Word & """ not found in simulation dictionary." &
            Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
            "Consider adding it in the " & Config_File_Name & " file.",
            Severity => Log.Error
            );
         Ada.Exceptions.Raise_Exception
           (E       => Undefined_Word'Identity,
            Message => "Word """ & Word & """ not found in simulation dictionary." &
            Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
            "Consider adding it in the " & Config_File_Name & " file");
      end if;

      return Ada.Strings.Unbounded.To_String (Definition);
   end Lookup;

   -----------------------------------------------------------------------------
   -- Lookup the given word in the dictionary, and return its definition. Note
   -- that the lookup is cached, so that subsequent lookups for the same word
   -- won't take as long.
   -----------------------------------------------------------------------------
   function Lookup (Word : in String) return Integer is

      Definition : Integer;
      Not_In_Dictionary : Boolean := False;
   begin
      Dictionary_Lock.Acquire;

      begin
         Definition :=
           Integer_Map.Value_Of (M => Integer_Dictionary,
                                 I => Ada.Strings.Unbounded.To_Unbounded_String (Word)
                                 );
      exception
         when BC.Not_Found =>
            Not_In_Dictionary := True;
         when others =>
            Dictionary_Lock.Release;
            raise;
      end;

      Dictionary_Lock.Release;

      if Not_In_Dictionary then
         -- Since this routine is often called during package elaborations,
         -- silently raising an exception is not sufficient.
         Log.Report
           (Event => "Word """ & Word & """ not found in simulation dictionary." &
            Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
            "Consider adding it in the " & Config_File_Name & " file.",
            Severity => Log.Error
            );
         Ada.Exceptions.Raise_Exception
           (E       => Undefined_Word'Identity,
            Message => "Word """ & Word & """ not found in simulation dictionary." &
            Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
            "Consider adding it in the " & Config_File_Name & " file");
      end if;

      return Definition;
   end Lookup;

   -----------------------------------------------------------------------------
   -- Lookup the given word in the dictionary, and return its definition. Note
   -- that the lookup is cached, so that subsequent lookups for the same word
   -- won't take as long.
   -----------------------------------------------------------------------------
   function Lookup
     (Word    : in String;
      Default : in String
     ) return String is

      Definition : Ada.Strings.Unbounded.Unbounded_String;
      Not_In_Dictionary : Boolean := False;
   begin
      Dictionary_Lock.Acquire;

      begin
         Definition :=
           String_Map.Value_Of (M => String_Dictionary,
                          I => Ada.Strings.Unbounded.To_Unbounded_String (Word)
                          );
      exception
         when BC.Not_Found =>
            Not_In_Dictionary := True;
         when others =>
            Dictionary_Lock.Release;
            raise;
      end;

      Dictionary_Lock.Release;

      if Not_In_Dictionary then
         Log.Report
           (Event => "Word """ & Word & """ not found in simulation dictionary." &
            " Default used."
            );
         return Default;
      end if;

      return Ada.Strings.Unbounded.To_String (Definition);
   end Lookup;

   -----------------------------------------------------------------------------
   -- Lookup the given word in the dictionary, and return its definition. Note
   -- that the lookup is cached, so that subsequent lookups for the same word
   -- won't take as long.
   -----------------------------------------------------------------------------
   function Lookup
     (Word    : in String;
      Default : in Integer
     ) return Integer is

      Definition : Integer;
      Not_In_Dictionary : Boolean := False;
   begin
      Dictionary_Lock.Acquire;

      begin
         Definition :=
           Integer_Map.Value_Of (M => Integer_Dictionary,
                                 I => Ada.Strings.Unbounded.To_Unbounded_String (Word)
                                 );
      exception
         when BC.Not_Found =>
            Not_In_Dictionary := True;
         when others =>
            Dictionary_Lock.Release;
            raise;
      end;

      Dictionary_Lock.Release;

      if Not_In_Dictionary then
         Log.Report
           (Event => "Word """ & Word & """ not found in simulation dictionary." &
            " Default used."
            );
         return Default;
      end if;

      return Definition;
   end Lookup;

   -----------------------------------------------------------------------------
   -- Set a default definition for the given Word in the dictionary. If there is
   -- already a definition for the word, the previous definition will be used
   -- instead of the one supplied here.
   --
   -- This routine is useful to prevent Undefined_Word exceptions from Lookup.
   -----------------------------------------------------------------------------
   procedure Default
     (Word       : in String;
      Definition : in String := ""
     ) is

      Key : constant Ada.Strings.Unbounded.Unbounded_String :=
        Ada.Strings.Unbounded.To_Unbounded_String (Word);
   begin

      Dictionary_Lock.Acquire;

      if not String_Map.Is_Bound
        (M => String_Dictionary,
         I => Key
         )
      then
         String_Map.Bind
           (M => String_Dictionary,
            I => Key,
            V => Ada.Strings.Unbounded.To_Unbounded_String (Definition)
            );
      end if;

      Dictionary_Lock.Release;
   end Default;

   -----------------------------------------------------------------------------
   -- Set a default definition for the given Word in the dictionary. If there is
   -- already a definition for the word, the previous definition will be used
   -- instead of the one supplied here.
   --
   -- This routine is useful to prevent Undefined_Word exceptions from Lookup.
   -----------------------------------------------------------------------------
   procedure Default
     (Word       : in String;
      Definition : in Integer := 0
     ) is

      Key : constant Ada.Strings.Unbounded.Unbounded_String :=
        Ada.Strings.Unbounded.To_Unbounded_String (Word);
   begin

      Dictionary_Lock.Acquire;

      if not Integer_Map.Is_Bound
        (M => Integer_Dictionary,
         I => Key
         )
      then
         Integer_Map.Bind
           (M => Integer_Dictionary,
            I => Key,
            V => Definition
            );
      end if;

      Dictionary_Lock.Release;
   end Default;

begin
   Initialize;

end Simulation_Dictionary;
