-------------------------------------------------------------------------------
--
--            FlightSafety International Simulation Systems Division
--                     Broken Arrow, OK  USA  918-259-4000
--
--                  JPATS T-6A Texan-II Flight Training Device
--
--
--   Engineer:  Ted Dennison
--
--  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--  technical or operational data or information from automatic dissemination
--  under the International Exchange Program or by other means.  This protection
--  covers information required solely for administrative or operational
--  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
--  WARNING:  This document contains technical data whose export is restricted
--  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--  Executive Order 12470.  Violation of these export control laws is subject
--  to severe criminal penalties.  Dissemination of this document is controlled
--  under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with System;
with Interfaces.C;

-------------------------------------------------------------------------------
-- This package implements an Ada interface to the ScopeProfile real-time
-- execution profiler. The current version supported by this package is 2.3
-------------------------------------------------------------------------------
package ScopeProfile is

   type Clock_Selection is (Aux_Clock, System_Clock, No_Clock);
   for Clock_Selection use
      (Aux_Clock    => 1,
       System_Clock => 2,
       No_Clock     => 3
       );
   for Clock_Selection'Size use Interfaces.C.Int'Size;

   All_Tasks : constant := 0;   -- Task ID implying *all* tasks
   Task_Only : constant := -1;  -- Task ID implying all tasks, but no function data

   type Granularity_Designation is
     (Flat_Normal,
      Flat_Exploded,
      Tree_Normal,
      Tree_Exploded
      );
   for Granularity_Designation'Size use Interfaces.C.Int'Size;

   type Index is new Integer;
   Uninitialized_Scope : constant Index := -1;

   -- Default sizes are to be used for the sample or signal buffers
   Default_Size : constant := 0;

   type Instance is private;

   -- An error occured during a profile call
   Error : exception;

   -----------------------------------------------------------------------------
   -- This routine initializes the profiler. It also optionally attaches the
   -- profilers to a clock and begins profile sampling.
   --
   -- This routine spawns an "analysis process" at a priority of 250. It also
   -- spawns a "sampler process" at priority 0. These priorities can be changed
   -- with taskPrioritySet if desired. It sets the sampling rate to 107.0 (or to
   -- the current system clock rate if there is no aux clock), and the analysis
   -- rate to 1.0 Hz. Both these can be changed as required.
   --
   -- This routine allocates and returns a data structure that indicates an
   -- active profile instance.
   -----------------------------------------------------------------------------
   function Initialize
     (Clock       : Clock_Selection         := No_Clock;
      Task_ID     : Interfaces.C.Int        := All_Tasks;
      Granularity : Granularity_Designation := Flat_Normal;
      Scope_Index : Index                   := Uninitialized_Scope;
      Max_Samples : Natural                 := Default_Size;
      Max_Stack   : Natural                 := Default_Size;
      Buffer_Size : Natural                 := Default_Size;
      Signal_Size : Natural                 := Default_Size
      ) return Instance;

   -----------------------------------------------------------------------------
   -- This routine need only be called for implementations utilizing a user
   -- clock. Most implementations (that supplied a clock to Initialize) should
   -- ignore this routine.
   -- Sample may be called at interrupt priority.
   -----------------------------------------------------------------------------
   procedure Sample (Profile : in Instance);

   -----------------------------------------------------------------------------
   -- Profile sampling is temporarily suspended. A specific execution period can
   -- be profilesd by calling "ProfileClear", executing the code, and then
   -- calling "ProfileFreeze". A specific section of code can be profiled by
   -- bracketing the code between calls to "ProfileFreeze".
   -- Freezing an already frozen profile or thawing an already thawed profile
   -- has no effect.
   -----------------------------------------------------------------------------
   procedure Freeze
     (Profile   : in Instance;
      Freeze_On : in Boolean := True
     );

   -----------------------------------------------------------------------------
   -- Return a string representation of the given profile.
   -----------------------------------------------------------------------------
   function Image (Profile : Instance) return String;

private
   type Instance is new System.Address;
end ScopeProfile;
