/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**  The information contained herein is the property of         **
**  FlightSafety Simulation Systems Division and shall          **
**  not be copied or used in any manner or disclosed to         **
**  others execpt as expressly authorized by FSI-SSD.           **
**                                                              **
******************************************************************
**                                                              **
**  Department: Navigation/Visual (65)                          **
**                                                              **								**
**  Task:       Receiver database searches                      **
**  Author:     Terry Tyler                                     **
**                                                              **								**
**  Revision:   1.18           Date: 16/Dec/97                  **
**  Revised by: T.Tyler                                         **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**  The nav_stn_srch program is responsible for the             **
**  database search for each of the nav receivers defined       **
**  in the active table.  The search algorithm used is a        **
**  basic binary search to find the start of the data to        **
**  be used.  A sequential search is then started until         **
**  all possibilities are exhausted.  The data found is         **
**  then unpacked/decoded into the station structures for       **
**  transmission to the host.                                   **
**                                                              **
**  The marker search is performed in a similar fashion,        **
**  but using a search on latitude/longitude instead of         **
**  frequency.                                                  **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revision History:                                           **
*****************************************************************/
/* Rev 1.00  02/28/90	T.Tyler-Initial program release.	*/
/* Rev 1.01  03/19/90	T.Tyler-Added Alignment channel type.	*/
/* Rev 1.02  04/20/90	T.Tyler-Added g/s to threshold distance	**
**			for ILS and MLS facilities.		*/
/* Rev 1.03  05/03/90	T.Tyler-Changed #include references.	*/
/* Rev 1.04  06/11/90   T.Tyler-Modified g/s to threshold code.	*/
/* Rev 1.05  08/08/90   T.Tyler-Modified station kill logic	**
**			added ils alignment w/ anytown logic	*/
/* Rev 1.06  12/06/90	T.Tyler-Corrected AWM logic in search	**
**			and unpack code.			*/
/* Rev 1.07  15/May/91	T.Tyler-Changed host_cmd references to	**
**			bitwise operations.			*/
/* Rev 1.08  18/Jul/91	T.Tyler-Changed rdb references for	**
**			dynamic memory allocation.		*/
/* Rev 1.09  01/Oct/91	T.Tyler-Added additional search for nav **
**			(VOR) receivers to find the closest	**
**			airport and load the airport ident into	**
**			the airport ident word for the host.	*/
/* Rev 1.10  23/Oct/91	T.Tyler-modified adf analog search.	*/
/* Rev 1.11  12/Mar/92	T.Tyler-No Changes.			*/
/* Rev 1.12  13/Jul/92	T.Tyler-changed checksum function to be	**
**			available to other modules.		*/
/* Rev 1.13  13/Nov/92	T.Tyler-Changed host airport/runway	**
**			select to support multiple selections.	**
**			The ILS priority is based on airport/	**
**			runway select #0.			*/
/* Rev 1.14  23/Sep/93	T.Tyler-No Changes.			*/
/* Rev 1.15  03/Jun/94	T.Tyler-Changed to run w/gcc compilers.	**
**			Added airport/runway idents to comm and	**
**			VOR facility searches.			*/
/* Rev 1.16  11/Jan/96	T.Tyler-Modified freq access to use type**
**			casts instead of '&' operation.		*/
/* Rev 1.17  27/Feb/96	T.Tyler-Modified binary search to stop	**
**			indexing past start of data.		*/
/* Rev 1.18  16/Dec/97	T.Tyler-Modified to run in Harris.	*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log: $ 
*
*****************************************************************/

/*********************************************************
**  External References                                 **
*********************************************************/

#include <string.h>
#include <stdio.h>

#include "nav_rdbdef.h"
#include "nav_navdef.h"

#include "nav_global.h"

/**  external functions  **/

extern long stn_kill_check( char *stn_idnt, long stn_type );
extern void stn_kill_mkr( char *stn_idnt, long *mkr_lat_tbl, long *mkr_lon_tbl );
extern void stn_kill_unpack( char *db_idnt, long *type1, long *type2 );

/*********************************************************
**  Function Prototypes for Other Modules               **
*********************************************************/

void nav_stn_srch( void );

void n_stn_srch( RCVR_REQUEST *rcvr, STN_STRUCT *stn );

/*********************************************************
**  Local Function prototypes                           **
*********************************************************/

static int stn_srch( long freq, long type, long adf_mode,
                      const char *apt_idnt, const char *rwy_idnt,
                      long ac_lat, long ac_lon,
                      STN_STRUCT *stn );

static RDB* frq_top( long rcvr_frq, long band_width );

static RDB* lat_top( long latitude );

static void mkr_srch( long latitude, long longitude, RDB* mkr_ptr[] );

static RDB* frq_srch( long frequency,  long ac_lat,
                      long ac_lon   ,  long type_mask,
                      const char *apt_idnt,  const char *rwy_idnt,
                      long band_width  );

static RDB* vhfapt_srch( const RDB* vhf_db_ptr );
static RDB* comapt_srch( const RDB* vhf_db_ptr );
static char *stn_rwy_srch( const RDB* airport_ptr, char *ident );

static void clr_stn_buffer( STN_STRUCT *stn );
static void vhf_unpack( register RDB* db_ptr, STN_STRUCT *stn,
                        RDB* apt_ptr, const char *ident );
static void mls_unpack( register RDB* db_ptr, STN_STRUCT *stn );
static void dme_unpack( register RDB* db_ptr, STN_STRUCT *stn );
static void ndb_unpack( register RDB* db_ptr, STN_STRUCT *stn );
static void com_unpack( register RDB* db_ptr, STN_STRUCT *stn,
                        RDB* apt_ptr, const char *ident );
static void mkr_unpack( register RDB* db_ptr, STN_STRUCT *stn, long rcvr_type );
static void awm_unpack( register RDB* db_ptr, STN_STRUCT *stn );

static long get_band_width( long frequency, long type );

/*********************************************************
**  Constants and definitions                           **
*********************************************************/

#define absl( val )  ( (val)<0 ?-(val):(val) )
#define RNG_MASK 0X000001FF
#define ASCII_BLANKS "    "

/*
**  The n_stn_srch supports the individual calling of the station search
**  function.
*/
void n_stn_srch( RCVR_REQUEST *rcvr, STN_STRUCT *stn )
{
  int valid = 0;


  if( rdb == 0 )  return;    /* exit if RDB not loaded */

  valid = stn_srch( rcvr->freq, rcvr->type, rcvr->bandwidth,
                    rcvr->apt_idnt, rcvr->rwy_idnt,
                    BAMS32(rcvr->ac_lat), BAMS32(rcvr->ac_lon), stn );

  return;
}

/*********************************************************
**                                                      **
**  nav_stn_srch.c -  Receiver & Facility Searches      **
**                                                      **
*********************************************************/
/*  The nav_stn_srch is called from the nav_main module.
**  It provides the interface between the nav_main module 
**  and the local functions that actually perform the
**  database searches.
*/
void nav_stn_srch( void ) {
  long rcvr_num=0;
  long ac_lat,ac_lon;
  long adf_mode;
  char apt_idnt[4],rwy_idnt[4];

  ac_lat = host_in.ac_lat;
  ac_lon = host_in.ac_lon;
  strncpy( apt_idnt, host_in.apt_idnt[ILS_APT_CHK], 4 );
  strncpy( rwy_idnt, host_in.rwy_idnt[ILS_APT_CHK], 4 );
  adf_mode = host_in.host_cmd.adf_mode;

  while( rcvr_num < MAX_RCVRS ) {	/* loop throught rcvrs */
    long frequency,type;
    STN_STRUCT *stn;

    stn = &host_out.stn[rcvr_num];
    frequency = host_in.rcvr_freq[ rcvr_num ];
    type = host_in.rcvr_type[ rcvr_num ];

    stn_srch( frequency, type, adf_mode, apt_idnt, rwy_idnt, ac_lat, ac_lon, stn );

    rcvr_num++;
    if( type == MKR )  rcvr_num += 2;         /* Skip other two markers */
    if( type == END )  rcvr_num = MAX_RCVRS;  /* Exit if at end */
  }
  return;
}


/*  The stn_srch functions searches the radio database for the
**  associated database record.  The search is based on frequency,
**  receiver type, and aircraft latitude/longitude.  The closest
**  facility to the aircraft location with a matching frequency
**  will be chosen.
*/
static int stn_srch( long freq, long type, long adf_mode,
                      const char *apt_idnt, const char *rwy_idnt,
                      long ac_lat, long ac_lon,
                      STN_STRUCT *stn ) {
  char ident[4];
  RDB  *db_ptr=0,*mkr_ptr[3],*apt_ptr;
  int i;

  switch( type ) {
    case OFF:
      clr_stn_buffer( stn );    break;
    case ALN:				/* Alignment Channel */
    case NAV:
      if( freq ) {
        long mask;

        if(( freq > 10800 ) && ( freq < 11200 ) && ( (freq/10) & 0x00000001 ))
          mask = ILS_MASK;
        else
          mask = VOR_MASK;

        db_ptr = frq_srch( freq, ac_lat, ac_lon, mask, apt_idnt, rwy_idnt, 0L );

        if( db_ptr ) {
          apt_ptr = vhfapt_srch( db_ptr );
          stn_rwy_srch( apt_ptr, ident );
          vhf_unpack( db_ptr , stn, apt_ptr, ident );
        }
        else
          clr_stn_buffer( stn );
      }
      else
        clr_stn_buffer( stn );
      break;

    case MLS:
      if( freq ) {
        db_ptr = frq_srch( freq, ac_lat, ac_lon, MLS_MASK, apt_idnt, rwy_idnt, 0L );

        if( db_ptr )
          mls_unpack( db_ptr, stn );
        else
          clr_stn_buffer( stn );
      }
      else
        clr_stn_buffer( stn );
      break;

    case DME:
      if( freq ) {
        db_ptr = frq_srch( freq, ac_lat, ac_lon, DME_MASK, apt_idnt, rwy_idnt, 0L );
        if( db_ptr )
          dme_unpack( db_ptr, stn );
        else
          clr_stn_buffer( stn );
      }
      else
        clr_stn_buffer( stn );
      break;

    case TAC:
      if( freq ) {
        db_ptr = frq_srch( freq, ac_lat, ac_lon, TAC_MASK, apt_idnt, rwy_idnt, 0L );
         if( db_ptr )
           dme_unpack( db_ptr , stn );
        else
          clr_stn_buffer( stn );
      }
      else
        clr_stn_buffer( stn );
      break;

    case NDB:
      if( freq ) {
        long band_width=0;
        if( adf_mode ) 
          band_width = get_band_width( freq, NDB );

        db_ptr = frq_srch( freq , ac_lat, ac_lon, NDB_MASK, apt_idnt, rwy_idnt, band_width );
        if( db_ptr )
          ndb_unpack( db_ptr , stn );
        else
          clr_stn_buffer( stn );
      }
      else
        clr_stn_buffer( stn );
      break;

    case COM:
      if( freq ) {
        db_ptr = frq_srch( freq, ac_lat, ac_lon, COM_MASK, apt_idnt, rwy_idnt, 0L );
        if( db_ptr ) {
          apt_ptr = comapt_srch( db_ptr );
          stn_rwy_srch( apt_ptr, ident );
          com_unpack( db_ptr , stn, apt_ptr, ident );
        }
        else
          clr_stn_buffer( stn );
      }
      else
        clr_stn_buffer( stn );
      break;

    case MKR:			/* all markers at once */
				/* OM=0,MM=1,IM/AWM=2	*/

      mkr_srch( ac_lat, ac_lon, mkr_ptr );

      for( i=0; i<3 ;i++ ) {		/* OM,MM,IM */
        if( mkr_ptr[i] ) {
          db_ptr = mkr_ptr[i];
          if( (db_ptr+RDB_TYP1)->l & ILS_MASK )
            mkr_unpack( db_ptr, stn, i );
          else
            awm_unpack( db_ptr, stn );
        }
        else
          clr_stn_buffer( stn );
        stn++;
      }
      break;

    case DMY:
    case END:
      clr_stn_buffer( stn );
      break;

    default:
      clr_stn_buffer( stn );
      break;
  }						/* end switch */
  return( db_ptr == 0 );
}						/* end  */

/*********************************************************
**                                                      **
**  frq_top -                                           **
**    locates the top of the matching                   **
**    frequency in the frequency table.                 **
**                                                      **
**  inputs: receiver frequency (long)                   **
**          receiver band widht (long)                  **
**                                                      **
**  output: frequency table index that corresponds	**
**          to the first (top) frequency within         **
**          the frequency table.                        **
**          NOTE: for receivers using a band width      **
**          model, the return is the lower station      **
**          within the band range.                      **
*********************************************************/

static RDB *frq_top( long rcvr_frq, long band_width )
{
  RDB  *tbl_ptr,*db_ptr;		/* database pointers	*/
  long incr_size;			/* search pattern size	*/
  long found=0;				/* station found flag	*/

  tbl_ptr   = frq_tbl.iptr;		/* set start search point */
  incr_size = frq_tbl.idxf;		/* set search size */
  rcvr_frq -= band_width;		/* get lower end of band*/

  while ( incr_size > 0 ) {
    long db_frq;

    db_ptr = rdb + ((tbl_ptr->l/sizeof(RDB)) + VHF_FRQ);
    db_frq = (long)((unsigned short)db_ptr->s[VHF_FRQ_S]);

    if( rcvr_frq == db_frq )  {
      for(;  (tbl_ptr >= frq_tbl.sptr) &&
        (rcvr_frq ==
          (long)((unsigned short)rdb[(tbl_ptr->l/sizeof(RDB)+VHF_FRQ)].s[VHF_FRQ_S]))
          ;  tbl_ptr-- );
      tbl_ptr++;			/*top index*/
      incr_size = 0;
      found = 1;
    }
    else if( rcvr_frq > db_frq ) {
      tbl_ptr += (incr_size /= 2);
      if( tbl_ptr > frq_tbl.eptr ) tbl_ptr = frq_tbl.eptr;
    }
    else
      tbl_ptr -= (incr_size /= 2);
  }
  if( found ) return( tbl_ptr );		/* discrete frq found */
  if( band_width ) return( tbl_ptr );		/* closest frequency */
  return 0;					/* not found */
}

/*********************************************************
**  frq_srch -                                          **
**    performs a station by station search              **
**    starting at the top of the matching               **
**    frequencies (previously found).                   **
**********************************************************
**  global variables -                                  **
**    struct frq_tbl	frequency ordered table data       **
**    rdb            radio database                     **
**                                                      **
**  inputs to function:                                 **
**    frequency:   receiver frequency                   **
**    latitude:    a/c laitutude, in BAMS               **
**    longitude:   a/c longitude, in BAMs               **
**    type-mask:   station data type code mask          **
**    airport id:  selected airport ident               **
**    runway  id:  selected runway ident                **
**    band_width:  tuning band width +/-                **
**                                                      **
**  returned from function:                             **
**    pointer to the frequency table that corresponds   **
**    to the station found.                             **	
**                                                      **
**  functions called:                                   **
**    frq_top:  finds the first matching freq           **
**              in the frequency table.                 **
*********************************************************/

#define WINDOW 0X38E38E4	/* (5deg/180) * (2**31-1) */
#define LAT_WINDOW WINDOW 
#define LON_WINDOW WINDOW*2
#define LOW_RANGE 1.0		/* 1.0 deg**2 */
#define ILS_BIAS  0.04112	/* 15,000 feet in degrees */

RDB *frq_srch( long frequency, long latitude, long longitude,
               long type_mask, const char *apt_idnt, const char *rwy_idnt, long band_width )
{
  long   db_freq;
  RDB    *db_ptr,*tbl_ptr,*closest_stn_ptr=0,*rwy_match_ptr=0;
  double  closest_stn_dist = 100.0;
  double  cos_lat;
/*
**  The index tables are used to search for the top of the stations
**  with a given frequency.  The pointer to the index table is
**  returned for a valid search, zero if the search fails.  For the
**  analog ADF algorithm, the pointer is returned regardless of
**  whether an exact match is found or not.
*/
  tbl_ptr = frq_top( frequency, band_width );
  if( tbl_ptr ) {
    db_ptr = rdb + ( tbl_ptr->l/sizeof(RDB) );
    db_freq = (long)((unsigned short)(db_ptr+VHF_FRQ)->s[VHF_FRQ_S]);
    cos_lat = cos((double)latitude * BAM32_TO_RAD);
/*
**  The table is searched sequentially until a non-matching frequency
**  is found or the end of table is reached.  For analog ADF's the
**  search is continued until the frequency+bandwidth is reached.
**  For each valid station, the distance to the station is computed
**  and compared to the closest station checked, and the closer of
**  of the two stations is saved for the next comparison.
*/
    while( ( tbl_ptr <= frq_tbl.eptr ) &&
           ( band_width ? ( (db_freq-frequency)<=band_width ) :
           ( frequency == db_freq ) ) ) {
      long db_type;

      if( (band_width ? (absl(db_freq-frequency)<=band_width) : 1 ) ){
        db_type = (db_ptr+RDB_TYP1)->l;
        if( stn_kill_active )
          db_type = stn_kill_check( db_ptr->c, (db_ptr+RDB_TYP1)->l );

        if( type_mask & db_type ) {		/*correct type?*/
          long delta_lat;

          delta_lat = absl( (latitude-((db_ptr+RDB_LAT)->l)) );

          if( delta_lat < LAT_WINDOW ) {
            long delta_lon;

            delta_lon = absl((longitude-((db_ptr+RDB_LON)->l)));

            if( delta_lon < LON_WINDOW ) {
              double dlat,dlon,thdg,range_sq;
/*
**  For ILS facilities, check for an airport/runway selection match.  If
**  the ILS in tune is for the selected runway, force the search to select
**  the corresponding ILS facility.  Otherwise, compute a point on the
**  aprroach end of the runway.  This should help in the situation where there
**  are multiple facilities utilizing the same frequency.
*/
              if( (ILS_MASK|MLS_MASK) & db_type ) {	/*ILS priority?*/
                thdg = db_ptr[ILS_HDG].s[ILS_HDG_S]/10.0*DEG_TO_RAD;
                dlat = DEGR32(latitude) -
                       (DEGR32( db_ptr[RDB_LAT].l ) - ILS_BIAS*cos(thdg));
                dlon = (DEGR32(longitude) -
                        (DEGR32( db_ptr[RDB_LON].l ) -
                        ILS_BIAS * sin( thdg ))) * cos_lat;
                range_sq = dlat * dlat + dlon * dlon;
                if(( strncmp( apt_idnt, db_ptr[ILS_AID].c, 4 ) == 0 ) &&
                   ( strncmp( rwy_idnt, db_ptr[ILS_RID].c, 4 ) == 0 ) &&
                   ( range_sq <= LOW_RANGE )) {
                  rwy_match_ptr = db_ptr;
                }				/*if( apt_idnt == ...)*/
              }
/*
**  For all non-ILS facilities, use the pure distance to facility.
*/
              else {				/* Non-ILS facility */
                dlat = DEGR32( delta_lat );
                dlon = DEGR32( delta_lon ) * cos_lat;
                range_sq = dlat*dlat + dlon*dlon;
              }
                        
              if( range_sq < closest_stn_dist ) {	/*closest?*/
                closest_stn_dist = range_sq;		/*yes, save*/
                closest_stn_ptr  = db_ptr;		/*station data*/
              }
            }				/*if( delta_lon...)*/
          }				/*if( delta_lat...)*/
        }				/*if( type_mask &..)*/
      }					/*if( band_width?...)*/
      db_ptr = rdb + ( (++tbl_ptr)->l/sizeof(RDB) );
      db_freq = (long)((unsigned short)db_ptr[VHF_FRQ].s[VHF_FRQ_S]);
    }					/* end while */
    if( rwy_match_ptr )        return( rwy_match_ptr );
    else if( closest_stn_ptr ) return( closest_stn_ptr );
    else                        return( 0 );
  }					/*if( tbl_ptr) */
  else return 0;
}

/*********************************************************
**                                                      **
**  lat_top -                                           **
**    locates the top of the matching                   **
**    latitude in the latitude table.                   **
**                                                      **
**  inputs:                                             **
**    aircraft latitude in BAMs                         **
**                                                      **
**  return:                                             **
**    latitude table index.                             **
**                                                      **
*********************************************************/
/*   The latitude/longitude ordered table contains three *
**    entries for each facility: latitude,longitude, and *
**    offset to the actual station data.  The TBL%%%'s   *
**    have been defined to ease the burden of mods.      *
*********************************************************/

#define TBLLAT 0		/* table lat offset	*/
#define TBLLON 1		/* table lon offset	*/
#define TBLIDX 2		/* table indx offset	*/
#define TBLSIZ 3		/* table data set size	*/

#define MKR_WINDOW    0x002468AC	/* 12 Nm (BAMs) */
#define MKR_LAT_WINDOW MKR_WINDOW
#define MKR_LON_WINDOW MKR_WINDOW*2	/* 1:1 at 60 Deg N */

static RDB *lat_top( long latitude )
{
  register RDB  *tbl_ptr;
  long incr_size;

  tbl_ptr   = lat_tbl.iptr;
  incr_size = lat_tbl.idxf;

  while( incr_size > 0 ) {
    if( latitude == tbl_ptr->l ) {		/* 1st is latitude */
      for( ;
           ( tbl_ptr >= lat_tbl.sptr ) && (latitude == tbl_ptr->l);
           tbl_ptr -= TBLSIZ );
      tbl_ptr += TBLSIZ;
      incr_size = 0;
    }
    else if( latitude > tbl_ptr->l ) {
      tbl_ptr += ( (incr_size/=2) * TBLSIZ );
      if( tbl_ptr > lat_tbl.eptr ) tbl_ptr = lat_tbl.eptr;
    }
    else
      tbl_ptr -= ( (incr_size/=2) * TBLSIZ );
  }
  return tbl_ptr;
}

/*********************************************************
**                                                      **
**  mkr_srch -                                          **
**    performs the station by station search            **
**    for all markers.  Begins at the upper             **
**    latitude point previously found, and              **
**    continues until the lower latitude                **
**    point is reached.                                 **
**    Order is OM,MM,IM/AWM                             **
**                                                      **
*********************************************************/

static void mkr_srch( long latitude, long longitude, RDB* mkr_ptr[] )
{
  long   uppr_lat;
  RDB    *tbl_ptr,*db_ptr;
  double cos_lat;
  int    i;

  RDB    *closest_mkr_ptr[3];		/* OM,MM,IM/AWM */
  double closest_mkr_dist[3];

  for( i=0; i<3; i++ ) {
    mkr_ptr[i] = closest_mkr_ptr[i] = 0;
    closest_mkr_dist[i] = 10.0;
  }
  uppr_lat = latitude + MKR_LAT_WINDOW;
  tbl_ptr  = lat_top( latitude-MKR_LAT_WINDOW );
  cos_lat  = cos((double)latitude * BAM32_TO_RAD);

  while ( ( tbl_ptr <= lat_tbl.eptr ) && ((tbl_ptr+TBLLAT)->l < uppr_lat ) ) {
    long delta_lat;

    delta_lat = absl( (latitude - (tbl_ptr+TBLLAT)->l) );

    if( delta_lat < MKR_LAT_WINDOW ) {
      long delta_lon;

      delta_lon = absl( (longitude-(tbl_ptr+TBLLON)->l) );

      if( delta_lon < MKR_LON_WINDOW ) {
        long mkr_lat[3],mkr_lon[3],db_type; 
        RDB *tmp_ptr;
        int  i,j;

        db_ptr  = rdb + ( (tbl_ptr+TBLIDX)->l/sizeof(RDB) );
        db_type = (db_ptr+RDB_TYP1)->l;
        for(i=0; i < 3; i++ )  mkr_lat[i]= mkr_lon[i]=0;

        if( db_type & ILS_MASK ) {
          tmp_ptr = db_ptr + ILS_ILAT;
          for( j=2; j >= 0 ; j-- ) {		/*load OM,MM,IM*/
            mkr_lat[j] = (tmp_ptr++)->l;
            mkr_lon[j] = (tmp_ptr++)->l;
          }
          if( stn_kill_active )			/*check for kill*/
            stn_kill_mkr( db_ptr->c, mkr_lat, mkr_lon );
        }
        else if ( db_type & AWM_MASK ) {
          long db_type2;

          db_type2 = (db_ptr+RDB_TYP2)->l;
          tmp_ptr  = db_ptr + RDB_LAT;

          if( db_type2 & 0x08000000 ) {			/* OM type */
            mkr_lat[0]=(tmp_ptr++)->l;
            mkr_lon[0]=(tmp_ptr  )->l; }
          else if( db_type2 & 0x10000000 ) {		/* MM type */
            mkr_lat[1]=(tmp_ptr++)->l;
            mkr_lon[1]=(tmp_ptr  )->l; }
          else {					/*IM/AM type*/
            mkr_lat[2]=(tmp_ptr++)->l;
            mkr_lon[2]=(tmp_ptr  )->l;
          }

        }					/*elseif(db_type &..)*/

         for( i=0; i < 3; i++ )	 {		/* check all mkrs **/
           if( mkr_lat[i] ) {
             double dlat,dlon,range_sq;

             dlat = DEGR32( absl((latitude -mkr_lat[i])) );
             dlon = DEGR32( absl((longitude-mkr_lon[i])) ) * cos_lat;
             range_sq = dlat*dlat + dlon*dlon;

             if( range_sq < closest_mkr_dist[i] ) {
               closest_mkr_dist[i] = range_sq;
               closest_mkr_ptr[i]  = db_ptr;
             }					/*if(range_sq <...)*/
           }					/*if(mkr_lat[i])*/
         }					/* end for */
       }				/* end if (delta-lon...) */
     }					/* end if (delta_lat...) */
     tbl_ptr += TBLSIZ;
   }						/* end while */

   mkr_ptr[0] = closest_mkr_ptr[0];			/* OM indx */
   mkr_ptr[1] = closest_mkr_ptr[1];			/* MM indx */
   mkr_ptr[2] = closest_mkr_ptr[2];			/* IM indx */

   return;
}

/*********************************************************
**                                                      **
**  vhfapt_srch -                                       **
**    searches for the closest airport to the           **
**    given navaid (with voice bits set).               **
**    Used for VOR ATIS/APPR/DEP types.                 **
**    Looks for any airport within a 12 NM              **
**    distance.  The station latitude+12 is             **
**    binary searched, then a sequential                **
**    search is performed until latitude-12             **
**    is reached.                                       **
**                                                      **
*********************************************************/

#define VHFAPT_WINDOW 0x001E573A		/* 20 Nm (BAMs)	*/
#define VHFAPT_LAT_WINDOW VHFAPT_WINDOW
#define VHFAPT_LON_WINDOW VHFAPT_WINDOW*2	/* 1:1 at N60 lat */
#define ATIS_MASK  0x02000000

static RDB* vhfapt_srch( const RDB* db_ptr )
{
  long   latitude,longitude,uppr_lat;
  RDB    *tbl_ptr,*closest_apt_ptr=0;
  double cos_lat,closest_apt_dist=0.1;

/*  If a null database pointer is passed to this function, return
**  a null airport pointer.  If the database pointer is valid but
**  the facility has no ATIS, then no search is necessary, and a
**  null airport pointer is returned.
*/
  if( !db_ptr ) return( 0 );
  if( ( !(db_ptr+RDB_TYP1)->l & VHF_MASK ) || 
      ( !(db_ptr+RDB_TYP2)->l & ATIS_MASK ) ) return( 0 );

/*  Generate the baseline variables for station lat/lon and upper
**  and lower limits for latitude.  Find the starting latitude point
**  in the index tables using the lat_top function.
*/

  latitude  = (db_ptr+RDB_LAT)->l;
  longitude = (db_ptr+RDB_LON)->l;
  uppr_lat = latitude + VHFAPT_LAT_WINDOW;
  cos_lat  = cos((double)latitude * BAM32_TO_RAD);
  tbl_ptr  = lat_top( latitude-VHFAPT_LAT_WINDOW );

/*  Loop through the remaining latitudes in the lat/lon index
**  tables until the upper latitude limit or the end of table
**  is reached.
*/
  while ( ( tbl_ptr <= lat_tbl.eptr ) && ((tbl_ptr+TBLLAT)->l < uppr_lat ) ) {

/*  Check to see if the record type is an airport.  Ignore all other
**  record types.
*/
    if( ((rdb+((tbl_ptr+TBLIDX)->l/sizeof(RDB)))+RDB_TYP1)->l&APT_MASK ) {
      long delta_lon;

/*  The record is now of the correct type (airport).  Check the 
**  longitude to see if the airport is within the valid window.
**  Note the latitude must be inside the window by default.
*/
      delta_lon = absl( (longitude-(tbl_ptr+TBLLON)->l) );

      if( delta_lon < VHFAPT_LON_WINDOW ) {
        double dlat,dlon,range_sq;
        RDB    *tmp_ptr;

/*  The airport is within the window.  Compute the range to station
**  for the airport, and if closer than previous checks, save the
**  range and pointer to the airport.  Ranges are computed in
**  degrees**2 to save computation requirements.
*/
        tmp_ptr = rdb + (tbl_ptr+TBLIDX)->l/sizeof(RDB);
        dlat = DEGR32(absl((latitude -(tmp_ptr+RDB_LAT)->l)));
        dlon = DEGR32(absl((longitude-(tmp_ptr+RDB_LON)->l)))
               *cos_lat;
        range_sq = dlat*dlat + dlon*dlon;

        if( range_sq < closest_apt_dist ) {
           closest_apt_dist = range_sq;
           closest_apt_ptr  = tmp_ptr;
        }
      }				/* end if (delta lon..) */
    }				/* end if (type check...) */
    tbl_ptr += TBLSIZ;		/* get next entry in table */
  }				/* end while */

  return( closest_apt_ptr );	/* 0 if no airport found */
				/* ptr to airport record if found */
}

/*********************************************************
**                                                      **
**  comapt_srch -                                       **
**    Finds the airport associated with a comm record   **
**    and returns a pointer to the airport record.      **
**                                                      **
*********************************************************/

static RDB* comapt_srch( const RDB *com_ptr )
{
  register RDB *tbl_ptr, *db_ptr;
  RDB *apt_ptr=0;
  long apt_ident, incr_size, found=0;

  incr_size = idt_tbl.idxf;			/* Search pattern size */
  tbl_ptr   = idt_tbl.iptr;			/* Starting search pointer */
  apt_ident = (com_ptr+RDB_IDNT)->l;		/* Airport Ident from comm */

/** Find the pointer to the top of the ident table for the airport ident **/

  while( incr_size > 0 ) {
    long db_idt;
    db_idt = rdb[ tbl_ptr->l/sizeof(RDB) ].l;

    if( apt_ident == db_idt ) {
      for( ;
           ( tbl_ptr >= idt_tbl.sptr ) &&
           ( apt_ident == rdb[ tbl_ptr->l/sizeof(RDB) ].l );
           tbl_ptr-- );				/* back to 1st ident in tbl */
      tbl_ptr++;				/* Correct pointer */
      incr_size = 0;				/* end of search */
      found = 1;
    }
    else if( apt_ident > db_idt ) {
      tbl_ptr += ( incr_size /= 2 );
      if( tbl_ptr > idt_tbl.eptr )  tbl_ptr = idt_tbl.eptr;
    }
    else
      tbl_ptr -= ( incr_size /= 2 );
  }

/*  At this point the initial binary search is complete, and the if found, the
**  table pointer points to the first ident in the ident table.  Use this
**  pointer to continue with the sequential search.  Step through the ident
**  table until the ident no longer matchs or the airport record is found.
*/
  if( found ) {
    db_ptr = rdb + (tbl_ptr->l/sizeof(RDB));
    while( ( tbl_ptr <= idt_tbl.eptr ) && ( apt_ident == db_ptr->l ) ) {
      if( (db_ptr+RDB_TYP1)->l & APT_MASK )
        apt_ptr = db_ptr;
      db_ptr = rdb + ((++tbl_ptr)->l/sizeof(RDB));
    }
  }
  else						/* Ident not found */
    apt_ptr = 0;
  return( apt_ptr );
}

/*********************************************************
**                                                      **
**  stn_rwy_srch -                                      **
**    Selects the longest runway for a given airport    **
**    based on the airport identifier.  The function    **
**    performs a sequential search starting with the    **
**    airport record.                                   **
**                                                      **							**
*********************************************************/

static char *stn_rwy_srch( const RDB *apt_ptr, char *ident )
{
  long rwy_length, longest=0;
  register RDB *db_ptr, *db_end;

  strncpy( ident, ASCII_BLANKS, 4 );

  if( !apt_ptr ) return( ident );	/* Check for invalid airport */

  db_end = frq_tbl.sptr - 1;			/* end of ident table */

/** Skip past the airport and waypoint records **/

  db_ptr = (RDB*)apt_ptr + APT_REC_LENGTH;
  while( ( db_ptr <= db_end ) && ((db_ptr+RDB_TYP1)->l & WPT_MASK) )
    db_ptr += WPT_REC_LENGTH;

/**  Search the runway records for the longest runway  **/

  while( ( db_ptr <= db_end ) && ((db_ptr+RDB_TYP1)->l & RWY_MASK) ) {
    if( (rwy_length=(db_ptr+RWY_LEN)->s[RWY_LEN_S]) > longest ) {
      longest = rwy_length;
      strncpy( ident, (db_ptr+RDB_IDNT)->c, 4 );
    }
    db_ptr += RWY_REC_LENGTH;
  }
  return( ident );
}

/*********************************************************
**                                                      **
**  stn_unpack.c -                                      **
**    unpacks the radio database data and places the    **
**    data into the appropriate structures.             **
**                                                      **
*********************************************************/

static void vhf_unpack( register RDB* db_ptr, STN_STRUCT *stn,
                        RDB* apt_ptr, const char *rwy_ident )
{
  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  if( ILS_MASK & (db_ptr+RDB_TYP1)->l ) {
    strncpy( stn->idnt, (db_ptr++)->c, 4 );
    strncpy( stn->icao, (db_ptr++)->c, 4 );
    stn->typ1 = (db_ptr  )->l;
    stn->rng  = (float)((db_ptr++)->l&RNG_MASK);
    stn->typ2 = (db_ptr++)->l;
    stn->lat  = DEGR32( (db_ptr++)->l );
    stn->lon  = DEGR32( (db_ptr++)->l );
    stn->var  = (float)((double)(db_ptr  )->s[ILS_VAR_S]*0.1);
    stn->elv  = (db_ptr++)->s[ILS_ELV_S];
    stn->frq  = (long)((unsigned short)(db_ptr  )->s[ILS_FRQ_S]);
    stn->hdg  = (float)((double)(db_ptr++)->s[ILS_HDG_S]*0.1);
    strncpy( stn->aid, (db_ptr++)->c, 4 );
    strncpy( stn->rid, (db_ptr++)->c, 4 );
    stn->lgd  = (db_ptr  )->s[ILS_LGD_S];
    stn->lbw  = (float)((double)(db_ptr++)->s[ILS_LBW_S]*0.01);
    stn->gsa  = (float)((double)(db_ptr  )->s[ILS_GSA_S]*0.01);
    stn->gsw  = (float)((double)(db_ptr++)->s[ILS_GBW_S]*0.01);
    db_ptr += 8;			/*skip mkr&dme lat/lons */
    stn->gtd  = (db_ptr  )->s[ILS_GTD_S];

    stn->bad  = 0.0;
    stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
    stn->mgpa = 0.0;
    stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
    stn->bhdg = 0.0;
 
  }
  else {					/*VOR type*/
    strncpy( stn->idnt, (db_ptr++)->c, 4 );
    strncpy( stn->icao, (db_ptr++)->c, 4 );
    stn->typ1 = (db_ptr  )->l;
    stn->rng  = (float)((db_ptr++)->l&RNG_MASK);
    stn->typ2 = (db_ptr++)->l;
    stn->lat  = DEGR32( (db_ptr++)->l );
    stn->lon  = DEGR32( (db_ptr++)->l );
    stn->var  = (float)((double)(db_ptr  )->s[VHF_VAR_S]*0.1);
    stn->elv  = (db_ptr++)->s[VHF_ELV_S];
    stn->frq  = (long)((unsigned short)(db_ptr  )->s[VHF_FRQ_S]);
    stn->hdg  = (float)((double)(db_ptr++)->s[VHF_DCL_S]*0.1);
    if( apt_ptr ) strncpy( stn->aid, (apt_ptr+RDB_IDNT)->c, 4 );
    else          strncpy( stn->aid, ASCII_BLANKS, 4 );
    strncpy( stn->rid, rwy_ident, 4 );
    stn->lgd  = 0;
    stn->lbw  = 0.0;
    stn->gsa  = 0.0;
    stn->gsw  = 0.0;
    stn->gtd  = 0;

    stn->bad  = 0.0;
    stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
    stn->mgpa = 0.0;
    stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
    stn->bhdg = 0.0;

  }
  if( stn_kill_active )
    stn_kill_unpack(  stn->idnt, &stn->typ1, &stn->typ2 );

  return;
}

/*************************************************
**  MLS station unpack                          **
*************************************************/

static void mls_unpack( register RDB* db_ptr, STN_STRUCT *stn )
{
  short j;

  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  strncpy( stn->idnt, (db_ptr++)->c, 4 );
  strncpy( stn->icao, (db_ptr++)->c, 4 );
  stn->typ1 = (db_ptr  )->l;
  stn->rng  = (float)((db_ptr++)->l & RNG_MASK);
  stn->typ2 = (db_ptr++)->l;
  stn->lat  = DEGR32( (db_ptr++)->l );
  stn->lon  = DEGR32( (db_ptr++)->l );
  stn->var  = (float)((double)(db_ptr  )->s[MLS_VAR_S]*0.1);
  stn->elv  = (db_ptr++)->s[MLS_ELV_S];
  stn->frq  = (long)((unsigned short)(db_ptr  )->s[MLS_FRQ_S]);
  stn->hdg  = (float)((double)(db_ptr++)->s[MLS_HDG_S]*0.1);
  strncpy( stn->aid, (db_ptr++)->c, 4 );
  strncpy( stn->rid, (db_ptr++)->c, 4 );
  stn->lgd  = (db_ptr  )->s[MLS_AED_S];
  j = (db_ptr++)->s[MLS_APA_S];
  stn->apal = (float)( ( j >> 8 ) & 0x00FF );
  stn->apar = (float)( j & 0x00FF );
  j = (db_ptr  )->s[MLS_ACA_S];
  stn->acal = (float)( ( j >> 8 ) & 0x00FF );
  stn->acar = (float)( j & 0x00FF );
  stn->gsw  = (float)((double)(db_ptr++)->s[MLS_EAS_S]*0.1);
  stn->gsa  = (float)((double)(db_ptr  )->s[MLS_NEA_S]*0.01);
  stn->mgpa = (float)((double)(db_ptr++)->s[MLS_MGPA_S]*0.01);
  stn->bad  = (float)((double)(db_ptr  )->s[MLS_BATD_S]);
  j = (db_ptr++)->s[MLS_BPA_S];
  stn->bpal = (float)( ( j >> 8 ) & 0x00FF );
  stn->bpar = (float)( j & 0x00FF );
  j = (db_ptr  )->s[MLS_BCA_S];
  stn->bpcl = (float)( ( j >> 8 ) & 0x00FF );
  stn->bpcr = (float)( j & 0x00FF );
  stn->bhdg = (float)((double)(db_ptr++)->s[MLS_BHDG_S]*0.1);
  stn->gtd  = (db_ptr  )->s[MLS_GTD_S ];
  if( stn_kill_active )
    stn_kill_unpack(  stn->idnt, &stn->typ1, &stn->typ2 );
  return;
}

/*************************************************
**  DME/TACAN station unpack                    **
*************************************************/

static void dme_unpack( register RDB* db_ptr, STN_STRUCT *stn )
{
  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  if( ILS_MASK & (db_ptr+RDB_TYP1)->l ) {
    strncpy( stn->idnt, (db_ptr++)->c, 4 );
    strncpy( stn->icao, (db_ptr++)->c, 4 );
    stn->typ1 = (db_ptr  )->l;
    stn->rng  = (float)((db_ptr++)->l&RNG_MASK);
    stn->typ2 = (db_ptr++)->l;
    db_ptr += 2;       /* skip ILS lat/lon */
    stn->var  = (float)((double)(db_ptr  )->s[ILS_VAR_S]*0.1);
    stn->elv  = (db_ptr++)->s[ILS_ELV_S];
    stn->frq  = (long)((unsigned short)(db_ptr  )->s[ILS_FRQ_S]);
    db_ptr++;          /* skip ILS hdg     */
    strncpy( stn->aid, (db_ptr++)->c, 4 );
    strncpy( stn->rid, (db_ptr++)->c, 4 );
    db_ptr += 2;       /* skip lgd,lbw,gsa,gsw */
    stn->lat  = DEGR32( (db_ptr++)->l );
    stn->lon  = DEGR32( (db_ptr++)->l );
/* skip all other elements */
    stn->hdg  = 0.0;
    stn->lgd  = stn->lbw = stn->gsa = stn->gsw = 0.0;
    stn->gtd  = 0;
    stn->bad  = 0.0;
    stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
    stn->mgpa = 0.0;
    stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
    stn->bhdg = 0.0;

  }
  else {			/** "free" DME or TACAN **/
    strncpy( stn->idnt, (db_ptr++)->c, 4 );
    strncpy( stn->icao, (db_ptr++)->c, 4 );
    stn->typ1 = (db_ptr  )->l;
    stn->rng  = (float)((db_ptr++)->l&RNG_MASK);
    stn->typ2 = (db_ptr++)->l;
    stn->lat  = DEGR32( (db_ptr++)->l );
    stn->lon  = DEGR32( (db_ptr++)->l );
    stn->var  = (float)((double)(db_ptr  )->s[VHF_VAR_S]*0.1);
    stn->elv  = (db_ptr++)->s[VHF_ELV_S];
    stn->frq  = (long)((unsigned short)(db_ptr  )->s[VHF_FRQ_S]);
    stn->hdg  = (float)((double)(db_ptr++)->s[VHF_DCL_S]*0.1);
    strncpy( stn->aid, ASCII_BLANKS, 4 );
    strncpy( stn->rid, ASCII_BLANKS, 4 );
/* skip all others (not used) */
    stn->hdg  = 0.0;
    stn->lgd  = stn->lbw = stn->gsa = stn->gsw = 0.0;
    stn->gtd  = 0;
    stn->bad  = 0.0;
    stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
    stn->mgpa = 0.0;
    stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
    stn->bhdg = 0.0;

  }
  if( stn_kill_active )
    stn_kill_unpack(  stn->idnt, &stn->typ1, &stn->typ2 );
  return;
}

/*************************************************
**  NDB station unpack                          **
*************************************************/

static void ndb_unpack( register RDB* db_ptr, STN_STRUCT *stn )
{
  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  strncpy( stn->idnt, (db_ptr++)->c, 4 );
  strncpy( stn->icao, (db_ptr++)->c, 4 );
  stn->typ1 = (db_ptr  )->l;
  stn->rng  = (float)((db_ptr++)->l & RNG_MASK);
  stn->typ2 = (db_ptr++)->l;
  stn->lat  = DEGR32( (db_ptr++)->l );
  stn->lon  = DEGR32( (db_ptr++)->l );
  stn->var  = (float)((double)(db_ptr  )->s[NDB_VAR_S]*0.1);
  stn->elv  = (db_ptr++)->s[NDB_ELV_S];
  stn->frq  = (long)((unsigned short)(db_ptr  )->s[NDB_FRQ_S]);
/* skip all others (not used) */
  stn->hdg  = 0.0;
  strncpy( stn->aid, ASCII_BLANKS, 4 );
  strncpy( stn->rid, ASCII_BLANKS, 4 );
  stn->lgd  = stn->lbw = stn->gsa = stn->gsw = 0.0;
  stn->gtd  = 0;
  stn->bad  = 0.0;
  stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
  stn->mgpa = 0.0;
  stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
  stn->bhdg = 0.0;

  if( stn_kill_active )
    stn_kill_unpack(  stn->idnt, &stn->typ1, &stn->typ2 );
  return;
}

/*************************************************
**  VHF comm station unpack                     **
*************************************************/

static void com_unpack( register RDB* db_ptr, STN_STRUCT *stn,
                        RDB* apt_ptr, const char *rwy_ident )
{
  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  strncpy( stn->idnt, (db_ptr++)->c, 4 );
  strncpy( stn->icao, (db_ptr++)->c, 4 );
  stn->typ1 = (db_ptr  )->l;
  stn->rng  = (float)((db_ptr++)->l & RNG_MASK);
  stn->typ2 = (db_ptr++)->l;
  stn->lat  = DEGR32( (db_ptr++)->l );
  stn->lon  = DEGR32( (db_ptr++)->l );
  stn->var  = (float)((double)(db_ptr  )->s[COM_VAR_S]*0.1);
  stn->elv  = (db_ptr++)->s[COM_ELV_S];
  stn->frq  = (long)((unsigned short)(db_ptr  )->s[COM_FRQ_S]);
/* If an associated airport is present, unpack the name */
  if( apt_ptr )  strncpy( stn->aid, stn->idnt, 4 );
  else           strncpy( stn->aid, ASCII_BLANKS, 4 );
  strncpy( stn->rid, rwy_ident, 4 );
/* skip all others (not used) */
  stn->hdg  = 0.0;
  stn->lgd  = stn->lbw = stn->gsa = stn->gsw = 0.0;
  stn->gtd  = 0;
  stn->bad  = 0.0;
  stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
  stn->mgpa = 0.0;
  stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
  stn->bhdg = 0.0;

  return;
}

/*************************************************
**  marker beacon station unpack                **
*************************************************/

static void mkr_unpack( register RDB* db_ptr, STN_STRUCT *stn, long rcvr_type )
{
  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  strncpy( stn->idnt, (db_ptr++)->c, 4 );
  strncpy( stn->icao, (db_ptr++)->c, 4 );
  stn->typ1 = (db_ptr  )->l | AWM_MASK;
  stn->rng  = (float)((db_ptr++)->l & RNG_MASK);
  db_ptr++;		/* skip type code */
  if     ( rcvr_type == OMKR )  stn->typ2 = 0X08000000;
  else if( rcvr_type == MMKR )  stn->typ2 = 0X10000000;
  else if( rcvr_type == IMKR )  stn->typ2 = 0X20000000;
  db_ptr += 2 ;      /* skip ILS lat,lon  */
  stn->var  = (float)((double)(db_ptr  )->s[ILS_VAR_S]*0.1);
  stn->elv  = (db_ptr++)->s[ILS_ELV_S];
                       /* skip ILS frequency */
  stn->hdg  = (float)((double)(db_ptr++)->s[ILS_HDG_S]*0.1);
  strncpy( stn->aid, (db_ptr++)->c, 4 );
  strncpy( stn->rid, (db_ptr++)->c, 4 );
            /* skip all others (not used) */
  db_ptr += 4;	/* skip lgd&lbw, gsa&gsw, dmelat, dmelon */
  if( rcvr_type == OMKR )      db_ptr += 4;
  else if( rcvr_type == MMKR ) db_ptr += 2;
  else if( rcvr_type == IMKR ) db_ptr += 0;
  stn->lat  = DEGR32( (db_ptr++)->l );
  stn->lon  = DEGR32( (db_ptr++)->l );
/* skip all others (not used) */
  stn->lgd  = stn->lbw = stn->gsa = stn->gsw = 0.0;
  stn->gtd  = 0;
  stn->bad  = 0.0;
  stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
  stn->mgpa = 0.0;
  stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
  stn->bhdg = 0.0;
/*  OM,MM,IM's are killed during the search   */

  return;
}

/*************************************************
**  Airway marker station unpack                **
*************************************************/

static void awm_unpack( register RDB* db_ptr, STN_STRUCT *stn )
{
  if( !db_ptr ) {
    clr_stn_buffer( stn );
    return;				/*exit if no pointer*/
  }

  strncpy( stn->idnt, (db_ptr++)->c, 4 );
  strncpy( stn->icao, (db_ptr++)->c, 4 );
  stn->typ1 = (db_ptr  )->l;
  stn->rng  = (float)((db_ptr++)->l & RNG_MASK);
  stn->typ2 = (db_ptr++)->l;
  stn->lat  = DEGR32( (db_ptr++)->l );
  stn->lon  = DEGR32( (db_ptr++)->l );
  stn->var  = (float)((double)(db_ptr  )->s[AWM_VAR_S]*0.1);
  stn->elv  = (db_ptr++)->s[AWM_ELV_S];
  stn->frq  = (long)((unsigned short)(db_ptr  )->s[AWM_FRQ_S]);
  stn->hdg  = (float)((double)(db_ptr++)->s[AWM_HDG_S]*0.1);
  strncpy( stn->aid, ASCII_BLANKS, 4 );
  strncpy( stn->rid, ASCII_BLANKS, 4 );
/* skip all others (not used) */
  stn->lgd  = stn->lbw = stn->gsa = stn->gsw = 0.0;
  stn->gtd  = 0;
  stn->bad  = 0.0;
  stn->apal = stn->apar = stn->acal = stn->acar = 0.0;
  stn->mgpa = 0.0;
  stn->bpal = stn->bpar = stn->bpcl = stn->bpcr = 0.0;
  stn->bhdg = 0.0;
/*  AWM's kill is done at search time */
  return;
}

/*************************************************
**  Clear station buffer function               **
*************************************************/

static void clr_stn_buffer( STN_STRUCT *stn )
{
  strncpy( stn->idnt, ASCII_BLANKS, 4 );
  strncpy( stn->icao, ASCII_BLANKS, 4 );
  stn->typ1 = stn->typ2 = 0;
  stn->rng  = 0.0;
  stn->lat  = stn->lon = 0.0;
  stn->var  = stn->elv = 0.0;
  stn->frq  = 0;
  stn->hdg  = 0.0;
  strncpy( stn->aid, ASCII_BLANKS, 4 );
  strncpy( stn->rid, ASCII_BLANKS, 4 );
  stn->lgd  = stn->lbw = 0.0;
  stn->gsa  = stn->gsw = 0.0;
  stn->gtd  = 0;
  stn->apal = stn->apar = 0.0;
  stn->acal = stn->acar = 0.0;
  stn->mgpa = 0.0;
  stn->bad  = 0.0;
  stn->bpal = stn->bpar = 0.0;
  stn->bpcl = stn->bpcr = 0.0;
  stn->bhdg = 0.0;
  return;
}

/*********************************************************
**  checksum -                                          **
**    computes an exclusive-or checksum of the          **
**    station data for use by the nav_host              **
**    module.                                           **
**  Inputs: db_ptr - pointer to start of data           **
**          size   - size of the data structure         **
**  Return: result - XOR of the data structure.         **
*********************************************************/
/*
static long checksum( long *ptr, int size )
{
  long i,result=0;

  for( i=0; i<(size/(int)sizeof(*ptr)); i++ )
    result ^= *(ptr++);
  return( result);
}
*/

/****************************************************************
**  get_band_width -                                           **
**    gets the band width for a specific receiver frequency    **
**                                                             **
**  inputs: long frequency - receiver frequency                **
**          long type      - receiver type code                **
**                                                             **
**  outputs: long band_width - the side band width that        **
**                             the search algorithm will tune. **
*****************************************************************/

/**  The ADF band width is used in conjunction with the adf_mode
**   flag from the host.  This is used in analog ADF simulation.
**   Each table entry denotes a band used on the receiver.
*/
struct { long lower; long upper; long width; }
  adf_band_table[3]= {
  { 1900,  4000,  50 },{ 4000,  8400, 075 },{ 8400, 17500, 100 }
  };

long get_band_width( freq, type ) 
long freq,type;
{
  long band_width=0;

  switch( type ) {
  case NDB:
    if( freq < adf_band_table[0].upper )
      band_width = adf_band_table[0].width;
    else if((freq <= adf_band_table[1].upper) &&
            (freq >= adf_band_table[1].lower))
      band_width = adf_band_table[1].width;
    else if( freq > adf_band_table[2].lower )
      band_width = adf_band_table[2].width;
    break;
  default:
    band_width = 0;
  }

  return( band_width );
}
/***  end of nav_stn_srch.c  ***/
