/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**  The information contained herein is the property of         **
**  FlightSafety Simulation Systems Division and shall          **
**  not be copied or used in any manner or disclosed to         **
**  others execpt as expressly authorized by FSI-SSD.           **
**                                                              **
******************************************************************
**                                                              **
**  Department: Navigation/Visual (65)                          **
**                                                              **
**  Task:       Radio Database Search                           **
**              Station Kill Module                             **
**                                                              **
**  Author:     Terry Tyler                                     **
**                                                              **
**  Revision:   1.18               Date: 16/Dec/97              **
**  Revised by: T.Tyler                                         **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**                                                              **
**  The nav_stn_kill module is responsible for the control      **
**  of the station kill identifiers and options that have       **
**  been received from the host.  The kill table and the        **
**  associated type code masks are generated for use by         **
**  the nav station search module.                              **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revision History:                                           **
*****************************************************************/
/*  Rev 1.00  02.28.90	T.Tyler - Initial program release.	*/
/*  Rev 1.01  05/03/90	T.Tyler - Changes #include references.	*/
/*  Rev 1.02  --/--/--  T.Tyler - No changes.			*/
/*  Rev 1.03  --/--/--  T.Tyler - No Changes.			*/
/*  Rev 1.04  06/11/90  T.Tyler - Corrected mask table.		*/
/*  Rev 1.05  08/80/90  T.Tyler - Added stn_kill_active label,	**
**			and modified kill tables to allow for	**
**			partial and marker kill.		*/
/*  Rev 1.06  12/06/90	T.Tyler-Corrected station kill logic	**
**			(again).				*/
/*  Rev 1.07  16/May/91	T.Tyler-No Changes			*/
/*  Rev 1.08  18/Jul/91	T.Tyler-No Changes.			*/
/*  Rev 1.09  03/Oct/91	T.Tyler-No Changes.			*/
/*  Rev 1.10  23/Oct/91	T.Tyler-Added option for ils kill to	**
**			kill the associated markers.		*/
/*  Rev 1.11  12/Mar/92	T.Tyler-No Changes.			*/
/*  Rev 1.12  13/Jul/92	T.Tyler-No Changes.			*/
/*  Rev 1.13  13/Nov/92	T.Tyler-Added two new options to support**
**			the mapping touch kill.  A problem had	**
**			arisen with multiple identifiers.	*/
/*  Rev 1.14  23/Sep/93	T.Tyler-No Changes.			*/
/*  Rev 1.15  03/Jun/94	T.Tyler-Changed to run w/gcc compilers.	*/
/*  Rev 1.16  11/Jan/96	T.Tyler-Change configuration file code.	*/
/*  Rev 1.17  28/Feb/96	T.Tyler-Simplified the module, and 	**
**			LOC kill function.			*/
/*  Rev 1.18  16/Dec/97	T.Tyler-Modified to run in Harris.	*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log: $ 
*
*****************************************************************/


/*********************************************************
**  External References                                 **
*********************************************************/

#include <stdio.h>
#include <ctype.h> 
#include <string.h>
#include <reusable_cat.h>

#include "nav_rdbdef.h"
#include "nav_navdef.h"
#include "navConfig.h"

/*********************************************************
**  Global Definitions/Prototypes                       **
*********************************************************/

#include "nav_global.h"

/*********************************************************
**  Local Definitions/Prototypes                       **
*********************************************************/

#define KILL_STN   1
#define KILL_RESET 2
#define RESET_ALL  3

#define ASCII_BLANKS "    "

/*****  AIOS Kill Mask Bit Assignments  *****/

/*   |       |       |       |       |       |       |       |       |
**   *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
**   | | | | | | | | | | | | | | | | | | | | | | | | | | | | | | | | |
**   |0|I|M|V|D|T| | |N|A|C| | | | | |A|R|E|E|T|H| | | |V|G|L|G|O|M|I|
**   | |L|L|O|M|A| | |D|W|O| | | | | |P|W|N|W|W|P| | | |D|L|O|S|M|M|M|
**   | |S|S|R|E|C| | |B|M|M| | | | | |T|Y|R|P|P| | | | |T|D|C| | | | |
**   | | | | | | | | | | | | | | | | | | | | | | | | | | | | | | | | |
**   *-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
**      x x x x x     x x                               m m s s s s s
**
**	x = Primary record types.
**	m = Mapping System types.
**	s = Secondary record types.
*/
#define VDT_KILL_MASK  0X00000040   /* Map kill mode	*/
#define GLD_KILL_MASK  0X00000020   /* Map kill mode	*/
#define LC_KILL_MASK   0X00000010   /* Type word 2: Loc kill */
#define GS_KILL_MASK   0X00000008   /* Type word 2: GS  kill */
#define OM_KILL_MASK   0X00000004   /* Type word 2: OM  kill */
#define MM_KILL_MASK   0X00000002   /* Type word 2: MM  kill */
#define IM_KILL_MASK   0X00000001   /* Type word 2: IM  kill */


#define TYPE_VDT   VOR_MASK | DME_MASK | TAC_MASK
#define TYPE_GLD   ILS_MASK | DME_MASK

static long mkr_kill=0;

/**  Function Prototypes  **/

static int stn_kill( char *kill_idnt, long kill_option,
                     long kill_mode, long mkr_kill_option );

void stn_kill_mkr( char *ils_idnt, long mkr_lat[], long mkr_lon[] );
long stn_kill_check( char *stn_idnt, long stn_type );
void stn_kill_unpack( char *stn_idnt, long *type_code_1, long *type_code_2 );

/*********************************************************
**                                                      **
**  Station Kill Control Interfaces                     **
**    Allows the station kill logic to be called either **
**    from the normal nav task or by explicit calls.    **
**                                                      **
*********************************************************/

int nav_stn_kill( void ) {
  int i;
  i = stn_kill( host_in.stn_kill_idnt[0], host_in.stn_kill_optn,
            host_in.stn_kill_mode, (long)host_in.host_cmd.mkr_kill );
  return( i );
}

void n_stn_kill( STN_KILL_REQUEST *kill  ) {
  kill->kill_mode = stn_kill( kill->stn_idnt, kill->kill_optn, kill->kill_mode, kill->mkr_kill );
}

/*********************************************************
**                                                      **
**  Station Kill Control                                **
**                                                      **
**********************************************************
**  Inputs: stn_kill_mode    from AIOS via host         **
**          stn_kill_idnt[2] from AIOS via host         **
**          stn_kill_optn    from AIOS via host         **
**                                                      **
**  Output: stn_kill_table[MAX_KILL]                    **
**          structure of killed stations                **
**                                                      **
*********************************************************/

static int stn_kill( char *kill_idnt, long kill_option,
                     long kill_mode, long mkr_kill_option )
{
  int i=0,j;
  int new_kill=1;
  long stn_kill_mode;
  KILL_TBL_STRUCT kill_data = { {0}, 0 };
  static KILL_TBL_STRUCT clear_kill_data={ {0}, 0 };

  if( !kill_mode ) return( kill_mode );

/*  Transfer the data from the input buffer to the local working
**  buffer.
*/
  stn_kill_mode = kill_mode;
  strncpy( kill_data.idnt, kill_idnt, 4 );
  /*  strncpy( kill_data.optn, ASCII_BLANKS, 4 ); */
  kill_data.mask = kill_option;
  mkr_kill = mkr_kill_option;

  switch( stn_kill_mode ) {
  case KILL_STN:

    for( i=0; (stn_kill_table[i].idnt[0] != '\0' ) && (i < MAX_KILL); i++ ) {
      if( strncmp( kill_data.idnt, stn_kill_table[i].idnt, 4 ) == 0 ) {
        new_kill = 0;
        stn_kill_table[i] = kill_data;
      }
    }

    if( new_kill && ( stn_kill_table[MAX_KILL-2].idnt[0] == '\0' ) ) {
      for( i=0; i < MAX_KILL-1; i++ ) {
        if( stn_kill_table[i].idnt[0] == '\0' ) {		/* open slot */
          stn_kill_table[i] = kill_data;
          i = MAX_KILL;		/* force exit	*/
        }
        else if( strncmp( kill_data.idnt, stn_kill_table[i].idnt, 4 ) < 0 ){
          for( j=(MAX_KILL-3); j>=i; j-- )	/* move table down */
            stn_kill_table[j+1] = stn_kill_table[j];
          stn_kill_table[i] = kill_data;	/* insert new data */
          i = MAX_KILL;			/* force exit	*/
        }
      }
    }
    break;

  case KILL_RESET:

    for( i=0; i < MAX_KILL; i++ ) {			/* search for ident*/
      if( strncmp( kill_data.idnt, stn_kill_table[i].idnt, 4 ) == 0 ) {
        for( ; i < (MAX_KILL-1); i++ ) {		/* move table up */
          stn_kill_table[i] = stn_kill_table[i+1];
        }
      }
    }
    break;

  case RESET_ALL:
    for( i=0; i<MAX_KILL; i++)
      stn_kill_table[i] = clear_kill_data;
    break;

  default:
    break;
  }
  stn_kill_active = ( stn_kill_table[0].idnt != 0 );
  return( stn_kill_mode );
}


/*****************************************************************
**  stn_kill_check                                              **
**    Checks the station ident against the kill table and       **
**    modifies the first type code accordingly.  The modified   **
**    type code is returned.  The secondary type code word is   **
**    not modified.                                             **
**                                                              **
**    external references:  stn_kill_table.                     **
**                                                              **
*****************************************************************/

long stn_kill_check( char *stn_idnt, long stn_type )
{
  int i=0;
  long mask=0;
 
  while( ( stn_kill_table[i].idnt[0] != '\0' ) &&
         ( strncmp( stn_idnt, stn_kill_table[i].idnt, 4 ) >= 0 ) ) {
    if( strncmp( stn_idnt, stn_kill_table[i].idnt, 4 ) == 0 ) {

    /* Mask the secondary flags before using on the primary type code */

      mask = ~( stn_kill_table[i].mask & 0XFFFFFF00 );

      if( stn_kill_table[i].mask == VDT_KILL_MASK ) {		/* Map mode */
        if( !(stn_type & ILS_MASK ) ) stn_type &= (~(TYPE_VDT));
      }
      else if( stn_kill_table[i].mask == GLD_KILL_MASK ) {	/* Map mode */
        if( (stn_type & ILS_MASK ) ) stn_type &= (~(TYPE_GLD));
      }
      else {
        stn_type &= mask;					/* Normal */
      }
    }
    i++;
  }
  return( stn_type );
}
 
/*****************************************************************
**  stn_kill_mkr -                                              **
**    Checks the station kill table for a ident match and       **
**    Clears the marker lat/lon array according to the option   **
**    selected.                                                 **
*****************************************************************/
 
void stn_kill_mkr( char *ils_idnt, long mkr_lat[], long mkr_lon[] )
{
  int i=0;

  while( ( stn_kill_table[i].idnt[0] != '\0' ) &&
         ( strncmp( ils_idnt, stn_kill_table[i].idnt, 4 ) >= 0 ) ) {
    if( strncmp( ils_idnt, stn_kill_table[i].idnt, 4 ) == 0 ) {
        if( stn_kill_table[i].mask & OM_KILL_MASK )
          mkr_lat[0] = mkr_lon[0] = 0;
        if( stn_kill_table[i].mask & MM_KILL_MASK )
          mkr_lat[1] = mkr_lon[1] = 0;
        if( stn_kill_table[i].mask & IM_KILL_MASK )
          mkr_lat[2] = mkr_lon[2] = 0;
    }
    i++;
  }
  return;
}

/*****************************************************************
**  stn_kill_unpack -                                           **
**    Checks the station data against the station kill table    **
**    and modifies both the first and second type code words    **
**    accordingly.                                              **
*****************************************************************/

void stn_kill_unpack( char *stn_idnt, long *type_code_1, long *type_code_2 )
{
  int i=0;

  while( ( strncmp( stn_idnt, stn_kill_table[i].idnt, 4 ) >= 0 ) &&
         ( stn_kill_table[i].idnt[0] != '\0' ) ) {
    if( strncmp( stn_idnt, stn_kill_table[i].idnt, 4 ) == 0 ) {
      *type_code_1 &= ( ~(stn_kill_table[i].mask & 0XFFFFFF00 ) );
      if( stn_kill_table[i].mask & GS_KILL_MASK )
        *type_code_2 &= ( ~TYPE_ILS_GS );
      if( stn_kill_table[i].mask & LC_KILL_MASK )
        *type_code_2 &= ( ~TYPE_ILS_LOC );
    }
    i++;
  }
  return;
}

/***** end of nav_stn_kill.c  *****/
