/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**  The information contained herein is the property of         **
**  FlightSafety Simulation Systems Division and shall          **
**  not be copied or used in any manner or disclosed to         **
**  others execpt as expressly authorized by FSI-SSD.           **
**                                                              **
******************************************************************
**                                                              **
**  Department: Navigation/Visual (65)                          **
**                                                              **
**  Task:       Radio Database Search                           **
**              Radio database loader.                          **
**                                                              **
**  Author:     Terry Tyler                                     **
**                                                              **
**  Revision:   1.18                Date: 16/Dec/97             **
**  Revised by: Terry Tyler                                     **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**                                                              **
**  The nav_rdb_init module is responsible for the reading      **
**  of the nav database from the disk into memory, and the      **
**  intialization of index tables used by the various           **
**  search algorithms.                                          **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revison History:                                            **
*****************************************************************/
/* Rev 1.00  02/28/90	Initial program release.		*/
/* Rev 1.01  03/20/90	T.Tyler-Modified Message posting	*/
/* Rev 1.02  --      	No changes				*/
/* Rev 1.03  05/03/90	T.Tyler-Cleaned up arguments		**
** 			Changed #include references.		*/
/* Rev 1.04  06/11/90   T.Tyler-No changes.			*/
/* Rev 1.05  10/04/90   T.Tyler-Added rdb re-load commands.	*/
/* Rev 1.06  12/06/90	T.Tyler-No changes.			*/
/* Rev 1.07  15/May/91	T.Tyler-Changed host_cmd references to	**
**			bitwise operation.			*/
/* Rev 1.08  18/Jul/91	T.Tyler-Changed database area from 	**
**			pre-defined to dynamic memory.		*/
/* Rev 1.09  02/Oct/91	T.Tyler-Changed database file path.	*/
/* Rev 1.10  23/oct/91	T.Tyler-No Changes.			*/
/* Rev 1.11  12/Mar/92	T.Tyler-No Changes.			*/
/* Rev 1.12  13/Jul/92	T.Tyler-Modified logic to prevent 	**
**			multiple allocation of memeory.		*/
/* Rev 1.13  13/Nov/92	T.Tyler-No significant changes.		*/
/* Rev 1.14  23/Sep/93	T.Tyler-No changes.			*/
/* Rev 1.15  03/Jun/94	T.Tyler-Changed to fit new configuration**
**			of the use of Config files, and to put	**
**			the database into a shared segment.	*/
/* Rev 1.16  11/Jan/96	T.Tyler-Brought up to 1.16 comments.	*/
/* Rev 1.17  28/Feb/96	T.Tyler-No Changes.			*/
/* Rev 1.18  16/Dec/97	T.Tyler-Modified to run in Harris.	*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log:  $ 
*
*****************************************************************/

/*****************************************************************
**                                                              **
**  nav_rdb_init.c -                                            **
**                                                              **
**  Function: Load the radio database into memory,              **
**            and initializes the search tables.                **
**            This function is only called whenever             **
**            the database is reloaded                          **
**                                                              **
*****************************************************************/

/*********************************************************
**  External References                                 **
*********************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "nav_rdbdef.h"
#include "nav_navdef.h"
#include "navConfig.h"
#include <reusable_cat.h>

/***** MessageLog handling  *****/

#define PROCNAME "[nav_main:nav_rdb_init]:"
#define RDB_FILENAME_KEY "RDB_Filename"

static char msgstr[100]; /* String for MessageLog */

/**************************************************
**  Global Data/References                       **
**************************************************/

#include "nav_global.h"


/***************************************************
**  Local Data/References                         **
***************************************************/

char * rdb_filename;


/***************************************************
**  Function Prototypes                           **
***************************************************/

/*****  Function Prototypes for externals *****/

void n_rdb_init( void );
long nav_rdb_init( void );

/*****  Local Function Prototypes  *****/

static RDB *get_rdb_memory( unsigned long rdb_size );

char *get_rdb_filename( );
static unsigned long get_rdbsize( char *filename );
static int read_rdb_file( RDB *rdb_ptr, long rdb_size, char *filename );


/*********************************************************
**  Function for external access
*********************************************************/

void n_rdb_init( void ) {
  nav_rdb_init();
}


/*********************************************************
**  rdb_init -                                          **
**                                                      **
**    Loads the radio database (rdb) and                **
**    returns the current revision level of rdb         **
**                                                      **
*********************************************************/

long nav_rdb_init( )
{
  static rdb_rev=-1;
  int size;

  if( rdb_rev < 0 ) {

    rdb_filename = get_rdb_filename();

    sprintf( msgstr,"%s NAV: Reading\n    RDB database.", PROCNAME);
    JPATS_Log_Report (msgstr, JPATS_Log_Informational);
    JPATS_Log_Wait ();

    size = get_rdbsize( rdb_filename );

    rdb = get_rdb_memory( size );

    read_rdb_file( rdb, size, rdb_filename );

    rdb_rev = rdb[2].l;

    sprintf( msgstr,"%s NAV:Radio\n    database %s load complete.",
             PROCNAME, rdb_filename);
    JPATS_Log_Report (msgstr, JPATS_Log_Informational);

    free (rdb_filename);

/*****************************************************************
**  Build the search table structures for other modules         **
**  Tables are built for Frequency, Identifier, and             **
**  Latitude.                                                   **
*****************************************************************/

/** frequency search table indicies  **/

    frq_tbl.start = rdb[29].l / 4 ;		/* long offset */
    frq_tbl.total = rdb[30].l;
    frq_tbl.end   = rdb[31].l / 4 ;		/* long offset */
    for( frq_tbl.idxf = 1; frq_tbl.idxf <= frq_tbl.total;
         frq_tbl.idxf *= 2 );
    frq_tbl.idxf /= 2;
    frq_tbl.indx = frq_tbl.start + frq_tbl.idxf - 1 ;
    frq_tbl.sptr = (RDB *) rdb+frq_tbl.start;
    frq_tbl.eptr = (RDB *) rdb+frq_tbl.end ;
    frq_tbl.iptr = (RDB *) rdb+frq_tbl.indx;

/**  ident search table indicies  **/

    idt_tbl.start = rdb[33].l / 4 ;
    idt_tbl.total = rdb[34].l;
    idt_tbl.end   = rdb[35].l / 4 ;
    for( idt_tbl.idxf = 1; idt_tbl.idxf <= idt_tbl.total;
         idt_tbl.idxf *= 2 );
    idt_tbl.idxf /= 2;
    idt_tbl.indx  = idt_tbl.start + idt_tbl.idxf - 1 ;
    idt_tbl.sptr  = (RDB *) rdb+idt_tbl.start;
    idt_tbl.eptr  = (RDB *) rdb+idt_tbl.end  ;
    idt_tbl.iptr  = (RDB *) rdb+idt_tbl.indx ;

/**  latitude/longitude search table indicies  **/

    lat_tbl.start = rdb[37].l / 4 ;
    lat_tbl.total = rdb[38].l;
    lat_tbl.end   = rdb[39].l / 4 ;
    for( lat_tbl.idxf = 1; lat_tbl.idxf <= lat_tbl.total;
         lat_tbl.idxf *= 2 );
    lat_tbl.idxf /= 2;
    lat_tbl.indx  = lat_tbl.start + (lat_tbl.idxf*3) - 3 ;
    lat_tbl.sptr  = (RDB *) rdb+lat_tbl.start;
    lat_tbl.eptr  = (RDB *) rdb+lat_tbl.end  ;
    lat_tbl.iptr  = (RDB *) rdb+lat_tbl.indx ;

/*********************************************************
**  Test mode code to post database specifics           **
*********************************************************/

  }				/* (rdb_rev != -1 )   */
  return( rdb[2].l );	/* return rev level of rdb */
}

/****************************************************************
**  get_rdb_filename -                                         **
**    build the filename of the radio database.                **
**    Returns:                                                 **
**      return:       pointer to the output character string   **
****************************************************************/

char *get_rdb_filename( )
{
  char * out_filename;
  char Default_RDB_Filename[] = "fsi_rdb.bin";

  out_filename = JPATS_Dictionary_Lookup (RDB_FILENAME_KEY);

  if (out_filename == NULL) {
    sprintf (msgstr, "NAV: Couldn't find %s in dictionary\n\
    (sim.cfg). Using default of %s", RDB_FILENAME_KEY, Default_RDB_Filename);
    JPATS_Log_Report (msgstr, JPATS_Log_Warning);
    return (strcpy (malloc (strlen (Default_RDB_Filename) + 1), Default_RDB_Filename));

  }

  return out_filename;
}

/*****************************************************************
**  get_rdbsize -                                               **
**    gets the size of the file in bytes.                       **
**    inputs:                                                   **
**      filename:  The file name.                               **
**    outputs:                                                  **
**      rdb_size:  Number of bytes found.                       **
*****************************************************************/

static unsigned long get_rdbsize( char *rdb_filename )
{
  FILE *fp; 
  long rdb_size;

  fp = fopen(rdb_filename,"rb");
  if( fp == (FILE *)0 ) 
  {
    sprintf( msgstr,"NAV:RDB Failed opening %s Radio Database file.",
	     rdb_filename);
    
    JPATS_Log_Report (msgstr, JPATS_Log_Error);
    rdb_size = 0;
  } 
   else {
    fseek( fp, 0L, SEEK_END );
    rdb_size = ftell( fp );

    if( fclose( fp ) == -1 ) {
      sprintf( msgstr,"NAV:RDB Failed closing %s Radio Database.",
                    rdb_filename);
      JPATS_Log_Report (msgstr, JPATS_Log_Warning);
      rdb_size = 0;
     }
  }
  return( rdb_size );
}

/*****************************************************************
**  read_rdb_file -                                             **
**    reads the radio database file into memory.                **
**    inputs:                                                   **
**      rdb_ptr:  RDB pointer to allocated memory.              **
**      rdb_size: size of database in bytes.                    **
**      filename: database filename.                            **
**      returns:                                                **
**        error code: 0 = valid,  <> 1 is error condition.      **
*****************************************************************/

static int read_rdb_file( RDB *rdb_ptr, long rdb_size, char *filename )
{
  FILE *fp;
  
  int rdb_error=0;

  if( ( fp = fopen( filename, "rb" )) == NULL) {
    sprintf( msgstr,"NAV:RDB Failed opening %s Radio Database file.",
                 filename );
    JPATS_Log_Report (msgstr, JPATS_Log_Error);
    exit( 1 );
  }
  fseek(  fp, 0L, 0 );
  fread( (char*)rdb_ptr, sizeof(char), rdb_size,  fp );

  if( fclose( fp ) == -1 ) {
    sprintf( msgstr,"NAV:RDB Failed closing %s Radio Database.",
                  filename);
    JPATS_Log_Report (msgstr, JPATS_Log_Warning);
    rdb_error = 1;
  }
  return( rdb_error );
}

/*********************************************************
**  get_rdb_mem -                                       **
**    attaches the radio database memory to a given     **
**    segment.                                          **
**    1) If the code is set up for a single             **
**       processor system, then use shared memory.      **
**    2) If the code is set up for TEST or a three      **
**       processor system, the use "malloc".            **
*********************************************************/

static RDB *get_rdb_memory( unsigned long size )
{
/*
**  If the radio database has not already been allocated, then
**  get the file size and allocate memory for the data.
*/
  if( !rdb ) {
    rdb = (RDB *)malloc( size );       /* allocate database memory*/
    if( rdb == 0 ) {                       /* 0 is error ! */
      sprintf( msgstr,"%s RDB memory allocation fail.", PROCNAME );
      JPATS_Log_Report (msgstr, JPATS_Log_Error);
    }
  }

  return( rdb );
}


/***** end of nav_rdb_init.c  *****/
