/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**	The information contained herein is the property of      **
**	FlightSafety Simulation Systems Division and shall       **
**	not be copied or used in any manner or disclosed to      **
**	others execpt as expressly authorized by FSI-SSD.        **
**                                                              **
******************************************************************
**                                                              **
**  Department: Navigation/Visual (65)                          **
**                                                              **
**  Task:       Radio Database Searches                         **
**              Nav magnetic variation lookup.                  **
**  Author:     Terry Tyler                                     **
**                                                              **
**  Revision:   1.18              Date: 10/Dec/97               **
**  Revised by: T.Tyler                                         **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**                                                              **
**  The nav_magvar function determines the magnetic variation   **
**  based on the Geomagnetic Field Model and IGRF database as   **
**  distributed by the National Oceanic and Atmospneric         **
**  Administration (NOAA).  The database should be updated      **
**  every five years in order to maintain accuracy.  Use of an  **
**  expired database will result in a Warning along with a      **
**  posting of the default database being used.  Using a        **
**  current database, the model is accurate to within 1/2       **
**  of one degree of angular measurement.                       **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revision History:                                           **
******************************************************************
** Rev 1.16  13/Jul/95  T.Tyler, J.Murray-Initial module release.
** Rev 1.17  28/Feb/96  T.Tyler-No changes.
** Rev 1.18  10/Dec/97  T.Tyler-Added configuration to run on a harris
**                      host.  Use the -DCONFIG_IN_HOST compile option.
**                      No changes to this module.
*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log: $ 
*
*****************************************************************/


/*********************************************************
**  nav_magvar.c -                                      **
*********************************************************/

#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <reusable_cat.h>

#include "nav_navdef.h"
#include "navConfig.h"

/*********************************************************
**  Global References/Definitions                       **
*********************************************************/

#ifdef TEST
extern void press_return( void );
#endif

/*********************************************************
**  Local References/Definitions                        **
*********************************************************/

#define IGRF_FILENAME_KEY "IGRF_Filename"

static char msgstr[80];			/* String for MessageLog */

#define ERAD 6371.2                /* Earth nominal radius, KM */
#define A2   40680925.0            /* Square of semi-major axis */
#define B2   40408588.0            /* Square of semi-minor axis */
#define GEOCENTRIC 0
#define GEODETIC   1
#define SN	0.0001
#define M_PI  3.1415926535         /* PII */

/* Ft to Km = Ft * ( 1 Sm / 5280 Ft ) * (1.609344 Km / Sm ) */
#define FT_TO_KM   0.0003048

/* Km to Ft = Km * (0.621371 Sm / Km) * ( 5280 Ft / Sm ) */
#define KM_TO_FT   3280.839895

/* File seek constants */

#ifndef SEEK_SET
#define SEEK_SET  0
#define SEEK_CURR 1
#define SEEK_END  2
#endif

#define MAX_IGRF_HDRS  40          /* Good until 2100 */
#define MAX_IGRF_DATA  200
struct igrf_rec_hdr_struct {
  long   offset;
  char   model_name[9];
  double epoch;
  long   max_deg_main;
  long   max_deg_secular;
  long   max_deg_accel;
  double min_date;
  double max_date;
  double min_alt;
  double max_alt;
  };

struct igrf_rec_data_struct {
  long   m;                 /* order of coefficients */
  long   n;                 /* degree of coefficients */
  double g;                 /* main field Gauss or epochal coeff.*/
  double h;                 /* main field Gauss or epochal coeff.*/
  double tg;                /* secular variation coeff.*/
  double th;                /* secular variation coeff.*/
  double ttg;               /* acceleration coeff.*/
  double tth;               /* acceleration coeff.*/
  char   model_name[9];     /* model name */
  };


/*********************************************************
**  Local Function Prototypes                           **
*********************************************************/

static double magvar( double ac_lat, double ac_lon );


/*  Get the system date in decimal form */
static double Nmvar_get_date( void );

/*  Get the file name from the Configuration File */
static char * Nmvar_get_filename( );

/*  Read the mvar data file header records */
static int Nmvar_read_hdr( struct igrf_rec_hdr_struct *hdr,
                           double *alt_min, double *alt_max,
                           double *date_min, double *date_max );

/*  Determine which data set is needed */
static int Nmvar_which_set( struct igrf_rec_hdr_struct *hdr,
                            int num_hdr, double date );

/*  Read the mvar data records from the data file */

static int Nmvar_read_recs( struct igrf_rec_hdr_struct *hdr,
                            double date, int set_num,
                            double *gh1, double *gh2,
                            double *gha, double *ghb );

/*  Read the individual records from the file and seperate the fields. */
static int Nmvar_getshc( FILE *fp, int mode, int offset, int max_n, double *ghn );

static char *getshc_field( char *out, char *in, int max );

/* interpolation of data */
void Nmvar_interpsh( double date, double date1, double nmax1,
                     double *gh1, double date2, double nmax2,
                     double *gh2, double nmax,  double *gh );

/* extrapolate the data */
static void Nmvar_extrapsh(  double date, double dte1, double nmax1,
                             double *gh1, double nmax2,
                             double *gh2, double nmax,
                             double *gh );

static void Nmvar_shval3( int mode, double lat, double lon, double elev_ft,
                         double erad_km, double a2, double b2, double nmax,
                         double *gh, double *x, double *y, double *z );

/*  Compute d/i/h/f from x/y/x */
static double Nmvar_dihf( double *x, double *y, double *z );


/*****  Local Variables  *****/

static struct igrf_rec_hdr_struct mvar_hdr[MAX_IGRF_HDRS];
static double nmax1,gh1[170],nmax2,gh2[170],nmax,gha[170],ghb[170];

char * igrf_filename;

/*
**  Functions for external reference (non nav_main scheduled task)
*/

void n_magvar( AC_STRUCT *ac, float * magvar_ptr  ) {
  *magvar_ptr = (float)magvar( ac->lat, ac->lon );
}

/*
**  Function for nav_main scheduled task
*/

double nav_magvar( long ac_lat, long ac_lon ) {
  return( magvar( DEGR32( ac_lat ), DEGR32( ac_lon ) ) );
};

/**************************************************************
**                                                           **
**  magvar -                                                 **
**    Magnetic Variation model -                             **
**                                                           **
**************************************************************/

static double magvar( double ac_lat, double ac_lon )
{
  static double ac_alt=0.0;		/* Assume always 0.0 */
  static double date=0.0;
  static double alt_min,alt_max,date_min,date_max;
  static int valid = 0;
  static double latitude,longitude,alt_km;
  static int num_hdr_recs,set_num;
  static double x,y,z,decl;

  latitude = ac_lat;
  longitude = ac_lon;
  alt_km   = ac_alt * FT_TO_KM;

  if( date <= 0.0 ) {                  /* first pass? */

    nmax = 9;
    date = Nmvar_get_date();			/* Get the system date */
    igrf_filename = Nmvar_get_filename();	/* Get data file name */

    num_hdr_recs = Nmvar_read_hdr( mvar_hdr, &alt_min, &alt_max,
                           &date_min, &date_max );

    set_num = Nmvar_which_set( mvar_hdr, num_hdr_recs, date );

    valid = Nmvar_read_recs( mvar_hdr, date, set_num, gh1, gh2, gha, ghb );
  }

  if( valid >= 0 ) { 
  
    Nmvar_shval3( GEODETIC, latitude, longitude, alt_km, ERAD, A2, B2,
                  nmax, gha, &x, &y, &z );
    decl = Nmvar_dihf( &x, &y, &z );

    if( decl >= 900.0  )
      decl = 999.0;
  }
  else
    decl = 999.0;

  return( decl );
};

/**********************************************************
**    Nmvar_get_date -
**      Returns the current system date in decimal form.
**
**      Inputs: None
**      Return: Decimal date in floating point format
*********************************************************/
/* As part of the Year 2000, some systems will not support the
** post 1999 dates.  To get around this, the clock may be reset
** to a 1971 baseline.  Because the magvar model does not have
** to run using older dates, 1998 is used as a trigger for the
** correction.
*/

static double Nmvar_get_date()
{
  time_t clock_time;
  struct tm *time_mdy;
  double  time_out;

  time( &clock_time );
  time_mdy = localtime( &clock_time );  /* convert to day/months/etc */
  time_out = ( (double)time_mdy->tm_year + 1900.0 )
	   + ( (double)time_mdy->tm_yday/365.0 );

  if( time_out <= 1998.0 ) time_out += 28.0;  /* Adjust if clock reset */

  return( time_out );
}

/*********************************************************
**	Nmvar_get_filename -				**
**	  Gets the magvar data file name from the	**
**	  configuration file.				**
*********************************************************/

char * Nmvar_get_filename()
{
  char * out_filename;
  char Default_IGRF_Filename[] = "nav_igrf.dat";

  out_filename = JPATS_Dictionary_Lookup (IGRF_FILENAME_KEY);

  if (out_filename == NULL) {
    sprintf (msgstr, "NAV: Couldn't find %s in dictionary\n\
    (sim.cfg). Using default of %s", IGRF_FILENAME_KEY, Default_IGRF_Filename);
    JPATS_Log_Report (msgstr, JPATS_Log_Warning);
    return (strcpy (malloc (strlen (Default_IGRF_Filename) + 1), Default_IGRF_Filename));

  }

  return out_filename;
}

/**********************************************************
**    Nmvar_read_hdr - reads the igrf.dat file header
**      records for future use.
**
**      hdr:  Input: pointer to array of header structures.
**
***********************************************************/

static int Nmvar_read_hdr( hdr, alt_min, alt_max, date_min, date_max )

struct igrf_rec_hdr_struct *hdr;
double *alt_min, *alt_max, *date_min, *date_max;
{
  FILE *fp;
  char buffer[132];
  char error_msg[80];
  long offset;
  int  i,num=0;

  if( (fp=fopen( igrf_filename, "r" )) == NULL ) {
    (void)sprintf(error_msg,"NAV: Error opening %s data file \n",
                  igrf_filename );
    (void)perror( error_msg );
#ifdef TEST
    press_return();
#endif
    return( 1 );
  }

/* The header record will follow one of the followin forms:
**---------------------------------------------------------------------------------
**          11111111112222222222333333333344444444445555555555666666666677777777778
**012345678901234567890123456789012345678901234567890123456789012345678901234567890
**---------------------------------------------------------------------------------
**     DGRF80  1980.00 10  0  0 1980.00 1980.00   -1.0  600.0          dgrf80    1
**     IGRF95  1995.00 10  8  0 1995.00 1995.00   -1.0  600.0            IGRF95  0
**   IGRF2000  2000.00 10  8  0 2000.00 2005.00   -1.0  600.0          IGRF2000  1
*/

  while( !feof( fp ) ) {
    offset = ftell( fp );
    fgets( buffer, sizeof(buffer)-1, fp );
    if( !strncmp( buffer, "   ", 3 ) ) {         /* Check for hdr record */
      hdr[num].offset = offset;

      /*      (void)strncpy( hdr[num].model_name, &buffer[4], 8 );
	      hdr[num].epoch = atof( &buffer[12] );
	      hdr[num].max_deg_main = atoi( &buffer[20] );
	      hdr[num].max_deg_secular = atoi( &buffer[23] );
	      hdr[num].max_deg_accel = atoi( &buffer[26] );
	      hdr[num].min_date = atof( &buffer[29] );
	      hdr[num].max_date = atof( &buffer[37] );
	      hdr[num].min_alt = atof( &buffer[45] );
	      hdr[num].max_alt = atof( &buffer[52] ); */

      sscanf(buffer," %s %lf %ld %ld %ld %lf %lf %lf %lf",
	     hdr[num].model_name,
	     &hdr[num].epoch, 
	     &hdr[num].max_deg_main,
	     &hdr[num].max_deg_secular, 
	     &hdr[num].max_deg_accel,
	     &hdr[num].min_date, 
	     &hdr[num].max_date,
	     &hdr[num].min_alt,  
	     &hdr[num].max_alt );

      if( num == 0 ) {                             /* 1st pass init max/mins */
	*date_min = hdr[num].min_date;
	*date_max = hdr[num].max_date;
	*alt_min  = hdr[num].min_alt;
	*alt_max  = hdr[num].max_alt;
      }
/*    Update max/min values if necessary */
      if( *alt_min  > hdr[num].min_alt )   *alt_min = hdr[num].min_alt;
      if( *alt_max  < hdr[num].max_alt )   *alt_max = hdr[num].max_alt;
      if( *date_min < hdr[num].min_date )  *date_min = hdr[num].min_date;
      if( *date_max > hdr[num].max_date )  *date_max = hdr[num].max_date;

      num++;
/*
**  If the header table is filled up, post an error to the simulation log.
**  Move the data in the table down, and keep reading the new headers into
**  the last location.
*/
      if( num >= MAX_IGRF_HDRS ) {
        sprintf( msgstr,"E:NAV: MagVar Header Buffer Overflow\n");
        JPATS_Log_Report (msgstr, JPATS_Log_Error);
        for( i=0; i<MAX_IGRF_HDRS-1; i++ ) {        /* Move list down! */
          hdr[i] = hdr[i+1];
        }
        num--;
      }                               /* end of if(num >= MAX_IGRF_HDRS)*/
    }                                 /* end if !strncmp */
  }                                   /* end while( !feof  ) */
/*
** Set all of the remaining offsets to -1 to indicate an unused header
*/
  for( i=num; i<MAX_IGRF_HDRS; i++ )
    hdr[i].offset = -1;

  fclose( fp );
  return( num );
}
/*****************************************************************
**	Nmvar_read_recs -					**
**	  Reads the data sets from the input file.		**
**								**
**	Inputs:							**
**	  hdr -	Structure of arrays of the data file headers.	**
**	  date-	Current floating point date.			**
**	Global:							**
**	  nmax-	(out) maximum number of coefficients.		**
**	Output:							**
**	  gha-  nterpolated array of coefficients.		**
**	  ghb-  interpolated array of coefficients.		**
**	  return-	-1 for error condition.			**
*****************************************************************/

static int Nmvar_read_recs( hdr, date, set_num, gh1, gh2, gha, ghb )

struct igrf_rec_hdr_struct *hdr;
int set_num;
double date, *gh1, *gh2, *gha, *ghb;
{
  FILE *ifp;
  int error=0;
  double date1,date2;

  if( (ifp = fopen( igrf_filename, "r" )) == NULL ) {
    sprintf( msgstr,"NAV: Error opening %s data file.\n", igrf_filename );
    JPATS_Log_Report (msgstr, JPATS_Log_Error);
    error = 1;
#ifdef TEST
    press_return();
#endif
  }
  else {
    if( hdr[set_num].max_deg_secular == 0 ) {

/**  Read in the records for model with no secular coefficients **/
      if( Nmvar_getshc( ifp, 1, hdr[set_num].offset,
                    hdr[set_num].max_deg_main, gh1 ) != 0 )
        error = -1;

      if( Nmvar_getshc( ifp, 1, hdr[set_num+1].offset,
                    hdr[set_num+1].max_deg_main, gh2 ) != 0 )
        error = -1;

/*  Interpolate the g and h coefficients for date based on g and h
**  for bounding epochs.  Second call is to determine secular change.
*/
      date1 = (double)(5 * (int)( date / 5.0 ));
      date2 = date1 + 5.0;
      if (set_num > 1) 
        nmax1 = set_num;
      else
        nmax1 = 1;
      if ((set_num+1) > 1)
        nmax2 = set_num+1;
      else
        nmax2 = 1;
      Nmvar_interpsh( date,  date1, nmax1, gh1, date2, nmax2, gh2, nmax, gha );
      Nmvar_interpsh( date+1,date1, nmax1, gh1, date2, nmax2, gh2, nmax, ghb );

    }
    else {                             /* Secular variations */

/**  Read in the records for model with secular coefficients  **/
      if( Nmvar_getshc( ifp, 1, hdr[set_num].offset,
                        hdr[set_num].max_deg_main, gh1 ) != 0 )
        error = -1;

      if( Nmvar_getshc( ifp, 0, hdr[set_num].offset,
                        hdr[set_num].max_deg_secular, gh2 ) != 0 )
        error = -1;
      if (set_num > 1) 
        nmax1 = set_num;
      else
        nmax1 = 1;
      if ((set_num+1) > 1)
        nmax2 = set_num;
      else
        nmax2 = 2;
      date1 = hdr[set_num].epoch;
      Nmvar_extrapsh( date,       date1, nmax1, gh1, nmax2, gh2, nmax, gha );
      Nmvar_extrapsh( (date+1.0), date1, nmax1, gh1, nmax2, gh2, nmax, ghb );
    }
  }

  return( error );
}

/************************************************************************
**  Nmvar_getshc - Read the coefficient file, arranged as follows:
**
**  N   M    G   H
** ---------------------
**                                   /  1    0    gh[0]  -
**                                 /    1    1    gh[1]  gh[2]
**                               /      2    0    gh[3]  -
**                             /        2    1    gh[4]  gh[5]
**       nmax*(nmax+3)/2     /          2    2    gh[6]  gh[7]
**          records          \          3    0    gh[8]  -
**                             \        .    .    .      .
**                               \      .    .    .      .
**       nmax*(nmax+2)             \    .    .    .      .
**        elements in gh             \  nmax nmax .      .
**
**   N and M are, respectively, the degree and order of the coefficient.
***********************************************************************/

static int Nmvar_getshc( FILE *fp, int mode, int offset, int max_n, double *ghn )
{
  char buffer[132],str[20];
  int i=0,nn,mm,error=0;
  int n,m;
  double g,h,tg,th;

  fseek( fp, offset, SEEK_SET );         /* Go to start of header */
  fgets( buffer, sizeof(buffer), fp );               /* Skip header record */

  for( nn=1; nn<=max_n; nn++ ) {
    for( mm=0; mm<=nn; mm++ ) {
      fgets( buffer, sizeof(buffer), fp );           /* get data record */
      if( mode == 1 ) {
        m = atoi( getshc_field( str, &buffer[0],  2 ) );
        n = atoi( getshc_field( str, &buffer[2],  2 ) );
        g = atof( getshc_field( str, &buffer[4],  8 ) );
        h = atof( getshc_field( str, &buffer[12], 8 ) );
	tg = atof( getshc_field( str, &buffer[20], 8 ) );
	th = atof( getshc_field( str, &buffer[28], 8 ) );
      }
      else {
        m = atoi( getshc_field( str, &buffer[0],  2 ) );
        n = atoi( getshc_field( str, &buffer[2],  2 ) );
        g = atof( getshc_field( str, &buffer[20],  8 ) );
        h = atof( getshc_field( str, &buffer[28], 8 ) );
	tg = atof( getshc_field( str, &buffer[20], 8 ) );
	th = atof( getshc_field( str, &buffer[28], 8 ) );
      }
      if( nn != n || mm != m )  return( -1 );
      ghn[i++] = g;
      if( m != 0 ) {
        ghn[i++] = h;
      }
    }
  }

  return( error );
}

char *getshc_field( out, in, max )

char *out, *in;
int max;
{
  (void)strncpy( out, in, max );
  out[max] = '\0';
  return( out );
}

/*****************************************************************
**	Nmvar_which_set -					**
**	  Determines which data set to use based on the date.	**
**								**
**	Inputs:							**
**	  struct igrf_rec_hdr_struct *hdr - Pointer to the	**
**	    header of each data set in the data file.		**
**	  int num_hdr - Number of header records in the file.	**
**	  float date - Decimal date.				**
**	Outputs:						**
**	  returns: int number of data set within the data file.	**
**	           -1 if error condition occurs.		**
**								**
*****************************************************************/

static int Nmvar_which_set( hdr, num_hdr, date )

struct igrf_rec_hdr_struct *hdr;
int num_hdr;
double date;
{
  int k=0,icount=0,index=0;
  int set_num=0;
  double delta_date=0;
  

/**  Determine which data set is needed **/
  if( num_hdr < 2 )
    set_num = num_hdr;                 /* Only one data set */
  else
    for( icount=0,k=0; k<num_hdr; k++ ) {
      if( hdr[k].min_date == hdr[k].max_date ) {
        delta_date = date - hdr[k].min_date;
        if( delta_date >= 0.0 && delta_date < 5.0 ) {
          icount++;
          if( icount < 2 )
	    index = k;
          else {
            sprintf(msgstr,"NAV: Multiple data sets found.\n");
	    JPATS_Log_Report (msgstr, JPATS_Log_Error);
          }
        }
      }
      else if( date >= hdr[k].min_date && date <= hdr[k].max_date ) {
        icount++;
        if( icount < 2 )
          index = k;
        else {
          sprintf(msgstr,"NAV: Multiple data sets found.\n");
	  JPATS_Log_Report (msgstr, JPATS_Log_Error);
        }
      }
      else if( date > hdr[k].max_date) {
        date = hdr[k].max_date-1;
        sprintf(msgstr,"NAV: MAGVAR database out of date.  Using %d as baseline date.\n", (int)date );
	JPATS_Log_Report (msgstr, JPATS_Log_Warning);
        index = k;
      }
      else {
        sprintf(msgstr,"NAV: No appropriate data set found.\n");
	JPATS_Log_Report (msgstr, JPATS_Log_Error);
        return(-1);
      }
    }
    if( icount < 2 )
      set_num = index;
    else {
      sprintf(msgstr,"NAV: Error on data set selection. %d\n",set_num);
      JPATS_Log_Report (msgstr, JPATS_Log_Error);
      return( -1 );
    }

/**  Check the date for validity  **/

  if( hdr[set_num].max_deg_secular == 0 ) {
    if( ( 5 * (int)( date/ 5.0 )) != hdr[set_num].min_date ) {
      sprintf( msgstr, "NAV: Date %f out of range.\n", date );
      JPATS_Log_Report (msgstr, JPATS_Log_Error);
     return( -1 );
    }
  }
  else {
    if( date < hdr[set_num].min_date || date > hdr[set_num].max_date ) {
      sprintf(msgstr, "NAV: Date %f out of range.\n", date );
      JPATS_Log_Report (msgstr, JPATS_Log_Error);
      return( -1 );
    }
  }
  sprintf( msgstr, "NAV: Mvar data set selected: %s\n",hdr[set_num].model_name);
  JPATS_Log_Report (msgstr, JPATS_Log_Informational);
#ifdef TEST
  press_return();
#endif
  return( set_num );
}

/**********************************************************
**    Nmvar_extrapsh - Extrapolates linearly a spherical
**      harmonic model with a rate of change model.
**
**      Inputs:
**        date - date of resulting model.
**        dte1 - date of base model.
**        nmax1- maximum degree and order of base model.
**        gh1  - schmidt quasi-normal internal shperical
**               harmonic coefficients of base model.
**        nmax2- maximum degree and order of rate of change
**               model.
**        gh2  - schmidt quasi-normal internal spherical
**               harmonic coefficients of rate of change model.
**
**      Outputs:
**        gh   - coefficients of resulting model.
**        nmax - maximum degree and order of resulting model.
**********************************************************/

void Nmvar_extrapsh(  double date,  double dte1,
                      double nmax1, double *gh1,
                      double nmax2, double *gh2,
                      double nmax,  double *gh )

{
  double factor;
  long  i,k,l;

  factor = date - dte1;
  if( nmax1 == nmax2 ) {
    k = nmax1 * ( nmax1 + 2 );
    nmax = nmax1;
  }
  else if( nmax1 > nmax2 ) {
    k = nmax2 * ( nmax2 + 2 );
    l = nmax1 * ( nmax1 + 2 );
    for( i=k; i < l; i++ )
      gh[i] = gh1[i];
    nmax = nmax1;
  }
  else {
    k = nmax1 * ( nmax1 + 2 );
    l = nmax2 * ( nmax2 + 2 );
    for( i=k; i<l; i++ )
      gh[i] = factor * gh2[i];
    nmax = nmax2;
  }
  for( i=0; i<k; i++ )
    gh[i] = gh1[i] + factor*gh2[i];

  return;
}
/***************************************************************
**    Nmvar_interpsh - Interpolates linearly, in time, between
**      two spherical harmonic models.
**
**    Inputs:
**      date  - Current date
**      date1 - Date of earlier model.
**      nmax1 - Maximum degree and order of earlier model.
**      gh1   - Schmidt quasi-normal internal spherical harmonic
**              coefficients of earlier model.
**      date2 - Date of later model.
**      nmax2 - Maximum degree and order of later model.
**      gh2   - Schmidt quasi-normal internal spherical harmonic
**              coefficients of later model.
**
**    Outputs:
**      gh    - coefficients of resulting model.
**      nmax  - maximum degree and order of resulting model.
**
**    The coefficients (gh) of the resulting model, at date
**    "date", are computed by linearly interpolating between the
**    coefficients of the earlier model (gh1), at date "date1",
**    and those of the later model (gh2), at date "dte2".  If one
**    model is smaller than the other, the interpolation is
**    performed with the missing coefficients assumed to be 0.
**
**    Derived from model by A. Zunde, USGS ms 964,
**      Box 25046 Federal Center, Denver, Co. 80225.
****************************************************************/

void Nmvar_interpsh( double date,
                     double date1, double nmax1, double *gh1,
                     double date2, double nmax2, double *gh2,
                     double nmax,  double *gh )
{
  double factor;
  long i,k,l;

  factor = ( date - date1 ) / ( date2 - date1 );
  if( nmax1 == nmax2 ) {
    k  = nmax1 * ( nmax1 + 2 );
    nmax = nmax1;
  }
  else if( nmax1 > nmax2 ) {
    k = nmax2 * ( nmax2 + 2 );
    l = nmax1 * ( nmax1 + 2 );
    for( i=k; i<l; i++ )
      gh[i] = gh1[i] + factor * (-gh1[i]);
    nmax = nmax1;
  }
  else {
    k = nmax1 * ( nmax1 + 2 );
    l = nmax2 * ( nmax2 + 2 );
    for( i=k; i<l; i++ )
      gh[i] = factor * gh2[i];
    nmax = nmax2;
  }
  for( i=0; i<k; i++ )
    gh[i] = gh1[i] + factor * ( gh2[i] - gh1[i] );
  return;
}

/**************************************************************
**  Nmvar_shval3 - Calculates filed components from spherical
**    harmonic (sh) models.
**
**  Inputs:
**    lat    - latitude, deg (+N)
**    lon    - longitude, deg (+E)
**    elev_ft- elevation, feet MSL (+Up)
**    erad_ft- value of earth's radius associated with the sh
**             coefficients, in same units as elev.
**    a2,b2  - Squares of semi-major and semi-minor axes of
**             the reference spheroid used for transformating
**             between geodetic and geocentric coordinates or
**             components.
**    nmax   - Maximum degree and order of coefficients.
**    gh[]   - Schmidt quasi-normal internal spherical
**             harmonic coefficients.
**
**  Outputs:
**    x      - Northward component
**    y      - Eastward component
**    z      - Vertically downward component.
**
**  The required sizes of the arrays used in this subroutine
**  depend on the value of nmax.  The minimum dimensions
**  needed are indicated in the tables below.
**
**  Based on a subroutine 'igrf' by D.R. Barraclough and S.R.C.
**  Malin, report No. 71/1, Institute of Geological Sciences, U.K.
**
**  Minimum Dimensions:
**    sl, cl  - nmax
**    p, q    - ( nmax * ( nmax + 3 )) / 2
**    gh      - ( nmax * ( nmax + 2 ))
**    ext     - 3
********************************************************************/

void Nmvar_shval3( mode, lat, lon, elev_ft, erad_km, a2, b2,
                   nmax, gh, x, y, z )
int mode;
double lat, lon, elev_ft, erad_km, a2, b2, gh[], nmax, *x, *y, *z ;
{
  double sl[14],cl[14];                 /* nmax */
  double p[119],q[119];                 /* ( nmax * ( nmax + 3 )) / 2 */
  double slat,clat,sd=0.0,cd=1.0;
  double r,aa,bb,cc,dd,elev_km;
  double xx=0.0,yy=0.0,zz=0.0;
  double fm;
  double fn = 0.0;
  double ratio;
  double rr = 0.0;

  int   i=0,i1,j=0,j1,k=0,k1,l=1,l1,n=0,n1,m=1,m1;
  int   npq;

  elev_km = elev_ft * FT_TO_KM;
  r = elev_km;
  slat = sin( lat * DEG_TO_RAD );      /* Sine of latitude */
  if( ( 90.0 - lat ) < 0.001 )         /* at north pole? */
    aa = 89.9999;
  else if( ( 90.0 + lat ) < 0.001 )    /* at south pole ? */
    aa = -89.9999;
  else
    aa = lat;

  clat = cos( aa * DEG_TO_RAD );       /* Cosine of latitude */

  sl[0] = sin( lon * DEG_TO_RAD );
  cl[0] = cos( lon * DEG_TO_RAD );
/* Note: xx,yy,zz,sd,cd,n,l,m are initialized above */
  n1 = n - 1;
  l1 = l - 1;
  m1 = m - 1;
  npq = ( nmax * ( nmax + 3 )) / 2;
  if( mode == GEODETIC ) {
    aa = a2 * clat * clat;
    bb = b2 * slat * slat;
    cc = aa + bb;
    dd = sqrt( cc );
    r = sqrt( elev_km * ( elev_km + 2.0 * dd )
            + ( a2 * aa + b2 *bb ) / cc );
    cd = ( elev_km + dd ) / r;
    sd = ( a2 - b2 ) / dd * slat * clat / r;
    aa = slat;
    slat = slat * cd - clat * sd;
    clat = clat * cd + aa * sd;
  }
  ratio = erad_km / r;
  aa = sqrt( 3.0 );
  p[0] = 2.0 * slat;
  p[1] = 2.0 * clat;
  p[2] = 4.5 * slat * slat - 1.5;
  p[3] = 3.0 * aa * clat * slat;
  q[0] = -clat;
  q[1] = slat;
  q[2] = -3.0 * clat * slat;
  q[3] = aa * ( slat * slat - clat * clat );
  for( k=1,k1=0; k<=npq; k++,k1++ ) {
    if( n < m ) {
      m = 0;  m1 = m-1;
      n++;    n1 = n-1;
      rr = pow( ratio,(double)(n + 2));
      fn = n;
    }
    fm = m;
    if( k >= 5 ) {
      if( m == n ) {
        aa = sqrt( 1.0 - 0.5 / fm );
	j = k - n - 1;
        j1 = j - 1;
	p[k1] = ( 1.0 + 1.0/fm ) * aa * clat * p[j1];
        q[k1] = aa * ( clat * q[j-1] + slat / fm * p[j1] );
	sl[m1] = sl[m1-1] * cl[0] + cl[m1-1] * sl[0];
        cl[m1] = cl[m1-1] * cl[0] - sl[m1-1] * sl[0];
      }
      else {
        aa = sqrt( fn * fn - fm * fm );
        bb = sqrt( ( fn - 1.0 ) * ( fn - 1.0 ) - fm * fm ) / aa;
        cc = ( 2.0 * fn - 1.0 ) / aa;
        i = k - n;
        i1 = i - 1;
        j = k - 2 * n + 1;
        j1 = j - 1;
        p[k1] = ( fn + 1.0 ) *
               ( cc * slat / fn * p[i1] - bb / ( fn - 1.0 ) * p[j1] );
        q[k1] = cc * ( slat * q[i1] - clat / fn * p[i1] ) - bb * q[j1];
      }
    }
    aa = rr * gh[l1];
    if( m == 0 ) {
      xx = xx + aa * q[k1];
      zz = zz - aa * p[k1];
      l++;
      l1 = l - 1;
    }
    else {
      bb = rr * gh[l];
      cc = aa * cl[m1] + bb * sl[m1];
      xx = xx + cc * q[k1];
      zz = zz - cc * p[k1];
      if( clat > 0.0 ) {
        yy = yy + ( aa * sl[m1] - bb * cl[m1] ) * fm * p[k1]
             / ( ( fn + 1.0 ) * clat );
      }
      else {
        yy = yy + ( aa * sl[m1] - bb *  cl[m1] ) * q[k1] * slat;
      }
      l += 2;
      l1 = l - 1;
    }
    m++;
    m1 = m - 1;
  }

  *x = xx * cd + zz * sd;
  *y = yy;
  *z = zz * cd - aa * sd;

  return;
}

/*****************************************************************
**	Nmvar_dihf - Computes the declination, inclination,	**
**	  horizontal intensity, and total intensity		**
**	  from the x, y, z components.				**
**								**
**	Inputs:							**
**	  x:	North component					**
**	  y:	Eastward component				**
**	  z:	Vertical component (+down)			**
**	Outputs:						**
**	  d:	Declination					**
*****************************************************************/

static double Nmvar_dihf(  x, y, z )

double *x, *y, *z;
{
  double hpx;
  double ld,lh,lf;

/**  If d and i cannot be determined, set to 999.0 **/

  lh = sqrt( *x**x + *y**y );
  lf = sqrt( *x**x + *y**y + *z**z );
  if ( lf < SN )
    ld = 999.0;
  else {
    if( lh < SN )
      ld = 999.0;
    else {
      hpx = lh + *x;
      if( hpx < 200.0 )
        ld = M_PI * RAD_TO_DEG;
      else
        ld = 2.0 * atan2( *y, hpx ) * RAD_TO_DEG;
    }
  }
  return( ld );
}

/***** end of nav_magvar.c  *****/
