/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**  The information contained herein is the property of         **
**  FlightSafety Simulation Systems Division and shall          **
**  not be copied or used in any manner or disclosed to         **
**  others execpt as expressly authorized by FSI-SSD.           **
**                                                              **
******************************************************************
**                                                              **
**  Department:  Navigation/Visual (65)                         **
**                                                              **
**  Task:        Reposition by ident searches                   **
**  Author:      Terry Tyler                                    **
**                                                              **
**  Revision:    1.18              Date: 16/Dec/97              **
**  Revised by:  T.Tyler                                        **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**                                                              **
**	The nav_ip_srch module is responsible for the database   **
**	search for the reposition functions.  The reposition     **
**	ident and option are received from the host, the         **
**	database is searched useing a binary search algorithm,   **
**	and the resulting data is unpacked and loaded into       **
**	the ip structure for transmission to the host.  The      **
**	allows for marker data to be included in ILS facility    **
**	data.                                                    **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revision History:                                           **
*****************************************************************/
/*  Rev 1.00  02/28/90  T.Tyler-Initial program release		*/
/*  Rev 1.01  03/21/90  T.Tyler-Modified reference to math.h	*/
/*  Rev 1.02  04/19/90  T.Tyler-General module cleanup.		*/
/*  Rev 1.03  05/03/90  T.Tyler-Changed #include references.	*/
/*  Rev 1.04  06/11/90  T.TYler-No changes.			*/
/*  Rev 1.05  --/--/--  T.Tyler-No Changes.			*/
/*  Rev 1.06  06/12/90  T.Tyler-No Changes.			*/
/*  Rev 1.07  16/May/91  T.Tyler-Added station search for the	**
**                       wx control.			*/
/*  Rev 1.08  18/Jul/91  T.Tyler-Changed rdb references for	**
**                       dynamic memory allocation.		*/
/*  Rev 1.09  03/Oct/91  T.Tyler-No Changes.			*/
/*  Rev 1.10  23/Oct/91  T.Tyler-No Changes.			*/
/*  Rev 1.11  12/Mar/92  T.Tyler-No Changes.			*/
/*  Rev 1.12  13/Jul/92  T.Tyler-Modifiedt IP search to be based **
**                       on airport selected first, then on the	**
**                       the aircraft latitude/longitude.	*/
/*  Rev 1.13  13/Nov/92  T.Tyler-Modified airport select idents	**
**                       to allow for multiple airport searches.	**
**                       The IP looks at airport number 0.	*/
/*  Rev 1.14  23/Sep/93  T.Tyler-No changes.			*/
/*  Rev 1.15  03/Jun/94  T.Tyler-Changed to work w/gcc compiler.	*/
/*  Rev 1.16  11/Jan/96  T.Tyler-Brought up to 1.16 comments.	*/
/*  Rev 1.17  28/Feb/96  T.Tyler-Rearranged the search loops to	**
**                       check for table boundries BEFORE using	**
**                       the data.				*/
/*  Rev 1.18  16/Dec/97  T.Tyler-Modified to run in Harris.	*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log: $ 
*
*****************************************************************/

/*********************************************************
**  External References                                 **
*********************************************************/

#include <string.h>

#include "nav_rdbdef.h"
#include "nav_navdef.h"

/*********************************************************
**  Global Variables                                    **
*********************************************************/

#include "nav_global.h"

/*********************************************************
**  Function prototypes                                 **
*********************************************************/

static void ip_srch( long ac_lat, long ac_lon,
                     char *ip_idnt, char *ip_optn, char *apt_idnt,
                     IP_STRUCT *ip );

static RDB* ip_idt_top( char *airport_ident );
static long get_ip_mask( char *ip_option );
static void ip_unpack( RDB *db_ptr, char *ip_ident, char *ip_optn, IP_STRUCT *ip );
static void ip_clr_unpack( IP_STRUCT *ip );
static int  wx_srch( long ac_lat, long ac_lon, char *wx_idnt, WX_STRUCT *wx );
static void wx_clr_unpack( WX_STRUCT *ip );
static void nav_wx_srch( void );

extern long leftify( long ident_input );
extern long checksum( long *pointer, int size_in_bytes );

/*********************************************************
**  Constants and Declarations                          **
*********************************************************/

#define absl( val )  ( (val)<0 ?-(val):(val) )

#define BLANK_ASCII  "    "
#define GENERAL_MASK 0X7CC00000		/*ILS/MLS/VOR/DME/TAC/NDB/AWM*/
#define ILS_ASCII    "ILS "		/* 3 characters = stn type option*/
#define  TO_ASCII    "TO  "		/* 2 characters = ILS option */
#define  TD_ASCII    "TD  "
#define  IM_ASCII    "IM  "
#define  MM_ASCII    "MM  "
#define  OM_ASCII    "OM  "
#define  IA_ASCII    "IA  "
#define  RA_ASCII    "RA  "
#define  LA_ASCII    "LA  "
#define MLS_ASCII    "MLS "
#define VOR_ASCII    "VOR "
#define DME_ASCII    "DME "
#define TAC_ASCII    "TAC "
#define NDB_ASCII    "NDB "
#define AWM_ASCII    "AWM "
#define WPT_ASCII    "WPT "
#define HLD_ASCII    "HLD "

/*********************************************************
**                                                      **
**  nav_ip_srch - External Interfaces                    **
**                                                      **
*********************************************************/

/*
**  The following function is used by the scheduled nav_main task
**  to perform the IP search function.
*/

void nav_ip_srch( void ) {

  ip_srch( host_in.ac_lat,  host_in.ac_lon,
           host_in.ip_idnt, host_in.ip_optn, host_in.apt_idnt[0],
           &host_out.ip );

/***  Weather Radar station ident search  ***/

  if( host_in.host_cmd.wx_srch )
    nav_wx_srch( );

  return;
}

/*
**  The following function is available for non-scheduled database
**  access.  The character identifiers MUST be a minimum of five
**  characters, CAPITAL, left justified, with trailing blanks, using
**  a NULL termination.
*/

void n_ip_srch( IP_REQUEST *ip_req, IP_STRUCT *ip ) {

  if( rdb == 0 ) {
    ip_clr_unpack( ip );
    return;
  }
  ip_srch( BAMS32(ip_req->lat), BAMS32(ip_req->lon),
           ip_req->ip_idnt, ip_req->ip_optn, ip_req->apt_idnt, ip );
  return;
}


/*********************************************************
**                                                      **
**  ip_srch -                                           **
**                                                      **
**********************************************************
**  global variables -                                  **
**    struct idt_tbl	frequency ordered table data	**
**    rdb		radio database			**
**    ip_idnt		ip ident                           **
**    ip_optn		ip option (OM,MM,IM...)            **
**                                                      **
**  returned from function:                             **
**    global host_out.ip  with the reposition data.     **
**                                                      **
*********************************************************/

static void ip_srch( long in_ac_lat, long in_ac_lon,
                     char *in_ip_idnt, char *in_ip_optn, char *in_apt_idnt,
                     IP_STRUCT *out_ip ) {


  long   latitude,longitude;
  char   apt_idnt[4];
  long   ip_mask;
  static char ip_idnt[4]="";
  static char ip_optn[4]="";
  RDB    *tbl_ptr,*db_ptr,*closest_stn_ptr=0,*apt_ptr=0;
  double  cos_lat,closest_stn_dist = 32500.0;

  strncpy( ip_idnt, in_ip_idnt, 4 );
  strncpy( ip_optn, in_ip_optn, 4 );
  latitude  = in_ac_lat;                 /*default a/c lat*/
  longitude = in_ac_lon;                 /*default a/c lon*/
  strncpy( apt_idnt, in_apt_idnt, 4 );

/**  Look for airport selected first  **/

  if( (tbl_ptr = ip_idt_top( apt_idnt )) != 0 ) {
    db_ptr = rdb + ( tbl_ptr->l/sizeof(RDB) );
    while( (tbl_ptr <= idt_tbl.eptr) && (strncmp( apt_idnt, db_ptr->c, 4 ) == 0 ) ) {
      if( (db_ptr+RDB_TYP1)->l & APT_MASK )
        apt_ptr = db_ptr;
      db_ptr = rdb + ((++tbl_ptr)->l/sizeof(RDB));
    }
    if( apt_ptr ) {
      latitude  = (apt_ptr+RDB_LAT)->l;	/*use airport lat*/
      longitude = (apt_ptr+RDB_LON)->l;	/*use airport lon*/
    }
  }
    
  tbl_ptr = ip_idt_top( ip_idnt );
  if( tbl_ptr ) {
    ip_mask = get_ip_mask( ip_optn );
    db_ptr  = rdb + ( tbl_ptr->l/sizeof(RDB) );
    cos_lat = cos((double)latitude * BAM32_TO_RAD);

    while( ( tbl_ptr <= idt_tbl.eptr ) && ( strncmp( ip_idnt, db_ptr->c, 4 ) == 0 ) ) {
      if( (db_ptr+RDB_TYP1)->l & ip_mask ) {
        double dlat,dlon,range_sq;
        long  dlat_bams,dlon_bams;

        dlat_bams = latitude  - (db_ptr+RDB_LAT)->l;
        dlon_bams = longitude - (db_ptr+RDB_LON)->l;
        dlat = DEGR32( absl( dlat_bams ) );
        dlon = DEGR32( absl( dlon_bams ) ) * cos_lat;
        if( (range_sq = dlat*dlat + dlon*dlon) < closest_stn_dist ) {
          closest_stn_dist = range_sq;
          closest_stn_ptr  = db_ptr;
        }
      }
      db_ptr = rdb + ( (++tbl_ptr)->l/sizeof(RDB) );
    }
    ip_unpack( closest_stn_ptr, ip_idnt, ip_optn, out_ip );
  }
  else
    ip_unpack( (RDB*) 0, ip_idnt, ip_optn, out_ip );

  return;
}

/*********************************************************
**  ip_idt_top                                          **
**********************************************************
**  locates the top of the idents in the ident          **
**  table.                                              **
**                                                      **
**  inputs: ASCII identifier (long)			**
**                                                      **
**  output: pointer to the identifier table of the      **
**          first matching ident.                       **
**                                                      **
**  global: Inputs from the radio database.		**
**                                                      **
*********************************************************/

static RDB *ip_idt_top( char *ident )
{
  register RDB *tbl_ptr;
  long         incr_size,found=0;

  incr_size = idt_tbl.idxf;
  tbl_ptr   = idt_tbl.iptr;

  while( incr_size > 0 ) {
    char *db_idt;
    db_idt = rdb[ tbl_ptr->l/sizeof(RDB) ].c;
    if( strncmp( ident, db_idt, 4 ) == 0 ) {
      for( ;
          (tbl_ptr>=idt_tbl.sptr) && (strncmp( ident, rdb[ tbl_ptr->l/sizeof(RDB) ].c, 4 ) == 0 );
          tbl_ptr-- );
      tbl_ptr++;
      incr_size = 0;
      found = 1;
    }
    else if( strncmp( ident, db_idt, 4 ) > 0 ) {
      tbl_ptr += ( incr_size /= 2 );
      if( tbl_ptr > idt_tbl.eptr ) tbl_ptr = idt_tbl.eptr;
    }
    else
      tbl_ptr -= ( incr_size /= 2 );
  }
  if( found ) return( tbl_ptr );
  else        return( 0 );
}

/*********************************************************
**  get_ip_mask -                                       **
**    Changes ASCII ip options into the                 **
**    station type code.                                **
*********************************************************/

static long get_ip_mask( char *option )
{
  if( strncmp( option, BLANK_ASCII, 4 ) == 0 )
    return( GENERAL_MASK );
  else if( ( strncmp( option, ILS_ASCII, 4 ) == 0 ) ||
           ( strncmp( option+2, "  ", 2 ) == 0 ) )
    return( ILS_MASK );
  else if( strncmp( option, MLS_ASCII, 4 ) == 0 )
    return( MLS_MASK );
  else if( strncmp( option, VOR_ASCII, 4 ) == 0 )
    return( VOR_MASK );
  else if( strncmp( option, DME_ASCII, 4 ) == 0 )
    return( DME_MASK );
  else if( strncmp( option, TAC_ASCII, 4 ) == 0 )
    return( TAC_MASK );
  else if( strncmp( option, NDB_ASCII, 4 ) == 0 )
    return( NDB_MASK );
  else if( strncmp( option, AWM_ASCII, 4 ) == 0 )
    return( AWM_MASK );
  else if( strncmp( option, WPT_ASCII, 4 ) == 0 )
    return( EWP_MASK | TWP_MASK );
  else if( strncmp( option, HLD_ASCII, 4 ) == 0 )
    return( HLD_MASK  );
  else
    return( GENERAL_MASK );
}

/*********************************************************
**  ip_unpack                                           **
*********************************************************/

static void ip_unpack( register RDB* db_ptr, char *ident, char *option,
                       IP_STRUCT *ip )
{

  if( db_ptr ) {			/* ptr to start of data */
    strncpy( ip->idnt, ident, 4 );
    strncpy( ip->optn, option, 4 );
    db_ptr+=2;				/* skip icao code */
    ip->typ1 = (db_ptr++)->l;
    ip->typ2 = (db_ptr++)->l;
    ip->slat = DEGR32( (db_ptr++)->l );
    ip->slon = DEGR32( (db_ptr++)->l );

    ip->var  = ip->elv  = ip->hdg = 0.0;
    ip->lgd  = ip->gsa  = 0.0;
    ip->mlat = ip->mlon = 0.0;

    if( ip->typ1 & ILS_MASK ) {
      ip->var  = (float) ((double)(db_ptr  )->s[ILS_VAR_S]*0.1);
      ip->elv  = (float) ((double)(db_ptr++)->s[ILS_ELV_S]);
      ip->hdg  = (float) ((double)(db_ptr++)->s[ILS_HDG_S]*0.1);
      ip->lgd  = (float) ((double)(db_ptr+=2)->s[ILS_LGD_S]);
      ip->gsa  = (float) ((double)(++db_ptr)->s[ILS_GSA_S]*0.01);

      if( strncmp( option, IM_ASCII, 2 ) == 0 ) {
        ip->mlat = DEGR32((db_ptr+=3)->l);
        ip->mlon = DEGR32((++db_ptr )->l);
      }
      else if( strncmp( option, MM_ASCII, 2 ) == 0 ) {
        ip->mlat = DEGR32((db_ptr+=5)->l);
        ip->mlon = DEGR32((++db_ptr )->l);
      }
      else {                                 /* Default to OM */
        ip->mlat = DEGR32((db_ptr+=7)->l);
        ip->mlon = DEGR32((++db_ptr )->l);
      }
    }
    else if( ip->typ1 & MLS_MASK ) {
      ip->var  = (float)((double)(db_ptr   )->s[MLS_VAR_S]*0.1);
      ip->elv  = (float)((double)(db_ptr++ )->s[MLS_ELV_S]);
      ip->hdg  = (float)((double)(db_ptr++ )->s[MLS_HDG_S]*0.1);
      ip->lgd  = (float)((double)(db_ptr+=2)->s[MLS_AED_S]);
      ip->gsa  = (float)((double)(db_ptr+=2)->s[MLS_NEA_S]*0.01);
    }
    else if( ip->typ1 & ( VHF_MASK | AWM_MASK ) ) {
      ip->var  = (float)((double)(db_ptr  )->s[VHF_VAR_S]*0.1);
      ip->elv  = (float)((double)(db_ptr++)->s[VHF_ELV_S]);
      ip->hdg  = (float)((double)(db_ptr  )->s[VHF_DCL_S]*0.1);
    }
    else if( ip->typ1 & ( NDB_MASK | APT_MASK ) ) {
      ip->var  = (float)((double)(db_ptr  )->s[NDB_VAR_S]*0.1);
      ip->elv  = (float)((double)(db_ptr++)->s[NDB_ELV_S]);
    }
    else if( ip->typ1 & WPT_MASK ) {
      ip->var  = (float)((double)(db_ptr  )->s[WPT_VAR_S]*0.1);
    }
    else if( ip->typ1 & HLD_MASK ) {
      ip->hdg  = (float)((double)((db_ptr  )->s[HP_CRS_S])*0.1);
    }
    else if( ip->typ1 & RWY_MASK ) {
      ip->hdg  = (float)((double)(db_ptr  )->s[RWY_HDG_S]*0.1);
      ip->elv  = (float) (ip->typ2 & 0X0000FFFF);
    }
  }
  else {				/* if !db_ptr */
    strncpy( ip->idnt, ident, 4 );
    strncpy(ip->optn, option, 4 );
    ip->slat = ip->slon = 0.0;
    ip->typ1 = ip->typ2 = 0;
    ip->var  = ip->elv  = ip->hdg = 0.0;
    ip->lgd  = ip->gsa  = 0.0;
    ip->mlat = ip->mlon = 0.0;
  }
  return;
}


/****************************************************************
**  ip_clr_unpack -                                            **
**    Clears/Initializes the IP_STRUCT.                        **
*****************************************************************/

static void ip_clr_unpack( IP_STRUCT *ip ) {
  strncpy( ip->idnt, "    ", 4 );
  strncpy( ip->optn, "    ", 4 );
  ip->slat = ip->slon = ip->mlat = ip->mlon = 0.0;
  ip->typ1 = ip->typ2 = 0;
  ip->var  = ip->elv  = ip->hdg = ip->lgd = ip->gsa = 0.0;
  return;
}

/****************************************************************
**  wx_srch -                                                  **
**    Search the database for the closest ident.  Used for     **
**    the positioning of storm cells.                          **
*****************************************************************/
/* Used by software external to the nav_main task */

long n_wx_srch( WX_REQUEST *wx_req, WX_STRUCT *wx ) {
  long valid;

  if( rdb == 0 ) { 
    wx_clr_unpack( wx );
    return( 1 );
  }
  valid = wx_srch( BAMS32(wx_req->lat), BAMS32(wx_req->lon),wx_req->wx_idnt, wx );
  return( valid );
}

/* Used by the nav_main task */

static void nav_wx_srch( void ) {
  int i;
  static char wx_idnt_n1[MAX_WX][4];		/*n-1 station id's*/

  for( i=0; i<MAX_WX; i++ ) {
    if( strncmp( host_in.wx_idnt[i], wx_idnt_n1[i], 4 ) ) {
      wx_srch( host_in.ac_lat, host_in.ac_lon, host_in.wx_idnt[i],
                &host_out.wx[i] );
      strncpy( wx_idnt_n1[i], host_in.wx_idnt[i], 4 );
    }
  }
  return;
}

/****************************************************************
**  wx_srch -                                                 **
**    Search the database for the closest ident.  Used for     **
**    the positioning of storm cells.                          **
*****************************************************************/

static int wx_srch( long latitude, long longitude, char *wx_idnt,
                     WX_STRUCT *wx ) {

  char idnt[4];
  RDB *tbl_ptr,*db_ptr;			/*local pointers  */
  double cos_lat;
  double closest_stn_dist = 32500.0;		/*closest station*/
  RDB *closest_stn_ptr=0;

  strncpy( idnt, wx_idnt, 4 );

  if( (tbl_ptr = ip_idt_top( idnt )) != 0 ) {
    db_ptr  = rdb + ( tbl_ptr->l/sizeof(RDB) );
    cos_lat = cos( (double)latitude * BAM32_TO_RAD );

    while( ( tbl_ptr <= idt_tbl.eptr ) && (strncmp( idnt, db_ptr->c, 4) == 0 ) ) {
      double dlat,dlon,range_sq;
      long dlat_bams,dlon_bams;

      dlat_bams = latitude  - (db_ptr+RDB_LAT)->l;
      dlon_bams = longitude - (db_ptr+RDB_LON)->l;
      dlat = DEGR32( absl( dlat_bams ) );
      dlon = DEGR32( absl( dlon_bams ) ) * cos_lat;
      if( (range_sq=(dlat*dlat+dlon*dlon)) < closest_stn_dist ) {
        closest_stn_dist = range_sq;
        closest_stn_ptr  = db_ptr;
      }					/*if(range_sq..)*/
      db_ptr = rdb + ( (++tbl_ptr)->l/sizeof(RDB));
    }
    if( closest_stn_ptr ) {			/*stn found? */
      db_ptr = closest_stn_ptr;		/*get data ptr*/
      db_ptr += 4;				/*skip data */
      wx->lat = DEGR32( (db_ptr++)->l );
      wx->lon = DEGR32( (db_ptr++)->l );
      wx->elv = db_ptr->s[RDB_ELV_S];
      strncpy( wx->idnt, idnt, 4 );
    }
    else {
      wx->lat = wx->lon = 0.0;
      wx->elv = 0.0;
      strncpy( wx->idnt, idnt, 4 );
    }				/*endif(closest)*/
  }					/*else(tbl_ptr) */
  else {
    wx->lat = wx->lon = 0.0;
    wx->elv = 0.0;
    strncpy( wx->idnt, idnt, 4 );
  }					/*endif(tbl_ptr)	*/
  return( closest_stn_ptr == 0 );
}


/****************************************************************
**  wx_clr_unpack -                                            **
**    Clears/Initializes the WX_STRUCT                         **
*****************************************************************/

static void wx_clr_unpack( WX_STRUCT *wx ) {
  wx->lat = wx->lon = 0.0;
  wx->elv = 0.0;
  strncpy( wx->idnt, "    ", 4 );
  return;
}
/*****  end of nav_ip_srch.c  *****/
