/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**  The information contained herein is the property of         **
**  FlightSafety Simulation Systems Division and shall          **
**  not be copied or used in any manner or disclosed to         **
**  others execpt as expressly authorized by FSI-SSD.           **
**                                                              **
******************************************************************
**                                                              **
**  Department: Navigation/Visual (65)                          **
**                                                              **
**  Task:       Radio Database Search                           **
**              Elevation/Variation Searches                    **
**                                                              **
**  Author:     Terry Tyler                                     **
**                                                              **
**  Revision:   1.18              Date: 16/Dec/97               **
**  Revised by: T.Tyler                                         **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**                                                              **
**  The elevation/variation program is responsible for the      **
**  database search and the resulting elevation and             **
**  variation computations with respect to the aircraft         **
**  position.  The latitude/longitude data is received          **
**  the host at scheduled updates.  The radio database is       **
**  then searched for the closest two facilities in each        **
**  of four quadrants.  The elevation and variation is          **
**  then computed using a distance weighted algorithm           **
**  described later in this module.  Airport data is given      **
**  priority over normal navaids to insure a flat terrain       **
**  around the airport area. (A requirement for some visual     **
**  systems.)  The resulting data is loaded into the            **
**  host_out structure for transmission to the host.            **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revision History:                                           **
*****************************************************************/
/*  Rev 1.00  02/28/90  T.Tyler - Initial program release	*/
/*  Rev 1.01  03/21/90  T.Tyler - Modified reference to the	**
**			BAM32_TO_RAD term (now in navdef.h)	*/
/*  Rev 1.02  05/03/90	T.Tyler-Changed #include references.	*/
/*  Rev 1.03  --/--/--  T.Tyler - No Changes.			*/
/*  Rev 1.04  06/11/90  T.Tyler - No Changes.			*/
/*  Rev 1.05  --/--/--	T.Tyler - No Changes.			*/
/*  Rev 1.06  12/06/90	T.Tyler - No Changes.			*/
/*  Rev 1.07  16/May/91	T.Tyler - No Changes.			*/
/*  Rev 1.08  18/Jul/91 T.Tyler - Changed rdb references to	**
**			dynamic memory allocation references.	*/
/*  Rev 1.09  03/Oct/91	T.Tyler - No Changes.			*/
/*  Rev 1.10  23/oct/91	T.Tyler - No Changes.			*/
/*  Rev 1.11  12/Mar/92	T.Tyler - No Changes.			*/
/*  Rev 1.12  13/Jul/92 T.Tyler - No Changes.			*/
/*  Rev 1.13  13/Nov/92	T.Tyler - No Changes.			*/
/*  Rev 1.14  23/Sep/93	T.Tyler - No Changes.			*/
/*  Rev 1.15  08/Jun/94	T.Tyler - No changes.			*/
/*  Rev 1.16  19/Jul/95	T.Tyler - Added magvar function.	*/
/*  Rev 1.17  28/Feb/96	T.Tyler - Corrected problems in search	**
**				  boundry limits, overflow in	**
**				  elevation computation.	*/
/*  Rev 1.18  16/Dec/97	T.Tyler- Modified to run in Harris.	*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log: $ 
*
*****************************************************************/

/*********************************************************
**  External References                                 **
*********************************************************/

#include "nav_rdbdef.h"
#include "nav_navdef.h"
#include <math.h>

/*********************************************************
**  Global References/Declarations                      **
*********************************************************/

#include "nav_global.h"

/*********************************************************
**  Function prototypes                                 **
*********************************************************/

extern double nav_magvar( long latitude, long longitude );

static void elevar( long lat, long lon, float *elv, float *var );
static RDB *elevar_top( long latitude );
static float variation( long latitude, long longitude );

#define absl( val )  ( (val)<0 ?-(val):(val) )

#define TBL_LAT  0			/* table latitude offset */
#define TBL_LON  1			/* table long. offset    */
#define TBL_IDX  2			/* table RDB data offset */
#define TBL_SIZE 3			/* table dataset size    */

#define MAX_NUM_STNS    8		/* Max number of stations    */
#define MAX_APT_DIST    0.0625		/* 15 NM in Degrees **2      */
#define SEARCH_WINDOW   0x01C71C71	/* BAMs for search size	*/
					/* 0X038E38E3 = 5.0 deg/300Nm*/
					/* 0X01C71C71 = 2.5 Deg/150Nm*/
					/* 0X012F6846 = 100 Nm. */
#define MAX_STN_DIST    6.25		/* Float search distance */
					/* Same as SEARCH_WINDOW */
#define SEARCH_MASK     0x7CE08000	/*VHF|NDB|AWM|APT|ILS|MLS|COM*/
#define ELV             VHF_ELV
#define ELV_S           VHF_ELV_S
#define VAR             VHF_VAR
#define VAR_S           VHF_VAR_S

/** nav_main interface **/

void nav_elevar( void ) {
  elevar( host_in.ac_lat, host_in.ac_lon, &host_out.elv, &host_out.var );
  host_out.var = variation( host_in.ac_lat, host_in.ac_lon );
  return;
}

/** Other interfaces:  ??? etc. **/

void n_elev( AC_STRUCT *ac, float * Elev_Ptr ) {
  float var;
  
  var = 0.0;

  if( rdb == 0 )  
  {
	  *Elev_Ptr = -9999.0;
  }
  else
  {
  
	  elevar( BAMS32(ac->lat), BAMS32(ac->lon), Elev_Ptr, &var );

  }

}

float n_elevar( AC_STRUCT *ac, float *elv, float *var ) {

  if( rdb == 0 )  return( -9999.0 );

  elevar( BAMS32(ac->lat), BAMS32(ac->lon), elv, var );  /* Database Model */
  *var = variation( BAMS32(ac->lat), BAMS32(ac->lon) );  /* GEOMAG Model */

  return( *elv );
}


/*********************************************************
**                                                      **
**  nav_elevar  -  Elevation/Variation Search           **
**                                                      **
*********************************************************/
/*  Global Inputs:                                      **
**    latitude = aircraft latitude, in BAMs             **
**    longitude= aircraft longitude, in BAMs            **
**    rdb = radio database                              **
**    lat_tbl = radio database latitude tbls            **
**                                                      **
**  Global Outputs:                                     **
**    elv = elevation, ft under a/c                     **
**    var = variation, deg. under a/c                   **
**                                                      **
**  Other functions called:                             **
**    elevar_top - Finds the start point in             **
**                 the latitude table.                  **
*********************************************************/

static void elevar( long latitude, long longitude, float *elv_out, float *mvar_out )
{
  float elev,var=0;
  long   uppr_lat,lon_window;
  register RDB *tbl_ptr,*db_ptr;
  double cos_lat;
  int    i;

  double closest_apt_dst=MAX_APT_DIST;
  long   closest_apt_elv=0,closest_apt_var=0;
  int    number_of_stns=0;
  double closest_stn_dst[MAX_NUM_STNS];
  long   closest_stn_var[MAX_NUM_STNS],closest_stn_elv[MAX_NUM_STNS];

  for( i=0; i<8; i++ )  closest_stn_dst[i] = MAX_STN_DIST;
  uppr_lat = latitude + SEARCH_WINDOW;
  cos_lat  = cos((double)latitude * BAM32_TO_RAD);
  lon_window = (long)( (double)SEARCH_WINDOW / cos_lat );
                       
  tbl_ptr  = elevar_top( (latitude-SEARCH_WINDOW) );

  while ( ( tbl_ptr <= lat_tbl.eptr ) && ((tbl_ptr+TBL_LAT)->l < uppr_lat ) )
  {
    long delta_lat,delta_lon;
    
    db_ptr = rdb + ( (tbl_ptr+TBL_IDX)->l/sizeof(RDB) );

    delta_lon = longitude-(tbl_ptr+TBL_LON)->l;

    if( absl(delta_lon) < lon_window ) {
      double dlat,dlon,range_sq;

      delta_lat = latitude - (tbl_ptr+TBL_LAT)->l;
      dlat = DEGR32( delta_lat );
      dlon = DEGR32( delta_lon ) * cos_lat;
      range_sq = dlat*dlat + dlon*dlon;
      db_ptr = rdb + ( (tbl_ptr+TBL_IDX)->l/sizeof(RDB) );

      if( ( (db_ptr+RDB_TYP1)->l & APT_MASK )  &&
          ( range_sq < closest_apt_dst )         ) {
        closest_apt_dst = range_sq;
        closest_apt_elv = (db_ptr+APT_ELV)->s[APT_ELV_S];
        closest_apt_var = (db_ptr+APT_VAR)->s[APT_VAR_S];
      }

      if( (db_ptr+RDB_TYP1)->l & SEARCH_MASK ) {
        int quadrant=0;
						/*  0  |  1  */
        if( delta_lat < 0 ) quadrant  = 2;	/* ---a/c--- */
        if( delta_lon < 0 ) quadrant += 1;	/*  2  |  3  */

        if( range_sq < closest_stn_dst[quadrant] ) {
          closest_stn_dst[quadrant+4] = closest_stn_dst[quadrant];
          closest_stn_elv[quadrant+4] = closest_stn_elv[quadrant];
          closest_stn_var[quadrant+4] = closest_stn_var[quadrant];
          closest_stn_dst[quadrant]   = range_sq;
          closest_stn_elv[quadrant]   = (db_ptr+ELV)->s[ELV_S];
          closest_stn_var[quadrant]   = (db_ptr+VAR)->s[VAR_S];
        }
        else if( range_sq < closest_stn_dst[quadrant+4] ) {
          closest_stn_dst[quadrant+4] = range_sq;
          closest_stn_elv[quadrant+4] = (db_ptr+ELV)->s[ELV_S];
          closest_stn_var[quadrant+4] = (db_ptr+VAR)->s[VAR_S];
        }
      }
    }				/* end if (delta_lat...) */
    tbl_ptr += TBL_SIZE;
  }					/* end while */

/*
**  Search is completed.. Now do the computations 
**
**
**         elv1   elv2   elv3   elv4   elv n
**         ---- + ---- + ---- + ---- + -----
**         dst1   dst2   dst3   dst4   dst n
** elv = --------------------------------------------------
**          1.0    1.0    1.0    1.0    1.0
**         ---- + ---- + ---- + ---- + -----
**         dst1   dst2   dst3   dst4   dst n
*/

  if( closest_apt_dst < MAX_APT_DIST ) {
    elev = (float)closest_apt_elv;
  }
  else  {
    double etop=0,vtop=0,bottom=0;

    for( i=0; i<MAX_NUM_STNS; i++ ) {
      if( closest_stn_dst[i] < MAX_STN_DIST ) {
        double dst;

        if( closest_stn_dst[i] < 0.001 )
          closest_stn_dst[i] = 0.001;
        dst = 1.0 / closest_stn_dst[i];
        bottom += dst;
        etop   += (double)closest_stn_elv[i] * dst;
        vtop   += (double)closest_stn_var[i] * dst * 0.1;
        number_of_stns++;
      }
    }

    if( number_of_stns >= 2 ) {
      elev = (float) (etop / bottom);
      var  = (float) ( vtop / bottom );
    }
    else {
      elev = 0.0;
      var  = var;
    }
  }
  *mvar_out = var;
  *elv_out  = elev;
  return;
}

/***************************************************************************
**  var - Variation Model
***************************************************************************/
/*  The test area is set to a magvar of zero
**  If the aircraft is outside the boundries defined by: N47 x N43 x W42 X w48
**  the normal magnetic variation is computed.  Inside of this region, the
**  magnetic variation is reduced to zero at the test area center, and for a
**  radius of approximately 100 miles around the test area.
*/
static float variation( long latitude, long longitude ) {
  float var=0;

  if( ( latitude  > 0X216C16C1 ) || ( latitude  < 0X1E93E93E ) ||
      ( longitude > 0XE2222222 ) || ( longitude < 0XDDDDDDDD ) ) {
    var = (float)nav_magvar( latitude, longitude );
  }
  else {					/* Inside test area */
    double dlat,dlon,dist,easy_on;
    dlat = DEGR32( latitude ) - 45.0;
    dlon = ( DEGR32( longitude ) - ( -45.0 ) ) * cos(DEGR32(latitude));
    dist = sqrt( dlat*dlat + dlon*dlon ) * 60.0;
    easy_on = ( dist - 100 ) * 0.05;		/* 0.0 @100Nm, 1.0 @120Nm */
    easy_on = LIMIT( easy_on, 1.0, 0.0 );
    var = (float)(nav_magvar( latitude, longitude ) * easy_on);
  }
  return( var );
}

/*********************************************************
**  elevar_top                                          **
**    Finds the top latitude in the latitude table.     **
**    returns the pointer to the latitude table.        **
*********************************************************/

static RDB *elevar_top( long latitude )
{
  register RDB *tbl_ptr;
  long incr_size;

  tbl_ptr   = lat_tbl.iptr;
  incr_size = lat_tbl.idxf;

  while( incr_size > 0 ) {
    if( latitude == (tbl_ptr+TBL_LAT)->l ) {
      for( ;
           (tbl_ptr >= lat_tbl.sptr) && (latitude == (tbl_ptr+TBL_LAT)->l);
           tbl_ptr -= TBL_SIZE );
      tbl_ptr += TBL_SIZE;
      incr_size = 0;
    }
    else if( latitude > (tbl_ptr+TBL_LAT)->l ) {
      tbl_ptr += ( (incr_size /= 2 ) * TBL_SIZE );
      if( tbl_ptr > lat_tbl.eptr ) tbl_ptr = lat_tbl.eptr;
    }
    else
      tbl_ptr -= ( (incr_size /= 2 ) * TBL_SIZE );
  }
  return( tbl_ptr );
}

/*****  end of nav_elevar.c  *****/
