/*****************************************************************
**                                                              **
**  FlightSafety International, Inc.                            **
**  2700 North Hemlock Circle                                   **
**  Broken Arrow, Oklahoma 74012                                **
**  (918) 251-0500                                              **
**                                                              **
******************************************************************
**                                                              **
**  The information contained herein is the property of         **
**  FlightSafety Simulation Systems Division and shall          **
**  not be copied or used in any manner or disclosed to         **
**  others except as expressly authorized by FSI-SSD.           **
**                                                              **
******************************************************************
**                                                              **
**  Department: Navigation/Visual (65)                          **
**                                                              **
**  Task:       Radio Database Search                           **
**              Airport/Runway Lookup                           **
**  Author:     Terry Tyler                                     **
**                                                              **
**  Revision:   1.18          Date: 16/Dec/97                   **
**  Revised by: T.Tyler                                         **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Program Description:                                        **
******************************************************************
**                                                              **
**  The airport/runway search module is responsible for         **
**  searching the database for the airport/runway selected      **
**  by the host.  A binary search is first done on the          **
**  airport ident, then a sequential search is performed        **
**  for the airport data and runway data.  The data is          **
**  unpacked, loaded into the airport and runway                **
**  structures for transmission to the host.                    **
**                                                              **
*****************************************************************/

/*****************************************************************
**  Revision History:                                           **
*****************************************************************/
/*  Rev 1.00  02/28/90	T.Tyler-Initial program release		*/
/*  Rev 1.01  03/21/90	T.Tyler-Added airport runway data.	*/
/*  Rev 1.02  04/06/90  T.Tyler-Modified ordering of the runway	*/
/*			pointer tables.  Removed logic that	*/
/*			clears out the airport data if the ident*/
/*			is not found.  Now leaves n-1 data.	*/
/*  Rev 1.03  05/03/90	T.Tyler-Changed #include references.	*/
/*  Rev 1.04  06/11/90  T.Tyler-No changes.                     */
/*  Rev 1.05  06/12/90	T.Tyler-No Chagnes.			*/
/*  Rev 1.07  16/May/91	T.Tyler-No Changes.			*/
/*  Rev 1.08  18/Jul/91	T.Tyler-Changed rdb references for 	**
**			dynamic memory allocation.		*/
/*  Rev 1.09  03/Oct/91	T.Tyler-No changes.			*/
/*  Rev 1.10  23/oct/91	T.Tyler-No Changes.			*/
/*  Rev 1.11  12/Mar/92	T.Tyler-No Changes.			*/
/*  Rev 1.12  13/Jul/92 T.Tyler-Added checksum computations.	*/
/*  Rev 1.13  13/Nov/92	T.Tyler-Added airport ATA idents, and	**
**			set up module to allow for multiple	**
**			airport searches.  Added host selectable**
**			runway list ordering			*/
/*  Rev 1.14  23/Sep/93	T.Tyler-No changes.			*/
/*  Rev 1.14a 28/Oct/93	T.Tyler-Fixed compatibility problem in	**
**			the runway list selection.		*/
/*  Rev 1.15  03/Jun/94	T.Tyler-Changed to run w/gcc compilers.	*/
/*  Rev 1.16  11/Jan/96	T.Tyler-Brought up to 1.16 comments.	*/
/*  Rev 1.17  28/Feb/96	T.Tyler-Restructured search constraints	**
**			to check table boundries BEFORE checking**
**			the data.				*/
/*  Rev 1.18  16/Dec/97	T.Tyler-Modified to run in Harris.	*/
/*****************************************************************
*  RCS Revision History
******************************************************************
*
* $Id:  $
* $Log: $ 
*
*****************************************************************/

/*****************************************************************
**                                                              **
**  nav_apt_srch.c -                                            **
**    calls the following internal functions:                   **
**    idt_top -    Find match within ident table                **
**    apt_srch -   Find airport record in database              **
**    rwy_srch -   Find runway record in database               **
**    apt_unpack - Unpack airport data record                   **
**    rwy_unpack - Unpack runway data record                    **
**                                                              **
*****************************************************************/

/*********************************************************
**  External References                                 **
*********************************************************/

#include <string.h>

#include "nav_navdef.h"
#include "nav_rdbdef.h"
#include "nav_global.h"

/*********************************************************
**  Local structure declarations                        **
*********************************************************/

#define MAX_ATIS  3
#define ATIS_MASK 0X02000000

#define ASCII_BLANKS "    "

struct rwy_tbl_struct { char id[4];  RDB *addr;
                        long ils_frq; long atis_frq[MAX_ATIS]; };
typedef struct rwy_tbl_struct RWY_TBL;

/*********************************************************
**  Function prototypes                                 **
*********************************************************/

extern long checksum( long *pointer, int size_in_bytes );

static RDB *idt_top( char *identifier );
static RDB *apt_srch( RDB *tbl_top, char *identifier );
static RDB *rwy_srch( RDB *db_ptr, RWY_TBL rwy_tbl[],
                      char *apt_ident, char *rwy_ident, long list_mode );
static void apt_unpack( register RDB *db_ptr, long num, RWY_TBL rwy_tbl[],
                        APT_STRUCT *apt_struct );
static void rwy_unpack( register RDB *db_ptr, long num, RWY_TBL rwy_tbl[],
                        RWY_STRUCT *rwy_struct );

/**  Local Definitions  **/

#define RWY_TBL_MAX  25

/*****************************************************************
**  n_aptrwy_srch -
**    Allows for external access to the apt/rwy search algorithm
**  Note(s):
**  1) The apt_ident and rwy_ident are 5 character NULL terminated
**     string, CAPITAL, left justified, with trailing blanks.
*****************************************************************/

void n_aptrwy_srch( APTRWY_REQUEST *req, APT_STRUCT *apt, RWY_STRUCT *rwy ) {
  long found=0;
  RDB  *tbl_ptr,*apt_db_ptr=0,*rwy_db_ptr=0;
  RWY_TBL rwy_tbl[RWY_TBL_MAX];

  if( rdb == 0 )  return;       /* Error if RDB is not loaded */

  tbl_ptr = idt_top( req->apt_idnt );

  apt_db_ptr = apt_srch( tbl_ptr, req->apt_idnt );
  rwy_db_ptr = rwy_srch( apt_db_ptr, rwy_tbl, req->apt_idnt, req->rwy_idnt, req->rwy_sort_mode );
  apt_unpack( apt_db_ptr, 0, rwy_tbl, apt );
  rwy_unpack( rwy_db_ptr, 0, rwy_tbl, rwy );
  if( apt_db_ptr == 0 || rwy_db_ptr == 0 )  found = 2;

}
/*****************************************************************
**                                                              **
**  nav_apt_srch -                                              **
**    Called by main task.  Searchs the database for            **
**    corresponding airport/runway data, and unpacks            **
**    the data for transmission to the host.                    **
**                                                              **
**    input parameters:  none                                   **
**    return parameter:  none                                   **
**    global inputs:     Radio database (rdb)                   **
**                       Rdb pointer tables (idt_tbl,frq_tbl)   **
**                       Host airport ident (host_in.apt_idnt)  **
**                       Host ruNway  ident (host_in.rwy_idnt)  **
**    global outputs:    Airport data structure (host_out.apt)  **
**                       Runway  data structure (host_out.rwy)  **
**                                                              **
*****************************************************************/

void nav_apt_srch( )
{
  long number,list_mode;
  RDB  *tbl_ptr,*apt_db_ptr=0,*rwy_db_ptr=0;
  char apt_ident[4],rwy_ident[4];
  static char apt_ident_n1[MAX_AIRPORTS][4];
  static char rwy_ident_n1[MAX_AIRPORTS][4];
  RWY_TBL rwy_tbl[RWY_TBL_MAX];

  for( number=0; number < MAX_AIRPORTS; number++ ) {

    if( strncmp( host_in.apt_idnt[number], apt_ident_n1[number], 4 ) ||
        strncmp( host_in.rwy_idnt[number], rwy_ident_n1[number], 4 ) )
    {
      strncpy( apt_ident, host_in.apt_idnt[number], 4 );
      strncpy( rwy_ident, host_in.rwy_idnt[number], 4 );
      list_mode = host_in.host_cmd.rwy_list;

      tbl_ptr = idt_top( apt_ident );

      apt_db_ptr = apt_srch( tbl_ptr, apt_ident );
      rwy_db_ptr = rwy_srch( apt_db_ptr, rwy_tbl, apt_ident, rwy_ident, list_mode);
      apt_unpack( apt_db_ptr, number, rwy_tbl, &host_out.apt[number] );
      rwy_unpack( rwy_db_ptr, number, rwy_tbl, &host_out.rwy[number] );
    }
  }
  return;
}
 
/*****************************************************************
**                                                              **
**  idt_top -                                                   **
**    locates the top of the matching                           **
**    ident in the ident table.                                 **
**                                                              **
**    inputs: ASCII airport identifier  (long)                  **
**                                                              **
**    output: identifier table pointer that                     **
**            corresponds to the first ident within             **
**            the identifier table.                             **
**                                                              **
**    global: Inputs from the radio data pointers.              **
**                                                              **
*****************************************************************/

static RDB *idt_top( char *ident )
{
  register RDB  *tbl_ptr;
  long incr_size;			/* search pattern size */
  long found=0;				/* station found flag  */

  incr_size  = idt_tbl.idxf;
  tbl_ptr    = idt_tbl.iptr;		/*start of search ptr */

  while( incr_size > 0 ) {
    char *db_idt;
    db_idt = rdb[ tbl_ptr->l/sizeof(RDB) ].c;
    if( strncmp( ident, db_idt, 4 ) == 0 )  {
      for( /*tbl_ptr*/;
          ( tbl_ptr >= idt_tbl.sptr ) &&
          ( strncmp( ident, rdb[ tbl_ptr->l/sizeof(RDB) ].c, 4 ) == 0 );
          tbl_ptr-- );			/* back top of idents */
      tbl_ptr++;			/* top index */
      incr_size = 0;			/* end of search */
      found = 1;			/* set found flag */
    }
    else if( strncmp( ident, db_idt, 4 ) > 0 ) {
      tbl_ptr += (incr_size/=2);
      if( tbl_ptr > idt_tbl.eptr ) tbl_ptr = idt_tbl.eptr;
    }
    else
      tbl_ptr -= (incr_size/=2);
  }
  if( found ) return(  tbl_ptr );		/* idt found */
  else        return 0;			/* not found */
}

/*****************************************************************
**                                                              **
**  apt_srch -                                                  **
**    performs a station by station search                      **
**    starting at the top of the matching                       **
**    identifiers (previously found).                           **
**                                                              **
******************************************************************
**  global variables -                                          **
**    struct idt_tbl	frequency ordered table data               **
**    rdb		radio database                             **
**                                                              **
**    inputs to function:                                       **
**      top_ptr:	address of top of table of                 **
**			matching identifiers.                      **
**      ident:		airport identifier                  **
**      latitude:	a/c laitutude, in BAMS                     **
**      longitude:	a/c longitude, in BAMs                     **
**                                                              **
**    returned from function:                                   **
**      pointer to the ident table that corresponds             **
**      to the airport found.                                   **	
**                                                              **
*****************************************************************/

static RDB *apt_srch( RDB *tbl_ptr, char *ident )
{
  RDB  *db_ptr,*closest_apt_ptr=0;

  if( tbl_ptr == 0 )  return( 0 );

  db_ptr = rdb + (tbl_ptr->l/sizeof(RDB));

  while( ( tbl_ptr <= idt_tbl.eptr ) && ( strncmp( ident, db_ptr->c, 4 ) == 0 ) ) {
    if( (db_ptr+RDB_TYP1)->l & APT_MASK )  closest_apt_ptr = db_ptr;
    db_ptr  = rdb + ((++tbl_ptr)->l/sizeof(RDB));
  }
  return ( closest_apt_ptr );
}

/*****************************************************************
**                                                              **
**  rwy_srch -                                                  **
**    performs a station by station search                      **
**    starting at the top of the matching                       **
**    identifiers (previously found).                           **
**                                                              **
******************************************************************
**  global variables -                                          **
**    struct idt_tbl	frequency ordered table data               **
**    rdb            radio database                             **
**                                                              **
**  inputs to function:                                         **
**    apt_tbl_ptr:  address of table entry of                   **
**                  airport record within the table             **
**    ident:        airport identifier                          **
**                                                              **
**  returned from function:                                     **
**    pointer to the runway record that corresponds             **
**    to the matching airport/runway identifier.                **	
**                                                              **
*****************************************************************/

static RDB *rwy_srch( RDB *apt_db_ptr, RWY_TBL rwy_tbl[],
                      char *apt_ident, char*rwy_ident, long list_mode )
{
  int i,rwy_count,atis_cnt;
  long rwy_len,rwy_longest=0;
  register RDB *db_ptr,*db_end,*rwy_ptr=0;

  if( apt_db_ptr == 0 )  return( 0 );

  db_end = frq_tbl.sptr - 1;			/* rdb end addr */

/** clear out runway table data **/
    
  for( i=0; i < RWY_TBL_MAX; i++ ) {
    strncpy( rwy_tbl[i].id, ASCII_BLANKS, 4 );
    rwy_tbl[i].addr = 0;
    rwy_tbl[i].ils_frq = 0;
    rwy_tbl[i].atis_frq[0]=rwy_tbl[i].atis_frq[1]=rwy_tbl[i].atis_frq[2] = 0;
  }

/** Skip the airport record **/

  db_ptr = apt_db_ptr + APT_REC_LENGTH;

/** Skip past the waypoint records **/

  while((db_ptr <= db_end) && ((db_ptr+RDB_TYP1)->l & ( TWP_MASK | EWP_MASK )))
    db_ptr += WPT_REC_LENGTH;
     
/*	Search the runway records	*/
/*  0) List runway is alpha order	*/
/*  1) Longest runway is first		*/
/*  2) Selected runway is first		*/
/*  3-7) Spares.			*/

  switch( list_mode ) {

/*  Mode 0:  Alpha ordered list of runways, starting with element 0.
**           Note that no duplication of runways will occur.
*/
  case 0:
    rwy_count = 0;				/* Start w/ element 0 */
    while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & RWY_MASK ) ) {
      if( ( rwy_count < RWY_TBL_MAX ) && ( strncmp( apt_ident, (db_ptr+RWY_AID)->c, 4 ) == 0 ) ) {
        if( strncmp( rwy_ident, (db_ptr+RDB_IDNT)->c, 4 ) == 0 )  rwy_ptr = db_ptr;
        strncpy( rwy_tbl[rwy_count  ].id, (db_ptr+RDB_IDNT)->c, 4 );
        rwy_tbl[rwy_count++].addr = db_ptr;
      }
      db_ptr += RWY_REC_LENGTH;
    }
    break;

/*  Mode 1:  The first element (element 0) contains the longest runway.
**           the runway list is placed in elements 1 through n.
**           Note the longest runway will occur twice in the overall list.
*/
  case 1:				/* longest runway first */
    rwy_count = 1;			/* don't start w/ first element */
    while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & RWY_MASK ) ) {
      if( ( rwy_count < RWY_TBL_MAX ) && ( strncmp( apt_ident, (db_ptr+RWY_AID)->c, 4 ) == 0 ) ) {
        if( strncmp( rwy_ident, (db_ptr+RDB_IDNT)->c, 4 ) == 0 )  rwy_ptr = db_ptr;
        if( (rwy_len=(db_ptr+RWY_LEN)->s[RWY_LEN_S]) > rwy_longest ) {
          strncpy( rwy_tbl[0].id, (db_ptr+RDB_IDNT)->c, 4 );
          rwy_tbl[0].addr = db_ptr;
          rwy_longest = rwy_len;
        }
      }
      strncpy( rwy_tbl[rwy_count].id, (db_ptr+RDB_IDNT)->c, 4 );
      rwy_tbl[rwy_count++].addr = db_ptr;
      db_ptr += RWY_REC_LENGTH;
    }
    break;

/*  Mode 2:  The first element (element 0) contains the selected runway.
**           the runway list is placed in elements 1 through n.
**           Note the selected runway will occur twice in the overall list.
*/
  case 2:				/* Selected runway first */
    rwy_count = 1;
    while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & RWY_MASK ) ) {
      if( ( rwy_count < RWY_TBL_MAX ) && ( strncmp( apt_ident, (db_ptr+RWY_AID)->c, 4 ) == 0 ) ) {
        if( strncmp( rwy_ident, (db_ptr+RDB_IDNT)->c, 4 ) == 0 )  {
          rwy_ptr = db_ptr;
          rwy_tbl[0].addr = db_ptr;
          strncpy( rwy_tbl[0].id, (rwy_ptr+RDB_IDNT)->c, 4 );
        }
        strncpy( rwy_tbl[rwy_count  ].id, (db_ptr+RDB_IDNT)->c, 4 );
        rwy_tbl[rwy_count++].addr = db_ptr;
      }
      db_ptr += RWY_REC_LENGTH;
    }
    break;

/*  Mode 3:  Runway list with the selected runway as the first element.
**	     There will be no duplicates in the overall list.
*/
  case 3:				/* For future use. */
    rwy_count = 1;			/* Start w/ first element */
    while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & RWY_MASK ) ) {
      if( ( rwy_count < RWY_TBL_MAX ) && ( strncmp( apt_ident, (db_ptr+RWY_AID)->c, 4 ) == 0 ) ) {
        if( strncmp( rwy_ident, (db_ptr+RDB_IDNT)->c, 4 ) == 0 ) {
          rwy_ptr = db_ptr;
          rwy_tbl[0].addr = db_ptr;
          strncpy( rwy_tbl[0].id, (db_ptr+RDB_IDNT)->c, 4 );
        }
        else {
          strncpy( rwy_tbl[rwy_count  ].id, (db_ptr+RDB_IDNT)->c, 4 );
          rwy_tbl[rwy_count++].addr = db_ptr;
        }
      }
      db_ptr += RWY_REC_LENGTH;
    }
    break;

/*  Default:  Same as mode 0 for now.
*/
  default:				/* For future use. */
    rwy_count = 0;			/* Start w/ first element */
    while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & RWY_MASK ) ) {
      if( ( rwy_count < RWY_TBL_MAX ) && ( strncmp( apt_ident, (db_ptr+RWY_AID)->c, 4 ) == 0 ) ) {
        if( strncmp( rwy_ident, (db_ptr+RDB_IDNT)->c, 4 ) == 0 )  rwy_ptr = db_ptr;
        strncpy( rwy_tbl[rwy_count  ].id, (db_ptr+RDB_IDNT)->c, 4 );
        rwy_tbl[rwy_count++].addr = db_ptr;
      }
      db_ptr += RWY_REC_LENGTH;
    }
    break;
  }					/* end switch( rwy_list ) */

/*  The ILS frequency for a given runway, and the ATIS frequencies are
**  created and passed to the host as part of the runway data.
*/
/*  Search the ILS records for the matching ILS frequency  */
/*  Check for ILS ident match or RWY ident match */

  while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & ILS_MASK ) ) {
    if( rwy_ptr != 0 ) {
      if( strncmp( (db_ptr+RDB_IDNT)->c, (rwy_ptr+RWY_LID)->c, 4 ) == 0 )
          rwy_tbl[0].ils_frq = (db_ptr+ILS_FRQ)->s[ILS_FRQ_S];
    }
    db_ptr += ILS_REC_LENGTH;
  }

/*  Skip the MLS records  */

  while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & MLS_MASK ) )
    db_ptr += MLS_REC_LENGTH;

/*  Search the COMM records for ATIS frequencies */

  atis_cnt = 0;
  while( ( db_ptr <= db_end ) && ( (db_ptr+RDB_TYP1)->l & COM_MASK ) ) {
    if( ( atis_cnt < MAX_ATIS ) &&
        ( (db_ptr+RDB_TYP2)->l & ATIS_MASK ) ) {
      rwy_tbl[0].atis_frq[atis_cnt++] = (long)((unsigned)(db_ptr+COM_FRQ)->s[COM_FRQ_S]);
    }
    db_ptr += COM_REC_LENGTH;
  }

  return ( rwy_ptr );
}

/*****************************************************************
**                                                              **
**  apt_unpack.c -                                              **
**    unpacks the airport station data for                      **
**    use in the host.                                          **
**                                                              **
**  inputs:  db_ptr - pointer to database element               **
**                    of the matching airport.                  **
**           apt_num - airport/runway index number              **
**                     (for future expansion)                   **
**           rwy_tbl - structure of runway idents               **
**                                                              **
**  output:  host_out.apt -structure of airport data            **
**                                                              **
**  global:  radio database (rdb)                               **
**                                                              **
*****************************************************************/

static void apt_unpack( register RDB *db_ptr, long num,
                        struct rwy_tbl_struct rwy_tbl[], APT_STRUCT *apt )
{
  int i;

  if( db_ptr ) {
    strncpy( apt->idnt, (db_ptr++)->c, 4 );
    strncpy( apt->icao, (db_ptr++)->c, 4 );
    apt->typ1 = (db_ptr++)->l;
    apt->typ2 = (db_ptr++)->l;
    apt->lat  = DEGR32( (db_ptr++)->l );
    apt->lon  = DEGR32( (db_ptr++)->l );
    apt->var  = (float)((double)(db_ptr  )->s[APT_VAR_S]*0.1);
    apt->elv  = (db_ptr++)->s[APT_ELV_S];
    strncpy( apt->ata, (db_ptr++)->c, 4 );
    ((short*)db_ptr)++;                             /* skip 16 bitword */
    strncpy( apt->name, (db_ptr++)->c, 30 );

    for( i=0; i<20; i++ )
      strncpy( apt->rwy[i], rwy_tbl[i].id, 4 );
  }
  return;
}

/*****************************************************************
**                                                              **
**  rwy_unpack.c -                                              **
**    unpacks the airport station data for                      **
**    use in the host.                                          **
**                                                              **
**  inputs:  db_ptr - pointer to rdb runway data                **
**           num- runway number(always 0)                       **
**                (for future expansion)                        **
**           rwy_tbl - runway tables                            **
**                                                              **
**  output:  host_out.rwy -structure of runway data.            **
**                                                              **
**  global:  rdb - radio database                               **
**                                                              **
*****************************************************************/

static void rwy_unpack( register RDB *db_ptr, long num,
    struct rwy_tbl_struct rwy_tbl[], RWY_STRUCT *rwy )
{
  int i;
  if( db_ptr ) {
    strncpy( rwy->idnt, (db_ptr++)->c, 4 );
    strncpy( rwy->icao, (db_ptr++)->c, 4 );
    rwy->typ1 = (db_ptr++)->l;
    rwy->typ2 = (db_ptr++)->l;
    rwy->lat  = DEGR32( (db_ptr++)->l );
    rwy->lon  = DEGR32( (db_ptr++)->l );
    rwy->hdg  = (float)((double)(db_ptr  )->s[RWY_HDG_S]*0.1);
    rwy->len  = (db_ptr++)->s[RWY_LEN_S];
    rwy->dtr  = (db_ptr  )->s[RWY_DTH_S];
    rwy->wid  = (db_ptr++)->s[RWY_WID_S];
    strncpy( rwy->lid, (db_ptr++)->c, 4 );
    strncpy( rwy->aid, (db_ptr++)->c, 4 );
    rwy->ifrq = rwy_tbl[0].ils_frq;
    for( i=0; i<MAX_ATIS; i++ )
      rwy->afrq[i] = rwy_tbl[0].atis_frq[i];
  }
  else {
    strncpy( rwy->idnt, ASCII_BLANKS, 4 );
    strncpy( rwy->icao, ASCII_BLANKS, 4 );
    rwy->typ1 = rwy->typ2 = 0;
    rwy->lat  = rwy->lon  = 0;
    rwy->hdg  = rwy->hdg  = 0;
    rwy->dtr  = rwy->wid  = 0;
    strncpy( rwy->lid, ASCII_BLANKS, 4 );
    strncpy( rwy->aid, ASCII_BLANKS, 4 );
    rwy->ifrq = 0;
    for( i=0; i<MAX_ATIS; i++ )  rwy->afrq[i] = 0;
  }
  return;
}

/***** end of nav_apt_srch.c  *****/
