-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

-------------------------------------------------------------------------------
-- A buffer in which old items are deleted to make way for new items only at
-- designated cutting points ("perforations")
-------------------------------------------------------------------------------
generic
   type Element is private;
   type Index is range <>;
   type Data is array (Index range <>) of Element;
package Perforated_Circular_Buffer is

   Buffer_Full      : exception;
   Invalid_Location : exception;
   Buffer_Empty     : exception;

   type Instance (Max_Size : Index) is limited private;

   -- Private types for direct access to the buffer
   type Write_Location is private;
   type Read_Location is private;

   -- The amount of buffer overhead required for each perforated block of data in the
   -- buffer.
   Data_Block_Overhead : constant Index;

   -------------------------------------------------------------------------------
   -- Write data to the buffer. If the form with Location is used, the location
   -- of the written data in the buffer is returned. This may be used in a future
   -- call to Rewrite. If there isn't enough room in the buffer, data will be
   -- removed from the head up to the next perforation. If enough space can't be
   -- freed up that way, Buffer_Full is raised.
   -------------------------------------------------------------------------------
   procedure Write
     (Buffer   : in out Instance;
      New_data : in     Data;
      Location :    out Write_Location
     );
   procedure Write
     (Buffer   : in out Instance;
      New_data : in     Data
     );

   -------------------------------------------------------------------------------
   -- Read data from the buffer. If there isn't enough data, Buffer_Empty is
   -- raised.
   -------------------------------------------------------------------------------
   procedure Read
     (Buffer   : in out Instance;
      New_data :    out Data;
      Location :    out Read_Location
     );
   procedure Read
     (Buffer   : in out Instance;
      New_data :    out  Data
     );

   -------------------------------------------------------------------------------
   -- Return the current Read Location of the buffer. This may be useful for a
   -- future Rewind operation. The location is rendered invalid by the next Write
   -- call.
   -------------------------------------------------------------------------------
   function Location (Buffer : in Instance) return Read_Location;

   -------------------------------------------------------------------------------
   -- Return the current Write Location of the buffer. This may be useful for a
   -- future Rewrite operation. The location is rendered invalid by the next Read
   -- call.
   -------------------------------------------------------------------------------
   function Location (Buffer : in Instance) return Write_Location;

   -------------------------------------------------------------------------------
   -- Place a "perforation" at the tail of the buffer. Old data may only be removed
   -- from the buffer at the head of a perforation.
   -------------------------------------------------------------------------------
   procedure Perforate (Buffer : in out Instance);

   -------------------------------------------------------------------------------
   -- Rewrite the data in the buffer at the given location. If the location does
   -- not hold data (in the given amount), Invalid_Location will be raised.
   -- Further writes will pick up where the last one (not this one) ended.
   --
   -- Note that the checking on this routine may not be foolproof. Any Location
   -- that was saved before a later Read call should be considered invalid.
   -------------------------------------------------------------------------------
   procedure Rewrite
     (Buffer   : in out Instance;
      New_data : in     Data;
      Location : in     Write_Location
     );

   -------------------------------------------------------------------------------
   -- Return the amount of elements currently stored in the buffer
   -------------------------------------------------------------------------------
   function Size (Buffer : in Instance) return Natural;

   -------------------------------------------------------------------------------
   -- Return the amount of elements that the buffer has room for (assuming one
   -- write call).
   -------------------------------------------------------------------------------
   function Available (Buffer : in Instance) return Integer;

   -------------------------------------------------------------------------------
   -- Flush all data from the buffer. Should also be used to initialize a freshly
   -- created buffer object.
   -------------------------------------------------------------------------------
   procedure Flush (Buffer : in out Instance);

   -------------------------------------------------------------------------------
   -- Rewind the input buffer to the given index. If the location has been reused
   -- to hold new data, Invalid_Location will be raised.
   --
   -- Note that the checking on this routine may not be foolproof. Any Location
   -- that was saved before a later Write call should be considered invalid.
   -------------------------------------------------------------------------------
   procedure Rewind
     (Buffer   : in out Instance;
      Location : in     Read_Location
     );

   -------------------------------------------------------------------------------
   -- Hold the calling task until there is data in the queue.
   -------------------------------------------------------------------------------
   procedure Wait_For_Data (Buffer : in out Instance);


   -------------------------------------------------------------------------------
   -- Hold the calling task until there is at least the given amount of space in
   -- the queue.
   -------------------------------------------------------------------------------
   procedure Wait_For_Space
     (Buffer : in out Instance;
      Amount : in     Natural := 1
     );

   -------------------------------------------------------------------------------
   -- Dump the given instance to the log.
   -------------------------------------------------------------------------------
   procedure Dump (Subject : in Instance);


private
   -------------------------------------------------------------------------------
   -- Protected type to provide waiting semantics
   -------------------------------------------------------------------------------
   protected type Size_Waiter (Max_Size : Index) is

      ----------------------------------------------------------------------------
      -- Set the size to the given value
      ----------------------------------------------------------------------------
      procedure Set_Size (New_Size     : in Natural;
                          New_Overhead : in Natural);

      ----------------------------------------------------------------------------
      -- Wait for the given amount of data
      ----------------------------------------------------------------------------
      entry Wait_For_Size;

      ----------------------------------------------------------------------------
      -- Wait for the given amount of data
      ----------------------------------------------------------------------------
      entry Wait_For (Size : Natural);

      ----------------------------------------------------------------------------
      -- Wait for the given amount of data
      ----------------------------------------------------------------------------
      entry Wait_For_Positive_Size;

   private
      Size         : Natural := 0;
      Overhead     : Natural := 0;
      Space_Needed : Index;
   end Size_Waiter;

   -- Different types of data that can be in the buffer
   type Data_Contents is (Perforation, Normal_Data);

   -- The header that goes on all data placed in the buffer
   type Data_Header is record
      Contents : Data_Contents;
      Size     : Natural;
   end record;

   -- The amount of elements needed to store one Data_Header
   Header_Element_Size : constant Index :=
      (Data_Header'Size + Element'Size - 1) / Element'Size;
   Data_Block_Overhead : constant Index := Header_Element_Size;

   subtype Data_Header_Elements is Data (Index'First..Index'First + Header_Element_Size - 1);

   type Instance (Max_Size : Index) is record
      Head        : Index   := 1;
      Tail        : Index   := 1;
      Last_Header : Index   := 1;
      Size        : Natural := 0;
      Overhead    : Natural := 0;
      Elements    : Data (1..Max_Size);
      Size_Sync   : Size_Waiter (Max_Size);

      -- Rewind fields. These are used to save data when a read is made so that later
      -- rewinds may operate correctly.

      -- Header_Moved is closely tied to Head. Whenever Head changes Header_Moved
      -- should be updated as well. (typically reset to False).
      Header_Moved : Boolean := False;
      Reread_Cache : Data_Header_Elements;

   end record;

   type Write_Location is record
      Location : Index;
      Header   : Index;
   end record;

   type Read_Location is record
      Location     : Index;
      Last_Header  : Index;
      Header_Cache : Data_Header_Elements;
      Overhead     : Natural;
   end record;

end Perforated_Circular_Buffer;
