-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Unchecked_Conversion;

-- Debugging support packages
with Log;
with Ada.Streams;
with Unchecked_Conversion;
with Ada.Exceptions;

-------------------------------------------------------------------------------
-- A buffer in which old items are deleted to make way for new items only at
-- designated cutting points ("perforations")
-------------------------------------------------------------------------------
package body Perforated_Circular_Buffer is

   --
   -- Declarations to allow placing headers in the buffer
   --

   function To_Header_Elements is new Unchecked_Conversion
     (Source => Data_Header,
      Target => Data_Header_Elements
     );

   function To_Data_Header is new Unchecked_Conversion
     (Source => Data_Header_Elements,
      Target => Data_Header
     );

   Null_Data_Header   : constant Data_Header_Elements :=
     To_Header_Elements (Data_Header'(Contents => Normal_Data, Size => 0));
   Perforation_Header : constant Data_Header_Elements :=
     To_Header_Elements (Data_Header'(Contents => Perforation, Size => 0));


   --
   -- Internal Routines
   --

   -------------------------------------------------------------------------------
   -- Dump the given instance to standard output.
   -------------------------------------------------------------------------------
   procedure Dump (Subject : in Instance) is
      Ind : Index := Subject.Head;
      Count : Natural := 0;

      function To_Stream_Element is new Unchecked_Conversion
        (Source => Element, Target => Ada.Streams.Stream_Element);
   begin
      Log.Report ("Head =>" & Index'Image(Subject.Head) &
                  " Tail =>" & Index'Image(Subject.Tail) &
                  " Last_Header =>" & Index'Image(Subject.Last_Header) &
                  " Size =>" & Natural'Image(Subject.Size) &
                  " Overhead =>" & Natural'Image(Subject.Overhead) &
                  " Header_Moved =>" & Boolean'Image(Subject.Header_Moved));
      if Subject.Header_Moved then
         Log.Report (" Reread_Cache =>");
         for Index in Subject.Reread_Cache'range loop
            Log.Report (Ada.Streams.Stream_Element'Image
                        (To_Stream_Element(Subject.Reread_Cache(Index))) & ' ');
         end loop;
      end if;

      Log.Report (" Elements =>");

      while Count < Subject.Size loop
         Log.Report (Ada.Streams.Stream_Element'Image (To_Stream_Element(Subject.Elements(Ind))) & ' ');

         Count := Count + 1;
         if Ind < Subject.Elements'Last then
            Ind := Ind + 1;
         else
            Ind := Subject.Elements'First;
         end if;
      end loop;
   end Dump;

   -------------------------------------------------------------------------------
   -- Move the index forward by the desired amount. The index value is wrapped
   -- back to the front if it goes past the end of the array.
   -------------------------------------------------------------------------------
   function Increment
     (Value  : in Index;
      Amount : in Integer;
      Buffer : in Instance
     ) return Index is
   begin
      return Index
        (
          ( (Integer(Value) + Amount - Integer(Buffer.Elements'First)) mod
            Integer(Buffer.Elements'Last)
          )
          + Integer(Buffer.Elements'First)
        );
   end Increment;

   -------------------------------------------------------------------------------
   -- Write the requested amount of data to the buffer at the given location.
   -- The rest of the contents of the buffer remain unchanged.
   -------------------------------------------------------------------------------
   procedure Write_Data
      (New_Data : in     Data;
       Location : in     Index;
       Buffer   :    out Data
      ) is

      Wrap_Offset : Natural;
   begin

      if New_Data'Length + Location > Buffer'Last then

         -- Find the offset into new_data where the Buffer's data wraps
         Wrap_Offset := Natural(Buffer'Last - Location);

         -- Copy over the first part of the data
         Buffer (Location..Buffer'Last) :=
           New_Data (New_Data'First .. Index(Integer(New_Data'First) + Wrap_Offset));

         -- Copy over the later (after the wrap) part of the data
         Buffer (1..Index(New_Data'Length - (Wrap_Offset + 1))) :=
           New_Data (Index( Integer(New_Data'First) + Wrap_Offset + 1) ..
                     Index( Integer(New_Data'First) + Integer(New_Data'Length) - 1));


      else
         Buffer (Location .. Location + New_Data'Length - 1) := New_Data;
      end if;

   end Write_Data;

   -------------------------------------------------------------------------------
   -- Read the requested amount of data from the buffer at the given location.
   -- The contents of the buffer remain unchanged.
   -------------------------------------------------------------------------------
   procedure Read_Data
      (New_Data :    out Data;
       Location : in     Index;
       Buffer   : in     Data
      ) is

      Wrap_Offset : Integer;
   begin

      if New_Data'Length + Location > Buffer'Last then

         -- Find the offset into new_data where the Buffer's data wraps
         Wrap_Offset := Integer(Buffer'Last) - Integer(Location);

         -- Copy over the first part of the data
         New_Data (New_Data'First .. Index(Integer(New_Data'First) + Wrap_Offset)) :=
           Buffer (Location..Buffer'Last);

         -- Copy over the later (after the wrap) part of the data
         New_Data (Index( Integer(New_Data'First) + Wrap_Offset + 1) ..
                   New_Data'Last) :=
           Buffer (1..Index (New_Data'Length - (Wrap_Offset + 1)));

      else
         New_Data := Buffer (Location .. Location + New_Data'Length - 1);
      end if;

   end Read_Data;

   -------------------------------------------------------------------------------
   -- Attempt to free up at least the given amount of space in the buffer. Space
   -- may only be freed at a perforation. Buffer_Full will be raised if it can't
   -- be done.
   -------------------------------------------------------------------------------
   procedure Tear_Off_Data
     (Buffer   : in out Instance;
      Required : in     Natural
     ) is

      Head              : Index   := Buffer.Head;
      Size              : Natural := Buffer.Size;
      Overhead          : Natural := Buffer.Overhead;
      Proposed_Head     : Index   := Head;
      Proposed_Size     : Natural := Size;
      Proposed_Overhead : Natural := Overhead;

      Header : Data_Header_Elements;
   begin

      while
         Proposed_Size + Required > Integer(Buffer.Max_Size) and
         Size > 0
      loop

         -- Read in the header
         Read_Data
           (New_Data => Header,
            Location => Head,
            Buffer   => Buffer.Elements
           );

         -- Locate the next Header
         Head := Increment
                   (Value  => Head,
                    Amount => Header'Length + To_Data_Header(Header).Size,
                    Buffer => Buffer
                   );
         Size := Size - Header'Length - To_Data_Header(Header).Size;
         Overhead := Overhead - Header'Length;

         -- If the header was a perforation, propose cutting here.
         if To_Data_Header(Header).Contents = Perforation then
            Proposed_Head     := Head;
            Proposed_Size     := Size;
            Proposed_Overhead := Overhead;
         end if;

      end loop;

      -- If we did find a perforation that would give enough space, tear the head off of the
      -- buffer after that point.
      if Proposed_Size + Required > Integer(Buffer.Max_Size) then
         --         raise Buffer_Full;
         Ada.Exceptions.Raise_Exception
           ( Buffer_Full'Identity,
             "Perforated_Circular_Buffer.Tear_Off_Data Proposed_Size " &
             Natural'Image ( Proposed_Size ) &
             " bytes + Required " &
             Natural'Image ( Required ) &
             " bytes > Buffer.Max_Size " &
             Integer'Image ( Integer(Buffer.Max_Size) ) &
             " bytes" );
      end if;

      Buffer.Head     := Proposed_Head;
      Buffer.Size     := Proposed_Size;
      Buffer.Overhead := Proposed_Overhead;

      Buffer.Header_Moved := False;

   end Tear_Off_Data;



   --
   -- External Routines
   --

   -------------------------------------------------------------------------------
   -- Write data to the buffer. If the form with Location is used, the location
   -- of the written data in the buffer is returned. This may be used in a future
   -- call to Rewrite. If there isn't enough room in the buffer, data will be
   -- removed from the head up to the next perforation. If enough space can't be
   -- freed up that way, Buffer_Full is raised.
   -------------------------------------------------------------------------------
   procedure Write
     (Buffer   : in out Instance;
      New_data : in     Data;
      Location :    out Write_Location
     )
   is
      Header : Data_Header_Elements;

   begin -- Write

      -- Check to see if there is enough space in the buffer to hold it all
      if Buffer.Size + New_Data'Length > Integer(Buffer.Max_Size) then

         Tear_Off_Data
           (Buffer   => Buffer,
            Required => New_Data'Length
           );

      end if;

      -- Update the data header with the new size of the data block
      Read_Data
        (New_Data => Header,
         Location => Buffer.Last_Header,
         Buffer   => Buffer.Elements
         );

      Write_Data
        (New_Data => To_Header_Elements
         (Data_Header'(Contents => Normal_Data,
                       Size     => To_Data_Header(Header).Size + New_Data'Length)),
         Location => Buffer.Last_Header,
         Buffer   => Buffer.Elements
        );

      -- Copy the data into the buffer
      Write_Data
        (New_Data => New_Data,
         Location => Buffer.Tail,
         Buffer   => Buffer.Elements
        );

      -- Save the location
      Location := (Location => Buffer.Tail,
                   Header   => Buffer.Last_Header);

      -- Move the tail
      Buffer.Tail := Increment
                       (Value  => Buffer.Tail,
                        Amount => New_Data'Length,
                        Buffer => Buffer
                       );

      -- Change the size
      Buffer.Size     := Buffer.Size + New_Data'Length;
      Buffer.Size_Sync.Set_Size
        (New_Size     => Buffer.Size,
         New_Overhead => Buffer.Overhead
        );

   end Write;

   procedure Write
     (Buffer   : in out Instance;
      New_data : in     Data
     )
   is
      Trash : Write_Location;
   begin -- Write
      Write
        (Buffer   => Buffer,
         New_data => New_Data,
         Location => Trash
        );
   end Write;


   -------------------------------------------------------------------------------
   -- Read data from the buffer. If there isn't enough data, Buffer_Empty is
   -- raised.
   -------------------------------------------------------------------------------
   procedure Read
     (Buffer   : in out Instance;
      New_data :    out Data;
      Location :    out Read_Location
     )
   is
      Header_Data : Data_Header_Elements;
      Header      : Data_Header;

      Data_Length : constant Natural := Header_Data'Length + New_Data'Length;

      Intermediate_Index : Index := New_Data'First;
      Header_Index       : Index;

   begin -- Read

      -- Save off the location data
      Location.Location := Buffer.Head;
      Location.Last_Header := Buffer.Last_Header;
      Location.Overhead := Buffer.Overhead;
      Read_Data
        (New_Data => Location.Header_Cache,
         Location => Buffer.Head,
         Buffer   => Buffer.Elements
         );


      while Intermediate_Index <= New_Data'Last loop

         -- Find the next data header
         loop
            if Buffer.Size = 0 then
               raise Buffer_Empty;
            end if;

            -- Read in the header
            Read_Data
              (New_Data => Header_Data,
               Location => Buffer.Head,
               Buffer   => Buffer.Elements
              );
            Header := To_Data_Header(Header_Data);

            -- Move the buffer head
            Header_Index := Buffer.Head;
            Buffer.Head := Increment
                             (Value  => Buffer.Head,
                              Amount => Header_Data'Length,
                              Buffer => Buffer);
            -- Change the size
            Buffer.Size     := Buffer.Size - Header_Data'Length;
            Buffer.Overhead := Buffer.Overhead - Header_Data'Length;

            exit when Header.Contents = Normal_Data;
         end loop;

         -- Read in the data
         if Header.Size >= Natural(New_Data'Last + 1 - Intermediate_Index) then
            -- There was enough data

            -- Read in the data
            Read_Data
              (New_Data => New_Data (Intermediate_Index..New_Data'Last),
               Location => Buffer.Head,
               Buffer   => Buffer.Elements
              );

            -- Move the data header to the front of the valid data. Make sure to
            -- save the old data that was in that location, and restore the old
            -- data (if any) that was in the header's previous location.

            -- Restore any old data that was in the header's current location
            if Buffer.Header_Moved then
               Write_Data
                 (New_Data => Buffer.Reread_Cache,
                  Location => Header_Index,
                  Buffer   => Buffer.Elements
                  );
            end if;

            -- Move the buffer's head
            Buffer.Head := Increment
                             (Value  => Header_Index,
                              Amount => Natural(New_Data'Last + 1 - Intermediate_Index),
                              Buffer => Buffer);
            Buffer.Size := Buffer.Size + Header_Data'Length -
              (Natural(New_Data'Last + 1 - Intermediate_Index) );
            Buffer.Overhead := Buffer.Overhead + Header_Data'Length;

            -- Save the data that is currently under the head
            Read_Data
              (New_Data => Buffer.Reread_Cache,
               Location => Buffer.Head,
               Buffer   => Buffer.Elements
               );

            -- Write the header to the buffer
            Write_Data
              (New_Data => To_Header_Elements
                             ((Contents => Normal_Data,
                               Size     => (Header.Size - Natural(New_Data'Last + 1 -
                                            Intermediate_Index))

                             )),
               Location => Buffer.Head,
               Buffer   => Buffer.Elements
               );
            Buffer.Header_Moved := True;

            -- If this was the last header in the buffer, move its marker.
            if Header_Index = Buffer.Last_Header then
               Buffer.Last_Header := Buffer.Head;
            end if;

            Intermediate_Index := New_Data'Last + 1;

         else

            -- Read the data into the users' buffer
            Read_Data
              (New_Data => New_Data(Intermediate_Index .. Intermediate_Index +
                                    Index(Header.Size) - 1
                                   ),
               Location => Buffer.Head,
               Buffer   => Buffer.Elements
              );

            if Buffer.Header_Moved then
               Write_Data
                 (New_Data => Buffer.Reread_Cache,
                  Location => Header_Index,
                  Buffer   => Buffer.Elements
                  );
            end if;

            -- Move the buffer head
            Buffer.Head := Increment
                             (Value  => Buffer.Head,
                              Amount => Header.Size,
                              Buffer => Buffer);
            Buffer.Header_Moved := False;

            -- Change the size
            Buffer.Size := Buffer.Size - Header.Size;

            -- Move the target buffer index
            Intermediate_Index := Intermediate_Index + Index(Header.Size);

         end if;

      end loop;

      Buffer.Size_Sync.Set_Size
        (New_Size     => Buffer.Size,
         New_Overhead => Buffer.Overhead
         );


   end Read;

   procedure Read
     (Buffer   : in out Instance;
      New_data :    out  Data
     ) is
      Dont_Care : Read_Location;
   begin

      Read
        (Buffer   => Buffer,
         New_Data => New_Data,
         Location => Dont_Care
        );

   end Read;

   -------------------------------------------------------------------------------
   -- Place a "perforation" at the tail of the buffer. Old data may only be removed
   -- from the buffer at the head of a perforation.
   -------------------------------------------------------------------------------
   procedure Perforate (Buffer : in out Instance)
   is
   begin -- Perforate

      -- Make sure there is enough space in the buffer to hold the perforation and a
      -- new data header
      if Buffer.Size + (Integer(Header_Element_Size) * 2) > Integer(Buffer.Max_Size) then

         Tear_Off_Data
           (Buffer   => Buffer,
            Required => Integer(Header_Element_Size) * 2
           );

      end if;

      -- Put the perforation into the buffer
      Write_Data
        (New_Data => Perforation_Header,
         Location => Buffer.Tail,
         Buffer   => Buffer.Elements
         );

      -- Move the tail
      Buffer.Tail := Increment
        (Value  => Buffer.Tail,
         Amount => Perforation_Header'Length,
         Buffer => Buffer
         );

      -- Put a null header into the buffer
      Write_Data
        (New_Data => Null_Data_Header,
         Location => Buffer.Tail,
         Buffer   => Buffer.Elements
         );
      Buffer.Last_Header := Buffer.Tail;

      -- Move the tail
      Buffer.Tail := Increment
        (Value  => Buffer.Tail,
         Amount => Null_Data_Header'Length,
         Buffer => Buffer
         );

      -- Change the size
      Buffer.Size     := Buffer.Size + (Integer(Header_Element_Size) * 2);
      Buffer.Overhead := Buffer.Overhead + (Integer(Header_Element_Size) * 2);
      Buffer.Size_Sync.Set_Size
        (New_Size     => Buffer.Size,
         New_Overhead => Buffer.Overhead
         );

   end Perforate;


   -------------------------------------------------------------------------------
   -- Rewrite the data in the buffer at the given location. If the location does
   -- not hold data (in the given amount), Invalid_Location will be raised.
   -------------------------------------------------------------------------------
   procedure Rewrite
     (Buffer   : in out Instance;
      New_data : in     Data;
      Location : in     Write_Location
     )
   is
     Header : Data_Header_Elements;
     Size_To_Location : Integer;
   begin -- Rewrite

      -- Verify that the given location is within the buffer
      Size_To_Location := Integer(Location.Location) - Integer (Buffer.Head);
      if Size_To_Location < 0 then
         Size_To_Location := Size_To_Location + Buffer.Elements'Length;
      end if;

      if Size_To_Location + New_Data'Length > Buffer.Size then
         raise Invalid_Location;
      end if;

      -- Verify that the new data fits in the old data
      Read_Data
        (New_Data => Header,
         Location => Index(Location.Header),
         Buffer   => Buffer.Elements
        );

      if To_Data_Header(Header).Size <
        Integer(Increment (Value  => Location.Location,
                   Amount => -Integer(Location.Header),
                   Buffer => Buffer)) +
        New_Data'Length - Header'Length
      then
         raise Invalid_Location;
      end if;

      -- Write the data
      Write_Data
        (New_Data => New_Data,
         Location => Location.Location,
         Buffer   => Buffer.Elements
        );

   end Rewrite;


   -------------------------------------------------------------------------------
   -- Return the amount of elements currently stored in the buffer
   -------------------------------------------------------------------------------
   function Size (Buffer : in Instance) return Natural
   is
   begin -- Size
      return Buffer.Size - Buffer.Overhead;
   end Size;


   -------------------------------------------------------------------------------
   -- Return the amount of elements that the buffer has room for (assuming one
   -- write call).
   -------------------------------------------------------------------------------
   function Available (Buffer : in Instance) return Integer
   is
   begin -- Size
      return Integer(Buffer.Max_Size) - Buffer.Size;
   end Available;


   -------------------------------------------------------------------------------
   -- Flush all data from the buffer.
   -------------------------------------------------------------------------------
   procedure Flush (Buffer : in out Instance)
   is
   begin -- Flush
      Buffer.Size        := Data_Header_Elements'Length;
      Buffer.Overhead    := Data_Header_Elements'Length;
      Buffer.Head        := Buffer.Elements'First;
      Buffer.Tail        := Buffer.Elements'First + Data_Header_Elements'Length;
      Buffer.Last_Header := Buffer.Elements'First;
      Buffer.Size_Sync.Set_Size
        (New_Size     => Buffer.Size,
         New_Overhead => Buffer.Overhead
         );
      Buffer.Header_Moved := False;

      -- Put an empty header at the start of the buffer
      Buffer.Elements(Buffer.Elements'First .. Buffer.Elements'First +
                      Header_Element_Size - 1) := Null_Data_Header;

   end Flush;

   -------------------------------------------------------------------------------
   -- Rewind the input buffer to the given index. If the location has been reused
   -- to hold new data, Invalid_Location will be raised.
   -------------------------------------------------------------------------------
   procedure Rewind
     (Buffer   : in out Instance;
      Location : in     Read_Location
     ) is
     Size_From_Location : Integer;
   begin

      -- Verify that the given location is not already within the buffer
      Size_From_Location := Integer (Buffer.Tail) - Integer(Location.Location);
      if Size_From_Location <= 0 then
         Size_From_Location := Size_From_Location + Buffer.Elements'Length;
      end if;

      if Size_From_Location <= Buffer.Size then
         raise Invalid_Location;
      end if;

      -- Replace any saved data under the head header
      if Buffer.Header_Moved then
         Write_Data
           (New_Data => Buffer.Reread_Cache,
            Location => Buffer.Head,
            Buffer   => Buffer.Elements
            );
      end if;

      -- Move the head
      Buffer.Head := Index(Location.Location);

      -- Replace the header at the rewound location (saving off any data there).
      Read_Data
        (New_Data => Buffer.Reread_Cache,
         Location => Buffer.Head,
         Buffer   => Buffer.Elements
         );
      Write_Data
        (New_Data => Location.Header_Cache,
         Location => Buffer.Head,
         Buffer   => Buffer.Elements
         );
      Buffer.Header_Moved := True;

      -- Change the size
      Buffer.Last_Header := Location.Last_Header;
      Buffer.Size        := Size_From_Location;
      Buffer.Overhead    := Location.Overhead;
      Buffer.Size_Sync.Set_Size
        (New_Size     => Buffer.Size,
         New_Overhead => Buffer.Overhead
        );

   end Rewind;

   -------------------------------------------------------------------------------
   -- Return the current Read Location of the buffer. This may be useful for a
   -- future Rewind operation. The location is rendered invalid by the next Write
   -- call.
   -------------------------------------------------------------------------------
   function Location (Buffer : in Instance) return Read_Location is
      Header : Data_Header_Elements;
   begin
      Read_Data
        (New_Data => Header,
         Location => Buffer.Head,
         Buffer   => Buffer.Elements
         );

      return (Location     => Buffer.Head,
              Last_Header  => Buffer.Last_Header,
              Header_Cache => Header,
              Overhead     => Buffer.Overhead);
   end Location;


   -------------------------------------------------------------------------------
   -- Return the current Write Location of the buffer. This may be useful for a
   -- future Rewrite operation. The location is rendered invalid by the next Read
   -- call.
   -------------------------------------------------------------------------------
   function Location (Buffer : in Instance) return Write_Location is
   begin
      return (Location => Buffer.Tail,
              Header   => Buffer.Last_Header);
   end Location;

   -------------------------------------------------------------------------------
   -- Hold the calling task until there is data in the queue.
   -------------------------------------------------------------------------------
   procedure Wait_For_Data (Buffer : in out Instance) is
   begin
      Buffer.Size_Sync.Wait_For_Positive_Size;
   end Wait_For_Data;

   -------------------------------------------------------------------------------
   -- Hold the calling task until there is at least the given amount of space in
   -- the queue.
   -------------------------------------------------------------------------------
   procedure Wait_For_Space
     (Buffer : in out Instance;
      Amount : in     Natural := 1
     ) is
   begin
      Buffer.Size_Sync.Wait_For (Amount);
   end Wait_For_Space;


   -------------------------------------------------------------------------------
   -- Protected type to provide waiting semantics
   -------------------------------------------------------------------------------
   protected body Size_Waiter is

      ----------------------------------------------------------------------------
      -- Set the size to the given value
      ----------------------------------------------------------------------------
      procedure Set_Size (New_Size     : in Natural;
                          New_Overhead : in Natural ) is
      begin
         Size     := New_Size;
         Overhead := New_Overhead;
      end Set_Size;

      ----------------------------------------------------------------------------
      -- Wait for the given amount of data
      ----------------------------------------------------------------------------
      entry Wait_For_Size
      when Max_Size - Index(Size) - Index(Overhead) >= Space_Needed is
      begin
         -- No work is done. This is entry is just for synchronization.
         null;
      end Wait_For_Size;

      ----------------------------------------------------------------------------
      -- Wait for the given amount of data.
      -- This routine just sets the size we are waiting for, and proceeds to wait
      -- on the Wait_For_Size queue.
      ----------------------------------------------------------------------------
      entry Wait_For (Size : Natural) when True is
      begin
         Space_Needed := Index(Size) + Header_Element_Size;
         requeue Wait_For_Size;
      end Wait_For;

      ----------------------------------------------------------------------------
      -- Wait for the given amount of data
      ----------------------------------------------------------------------------
      entry Wait_For_Positive_Size when Integer(Size) - Integer(Overhead) > 0 is
      begin
         -- No work is done. This is entry is just for synchronization.
         null;
      end;

   end Size_Waiter;

end Perforated_Circular_Buffer;
