-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

-- Generic package for ordered lists of real numbers, such as a 3-vector or
-- a quaternion.
with Ada.Numerics.Generic_Elementary_Functions;
package body Ordered_Tuple is

   -- instantiate standard elementary functions for the
   -- float type specified

   package Float_Type_Functions is new
     Ada.Numerics.Generic_Elementary_Functions (Float_Type);

   -- addition and subtraction

   function "+"
     ( A : in Instance;
       B : in Instance )
      return Instance is

      Result : Instance;

   begin

      for I in Index loop
         Result(I) := A(I) + B(I);
      end loop;

      return Result;

   end "+";

   function "-"
     ( A : in Instance;
       B : in Instance )
      return Instance is

      Result : Instance;

   begin

      for I in Index loop
         Result(I) := A(I) - B(I);
      end loop;

      return Result;

   end "-";

   -- dot product [ A1 * B1 + A2 * B2 + ... AN * BN ]

   function "*"
     ( A : in Instance;
       B : in Instance )
      return Float_Type is

      Result : Float_Type := 0.0;

   begin

      for I in Index loop
         Result := Result + A(I) * B(I);
      end loop;

      return Result;

   end "*";

   -- scalar multiplication and division
   -- For multiplication you can have scalar * tuple or tuple * scalar
   -- A tuple can be divided by a scalar but not vice versa

   function "*"
     ( A : in Instance;
       N : in Float_Type )
      return Instance is

      Result : Instance;

   begin

      for I in Index loop
         Result(I) := A(I) * N;
      end loop;

      return Result;

   end "*";

   function "/"
     ( A : in Instance;
       N : in Float_Type )
      return Instance is

      Result : Instance;

   begin

      for I in Index loop
         Result(I) := A(I) / N;
      end loop;

      return Result;

   end "/";

   function "*"
     ( N : in Float_Type;
       A : in Instance )
      return Instance is

      Result : Instance;

   begin

      for I in Index loop
         Result(I) := A(I) * N;
      end loop;

      return Result;

   end "*";


   function Magnitude
     ( A : in Instance )
     return Float_Type is

   begin

      -- magnitude is the square root of the dot product of the vector and
      -- itself

      return Float_Type_Functions.Sqrt ( A * A );

   end Magnitude;


   -- Tuple integration using predictive and corrective methods
   -- Each component is integrated separately.
   -- The functions return the change over the period of the integration
   -- constant.

   -- Second-order Adams Open (predictive) formula
   -- For integrating acceleration to get velocity

   function Integrate_Predictive
     ( An_Integration_Constant : Float_Type;
       This_Pass_Derivative    : Instance;
       Last_Pass_Derivative    : Instance )
     return Instance is

   begin

      return
        An_Integration_Constant * 0.5
        * ( 3.0 * This_Pass_Derivative - Last_Pass_Derivative );

   end Integrate_Predictive;

   -- Second-order Adams Closed (corrective) formula
   -- For integrating velocity to get position

   function Integrate_Corrective
     ( An_Integration_Constant : Float_Type;
       This_Pass_Derivative : Instance;
       Last_Pass_Derivative : Instance )
     return Instance is

   begin

      return
        An_Integration_Constant * 0.5
        * ( This_Pass_Derivative + Last_Pass_Derivative );

   end Integrate_Corrective;

end Ordered_Tuple;



