-------------------------------------------------------------------------------
--
-- Copyright (C) 1999 Christoph Karl Walter Grein
--
-- This file is part of the OpenToken package.
--
-- The OpenToken package is free software; you can redistribute it and/or
-- modify it under the terms of the  GNU General Public License as published
-- by the Free Software Foundation; either version 2, or (at your option)
-- any later version. The OpenToken package is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY; without even the implied
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for  more details.  You should have received
-- a copy of the GNU General Public License  distributed with the OpenToken
-- package;  see file GPL.txt.  If not, write to  the Free Software Foundation,
-- 59 Temple Place - Suite 330,  Boston, MA 02111-1307, USA.
--
-- As a special exception,  if other files  instantiate  generics from this
-- unit, or you link this unit with other files to produce an executable,
-- this unit does not by itself cause the resulting executable to be
-- covered by the GNU General Public License.  This exception does not
-- however invalidate any other reasons why the executable file might be
-- covered by the GNU Public License.
--
-- Maintainer: Christoph K. W. Grein (Christ-Usch.Grein@T-Online.de)
--
-- Update History:
-- $Log: ada_lexer.ads,v $
-- Revision 1.2  1999/10/08 23:19:01  Ted
-- Disable sign recognition in integer and real literals
--
-- Revision 1.1  1999/08/17 03:40:24  Ted
-- Initial Version
--
--
-- 1.0 -  8 August 1999  Final complete version
-- 0.3 - 26 June 1999    Added character literals
-- 0.2 - 25 June 1999    Added based numbers
--                       (still missing: character literals and strings)
-- 0.1 - 23 June 1999    Bug fix (numeric literals)
-- 0.0 - 22 June 1999    First preliminary release
-------------------------------------------------------------------------------

with Token.Analyzer;
with Token.Keyword, Token.Separator;
with Token.Identifier;
with Token.Graphic_Character, Token.String;
with Token.Integer, Token.Based_Integer_Ada_Style,
     Token.Real, Token.Based_Real_Ada_Style;
with Token.Character_Set;
with Token.Line_Comment;
with Token.End_Of_File;

pragma Elaborate_All (Token.Analyzer,
                      Token.Keyword, Token.Separator,
                      Token.Identifier,
                      Token.Graphic_Character, Token.String,
                      Token.Integer, Token.Based_Integer_Ada_Style,
                      Token.Real, Token.Based_Real_Ada_Style,
                      Token.Character_Set,
                      Token.Line_Comment,
                      Token.End_Of_File);

package Ada_Lexer is

  ---------------------------------------------------------------------
  -- This ia a lexical analyser for the Ada language.
  --
  -- There is another lexer for the Ada and Java languages at:
  --   <http://home.T-Online.de/home/Christ-Usch.Grein/Ada/Lexer.html>
  ---------------------------------------------------------------------

  type Ada_Token is
    (-- Reserved words ARM 2.9 (2)
     Abort_T, Abs_T, Abstract_T, Accept_T, Access_T, Aliased_T, All_T, And_T, Array_T, At_T,
     Begin_T, Body_T,
     Case_T, Constant_T,
     Declare_T, Delay_T, Delta_T, Digits_T, Do_T,
     Else_T, Elsif_T, End_T, Entry_T, Exception_T, Exit_T,
     For_T, Function_T,
     Generic_T, Goto_T,
     If_T, In_T, Is_T,
     Limited_T, Loop_T,
     Mod_T,
     New_T, Not_T, Null_T,
     Of_T, Or_T, Others_T, Out_T,
     Package_T, Pragma_T, Private_T, Procedure_T, Protected_T,
     Raise_T, Range_T, Record_T, Rem_T, Renames_T, Requeue_T, Return_T, Reverse_T,
     Select_T, Separate_T, Subtype_T,
     Tagged_T, Task_T, Terminate_T, Then_T, Type_T,
     Until_T, Use_T,
     When_T, While_T, With_T,
     Xor_T,
     -- Delimiters ARM 2.2 (9)
     -- & ' ( ) * + , - . / : ; < = > |
     -- Compound delimiters ARM 2.2 (14)
     -- => .. ** := /= >= <= << >> <>
     Colon_T, Comma_T, Dot_T, Semicolon_T, Tick_T,         -- : , . ; '
     Left_Parenthesis_T, Right_Parenthesis_T,              -- ( )
     Concatenate_T,                                        -- &
     Alternative_T,                                        -- |
     Equal_T, Not_Equal_T, Greater_Than_T, Less_Than_T,    -- = /= > <
     Greater_Equal_T, Less_Equal_T,                        -- >= <=
     Plus_T, Minus_T, Times_T, Divide_T,                   -- + - * /
     Arrow_T, Assignment_T, Double_Dot_T, Exponentiate_T,  -- => := .. **
     Left_Label_Bracket_T, Right_Label_Bracket_T, Box_T,   -- << >> <>
     -- Literals ARM 2.4 .. 2.6
     Integer_T,               -- 1, 1E+10
     Based_Integer_T,         -- 13#C#, 13#C#E+10
     Real_T,                  -- -3.141, 1.0E+10
     Based_Real_T,            -- 13#C.B#, 13#C.B#E+5
     Character_T, String_T,
     -- Other tokens
     Identifier_T,
     Comment_T,
     Whitespace_T,
     -- Syntax error
  -- Bad_Token_T,
     --
     End_of_File_T);

  package Tokenizer is new Token.Analyzer (Ada_Token);

  Syntax : constant Tokenizer.Syntax :=
    (Abort_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("abort")),
     Abs_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("abs")),
     Abstract_T  => new Token.Keyword.Instance'(Token.Keyword.Get ("abstract")),
     Accept_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("accept")),
     Access_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("access")),
     Aliased_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("aliased")),
     All_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("all")),
     And_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("and")),
     Array_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("array")),
     At_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("at")),
     Begin_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("begin")),
     Body_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("body")),
     Case_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("case")),
     Constant_T  => new Token.Keyword.Instance'(Token.Keyword.Get ("constant")),
     Declare_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("declare")),
     Delay_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("delay")),
     Delta_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("delta")),
     Digits_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("digits")),
     Do_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("do")),
     Else_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("else")),
     Elsif_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("elsif")),
     End_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("end")),
     Entry_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("entry")),
     Exception_T => new Token.Keyword.Instance'(Token.Keyword.Get ("exception")),
     Exit_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("exit")),
     For_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("for")),
     Function_T  => new Token.Keyword.Instance'(Token.Keyword.Get ("function")),
     Generic_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("generic")),
     Goto_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("goto")),
     If_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("if")),
     In_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("in")),
     Is_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("is")),
     Limited_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("limited")),
     Loop_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("loop")),
     Mod_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("mod")),
     New_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("new")),
     Not_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("not")),
     Null_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("null")),
     Of_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("of")),
     Or_T        => new Token.Keyword.Instance'(Token.Keyword.Get ("or")),
     Others_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("others")),
     Out_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("out")),
     Package_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("package")),
     Pragma_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("pragma")),
     Private_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("private")),
     Procedure_T => new Token.Keyword.Instance'(Token.Keyword.Get ("procedure")),
     Protected_T => new Token.Keyword.Instance'(Token.Keyword.Get ("protected")),
     Raise_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("raise")),
     Range_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("range")),
     Record_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("record")),
     Rem_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("rem")),
     Renames_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("renames")),
     Requeue_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("requeue")),
     Return_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("return")),
     Reverse_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("reverse")),
     Select_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("select")),
     Separate_T  => new Token.Keyword.Instance'(Token.Keyword.Get ("separate")),
     Subtype_T   => new Token.Keyword.Instance'(Token.Keyword.Get ("subtype")),
     Tagged_T    => new Token.Keyword.Instance'(Token.Keyword.Get ("tagged")),
     Task_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("task")),
     Terminate_T => new Token.Keyword.Instance'(Token.Keyword.Get ("terminate")),
     Then_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("then")),
     Type_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("type")),
     Until_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("until")),
     Use_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("use")),
     When_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("when")),
     While_T     => new Token.Keyword.Instance'(Token.Keyword.Get ("while")),
     With_T      => new Token.Keyword.Instance'(Token.Keyword.Get ("with")),
     Xor_T       => new Token.Keyword.Instance'(Token.Keyword.Get ("xor")),
     Colon_T               => new Token.Separator.Instance'(Token.Separator.Get (":")),
     Comma_T               => new Token.Separator.Instance'(Token.Separator.Get (",")),
     Dot_T                 => new Token.Separator.Instance'(Token.Separator.Get (".")),
     Semicolon_T           => new Token.Separator.Instance'(Token.Separator.Get (";")),
     Tick_T                => new Token.Separator.Instance'(Token.Separator.Get ("'")),
     Left_Parenthesis_T    => new Token.Separator.Instance'(Token.Separator.Get ("(")),
     Right_Parenthesis_T   => new Token.Separator.Instance'(Token.Separator.Get (")")),
     Concatenate_T         => new Token.Separator.Instance'(Token.Separator.Get ("&")),
     Alternative_T         => new Token.Separator.Instance'(Token.Separator.Get ("|")),
     Equal_T               => new Token.Separator.Instance'(Token.Separator.Get ("=")),
     Not_Equal_T           => new Token.Separator.Instance'(Token.Separator.Get ("/=")),
     Greater_Than_T        => new Token.Separator.Instance'(Token.Separator.Get (">")),
     Less_Than_T           => new Token.Separator.Instance'(Token.Separator.Get ("<")),
     Greater_Equal_T       => new Token.Separator.Instance'(Token.Separator.Get (">=")),
     Less_Equal_T          => new Token.Separator.Instance'(Token.Separator.Get ("<=")),
     Plus_T                => new Token.Separator.Instance'(Token.Separator.Get ("+")),
     Minus_T               => new Token.Separator.Instance'(Token.Separator.Get ("-")),
     Times_T               => new Token.Separator.Instance'(Token.Separator.Get ("*")),
     Divide_T              => new Token.Separator.Instance'(Token.Separator.Get ("/")),
     Arrow_T               => new Token.Separator.Instance'(Token.Separator.Get ("=>")),
     Assignment_T          => new Token.Separator.Instance'(Token.Separator.Get (":=")),
     Double_Dot_T          => new Token.Separator.Instance'(Token.Separator.Get ("..")),
     Exponentiate_T        => new Token.Separator.Instance'(Token.Separator.Get ("**")),
     Left_Label_Bracket_T  => new Token.Separator.Instance'(Token.Separator.Get ("<<")),
     Right_Label_Bracket_T => new Token.Separator.Instance'(Token.Separator.Get (">>")),
     Box_T                 => new Token.Separator.Instance'(Token.Separator.Get ("<>")),
     Integer_T       => new Token.Integer.Instance'(Token.Integer.Get (Allow_Signs => False)),
     Based_Integer_T => new Token.Based_Integer_Ada_Style.Instance'(Token.Based_Integer_Ada_Style.Get),
     Real_T          => new Token.Real.Instance'(Token.Real.Get (Allow_Signs => False)),
     Based_Real_T    => new Token.Based_Real_Ada_Style.Instance'(Token.Based_Real_Ada_Style.Get),
     Character_T     => new Token.Graphic_Character.Instance'(Token.Graphic_Character.Get),
     String_T        => new Token.String.Instance'(Token.String.Get),
     Identifier_T    => new Token.Identifier.Instance'(Token.Identifier.Get),
     Comment_T       => new Token.Line_Comment.Instance'(Token.Line_Comment.Get ("--")),
     Whitespace_T    => new Token.Character_Set.Instance'(Token.Character_Set.Get
                                      (Token.Character_Set.Standard_Whitespace)),
     End_of_File_T   => new Token.End_Of_File.Instance'(Token.End_Of_File.Get));

   Analyzer: Tokenizer.Instance := Tokenizer.Initialize (Syntax);

end Ada_Lexer;
