-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Calendar;
with Ada.Exceptions;
with Ada.Text_Io;
with Ada.Unchecked_Deallocation;
with Interfaces.C;
with Interfaces.C.Strings;
with Time_Stamp;

-------------------------------------------------------------------------------
-- This package allows clients to record descriptions of events without having
-- to wait for the output to be performed.
-------------------------------------------------------------------------------
package body Log is

   -- The maximum amount of bytes that may be allocated for Event_Nodes. Attempts
   -- to allocate any more than this will cause Storage_Error.
   Max_Message_Queue_Size : constant := 32_000;


   -- The name of the log file
   Log_File_Name : constant String := "log.txt";
   Log_Directory : constant String := "/Host_Data/";

   Last_Logged : Boolean := True;

   type Event_Node (Message_Length : Natural);

   type Event_Pointer is access Event_Node;

   type Event_Node (Message_Length : Natural) is record
      Message  : String (1..Message_Length);
      Time     : Ada.Calendar.Time;
      Severity : Event_Type;
      Next     : Event_Pointer := null;
   end record;

   Console_Logging : Boolean := True;
   for Console_Logging'Size use 32;
   pragma Export (C, Console_Logging, "console_logging");


   for Event_Pointer'Storage_Size use Max_Message_Queue_Size;

   procedure Dispose is new Ada.Unchecked_Deallocation (Object => Event_Node,
                                                       Name   => Event_Pointer);

   ----------------------------------------------------------------------------
   -- Report an event to the log from a C unit. This is a C wrapper to the Ada
   -- Report routine
   ----------------------------------------------------------------------------
   procedure C_Report
     (Event    : in Interfaces.C.Strings.Chars_Ptr;
      Severity : in Interfaces.C.Int);
   pragma Export (C, C_Report, "JPATS_Log_Report");


   ----------------------------------------------------------------------------
   -- Wait for all pending long messages to be processed. This is a C wrapper
   -- to the Ada Wait routine.
   ----------------------------------------------------------------------------
   procedure C_Wait;
   pragma Export (C, C_Wait, "JPATS_Log_Wait");


   ----------------------------------------------------------------------------
   -- Report an event to the log from a C unit. This is a C wrapper to the Ada
   -- Report routine
   ----------------------------------------------------------------------------
   procedure C_Report
     (Event    : in Interfaces.C.Strings.Chars_Ptr;
      Severity : in Interfaces.C.Int) is
   begin
      Report
        (Event    => Interfaces.C.Strings.Value(Event),
         Severity => Event_Type'Val(Severity)
         );
   end C_Report;

   ----------------------------------------------------------------------------
   -- Wait for all pending long messages to be processed. This is a C wrapper
   -- to the Ada Wait routine.
   ----------------------------------------------------------------------------
   procedure C_Wait is
   begin
      Wait;
   end C_Wait;

   ----------------------------------------------------------------------------
   -- The actual queue Object. Since is can be asynchronously updated, it needs
   -- to be a protected object.
   ----------------------------------------------------------------------------
   protected type Event_Queue is

      -------------------------------------------------------------------------
      -- Put an event on the end of the queue.
      -------------------------------------------------------------------------
      procedure Enqueue (Event : in Event_Pointer);

      -------------------------------------------------------------------------
      -- Report that we couldn't create a new event node for an event.
      -------------------------------------------------------------------------
      procedure Lost_One;

      -------------------------------------------------------------------------
      -- Remove an event from the queue. If no event is available, it will
      -- wait for one. It is the caller's responsibility to deallocate the
      -- memory associated with this pointer.
      -- Num_Lost holds 0 for normal messages, and the number of lost messages
      -- for "message dropped" messages. Those messages are *not* to be
      -- deallocated.
      -------------------------------------------------------------------------
      entry Dequeue (Event    : out Event_Pointer;
                     Num_Lost : out Natural);

      -------------------------------------------------------------------------
      -- Hold the calling task until the event queue is empty.
      -------------------------------------------------------------------------
      entry Is_Empty;

   private
      Head  : Event_Pointer := null;
      Tail  : Event_Pointer;

      -- A pre-allocated event for specifing a number of dropped messages.
      Messages_Dropped_Event : Event_Pointer := new Event_Node'(Message_Length => 0,
                                                                Message        => "",
                                                                Time           => Ada.Calendar.Clock,
                                                                Severity       => Warning,
                                                                Next           => null);
      Lossage_Tally : Integer := 0;
   end Event_Queue;

   ----------------------------------------------------------------------------
   -- The queue invarient is that when the queue is non-empty, Head points to
   -- the first element placed in the queue and Tail points to the last element.
   -- When the queue is empty, Head points to null and Tail is undefined.
   ----------------------------------------------------------------------------
   protected body Event_Queue is

      -------------------------------------------------------------------------
      -- Put an event on the end of the queue.
      -------------------------------------------------------------------------
      procedure Enqueue (Event : in Event_Pointer) is
      begin

         if Head = null then

            -- Handle Queue empty condition
            Head := Event;
         else

            Tail.Next := Event;
         end if;

         Tail := Event;

      end Enqueue;

      -------------------------------------------------------------------------
      -- If the messages dropped event isn't on the queue, it is put on.
      -- The number of dropped messages is incremented.
      -------------------------------------------------------------------------
      procedure Lost_One is
      begin
         if Lossage_Tally = 0 then
            Enqueue (Messages_Dropped_Event);
         end if;

         Lossage_Tally := Lossage_Tally + 1;
      end Lost_One;


      -------------------------------------------------------------------------
      -- Remove an event from the queue. Only accept this entry when an event
      -- is available.
      -------------------------------------------------------------------------
      entry Dequeue (Event    : out Event_Pointer;
                     Num_Lost : out Natural) when Head /= null is
      begin

         Event := Head;

         Head := Head.Next;

         -- If it was a message dropped event, report the number dropped.
         if Event = Messages_Dropped_Event then
            Num_Lost := Lossage_Tally;
            Lossage_Tally := 0;
            Messages_Dropped_Event.Next := null;
         else
            Num_Lost := 0;
         end if;

      end Dequeue;

      -------------------------------------------------------------------------
      -- Hold the calling task until the event queue is empty.
      -- This entry is only accepted when there are no elements to process and
      -- the logging task is waiting for more elements.
      -------------------------------------------------------------------------
      entry Is_Empty when Head = null and Dequeue'Count > 0 is
      begin
         null;
      end;


   end Event_Queue;

   Log_Event_Queue : Event_Queue;

   ----------------------------------------------------------------------------
   -- It will first attempt to append to a log file in the host data directory.
   -- If that can't be done, it will attempt it to a log file in the local
   -- directory. If neither file can be opened, Name_Error will propagate out
   -- of this routine.
   ----------------------------------------------------------------------------
   procedure Open_Log_File (File : in out Ada.Text_Io.File_Type) is
   begin
      -- Attempt to open the log file in the log directory
      begin
         Ada.Text_Io.Open
           (File => File,
            Name => Log_Directory & Log_File_Name,
            Mode => Ada.Text_Io.Append_File
            );

         return;
      exception
         when Ada.Text_IO.Name_Error =>
            null;
      end;

      -- Attempt to create the log file in the log directory
      begin
         Ada.Text_Io.Create
           (File => File,
            Name => Log_Directory & Log_File_Name,
            Mode => Ada.Text_Io.Out_File
            );
         return;
      exception
         when Ada.Text_IO.Name_Error =>
            null;
      end;

      -- Attempt to open the log file on the local directory
      begin
         Ada.Text_Io.Open
           (File => File,
            Name => Log_File_Name,
            Mode => Ada.Text_Io.Append_File
            );

         return;
      exception
         when Ada.Text_IO.Name_Error =>
            null;
      end;

      -- Attempt to create the log file in the local directory
      Ada.Text_Io.Create
        (File => File,
         Name => Log_Directory & Log_File_Name,
         Mode => Ada.Text_Io.Out_File
         );

   end Open_Log_File;

   ----------------------------------------------------------------------------
   -- Attempt to write the given value to a log file and to standard output.
   -- If no log file can be opened, the message will still get printed on
   -- standard output.
   ----------------------------------------------------------------------------
   procedure Write_To_Log_File (Value : in String) is
      Log_File    : Ada.Text_Io.File_Type;

   begin
      if Console_Logging then
         Ada.Text_Io.Put (Value);
         Ada.Text_Io.Flush;
      end if;

      begin
         Open_Log_File (Log_File);

         Ada.Text_Io.Put_Line
           (File => Log_File,
            Item => Value & Ada.Characters.Latin_1.CR
            );

         Ada.Text_Io.Close (Log_File);

         -- The new line is put here to help diagnose NFS problems with the logger.
         if Console_Logging then
            Ada.Text_Io.New_Line;
         end if;

         if not Last_Logged then
               Ada.Text_Io.Put_Line (Time_Stamp & " " & Event_Type'Image(Informational) &
                                     ": Logging to disk resumed.");
               Last_Logged := True;
         end if;

      exception
         when Ada.Text_IO.Name_Error =>
            Ada.Text_Io.New_Line;

            if Last_Logged then
               Ada.Text_Io.Put_Line (Time_Stamp & " " & Event_Type'Image(Warning) &
                                     ": Unable to log message(s) to disk.");
               Last_Logged := False;
            end if;
      end;

   end Write_To_Log_File;


   ----------------------------------------------------------------------------
   -- The following task removes messages from the log event queue and outputs
   -- them.
   ----------------------------------------------------------------------------
   task Log_Server;
   task body Log_Server is
      Event    : Event_Pointer;
      Num_Lost : Natural;
   begin

      loop
         -- Get the next message to log
         Log_Event_Queue.Dequeue (Event    => Event,
                                  Num_Lost => Num_Lost);

         -- if its a normal message, print it out and with its time and severity, then
         -- deallocate it.
         if Num_Lost = 0 then
            Write_To_Log_File (Time_Stamp (Event.Time) & " " &
              Event_Type'Image(Event.Severity) & ": " & Event.Message);

            Dispose (Event);

         -- if it is a "messages dropped" message, print the time range and number dropped.
         -- Do *not* deallocate it!
         else
            Write_To_Log_File (Time_Stamp (Event.Time) & "...");
            Write_To_Log_File (Time_Stamp & " " &
               Event_Type'Image(Warning) & ":" & Integer'Image(Num_Lost) &
               " Log messages were dropped. Consider changing Log.Max_Message_Queue_Size");
         end if;
      end loop;
   exception
      when Error : others =>
         -- Try to put one last-ditch message to the screen saying that the logger died
         Ada.Text_Io.Put_Line (Time_Stamp & " " & Event_Type'Image(Fatal) &
                               ": Log_Server died with the following exception:" &
                               Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
                               Ada.Exceptions.Exception_Information (Error));
   end Log_Server;

   ----------------------------------------------------------------------------
   -- Report an event to the log. The logger (see body) will output the event
   -- along with the severity and time-stamp as soon as it can. If the logger
   -- gets behind it is possible that new reports will dropped.
   ----------------------------------------------------------------------------
   procedure Report (Event    : in String;
                     Severity : in Event_Type := Informational) is

   begin

      Log_Event_Queue.Enqueue (new Event_Node'(Message_Length => Event'Length,
                                               Message        => Event,
                                               Time           => Ada.Calendar.Clock,
                                               Severity       => Severity,
                                               Next           => Null));



   exception
      -- We can't allocate any more Events until some more are processed. Report
      -- the loss to the event queue and move on.
      when Storage_Error =>
         Log_Event_Queue.Lost_One;
   end Report;

   ----------------------------------------------------------------------------
   -- Wait for all pending log entries to be serviced.
   ----------------------------------------------------------------------------
   procedure Wait is
   begin
      Log_Event_Queue.Is_Empty;
   end Wait;

end Log;
