-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Command_Line;
with Ada.Exceptions;
with Ada.Text_Io;
with Ada.Strings.Fixed;
with Ada.Strings.Maps;
with Ada.Strings.Maps.Constants;
with Ada.Strings;
with Ada.Streams.Stream_Io;

with Interpolation_Table;
with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Interpolation_Table.Multiply_Indexed;
with Token.Csv_Field;
with Token.Analyzer;
with Token.Keyword;
with Token.Line_Comment;
with Token.Character_Set;
with Token.End_Of_File;

--------------------------------------------------------------------------------
-- This routine compiles interpolation tables into a binary form that can be
-- loaded into a table instance with *.Read. The tables will be written into
-- files with the extension ".ito"
--
-- By default it will create either a single, double, or multiple table based
-- on which type will fill all colums, with one dependent. A specific table
-- type can be forced with the command line parameters:
--    -s or -single, -d or -double, -m or -multiple. A specific number of tables
-- can be forced with the parameter -#, where # is the number of tables to make
-- from each input file.
-- The default behavior is -b -1. The behavior will apply to all tables listed
-- after it until a new behavior is specified.
--------------------------------------------------------------------------------
procedure Itac is

   Input_Extension  : constant String := ".csv";
   Output_Extension : constant String := ".ito";

   type Table_Id is (Best, Single, Double, Multiple);

   Table_Creation       : Table_Id := Best;
   Table_Amount         : Natural := 1;
   Actual_Creation_Mode : Table_Id;

   Num_First : Natural;
   Num_Last  : Natural;

   -- Global text file for reading parse data
   File : Ada.Text_Io.File_Type;

   Not_Enough_Files : exception;

   Verbose : Boolean := False;

   ------------------------------------------------------------------------------------
   -- The routine used to feed data into the token analizer. This routine reads from
   -- the analizer file one line at a time until EOF is reached.
   ------------------------------------------------------------------------------------
   function Get_Text_Line return String is
      Buffer : String(1..512);
      Data_Length : Integer;
   begin

      Ada.Text_IO.Get_Line (File => File, Item => Buffer, Last => Data_Length);

      if Ada.Text_IO.End_Of_File(File) then
         return Buffer(1..Data_Length) & Token.EOF_Character;
      else
         return Buffer(1..Data_Length) & Token.EOL_Character;
      end if;
   exception
      when Ada.Text_IO.End_Error =>
         return (1 => Token.EOF_Character);

   end Get_Text_Line;

   ------------------------------------------------------------------------------------
   -- Strip off the input file extension, if it appears in the given file.
   ------------------------------------------------------------------------------------
   function Strip_Csv (File_Name : in String) return String is
      Last_Dot : constant Natural :=
        Ada.Strings.Fixed.Index
        (Source  => File_Name,
         Pattern => ".",
         Going   => Ada.Strings.Backward
         );
   begin

      if
        Last_Dot > 0 and then
        Ada.Strings.Fixed.Translate
        (Source  => File_Name(Last_Dot..File_Name'Last),
         Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map
         ) = Input_Extension
      then
         return File_Name (File_Name'First..Last_Dot - 1);
      else
         return File_Name;
      end if;

   end Strip_Csv;

   ------------------------------------------------------------------------------------
   -- This function parses the first line in a supposed CSV file to determine the
   -- lowest type of table it can use. Single, Double, or Multiple will be returned.
   ------------------------------------------------------------------------------------
   function Best_Creation_Mode
     (File_Name  : in String;
      Num_Tables : in Natural
     ) return Table_Id is

      type CSV_Token_ID is (Comma, Comment, Identifier, Whitespace, EOF);

      -- Instantiate a table file tokenizer
      package Table_Tokenizer is new Token.Analyzer(CSV_Token_ID);

      -- The tokens
      Comma_Token       : aliased Token.Class := Token.Keyword.Get(",");
      Comment_Token     : aliased Token.Class := Token.Line_Comment.Get("#", True);
      Identifier_Token  : aliased Token.Class := Token.CSV_Field.Get;
      Whitespace_Token  : aliased Token.Class := Token.Character_Set.Get
        (Token.Character_Set.Standard_Whitespace);
      EOF_Token         : aliased Token.Class := Token.End_Of_File.Get;

      -- The specification of the syntax of an ASCII table
      Table_Syntax : Table_Tokenizer.Syntax :=
        (Comma       => Comma_Token'Access,
         Identifier  => Identifier_Token'Access,
         Comment     => Comment_Token'Access,
         Whitespace  => Whitespace_Token'Access,
         EOF         => EOF_Token'Access);

      Tokenizer : Table_Tokenizer.Instance := Table_Tokenizer.Initialize
        (Language_Syntax => Table_Syntax,
         Feeder          => Get_Text_Line'access);

      Num_Fields : Natural := 0;

      Input_File_Name : constant String := Strip_Csv(File_Name) & Input_Extension;
   begin

      Ada.Text_IO.Open (Name => Input_File_Name,
                        Mode => Ada.Text_IO.In_File,
                        File => File);

      -- Ignore leading comments
      loop
         Table_Tokenizer.Find_Next (Tokenizer);
         exit when Table_Tokenizer.Token (Tokenizer) /= Comment;
      end loop;

      -- Turn off comment reporting, and parse the rest of the table file
      Comment_Token.Report := False;

      -- Find the number of fields in the first row
      loop

         -- Expecting a field (or eof)
         case Table_Tokenizer.Token (Tokenizer) is
            when Identifier =>
               Num_Fields := Num_Fields + 1;
               Table_Tokenizer.Find_Next (Tokenizer);
            when Comma =>
               Num_Fields := Num_Fields + 1;
            when others =>
               Num_Fields := Num_Fields + 1;
               exit;
         end case;

         -- Expecting a field searator (comma)
         if Table_Tokenizer.Token (Tokenizer) /= Comma then
            exit;
         end if;
         Table_Tokenizer.Find_Next (Tokenizer);

      end loop;

      Ada.Text_Io.Close (File);

      case Num_Fields - Num_Tables is
         when 1 =>
            return Single;
         when 2 =>
            return Double;
         when 0 =>
            Ada.Exceptions.Raise_Exception
              (Not_Enough_Files'Identity, "There are not enough fields in " &
               Input_File_Name & " to make " & Integer'Image(Num_Fields) & " tables");
         when others =>
            return Multiple;
      end case;
   exception
      when Ada.Text_Io.Name_Error =>
         Ada.Text_Io.Put_Line ("Error: The file """ & Input_File_Name & """ does not " &
                               "exist.");
         raise;
   end Best_Creation_Mode;

   ------------------------------------------------------------------------------------
   -- Create the given amount singly-indexed interpolation tables in an output file
   ------------------------------------------------------------------------------------
   procedure Create_Single_Tables
     (Name   : in String;
      Amount : in Natural
     ) is
      Table_List : Interpolation_Table.Singly_Indexed.Table_List(1..Amount);

      Input_File_Name : constant  String := Strip_Csv(Name) & Input_Extension;
      Output_File_Name : constant String := Strip_Csv(Name) & Output_Extension;
   begin
      Interpolation_Table.Singly_Indexed.Read_Ascii
        (File_Name => Input_File_Name,
         Table     => Table_List
         );

      Interpolation_Table.Singly_Indexed.Write
        (File_Name => Output_File_Name,
         Table     => Table_List
         );

      if Verbose then
         Ada.Text_Io.Put_Line
           (Integer'Image (Amount) & " singly indexed => " & Output_File_Name);
      end if;

   end Create_Single_Tables;

   ------------------------------------------------------------------------------------
   -- Create the given amount doubly-indexed interpolation tables in an output file
   ------------------------------------------------------------------------------------
   procedure Create_Double_Tables
     (Name   : in String;
      Amount : in Natural
     ) is
      Table_List : Interpolation_Table.Doubly_Indexed.Table_List(1..Amount);

      Output_File_Name : constant String := Strip_Csv(Name) & Output_Extension;
      Input_File_Name  : constant String := Strip_Csv(Name) & Input_Extension;
   begin
      Interpolation_Table.Doubly_Indexed.Read_Ascii
        (File_Name => Input_File_Name,
         Table     => Table_List
         );

      Interpolation_Table.Doubly_Indexed.Write
        (File_Name => Output_File_Name,
         Table     => Table_List
         );

      if Verbose then
         Ada.Text_Io.Put_Line
           (Integer'Image (Amount) & " doubly indexed => " & Output_File_Name);
      end if;

   end Create_Double_Tables;

   ------------------------------------------------------------------------------------
   -- Create the given amount Multiply-indexed interpolation tables in an output file
   ------------------------------------------------------------------------------------
   procedure Create_Multiple_Tables
     (Name   : in String;
      Amount : in Natural
     ) is
      Table_List : Interpolation_Table.Multiply_Indexed.Table_List(1..Amount);

      Output_File_Name : constant String := Strip_Csv(Name) & Output_Extension;
      Input_File_Name  : constant String := Strip_Csv(Name) & Input_Extension;
   begin
      Interpolation_Table.Multiply_Indexed.Read_Ascii
        (File_Name => Input_File_Name,
         Table     => Table_List
         );

      Interpolation_Table.Multiply_Indexed.Write
        (File_Name => Output_File_Name,
         Table     => Table_List
         );

      if Verbose then
         Ada.Text_Io.Put_Line
           (Integer'Image (Amount) & " multiply indexed => " & Output_File_Name);
      end if;

   end Create_Multiple_Tables;

begin

   -- If nothing was passed in, print a usage string
   if Ada.Command_Line.Argument_Count = 0 then
      Ada.Text_Io.New_Line;
      Ada.Text_Io.Put_Line ("usage: " & Ada.Command_Line.Command_Name &
                            " [flags] inputfilename [[flags] inputfilename]...");
      Ada.Text_Io.Put_Line ("    flag definitions:");
      Ada.Text_Io.Put_Line ("      -v[erbose]  display information on files " &
                            "processed and created");
      Ada.Text_Io.Put_Line ("      -s[ingle]   create singly-indexed tables");
      Ada.Text_Io.Put_Line ("      -d[ouble]   create doubly-indexed tables");
      Ada.Text_Io.Put_Line ("      -m[ultiple] create multiply-indexed tables");
      Ada.Text_Io.Put_Line ("      -b[est]     create single or double " &
                            "tables if possible, otherwise");
      Ada.Text_Io.Put_Line ("                  multiple. {default}");
      Ada.Text_Io.Put_Line ("      -n          create n tables per input file " &
                            "{default = 1}");
   end if;


   -- Process the argument flags
   for Arg_Num in 1..Ada.Command_Line.Argument_Count loop

      if Ada.Command_Line.Argument(Arg_Num) = "-s" or
        Ada.Command_Line.Argument(Arg_Num) = "-single"
      then
         Table_Creation := Single;
      elsif
        Ada.Command_Line.Argument(Arg_Num) = "-d" or
        Ada.Command_Line.Argument(Arg_Num) = "-double"
      then
         Table_Creation := Double;
      elsif
        Ada.Command_Line.Argument(Arg_Num) = "-m" or
        Ada.Command_Line.Argument(Arg_Num) = "-multiple"
      then
         Table_Creation := Multiple;
      elsif
        Ada.Command_Line.Argument(Arg_Num) = "-v" or
        Ada.Command_Line.Argument(Arg_Num) = "-verbose"
      then
         Verbose := True;
      elsif Ada.Command_Line.Argument(Arg_Num)(1) = '-' then
         Ada.Strings.Fixed.Find_Token
           (Source => Ada.Command_Line.Argument(Arg_Num),
            Set    => Ada.Strings.Maps.To_Set ("0123456789"),
            Test   => Ada.Strings.Inside,
            First  => Num_First,
            Last   => Num_Last
            );
         if Num_First = 2 and Num_Last = Ada.Command_Line.Argument(Arg_Num)'Last then
            Table_Amount := Natural'Value(Ada.Command_Line.Argument(Arg_Num)
                                          (Num_First..Num_Last));
         else
            Ada.Text_Io.Put_Line
              ("WARNING: Unknown flag """ & Ada.Command_Line.Argument(Arg_Num) &
               """ ignored");
         end if;
      else

         -- Find the actual table type
         if Table_Creation = Best then
            Actual_Creation_Mode := Best_Creation_Mode
              (File_Name  => Ada.Command_Line.Argument(Arg_Num),
               Num_Tables => Table_Amount
               );
         else
            Actual_Creation_Mode := Table_Creation;
         end if;

         -- Parse the table file
         case Actual_Creation_Mode is
            when Single =>
               Create_Single_Tables
                 (Name   => Ada.Command_Line.Argument(Arg_Num),
                  Amount => Table_Amount
                  );
            when Double =>
               Create_Double_Tables
                 (Name   => Ada.Command_Line.Argument(Arg_Num),
                  Amount => Table_Amount
                  );
            when Multiple =>
               Create_Multiple_Tables
                 (Name   => Ada.Command_Line.Argument(Arg_Num),
                  Amount => Table_Amount
                  );
            when others =>
               Ada.Exceptions.Raise_Exception
                 (Not_Enough_Files'Identity, "There are not enough fields in " &
                  Ada.Command_Line.Argument(Arg_Num) & " to make " &
                  Integer'Image(Table_Amount) & " tables");
         end case;

      end if;

   end loop;
end Itac;
