-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Streams.Stream_Io;  -- I/O for instances

--------------------------------------------------------------------------------
-- This package is an abstract class describing the interface for interpolation
-- tables. It can only be used as a basis for deriving other Interpolation Table
-- classes.
--------------------------------------------------------------------------------
package Interpolation_Table is

   -----------------------
   -- Tunable constants --
   -----------------------

   -- Limit on number of indices any table may have.
   Max_Indices : constant := 5;

   -- Maximum number of independent values in a table
   Max_Independents : constant := 1_500;

   -- The maximum number of tables allowed in a CSV file
   Max_Tables : constant := 5;

   ------------------------------
   -- (end of tunable section) --
   ------------------------------

   -- Index for Table indpendent variables
   subtype Independent_Index is Positive range 1..Max_Independents;

   -- Commonly(internally) used type for lists of independent values
   type Independent_List is array (Independent_Index range <>) of Float;
   subtype Full_Independent_List is Independent_List(Independent_Index'Range);

   subtype Index_Range is Positive range 1..Max_Indices;

   type Index_Vector is array (Index_Range range <>) of Float;

   subtype Table_Index_Range is Positive range 1..Max_Tables;

   type Instance is abstract tagged private;

   type Link is access all Instance'Class;

   --------------------------------------------------------------------------------
   -- Abstract declaration of a procedure to Read in the table from an ASCII file.
   -- Table should be set to the new table if no errors are encountered.
   --------------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        : in out Instance) is abstract;

   --------------------------------------------------------------------------------
   -- Abstract routine to put out the table to an ASCII file.
   --------------------------------------------------------------------------------
   procedure Write_ASCII( File_Name   : in     String;
                          Table       : in     Instance) is abstract;

   --------------------------------------------------------------------------------
   -- Procedure to read a table instance from a table object file.
   -- It raises CONSTRAINT_ERROR if the instance saved in the object file is not of
   -- the same type as the given Table.
   --------------------------------------------------------------------------------
   procedure Read ( File_Name : in     String;
                    Table     :    out Instance'Class);

   --------------------------------------------------------------------------------
   -- Procedure to write a table instance into a table object file.
   -- It saves the instance's tag into the file.
   --------------------------------------------------------------------------------
   procedure Write ( File_Name   : in     String;
                     Table       : in     Instance'Class);

   --------------------------------------------------------------------------------
   -- Abstract routine to retrieve a machine-independent tag for the given
   -- instance.
   --------------------------------------------------------------------------------
   function Machine_Independent_Tag ( Table : in Instance) return String is abstract;

private


   --------------------------------------------------------------------------------
   -- Open a file as a stream. If there is no file with the given name, a
   -- case-insensitive open will be attempted. If this succeeds, a warning message
   -- will be printed.
   --------------------------------------------------------------------------------
   procedure Open_Stream
     ( File : in out Ada.Streams.Stream_Io.File_Type;
       Name : in     String;
       Mode : in     Ada.Streams.Stream_Io.File_Mode
     );

   --------------------------------------------------------------------------------
   -- Routine to (portably) output a string to a stream.
   --------------------------------------------------------------------------------
   procedure Write_String ( Value       : in     String;
                            Destination : access Ada.Streams.Root_Stream_Type'Class);

   --------------------------------------------------------------------------------
   -- Routine to read a string from a stream that has been written with
   -- Write_String.
   --------------------------------------------------------------------------------
   function Read_String ( Source : access Ada.Streams.Root_Stream_Type'Class) return String;

   type Instance is abstract tagged null record;

end Interpolation_Table;

