-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Tags;                   -- Getting unique tag for instance type
with Ada.Exceptions;             -- Custom exception messages
with Ada.Characters.Latin_1;     -- CR and LF
with Ada.Characters.Handling;    -- Character handling
with Ada.Strings.Fixed;          -- String handling
with Ada.Strings.Maps.Constants;
with Log;                        -- Printing of warnings

--------------------------------------------------------------------------------
-- This package is an abstract class describing the interface for interpolation
-- tables. It can only be used as a basis for deriving other Interpolation Table
-- classes.
--------------------------------------------------------------------------------
package body Interpolation_Table is

   --------------------------------------------------------------------------------
   -- Procedure to read a table instance from a table object file.
   -- It raises CONSTRAINT_ERROR if the instance saved in the object file is not of
   -- the same type as the given Table.
   --------------------------------------------------------------------------------
   procedure Read ( File_Name : in     String;
                    Table     :    out Instance'Class) is

      File : Ada.Streams.Stream_Io.File_Type;
   begin

--      Ada.Streams.Stream_Io.Open
      Open_Stream
        (File => File,
         Name => File_Name,
         Mode => Ada.Streams.Stream_Io.In_File
         );


      if
        Read_String (Ada.Streams.Stream_Io.Stream(File)) =
        Machine_Independent_Tag (Table)
      then
         Instance'Class'Read (Ada.Streams.Stream_Io.Stream(File), Table);
         Ada.Streams.Stream_Io.Close (File);
      else
         Ada.Streams.Stream_Io.Close (File);
         Ada.Exceptions.Raise_Exception
           (Constraint_Error'Identity, "Table tag in file """ & File_Name & """ is not " &
            Ada.Tags.External_Tag(Table'Tag));
      end if;

   exception
   when Error : Ada.Streams.Stream_Io.Name_Error =>
      Ada.Exceptions.Raise_Exception
        (Ada.Exceptions.Exception_Identity(Error),
         Ada.Exceptions.Exception_Message(Error) &
         Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
         "File " & '"' & File_Name & '"' &
         " not found");
   end Read;

   --------------------------------------------------------------------------------
   -- Procedure to write a table instance into a table object file.
   -- It saves the instance's tag into the file.
   --------------------------------------------------------------------------------
   procedure Write ( File_Name   : in     String;
                     Table       : in     Instance'Class) is

      File : Ada.Streams.Stream_Io.File_Type;
   begin

      Ada.Streams.Stream_Io.Create
        (File => File,
         Name => File_Name
         );

      Write_String
        (Value       => Machine_Independent_Tag (Table),
         Destination => Ada.Streams.Stream_Io.Stream(File)
         );
      Instance'Class'Write (Ada.Streams.Stream_Io.Stream(File), Table);

      Ada.Streams.Stream_Io.Close (File);

   end Write;

   --------------------------------------------------------------------------------
   -- Routine to (portably) output a string to a stream.
   --------------------------------------------------------------------------------
   procedure Write_String ( Value       : in     String;
                            Destination : access Ada.Streams.Root_Stream_Type'Class) is
   begin

      Integer'Write (Destination, Value'Length);
      for Char in Value'Range loop
         Character'Write(Destination, Value(Char));
      end loop;

   end Write_String;

   --------------------------------------------------------------------------------
   -- Routine to read a string from a stream that has been written with
   -- Write_String.
   --------------------------------------------------------------------------------
   function Read_String ( Source : access Ada.Streams.Root_Stream_Type'Class)
     return String
   is
      Size  : constant Integer := Integer'Input (Source);
      Value : String (1..Size) := (others => Character'Input(Source));
   begin
      return Value;
   end Read_String;

   --------------------------------------------------------------------------------
   -- Open a file as a stream. If there is no file with the given name, a
   -- case-insensitive open will be attempted. If this succeeds, a warning message
   -- will be printed.
   --------------------------------------------------------------------------------
   procedure Open_Stream
     ( File : in out Ada.Streams.Stream_Io.File_Type;
       Name : in     String;
       Mode : in     Ada.Streams.Stream_Io.File_Mode
     ) is

      Working_Name : String := Ada.Strings.Fixed.Translate
        (Source  => Name,
         Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map
         );

      -----------------------------------------------------------------------------
      -- Attempt to open the given file with every permuation of its name from the
      -- given character index to the end.
      -----------------------------------------------------------------------------
      function Open_Stream_Permutation (Index : Natural) return Boolean is
      begin

         -- If this isn't the last character, return true if either case of this
         -- character combined with all case permuations of the following characters
         -- can open the stream file.
         if Index < Working_Name'Last then

            if Open_Stream_Permutation (Index + 1) then
               return True;
            else
               Working_Name (Index) := Ada.Characters.Handling.To_Upper(Working_Name(Index));
               if Open_Stream_Permutation (Index + 1) then
                  return True;
               end if;
            end if;

         -- If this is the last character, return true if either case of this
         -- character allows us to successfully open a file.
         else
            begin
               Ada.Streams.Stream_Io.Open
                 (File => File,
                  Name => Working_Name,
                  Mode => Ada.Streams.Stream_Io.In_File
                  );
               return True;
            exception
               when Ada.Streams.Stream_Io.Name_Error =>
                  null;
            end;
            Working_Name (Index) := Ada.Characters.Handling.To_Upper(Working_Name(Index));
            begin
               Ada.Streams.Stream_Io.Open
                 (File => File,
                  Name => Working_Name,
                  Mode => Ada.Streams.Stream_Io.In_File
                  );
               return True;
            exception
               when Ada.Streams.Stream_Io.Name_Error =>
                  null;
            end;
         end if;

         -- Failed to open the file. Put the character back to lower case and report the
         -- failure.
         Working_Name (Index) := Ada.Characters.Handling.To_Lower(Working_Name(Index));
         return False;

      end Open_Stream_Permutation;
   begin

      -- Attempt an open with the given name. If it succeeds, we are done.
      begin
         Ada.Streams.Stream_Io.Open
           (File => File,
            Name => Name,
            Mode => Ada.Streams.Stream_Io.In_File
            );
         return;
      exception
         when Ada.Streams.Stream_Io.Name_Error =>
            null;
      end;

      Log.Report
        (Event    => "Couldn't find table file """ & Name & """." &
         Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
         "     Attempting case-sense...""",
         Severity => Log.Warning
         );

      -- Go through every possible capitalization of the given file name looking for
      -- a match
      if Open_Stream_Permutation (Working_Name'First) then
         Log.Report
           (Event    => "Table file """ & Name & """" &
            Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
            "     does not exist, so file """ &
            Working_Name & """ was used instead. " &
            Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
            "     Please change one or the other to match.",
            Severity => Log.Warning
            );
      else
         raise Ada.Streams.Stream_Io.Name_Error;
      end if;
   end Open_Stream;

end Interpolation_Table;



