-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Text_IO;
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Strings.Unbounded;
with Ada.Streams.Stream_Io;
with Ada.Tags;
with Interpolation_Table.CSV_Parsing;

use type Ada.Strings.Unbounded.Unbounded_String;

--------------------------------------------------------------------------------
-- This package is implemented as a high speed lookup table of a one
-- dimensional array of independent and dependent pairs.  To operate the
-- user provides an ASCII text file of number pairs, compiles the
-- ASCII file which will test for syntax and content errors, if successful
-- will store the resulting data in a binary datafile.  Then during
-- normal operation the user calls the VALUE routine with an independent
-- input value which returns an interpolated dependent value based on the
-- values stored in the table.
--------------------------------------------------------------------------------
package body Interpolation_Table.Singly_Indexed is

   Invalid_Table : exception;

                          -----------------------
                          -- Local subprograms --
                          -----------------------

   --------------------------------------------------------------------------------
   -- Pre-compute the slope values between successive dependent values, this
   -- is so run time computation of the resulting interpolated values can
   -- run as fast as possible.
   --
   -- See Extrapolate below to see exactly how this slope is used.
   --------------------------------------------------------------------------------
   procedure  Calculate_Slopes (Table  : in out Instance) is

      Independent_Diff  : Float;
   begin

      -- Check for infinite slope and verify that the independent variable
      -- is monotonically increasing.
      for Independent in  1  ..  Table.Independent_Count-1  loop

         -- calculate slopes for this independent variable.
         Independent_Diff  :=  Table.Independent(Independent+1) -
           Table.Independent(Independent);

         if Independent_Diff  <=  0.0  then
            -- the independend values must be in increasing order, if not
            -- raise Invalid_Table.
            Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
              "independent values must be monotonically increasing");
            raise Invalid_Table;

         else

            -- for each set of point value, pre calculate the
            -- slope value, this is done to speed up the run time
            -- calculation.
            -- The calculation is
            --    Slope  =  (dependent+1    -  dependent)  /
            --              (independent+1  -  independent)
            Table.Slope(Independent) :=
              (Table.Dependent(Independent+1) - Table.Dependent(Independent)) /
              Independent_Diff;
         end if;
      end loop;

   end Calculate_Slopes;

   -----------------------------------------------------------------------------
   --
   -- Calculate the output value from the input parameters as follows:
   --   Output_Start + ( (Input_Value - Input_Start) * Slope)
   --
   -----------------------------------------------------------------------------
   function Extrapolate (Input_Value  : in Float;
                         Input_Start  : in Float;
                         Output_Start : in Float;
                         Slope        : in Float) return Float is
   begin

      return ( (Input_Value - Input_Start) * Slope ) + Output_Start;

   end Extrapolate;


                       ------------------------
                       -- Public Subprograms --
                       ------------------------

   --------------------------------------------------------------------------------
   -- Return the value of the first (lowest) and last (highest) independents.
   --------------------------------------------------------------------------------
   function First_Independent (Table : in Instance) return Float is
   begin
      return Table.Independent(Full_Independent_List'First);
   end First_Independent;

   function Last_Independent (Table : in Instance) return Float is
   begin
      return Table.Independent(Table.Independent_Count);
   end Last_Independent;

   --------------------------------------------------------------------------------
   -- Read in a number of tables from an ASCII file. The number of tables
   -- available in the file must match the size of the Table array.
   --------------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        :    out Table_List) is

      -- The CSV parser pacakge
      package CSV is new Interpolation_Table.CSV_Parsing
        (Max_Dependents   => Table(1).Dependent'length,
         Max_Dimensions   => 1);

      -- The lines of the text file that were read in
      Source_Lines    : CSV.CSV_Line_List;
      Number_Of_Lines : Integer;

      -- The names of the dimension headers
      Header : CSV.Dimension_Strings := (others => Ada.Strings.Unbounded.Null_Unbounded_String);
      Number_Of_Headers : Integer;

      Total_Dependents : Integer := 0;

      Dimensions : Integer := 1;

      --------------------------------------------------------------------------------
      -- Load all the dependent values for all the independents starting at the given
      -- dimension. If there is an entry missing, a message indicating the missing
      -- entry will be tacked onto the end of Missing_Dependents.
      --
      -- The algorithm used to match up input rows with expected independent values
      -- depends on the Rows being sorted so that their independents are in
      -- lexigraphical order.
      --------------------------------------------------------------------------------
      procedure Load_CSV_Dependents (Table       : in out Table_List;
                                     Num_Rows    : in     Positive;
                                     Sorted_Rows : in     CSV.CSV_Line_List) is

         -- The current index of the dependent
         Dependent_Index : Integer := 1;

         Error_Message : Ada.Strings.Unbounded.Unbounded_String
           := Ada.Strings.Unbounded.Null_Unbounded_String;
         Error_Length : Integer := 0;

         Dependent : CSV.Dependent_List;

      begin

         -- This loop runs through every combination of independent indices in the sorted order
         -- and copies their dependent from the sorted rows into the table.
         for Index in 1..Table(1).Independent_Count loop

            -- Attempt to read a dependent from the row with the specified independents.
            CSV.Get_Expected_Dependents
              (Independent => (1 => Table(1).Independent(Index)),
               Sorted_Row  => Sorted_Rows(Dependent_Index),
               Dependent   => Dependent,
               Error       => Error_Message
              );

            -- If there was no error, copy the dependents into their tables
            if Error_Length = Ada.Strings.Unbounded.Length(Error_Message) then

               -- Copy the dependents into their tables
               for Table_Index in 1..Table'Length loop
                  Table(Table_Index).Dependent(Index) := Dependent(Table_Index);
               end loop;

               Dependent_Index := Dependent_Index + 1;

            end if;
            Error_Length := Ada.Strings.Unbounded.Length(Error_Message);

         end loop;

         -- If there were missing rows, raise an exception and report the error.
         if Error_Message /= Ada.Strings.Unbounded.Null_Unbounded_String then
            Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
              Ada.Strings.Unbounded.To_String(Error_Message));
         end if;

      end Load_CSV_Dependents;


   begin

      -- Parse the table file into CSV lines
      CSV.Parse
        (File_Name       => File_Name,
         Name_List       => Header,
         Number_Of_Names => Number_Of_Headers,
         Dimensions      => Dimensions,
         Dependents      => Table'Length,
         Num_Rows        => Number_Of_Lines,
         CSV_Rows        => Source_Lines);

      -- Load in the independents
      CSV.Load_Independents (File_Name        => File_Name,
                             Rows             => Source_Lines,
                             Num_Rows         => Number_Of_Lines,
                             Dimension        => 1,
                             Independents     => Table(1).Independent,
                             Num_Independents => Table(1).Independent_Count);

      -- Copy over the independents to the other tables
      for Table_Index in 2..Table'Length loop

         Table(Table_Index).Independent (1..Table(1).Independent_Count) :=
           Table(1).Independent (1..Table(1).Independent_Count);
         Table(Table_Index).Independent_Count := Table(1).Independent_Count;
      end loop;

      -- Load all the dependents into the table
      Load_CSV_Dependents
        (Table              => Table,
         Num_Rows           => Number_Of_Lines,
         Sorted_Rows        => Source_Lines);

      -- Precalculate the slopes, etc.
      for Table_Index in Table'range loop
         Calculate_Slopes(Table(Table_Index));
      end loop;

   end Read_ASCII;

   --------------------------------------------------------------------------------
   -- Read in the single point table from an ASCII file, table pointer will
   -- be set to the new table if no errors are encountered.
   --
   -- To perform binary reads and writes, simply use the stream facilities
   -- (eg: Interpolation_Table.Instance'Read)
   --
   -- Exceptions: Text_IO exceptions.
   --------------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        : in out Instance) is
      Single_Table : Table_List (1..1);
   begin

      Read_ASCII ( File_Name => File_Name,
                   Table     => Single_Table
                   );

      Table := Single_Table(1);

   end Read_ASCII;

   -----------------------------------------------------------------------------
   -- Write out the ASCII version of the double point table to a file.
   --
   -- It should be written in the same format expected by the Read_ASCII
   -- routine.
   -----------------------------------------------------------------------------
   procedure Write_ASCII( File_Name   : in     String;
                          Table       : in     Instance) is


      Table_File     : Ada.Text_IO.File_Type;

   begin
      Ada.Text_IO.Create(File => Table_File,
                         Name => File_Name);

      -- Start with a comment naming the file, and listing the number of independents
      Ada.Text_IO.Put_Line(File => Table_File, Item => "# single point lookup data table " & File_Name);
      Ada.Text_IO.Put(File => Table_File, Item => "# Number of Independents =" &
        Integer'Image(Table.Independent_Count));
      Ada.Text_IO.New_Line(File => Table_File, Spacing => 2);


      -- Write out all the dependents with their independent values
      for Index in 1..Table.Independent_Count loop

         Ada.Text_IO.Put_Line
           (File => Table_File,
            Item => Float'Image (Table.Independent(Index)) & ", " &
                    Float'Image (Table.Dependent(Index))
           );

      end loop;

      Ada.Text_IO.Close(Table_File);

   end Write_ASCII;

   --------------------------------------------------------------------------------
   -- Compute the interpolated value for the given input value base on the
   -- values read into the table.
   --
   -- This routine is built for speed. But I do see two things that could be done
   -- to further speed it up if nessecary.
   -- o  Change the Table parameter to an "in out". The compiler may be able to
   --    do a much better job optimizing the code when it is not dealing with
   --    an aliased object. This would require the function to be changed to a
   --    a procedure though. Yuk!
   -- o  Change the "zone search algorithm". Currently it uses a kind of linear
   --    (although the top level is binary). A full binary search could speed
   --    things up when the correct zone is far from the last zone. However, if the
   --    zones don't change much (which is probably the case) then a binary
   --    search could actually slow it down.
   --------------------------------------------------------------------------------
   function Interpolate( Input      : in     Float;
                         Table      : access Instance )
             return  Float is

      Independent : Float := Input;
   begin

      -- Make sure the table has been initialized
      if (Table.Independent_Count = 0) then
         raise Invalid_Table;
      end if;

      if ((Input >=  Table.Independent(Table.Start_Zone) and then
           (Input <=  Table.Independent(Table.Start_Zone+1) )))
      then
         -- input value falls within range of the previous call's current zone, calculate the
         -- return value.
         null;

      elsif (Input >  Table.Independent(Table.Independent_Count)) then
         -- Input value is greater than any in the table. Calculate from
         -- the values at the top of the table.
         Table.Start_Zone  := Table.Independent_Count - 1;
         Independent       := Table.Independent(Table.Independent_Count);

      elsif (Input <  Table.Independent(Table.Independent'First)) then
         -- Input value is smaller than any in the table. Calculate from the
         -- values at the bottom of the table.
         Table.Start_Zone  := Independent_Index'First;
         Independent       := Table.Independent(Table.Independent'First);

      elsif  Input <  Table.Independent(Table.Start_Zone)  then
         -- Input value is smaller than any in the previous current zone. Find
         -- the smaller zone it fits in and calculate from there.
         loop
            Table.Start_Zone := Table.Start_Zone - 1;
            exit when Input >= Table.Independent(Table.Start_Zone);
         end loop;

      elsif (Input > Table.Independent(Table.Start_Zone+1)) then
         -- Input value is larger than any in the previous current zone. Find
         -- the larger zone it fits in and calculate from there
         loop
            Table.Start_Zone := Table.Start_Zone + 1;
            exit when Input <= Table.Independent(Table.Start_Zone+1);
         end loop;

      end if;

      return Extrapolate (Input_Value  => Independent,
                          Input_Start  => Table.Independent (Table.Start_Zone),
                          Output_Start => Table.Dependent (Table.Start_Zone),
                          Slope        => Table.Slope (Table.Start_Zone) );

   end Interpolate;

   --------------------------------------------------------------------------------
   -- Procedure to read a table list from a table object file.
   -- It raises CONSTRAINT_ERROR if the list saved in the object file is not of
   -- the same type as the given Table.
   --------------------------------------------------------------------------------
   procedure Read ( File_Name : in     String;
                    Table     :    out Table_List) is
      File : Ada.Streams.Stream_Io.File_Type;
   begin

--      Ada.Streams.Stream_Io.Open
      Open_Stream
        (File => File,
         Name => File_Name,
         Mode => Ada.Streams.Stream_Io.In_File
         );

      -- Verify that the tag matches Instance
      declare
         Tag : constant String := Read_String (Ada.Streams.Stream_Io.Stream(File));
      begin
         if Machine_Independent_Tag (Table(Table'First)) /= Tag then

            Ada.Streams.Stream_Io.Close (File);
            Ada.Exceptions.Raise_Exception
              (Constraint_Error'Identity, "Table tags do not match." &
               Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
               "Expected " & Machine_Independent_Tag (Table(Table'First)) &
               " but file " & File_Name & " contains " & Tag & "."
               );
         end if;
      end;

      -- Read in all the table objects
      for Index in Table'Range loop
         Instance'Read (Ada.Streams.Stream_Io.Stream(File), Table(Index));
         Ada.Streams.Stream_Io.Close (File);
      end loop;

   exception
   when Error : Ada.Streams.Stream_Io.Name_Error =>
      Ada.Exceptions.Raise_Exception
        (Ada.Exceptions.Exception_Identity(Error),
         Ada.Exceptions.Exception_Message(Error) &
         Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
         "File " & '"' & File_Name & '"' &
         " not found");

   end Read;

   --------------------------------------------------------------------------------
   -- Procedure to write a table list into a table object file.
   -- It saves the instance's tag into the file.
   --------------------------------------------------------------------------------
   procedure Write ( File_Name   : in     String;
                     Table       : in     Table_List) is
      File : Ada.Streams.Stream_Io.File_Type;
   begin

      Ada.Streams.Stream_Io.Create
        (File => File,
         Name => File_Name
         );

      -- Write out the tag of Instance
      Write_String
        (Value       => Machine_Independent_Tag (Table(Table'First)),
         Destination => Ada.Streams.Stream_Io.Stream(File)
         );

      -- Write out all the table objects
      for Index in Table'Range loop
         Instance'Write (Ada.Streams.Stream_Io.Stream(File), Table(Index));
      end loop;

      Ada.Streams.Stream_Io.Close (File);

   end Write;

   --------------------------------------------------------------------------------
   -- Abstract routine to retrieve a machine-independent tag for the given
   -- instance.
   -- The scheme used to prevent clashes is to return the fully-qualified name of
   -- the type.
   --------------------------------------------------------------------------------
   function Machine_Independent_Tag ( Table : in Instance) return String is
   begin
      return "interpolation_table.singly_indexed.instance";
   end Machine_Independent_Tag;

   --------------------------------------------------------------------------------
   -- Primitive routine to write an instance to a stream. This routine should
   -- write in a somewhat portable and storage-efficent manner.
   --------------------------------------------------------------------------------
   procedure Write ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                     Item   : in     Instance) is
   begin
      -- Write the start zone and independent count
      Independent_Index'Write(Stream,Item.Start_Zone);
      Integer'Write(Stream, Item.Independent_Count);

      -- Write the independents, dependents, and slopes. Only write valid bytes (not the
      -- entire array). In the case of the slope, that's one less than for the other
      -- arrays.
      -- Note that we *should* be able to write out entire arrays at once instead of
      -- using loops like is done below. Unfonrtunately GreenHills 1.8.9B has a bug
      -- where it writes arrays backwards.

      -- Write the independents
      for Index in Item.Independent'First..Item.Independent'First + Item.Independent_Count - 1
      loop
         Float'Write(Stream,Item.Independent(Index));
      end loop;

      -- Write the dependents
      for Index in Item.Dependent'First..Item.Dependent'First + Item.Independent_Count - 1
      loop
         Float'Write(Stream,Item.Dependent(Index));
      end loop;

      -- Write the slopes
      for Index in Item.Slope'First..Item.Slope'First + Item.Independent_Count - 2
      loop
         Float'Write(Stream,Item.Slope(Index));
      end loop;
   end Write;

   --------------------------------------------------------------------------------
   -- Primitive routine to read an instance from a stream. This routine should
   -- be able to read the output of the Write command above.
   --------------------------------------------------------------------------------
   procedure Read ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                    Item   :    out Instance) is
   begin

      -- Read the start zone and independent count
      Independent_Index'Read(Stream,Item.Start_Zone);
      Integer'Read(Stream, Item.Independent_Count);

      -- Read the independents, dependents, and slopes. Only read valid bytes (not the
      -- entire array).
      for Index in Item.Independent'First..Item.Independent'First + Item.Independent_Count - 1
      loop
         Float'Read(Stream,Item.Independent(Index));
      end loop;
      for Index in Item.Dependent'First..Item.Dependent'First + Item.Independent_Count - 1
      loop
         Float'Read(Stream,Item.Dependent(Index));
      end loop;
      for Index in Item.Slope'First..Item.Slope'First + Item.Independent_Count - 2
      loop
         Float'Read(Stream,Item.Slope(Index));
      end loop;

   end Read;


 end Interpolation_Table.Singly_Indexed;
