-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Strings.Unbounded;    -- dynamic strings with dynamic operations
with Ada.Text_IO;
with Ada.Exceptions;
with Interpolation_Table.CSV_Parsing;
with Ada.Streams.Stream_Io;
with Ada.Tags;
with Ada.Characters.Latin_1;

use type Ada.Strings.Unbounded.Unbounded_String;

--------------------------------------------------------------------------------
-- This package is implemented as a high speed lookup table of an arbitrary
-- dimensional array of independent and dependent pairs.  To operate the
-- user provides an ASCII text file of number pairs, compiles the
-- ASCII file which will test for syntax and content errors, if successful
-- will store the resulting data in a binary datafile.  Then during
-- normal operation the user calls the Interpolate routine with an independent
-- input value which returns an interpolated dependent value based on the
-- values stored in the table.
--
-- The actual algorithm used is a piecewise linear interpolation algorithm.
--------------------------------------------------------------------------------
package body Interpolation_Table.Multiply_Indexed is

   Invalid_Table : exception;

                          -----------------------
                          -- Local subprograms --
                          -----------------------

   --------------------------------------------------------------------------------
   -- Return the dimension index for a value at the given Value_Matrix index, where
   -- Cap designates the largest possible value of the index for that dimension and
   -- Offset designates the smallest. This routine retrieves the implicit index for
   -- a dimension from a linear array index.
   --------------------------------------------------------------------------------
   function Index_For (Index  : in Integer;
                       Cap    : in Integer;
                       Offset : in Integer) return Integer is
   begin

      -- strip off the higher index info, and divide off the lower indices.
      return (((Index - 1) mod Cap) / Offset) + 1;
   end Index_For;

   ----------------------------------------------------------------------------------
   -- Get the dependent index with the given indices. Only works on initialized
   -- tables.
   ----------------------------------------------------------------------------------
   function Dependent_Index (Table   : in Instance;
                             Indices : in Independent_Index_List) return Integer is
      Master_Index : Integer := 1;
   begin

      for Dimension in 1..Table.Dimensions loop

         Master_Index := Master_Index + (Table.Offset(Dimension) * (Indices(Dimension) - 1) );
      end loop;

      return Master_Index;

   end Dependent_Index;

   ------------------------------------------------------------------------------
   -- Increment the given independent indices.
   -- Either the last dimension is incremented, or (if it's at its limit) it is
   -- set to 1 and the next to last dimension is incremented in the same manner.
   ------------------------------------------------------------------------------
   procedure Increment (Indices    : in out Independent_Index_List;
                        Table      : in     Instance) is
   begin
      -- The loop goes through the indices in reverse either incrementing them or
      -- resetting them the first index value (when they are at the last). It exits when
      -- one is incremented or when all have been reset.
      for Dimension in reverse 1..Table.Dimensions loop
         if Indices(Dimension) < Table.Count(Dimension) then
            Indices(Dimension) := Indices(Dimension) + 1;
            exit;
         end if;

         Indices(Dimension) := Independent_Index'First;
      end loop;
   end Increment;

   ------------------------------------------------------------------------------
   -- Assuming a Dimension dimensional hypercube, return the indices of the next
   -- node in the given dimension from the given origin. The intent of this
   -- procedure is to provide a method of visiting every node in a hypercube.
   -- For the first call Indices should be the same as Origin. Thereafter,
   -- Origin should remain unchanged.
   ------------------------------------------------------------------------------
   procedure Next_Node (Indices : in out Independent_Index_List;
                        Origin  : in     Independent_Index_List) is

   begin

      -- The next node is found by "incrementing" the given Indices. Since there
      -- are only 2 values in any dimension in a hypercube, this can be done by
      -- comparing the current value to the Origin's value.
      for Dimension in Indices'range loop
         if Indices(Dimension) = Origin(Dimension) then
            Indices(Dimension) := Indices(Dimension) + 1;
            exit;
         end if;

         Indices(Dimension) := Origin(Dimension);
      end loop;

   end Next_Node;


   ------------------------------------------------------------------------------
   -- Get the values of the given independents
   ------------------------------------------------------------------------------
   function Independent_Values (Indices : in Independent_Index_List;
                                Table   : in Instance) return Index_Vector is
      Result : Full_Index_Vector;
   begin
      for Dimension in 1..Table.Dimensions loop

         Result(Dimension) := Table.Independent(Dimension)(Indices(Dimension));

      end loop;

      return Result(1..Table.Dimensions);
   end Independent_Values;

   --------------------------------------------------------------------------------
   -- Pre-compute the slope values between successive dependent values, this
   -- is so run time computation of the resulting interpolated values can
   -- run as fast as possible.
   --
   -- See Basis_Function below to see exactly how this slope is used.
   --------------------------------------------------------------------------------
   procedure  Initialize (Table  : in out Instance) is

   begin

      -- for each dimension...
      for Dimension in reverse 1..Table.Dimensions loop

         -- Calculate the matrix indexing factors
         if Dimension = Table.Dimensions then
            Table.Offset(Dimension) := 1;
         else
            Table.Offset(Dimension) := Table.Offset(Dimension + 1) * Table.Count(Dimension + 1);
         end if;

         -- Calculate the index deltas
         for Index_Index in 1 .. Table.Count(Dimension) - 1 loop

            Table.Index_Delta(Dimension)(Index_Index)  :=
              Table.Independent (Dimension)(Index_Index + 1) -
              Table.Independent  (Dimension)(Index_Index);

         end loop;


      end loop;

      -- Set up the zones
      Table.Start_Zone (1..Table.Dimensions) := (others => 1);

   end Initialize;

   -----------------------------------------------------------------------------
   -- Find the indices of the zones in which the given value lies. The search is
   -- structured so that the zones indicated by "Zone" are checked first. If
   -- the value is outside the given list, the zone at the closest edge of the
   -- list is returned.
   --
   -- This routine is called from the interpolation routine, which itself
   -- liable to be called multiple times every cycle by 60Hz real-time processes.
   -- Thus, the speed of this routine is very important.
   --
   -- It may be possible to speed this routine up by changing the "zone search
   -- algorithm". Currently it uses a kind of linear search (although the top
   -- level is binary). A full binary search could speed things up when the
   -- correct zone is far from the last zone. However, if the zones don't change
   -- much (which is probably the case) then a binary search could actually
   -- slow it down.
   -----------------------------------------------------------------------------
   procedure Find_Zone_Indices
     (Independents     : in     Value_List;
      Last_Independent : in     Independent_Index_List;
      Value            : in out Index_Vector;
      Zone             : in out Independent_Index_List) is

   begin

      for Dimension in Value'range loop


         if ((Value (Dimension) >= Independents (Dimension)(Zone(Dimension)) and then
             (Value (Dimension) <= Independents (Dimension)(Zone(Dimension)+1))))
         then
            -- input value falls within range of the previous call's current zone, calculate the
            -- return value.
            null;

         elsif (Value (Dimension) >  Independents (Dimension)(Last_Independent(Dimension))) then
            -- Input value is greater than any in the table. Calculate from
            -- the values at the top of the table.
            Zone(Dimension)   := Last_Independent(Dimension) - 1;
            Value (Dimension) := Independents (Dimension)(Last_Independent(Dimension));

         elsif (Value (Dimension) <  Independents (Dimension)(Independents'First)) then
            -- Input value is smaller than any in the table. Calculate from the
            -- values at the bottom of the table.
            Zone (Dimension)  := Independents'First;
            Value (Dimension) := Independents (Dimension)(Independents'First);

         elsif  Value (Dimension) <  Independents (Dimension)(Zone(Dimension)) then
            -- Input value is smaller than any in the previous current zone. Find
            -- the smaller zone it fits in and calculate from there.
            loop
               Zone(Dimension) := Zone(Dimension) - 1;
               exit when Value(Dimension) >= Independents (Dimension)(Zone (Dimension));
            end loop;

         elsif (Value (Dimension) > Independents (Dimension)(Zone (Dimension) + 1)) then
            -- Input value is larger than any in the previous current zone. Find
            -- the larger zone it fits in and calculate from there
            loop
               Zone (Dimension) := Zone (Dimension) + 1;
               exit when Value (Dimension) <= Independents (Dimension)(Zone (Dimension) + 1);
            end loop;

         end if;

      end loop;

   end Find_Zone_Indices;

   -----------------------------------------------------------------------------
   -- Returns the "Basis Function" for the interior point in the table at the
   -- given node.
   --
   -- The formula is the product over all dimensions of the distance from the
   -- other node in that dimension divided by the distance between the nodes.
   --
   -- This formula assumes that the lower node from which the slope is to be
   -- calclulated is loaded into Table.Start_Zone.
   -----------------------------------------------------------------------------
   function Basis_Function (Table          : in Instance;
                            Interior_Point : in Index_Vector;
                            Node           : in Independent_Index_List
                           ) return Float is
      Result : Float := 1.0;
   begin

      for Dimension in 1..Table.Dimensions loop

         Result := Result *
           (1.0 -
            (abs (Interior_Point(Dimension) -
                  Table.Independent(Dimension)(Node(Dimension))) /
             Table.Index_Delta (Dimension)
             (Table.Start_Zone(Dimension))));

      end loop;

      return Result;

   end Basis_Function;

                        ---------------------
                        -- Public Routines --
                        ---------------------


   --------------------------------------------------------------------------------
   -- Return the value of the first (lowest) and last (highest) independents.
   --------------------------------------------------------------------------------
   function First_Independent (Table : in Instance) return Index_Vector is
      Independents : Index_Vector(Full_Index_Vector'First..Table.Dimensions);
   begin
      for Index in Independents'Range loop
         Independents(Index) := Table.Independent(Index)(Table.Independent(Index)'First);
      end loop;
      return Independents;
   end First_Independent;

   function Last_Independent (Table : in Instance) return Index_Vector is
      Independents : Index_Vector(Full_Index_Vector'First..Table.Dimensions);
   begin
      for Index in Independents'Range loop
         Independents(Index) := Table.Independent(Index)(Table.Count(Index));
      end loop;
      return Independents;
   end Last_Independent;

   --------------------------------------------------------------------------------
   -- Read in a number of tables from an ASCII file. The number of tables
   -- available in the file must match the size of the Table array.
   --------------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        :    out Table_List) is

      -- The CSV parser pacakge
      package CSV is new Interpolation_Table.CSV_Parsing
        (Max_Dependents   => Max_Dependents,
         Max_Dimensions   => Max_Indices);

      -- The lines of the text file that were read in
      Source_Lines    : CSV.CSV_Line_List;
      Number_Of_Lines : Integer;

      -- The names of the dimension headers
      Header : CSV.Dimension_Strings := (others => Ada.Strings.Unbounded.Null_Unbounded_String);
      Number_Of_Headers : Integer;

      Total_Dependents : Integer := 0;

      --------------------------------------------------------------------------------
      -- Load all the dependent values for all the independents starting at the given
      -- dimension. If there is an entry missing, a message indicating the missing
      -- entry will be tacked onto the end of Missing_Dependents.
      --
      -- The algorithm used to match up input rows with expected independent values
      -- depends on the Rows being sorted so that their independents are in
      -- lexigraphical order.
      --------------------------------------------------------------------------------
      procedure Load_CSV_Dependents (Table       : in out Table_List;
                                     Num_Rows    : in     Positive;
                                     Sorted_Rows : in     CSV.CSV_Line_List) is

         -- The current indices of the expected independent values
         Independent_Indices : Independent_Index_List := (others => Independent_Index'First);

         -- The current index of the dependent
         Dependent_Index : Integer := 1;

         Error_Message : Ada.Strings.Unbounded.Unbounded_String
           := Ada.Strings.Unbounded.Null_Unbounded_String;
         Error_Length : Integer := 0;

         Dependent : Csv.Dependent_List;

      begin

         -- This loop runs through every combination of independent indices in the sorted order
         -- and copies their dependent from the sorted rows into the table.
         loop

            -- Attempt to read a dependent from the row with the specified independents.
            CSV.Get_Expected_Dependents
              (Independent => Independent_Values(Indices => Independent_Indices,
                                                 Table   => Table(1)),
               Sorted_Row  => Sorted_Rows(Dependent_Index),
               Dependent   => Dependent,
               Error       => Error_Message
              );

            -- If there was no error, check the next dependent value next loop
            if Error_Length = Ada.Strings.Unbounded.Length(Error_Message) then

               -- Copy the dependents into their tables
               for Table_Index in 1..Table'Length loop
                  Table(Table_Index).Dependent(Dependent_Index) := Dependent(Table_Index);
               end loop;

               Dependent_Index := Dependent_Index + 1;
            end if;
            Error_Length := Ada.Strings.Unbounded.Length(Error_Message);

            -- Increment the independent indices by one
            Increment (Indices => Independent_Indices,
                       Table   => Table(1));

            -- exit the loop when we run out of combinations of independents
            exit when Independent_Indices(1..Table(1).Dimensions) =
              (1..Table(1).Dimensions => Independent_Index'First);
         end loop;

         -- If there were missing rows, raise and exception and report the error.
         if Error_Message /= Ada.Strings.Unbounded.Null_Unbounded_String then
            Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
              Ada.Strings.Unbounded.To_String(Error_Message));
         end if;

      end Load_CSV_Dependents;


   begin  -- read ascii table routine.

      Table(1).Count      := (others => 0);
      Table(1).Dimensions := 0;

      -- Parse the table file into CSV lines
      CSV.Parse
        (File_Name       => File_Name,
         Name_List       => Header,
         Number_Of_Names => Number_Of_Headers,
         Dimensions      => Table(1).Dimensions,
         Dependents      => Table'Length,
         Num_Rows        => Number_Of_Lines,
         CSV_Rows        => Source_Lines);

      -- Copy over the table dimensions to the other tables
      for Table_Index in 2..Table'Length loop
         Table(Table_Index).Dimensions := Table(1).Dimensions;
      end loop;

      -- Load in the independents for all the dimensions
      for Dimension in 1..Table(1).Dimensions loop

         CSV.Load_Independents (File_Name        => File_Name,
                                Rows             => Source_Lines,
                                Num_Rows         => Number_Of_Lines,
                                Dimension        => Dimension,
                                Independents     => Table(1).Independent(Dimension),
                                Num_Independents => Table(1).Count(Dimension));

         -- Copy over the independents to the other tables
         for Table_Index in 2..Table'Length loop

            Table(Table_Index).Independent(Dimension) (1..Table(1).Count (Dimension)) :=
              Table(1).Independent(Dimension) (1..Table(1).Count (Dimension) );
            Table(Table_Index).Count(Dimension) := Table(1).Count (Dimension);

         end loop;

      end loop;

      -- Load the all dependents into the table
      Load_CSV_Dependents
        (Table              => Table,
         Num_Rows           => Number_Of_Lines,
         Sorted_Rows        => Source_Lines);

      -- Precalcuate the slopes, etc
      for Table_Index in Table'range loop
         Initialize(Table(Table_Index));
      end loop;

   end Read_ASCII;

   -----------------------------------------------------------------------------
   -- Read in the ASCII table and store in the Table.
   -----------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        : in out Instance) is
      Single_Table : Table_List (1..1);
   begin

      Read_ASCII ( File_Name => File_Name,
                   Table     => Single_Table
                   );

      Table := Single_Table(1);

   end Read_ASCII;

   -----------------------------------------------------------------------------
   -- Write out the ASCII version of the double point table to a file.
   --
   -- It should be written in the same format expected by the Read_ASCII
   -- routine.
   -----------------------------------------------------------------------------
   procedure Write_ASCII( File_Name   : in     String;
                          Table       : in     Instance) is

      Table_File     : Ada.Text_IO.File_Type;

      Global_Element : Integer := 1;
      package CSV is new Interpolation_Table.CSV_Parsing
        (Max_Dependents   => Max_Dependents,
         Max_Dimensions   => Max_Indices);

      Coordinates : Independent_Index_List := (others => 1);
   begin
      Ada.Text_IO.Create(File => Table_File,
                         Name => File_Name);

      -- Start with a comment naming the file, and listing the number of independents
      Ada.Text_IO.Put_Line(File => Table_File, Item => "# multiple point lookup data table " & File_Name);
      Ada.Text_IO.Put(File => Table_File, Item => "# Number of Independents =");
      for Dimension in 1..Table.Dimensions-1 loop
         Ada.Text_IO.Put(File => Table_File,
                         Item => Integer'Image(Integer( Table.Count(Dimension))) & ",");
      end loop;

      Ada.Text_IO.Put(File => Table_File,
                      Item => "and " & Integer'Image(Integer( Table.Count(Table.Dimensions))));
      Ada.Text_IO.New_Line(File => Table_File, Spacing => 2);


      -- Write out all the dependents with their independent values
      for Line in 1..Table.Offset(1) * Table.Count(1) loop

         Ada.Text_IO.Put_Line
           (File => Table_File,
            Item => CSV.Index_Images
                      (Independent_Values
                         (Indices => Coordinates,
                          Table   => Table
                          )
                      ) & Float'Image (Table.Dependent(Line))
           );

         Increment (Indices => Coordinates,
                    Table   => Table);


      end loop;

      Ada.Text_IO.New_Line (File => Table_File);
      Ada.Text_IO.Close(Table_File);

   end Write_ASCII;

   --------------------------------------------------------------------------------
   -- Compute the interpolated value for the given input value base on the
   -- values read into the table.
   --
   -- The actual algorithm used is a piecewise linear interpolation. There is an
   -- explanation of this algorithm using hypercubes in the SDF. Scaling factors
   -- were added to support the general case of non-equal sides.
   --
   -- This routine is built for speed. But I do see two things that could be done
   -- to further speed it up if nessecary.
   -- o  Change the Table parameter to an "in out". The compiler may be able to
   --    do a much better job optimizing the code when it is not dealing with
   --    an aliased object. This would require the function to be changed to a
   --    a procedure though. Yuk!
   -- o  Change the "zone search algorithm". Currently it uses a kind of linear
   --    (although the top level is binary). A full binary search could speed
   --    things up when the correct zone is far from the last zone. However, if the
   --    zones don't change much (which is probably the case) then a binary
   --    search could actually slow it down.
   --------------------------------------------------------------------------------
   function Interpolate( Indices : in     Index_Vector;
                         Table   : access Instance ) return  Float is

      -- The input indices, or the closest available in the table's range
      Indices_In_Range : Index_Vector := Indices;

      Node_Index : Independent_Index_List;

      Interpolated_Value : Float := 0.0;

   begin

      -- Make sure the table has been initialized
      if (Table.Dimensions = 0) then
         raise Invalid_Table;
      end if;

      -- Find the Zone which the input vector lies in. The zone is defined by
      -- the indices of the lowest-valued independent in every dimension.
      Find_Zone_Indices (Independents     => Table.Independent,
                         Last_Independent => Table.Count,
                         Value            => Indices_In_Range,
                         Zone             => Table.Start_Zone);

      Node_Index := Table.Start_Zone;

      -- For every node in the bounding hypercube
      for Loops in 1..2**Indices'Last loop

         -- Increment the value by the size of the dependent at that node times
         -- the Basis Function for the table at that node.
         Interpolated_Value := Interpolated_Value +
           Table.Dependent (Dependent_Index (Table   => Table.all,
                                             Indices => Node_Index)) *
           Basis_Function (Table          => Table.all,
                           Node           => Node_Index,
                           Interior_Point => Indices_In_Range);

         -- Increment the Node_Index for the next iteration
         Next_Node (Indices => Node_Index,
                    Origin  => Table.Start_Zone);

      end loop;


      return Interpolated_Value;

   end Interpolate;

   --------------------------------------------------------------------------------
   -- Procedure to read a table list from a table object file.
   -- It raises CONSTRAINT_ERROR if the list saved in the object file is not of
   -- the same type as the given Table.
   --------------------------------------------------------------------------------
   procedure Read ( File_Name : in     String;
                    Table     :    out Table_List) is
      File : Ada.Streams.Stream_Io.File_Type;
   begin

--      Ada.Streams.Stream_Io.Open
      Open_Stream
        (File => File,
         Name => File_Name,
         Mode => Ada.Streams.Stream_Io.In_File
         );

      -- Verify that the tag matches Instance
      declare
         Tag : constant String := Read_String (Ada.Streams.Stream_Io.Stream(File));
      begin
         if Machine_Independent_Tag (Table(Table'First)) /= Tag then

            Ada.Streams.Stream_Io.Close (File);
            Ada.Exceptions.Raise_Exception
              (Constraint_Error'Identity, "Table tags do not match." &
               Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
               "Expected " & Machine_Independent_Tag (Table(Table'First)) &
               " but file " & File_Name & " contains " & Tag & "."
               );
         end if;
      end;

      -- Read in all the table objects
      for Index in Table'Range loop
         Instance'Read (Ada.Streams.Stream_Io.Stream(File), Table(Index));
         Ada.Streams.Stream_Io.Close (File);
      end loop;

   exception
   when Error : Ada.Streams.Stream_Io.Name_Error =>
      Ada.Exceptions.Raise_Exception
        (Ada.Exceptions.Exception_Identity(Error),
         Ada.Exceptions.Exception_Message(Error) &
         Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
         "File " & '"' & File_Name & '"' &
         " not found");

   end Read;

   --------------------------------------------------------------------------------
   -- Procedure to write a table list into a table object file.
   -- It saves the instance's tag into the file.
   --------------------------------------------------------------------------------
   procedure Write ( File_Name   : in     String;
                     Table       : in     Table_List) is
      File : Ada.Streams.Stream_Io.File_Type;
   begin

      Ada.Streams.Stream_Io.Create
        (File => File,
         Name => File_Name
         );

      -- Write out the tag of Instance
      Write_String
        (Value       => Machine_Independent_Tag (Table(Table'First)),
         Destination => Ada.Streams.Stream_Io.Stream(File)
         );

      -- Write out all the table objects
      for Index in Table'Range loop
         Instance'Write (Ada.Streams.Stream_Io.Stream(File), Table(Index));
      end loop;

      Ada.Streams.Stream_Io.Close (File);

   end Write;

   --------------------------------------------------------------------------------
   -- Abstract routine to retrieve a machine-independent tag for the given
   -- instance.
   -- The scheme used to prevent clashes is to return the fully-qualified name of
   -- the type.
   --------------------------------------------------------------------------------
   function Machine_Independent_Tag ( Table : in Instance) return String is
   begin
      return "interpolation_table.multiply_indexed.instance";
   end Machine_Independent_Tag;

   --------------------------------------------------------------------------------
   -- Primitive routine to write an instance to a stream. This routine should
   -- write in a somewhat portable and storage-efficent manner.
   --
   -- Note that we *should* be able to write out entire arrays at once instead of
   -- using loops like is done below. Unfonrtunately GreenHills 1.8.9B has a bug
   -- where it writes arrays backwards.
   --------------------------------------------------------------------------------
   procedure Write ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                     Item   : in     Instance) is
   begin
      -- Write the dimensions, independent counts, offsets, and start zones.
      Integer'Write(Stream, Item.Dimensions);

      for Dimension in Item.Count'First .. Item.Count'First + Item.Dimensions - 1 loop
         Integer'Write (Stream,Item.Count(Dimension));
      end loop;

      for Dimension in Item.Offset'First .. Item.Offset'First + Item.Dimensions - 1 loop
         Integer'Write (Stream,Item.Offset(Dimension));
      end loop;

      for
        Dimension in Item.Start_Zone'First .. Item.Start_Zone'First + Item.Dimensions - 1
      loop
         Integer'Write(Stream, Item.Start_Zone(Dimension));
      end loop;

      -- Write the independents, dependents, and slopes. Only write valid bytes (not the
      -- entire array). In the case of the slope, that's one less than for the other
      -- arrays.

      -- Write the independents
      for
        Dimension in Item.Independent'First..Item.Independent'First + Item.Dimensions - 1
      loop
         for
           Index in Item.Independent(Dimension)'First..Item.Independent(Dimension)'First
           + Item.Count(Dimension) - 1
         loop
            Float'Write(Stream,Item.Independent(Dimension)(Index));
         end loop;
      end loop;

      -- Write the dependents
      for Index in Item.Dependent'First..Item.Dependent'First +
        (Item.Offset(Item.Offset'First) * Item.Count(Item.Count'First)) - 1
      loop
         Float'Write(Stream,Item.Dependent(Index));
      end loop;

      -- Write the deltas
      for
        Dimension in Item.Index_Delta'First..Item.Index_Delta'First + Item.Dimensions - 1
      loop
         for
           Index in Item.Index_Delta(Dimension)'First..Item.Index_Delta(Dimension)'First
           + Item.Count(Dimension) - 2
         loop
            Float'Write(Stream,Item.Index_Delta(Dimension)(Index));
         end loop;
      end loop;

   end Write;

   --------------------------------------------------------------------------------
   -- Primitive routine to read an instance from a stream. This routine should
   -- be able to read the output of the Write command above.
   --------------------------------------------------------------------------------
   procedure Read ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                    Item   :    out Instance) is
   begin

      -- Read the dimensions, independent counts, offsets, and start zones.
      Integer'Read(Stream, Item.Dimensions);

      for Dimension in Item.Count'First .. Item.Count'First + Item.Dimensions - 1 loop
         Integer'Read (Stream,Item.Count(Dimension));
      end loop;

      for Dimension in Item.Offset'First .. Item.Offset'First + Item.Dimensions - 1 loop
         Integer'Read (Stream,Item.Offset(Dimension));
      end loop;

      for
        Dimension in Item.Start_Zone'First .. Item.Start_Zone'First + Item.Dimensions - 1
      loop
         Integer'Read(Stream, Item.Start_Zone(Dimension));
      end loop;

      -- Read the independents, dependents, and slopes. Only read valid bytes (not the
      -- entire array). In the case of the slope, that's one less than for the other
      -- arrays.

      -- Read the independents
      for
        Dimension in Item.Independent'First..Item.Independent'First + Item.Dimensions - 1
      loop
         for
           Index in Item.Independent(Dimension)'First..Item.Independent(Dimension)'First
           + Item.Count(Dimension) - 1
         loop
            Float'Read(Stream,Item.Independent(Dimension)(Index));
         end loop;
      end loop;

      -- Read the dependents
      for Index in Item.Dependent'First..Item.Dependent'First +
        (Item.Offset(Item.Offset'First) * Item.Count(Item.Count'First)) - 1
      loop
         Float'Read(Stream,Item.Dependent(Index));
      end loop;

      -- Read the deltas
      for
        Dimension in Item.Index_Delta'First..Item.Index_Delta'First + Item.Dimensions - 1
      loop
         for
           Index in Item.Index_Delta(Dimension)'First..Item.Index_Delta(Dimension)'First
           + Item.Count(Dimension) - 2
         loop
            Float'Read(Stream,Item.Index_Delta(Dimension)(Index));
         end loop;
      end loop;

   end Read;

end Interpolation_Table.Multiply_Indexed;
