-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Interpolation_Table;

--------------------------------------------------------------------------------
-- This package implements a class of double variable lookup tables.
-- (ie. lookups involve two independent variables). Its can return multiple
-- dependent values during each call.  This is a high speed lookup routine
-- that computes all slopes during datafile initialization. In addition, this
-- class is capable of remembering the zones from previous calls.
-- Also, the method will post error messages during initialization
-- if the independent variables are not monotonically increasing.
--
-- User's should be aware that this lookup method consumes large quantities
-- of memory.  In addition, saving the data on the slopes for the table
-- adds to the memory requirements.  For these reasons, this routine should
-- only be used in high-frequency software.  Slower software should use a
-- routine that makes more efficient use of memory.
--------------------------------------------------------------------------------

package Interpolation_Table.Doubly_Indexed is

   type Instance is new Interpolation_Table.Instance with private;

   type Link is access Instance;

   type Table_List is array (Table_Index_Range range <>) of aliased Instance;

   --------------------------------------------------------------------------------
   -- Return the dependent value based on the input value.
   --------------------------------------------------------------------------------
   function Interpolate( X,Y   : in     Float;
                         Table : access Instance )
             return  Float;

   --------------------------------------------------------------------------------
   -- Return the value of the first (lowest) and last (highest) independents.
   --------------------------------------------------------------------------------
   function First_X_Independent (Table : in Instance) return Float;
   function First_Y_Independent (Table : in Instance) return Float;
   function Last_X_Independent  (Table : in Instance) return Float;
   function Last_Y_Independent  (Table : in Instance) return Float;

   --------------------------------------------------------------------------------
   -- Read in a number of tables from an ASCII file. The number of tables
   -- available in the file must match the size of the Table array.
   --------------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        :    out Table_List);

   --------------------------------------------------------------------------------
   -- Procedure to read a table list from a table object file.
   -- It raises CONSTRAINT_ERROR if the list saved in the object file is not of
   -- the same type as the given Table.
   --------------------------------------------------------------------------------
   procedure Read ( File_Name : in     String;
                    Table     :    out Table_List);

   --------------------------------------------------------------------------------
   -- Procedure to write a table list into a table object file.
   -- It saves the instance's tag into the file.
   --------------------------------------------------------------------------------
   procedure Write ( File_Name   : in     String;
                     Table       : in     Table_List);

private
   -----------------------
   -- Tunable constants --
   -----------------------

   -- Maximum number of independents in any one dimension. This overrides the value in
   -- the parent package to keep down the size of the Instances.
   Max_Independents : constant := 150;

   ------------------------------
   -- (end of tunable section) --
   ------------------------------


                  ----------------------------------
                  -- Stream I/O override routines --
                  ----------------------------------

   --------------------------------------------------------------------------------
   -- Primitive routine to write an instance to a stream. This routine should
   -- write in a somewhat portable and storage-efficent manner.
   --------------------------------------------------------------------------------
   procedure Write ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                     Item   : in     Instance);

   --------------------------------------------------------------------------------
   -- Primitive routine to read an instance from a stream. This routine should
   -- be able to read the output of the Write command above.
   --------------------------------------------------------------------------------
   procedure Read ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                    Item   :    out Instance);



                       -----------------------
                       -- The instance type --
                       -----------------------

   -- Independant and dependent variable Array types
   subtype Independent_Index is Positive range 1..Max_Independents;

   subtype Full_Independent_List is Independent_List(Independent_Index);

   type Value_Matrix is array (Independent_Index) of Full_Independent_List;

   -- Definintion of an instance of a doubly-indexed table object
   type Instance is new Interpolation_Table.Instance with
   record
      -- Cache for holding the last zone used
      Start_X_Zone : Independent_Index := Independent_Index'First;
      Start_Y_Zone : Independent_Index := Independent_Index'First;

      -- Current number of dependent and independent variables
      X_Count : Integer := 0;
      Y_Count : Integer := 0;

      -- Independent and dependent varaible lists
      Independent_X : Full_Independent_List;
      Independent_Y : Full_Independent_List;
      Dependent     : Value_Matrix;

      -- Intermediate calculations stored here to speed up calculation of
      -- final interpolated value.
      Slope_X       : Value_Matrix;
      Slope_Y       : Value_Matrix;
      Delta_X       : Full_Independent_List;
      Delta_Y       : Full_Independent_List;

   end record;

                   -------------------------------------
                   -- Overriding of inherited methods --
                   -------------------------------------

   for Instance'Write use Write;
   for Instance'Read use Read;

   --------------------------------------------------------------------------------
   -- Read in the single point table from an ASCII file, table pointer will
   -- be set to the new table if no errors are encountered.
   --
   -- To perform binary reads and writes, simply use the stream facilities
   -- (eg: Interpolation_Table.Instance'Read)
   --
   -- Exceptions: Text_IO exceptions.
   --------------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        : in out Instance);

   --------------------------------------------------------------------------------
   -- Write out the single point table to an ASCII file,
   --
   -- Binary reads and writes should be used in preference to the ASCII ones, as
   -- they should be much faster. To perform binary reads and writes, simply use
   -- the stream facilities (eg: Interpolation_Table.Instance'Read)
   --
   -- Exceptions: Text_IO exceptions.
   --
   --------------------------------------------------------------------------------
   procedure Write_ASCII( File_Name   : in     String;
                          Table       : in     Instance);

   --------------------------------------------------------------------------------
   -- Abstract routine to retrieve a machine-independent tag for the given
   -- instance.
   --------------------------------------------------------------------------------
   function Machine_Independent_Tag ( Table : in Instance) return String;

end Interpolation_Table.Doubly_Indexed;
