-------------------------------------------------------------------------------
--
--           Flightsafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Text_IO;
with Ada.Exceptions;
with Ada.Strings.Unbounded;  -- Dynamic strings (for error handling only)
with Interpolation_Table.CSV_Parsing;
with Ada.Streams.Stream_Io;
with Ada.Tags;
with Ada.Characters.Latin_1;

use type Ada.Strings.Unbounded.Unbounded_String;

--------------------------------------------------------------------------------
-- This package is implemented as a high speed lookup table of a two
-- dimensional array of independent and dependent pairs.  To operate the
-- user provides an ASCII text file of number pairs, compiles the
-- ASCII file which will test for syntax and content errors, if successful
-- will store the resulting data in a binary datafile.  Then during
-- normal operation the user calls the Interpolate routine with an independent
-- input value which returns an interpolated dependent value based on the
-- values stored in the table.
--
-- The actual algorithm used is a piecewise linear interpolation algorithm.
--------------------------------------------------------------------------------
package body Interpolation_Table.Doubly_Indexed is

   Invalid_Table : exception;

                          -----------------------
                          -- Local subprograms --
                          -----------------------

   --------------------------------------------------------------------------------
   -- Pre-compute the slope values between successive dependent values, this
   -- is so run time computation of the resulting interpolated values can
   -- run as fast as possible.
   --
   -- See Extrapolate below to see exactly how this slope is used.
   --------------------------------------------------------------------------------
   procedure  Calculate_Slopes (Table  : in out Instance) is

   begin

      -- Precalculate and verify the Y diffs to save some time
      for Y in 1 .. Table.Y_Count-1 loop

         -- calculate slopes for the independent variable.
         Table.Delta_Y(Y)  :=  Table.Independent_Y (Y+1) - Table.Independent_Y(Y);

         if Table.Delta_Y(Y) <=  0.0  then
            -- the independend values must be in increasing order, if not
            -- raise Invalid_Table.
            Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
                                            "independent values must be monotonically increasing");
            raise Invalid_Table;

         end if;

      end loop;

      for X in  1  ..  Table.X_Count  loop

         if X < Table.X_Count then
            -- calculate slopes for the independent variable.
            Table.Delta_X(X) := Table.Independent_X (X+1) - Table.Independent_X(X);

            if Table.Delta_X(X)  <=  0.0  then
               -- the independend values must be in increasing order, if not
               -- raise Invalid_Table.
               Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
                 "independent values must be monotonically increasing");
            end if;

         end if;

         for Y in  1  ..  Table.Y_Count  loop

            -- for each set of point value, pre calculate the
            -- slope values, this is done to speed up the run time
            -- calculation.
            -- The calculation is
            --    Slope  =  (dependent+1    -  dependent)
            if X < Table.X_Count then
               Table.Slope_X (X)(Y) :=
                 (Table.Dependent(X+1)(Y) - Table.Dependent(X)(Y));
            end if;

            if Y < Table.Y_Count then
               Table.Slope_Y (X)(Y) :=
                 (Table.Dependent(X)(Y+1) - Table.Dependent(X)(Y));

            end if;
         end loop;
      end loop;

   end Calculate_Slopes;

   -----------------------------------------------------------------------------
   -- Find the index of the zone in which the given value lies. The search is
   -- structured so that the zone indicated by "Old_Zone" is checked first. If
   -- the value is outside the given list, the zone at the closest edge of the
   -- list is returned. In that case the Value is clamped down to the independent
   -- value at the edge as well.
   --
   -- This routine is called twice from the interpolation routine, which itself
   -- liable to be called multiple times every cycle by 60Hz real-time processes.
   -- Thus, the speed of this routine is very important.
   --
   -- It may be possible to speed this routine up by changing the "zone search
   -- algorithm". Currently it uses a kind of linear search (although the top
   -- level is binary). A full binary search could speed things up when the
   -- correct zone is far from the last zone. However, if the zones don't change
   -- much (which is probably the case) then a binary search could actually
   -- slow it down.
   -----------------------------------------------------------------------------
   procedure Zone_Index
     (Independents     : in     Independent_List;
      Value            : in out Float;
      Old_Zone         : in     Integer;
      New_Zone         :    out Integer) is

   begin

      if ((Value >=  Independents(Old_Zone) and then
          (Value <=  Independents(Old_Zone+1) )))
      then
         -- input value falls within range of the previous call's current zone, calculate the
         -- return value.
         New_Zone := Old_Zone;

      elsif (Value >  Independents(Independents'Last)) then
         -- Input value is greater than any in the table. Calculate from
         -- the values at the top of the table.
         New_Zone  :=  Independents'Last - 1;
         Value     :=  Independents(Independents'Last);

      elsif (Value <  Independents(Independents'First)) then
         -- Input value is smaller than any in the table. Calculate from the
         -- values at the bottom of the table.
         New_Zone := Independents'First;
         Value    := Independents(Independents'First);

      elsif  Value <  Independents(Old_Zone)  then
         -- Input value is smaller than any in the previous current zone. Find
         -- the smaller zone it fits in and calculate from there.
         New_Zone := Old_Zone - 1;
         while Value < Independents(New_Zone) loop
            New_Zone := New_Zone - 1;
         end loop;

      elsif (Value > Independents(Old_Zone+1)) then
         -- Input value is larger than any in the previous current zone. Find
         -- the larger zone it fits in and calculate from there
         New_Zone := Old_Zone + 1;
         while Value > Independents(New_Zone+1) loop
            New_Zone := New_Zone + 1;
         end loop;

      end if;

   end Zone_Index;

                        ---------------------
                        -- Public Routines --
                        ---------------------

   --------------------------------------------------------------------------------
   -- Return the value of the first (lowest) and last (highest) independents.
   --------------------------------------------------------------------------------
   function First_X_Independent (Table : in Instance) return Float is
   begin
      return Table.Independent_X(Full_Independent_List'First);
   end First_X_Independent;

   function First_Y_Independent (Table : in Instance) return Float is
   begin
      return Table.Independent_Y(Full_Independent_List'First);
   end First_Y_Independent;

   function Last_X_Independent (Table : in Instance) return Float is
   begin
      return Table.Independent_X(Table.X_Count);
   end Last_X_Independent;

   function Last_Y_Independent (Table : in Instance) return Float is
   begin
      return Table.Independent_Y(Table.Y_Count);
   end Last_Y_Independent;

   -----------------------------------------------------------------------------
   -- Interpolate the value at the given points from the given table from the
   -- given Start coordinates in the table.
   --
   -- The algorithm used is a variant of the multilinear interpolation
   -- algorithm. It looks a bit different than the method used in the
   -- multiply_indexed package, but they are mathematicly equivalent in the
   -- two-dimensional case. But this formulation breaks the algorithm into
   -- multiple stages, and allows more terms to be precalculated, so it
   -- should be quicker. The multiple formuation requires about 24
   -- floating-point operations in the two-dimensional case, whereas this
   -- routine performs only 8.
   -----------------------------------------------------------------------------
   function Interpolate
     (Input_X : in Float;
      Input_Y : in Float;
      X_Start : in Integer;
      Y_Start : in Integer;
      Table   : in Instance
     ) return Float is

      ---------------------------------------------------------------------
      -- The following constants represent the input X and Y values
      -- normalized into [0,1] space, where 0 and 1 are represented by the
      -- surrounding independent values.
      Normalized_X : constant Float :=
        (Input_X - Table.Independent_X (X_Start)) / Table.Delta_X (X_Start);

      Normalized_Y : constant Float :=
        (Input_Y - Table.Independent_Y (Y_Start)) / Table.Delta_Y (Y_Start);

      ---------------------------------------------------------------------
      -- The following constant represents a one-dimensional
      -- interpolatation based on the values at (X,Y) and (X+1,Y)
      Low_X_Component : constant Float :=
        ( Normalized_X * Table.Slope_X (X_Start)(Y_Start) ) +
        Table.Dependent(X_Start)(Y_Start);

      ---------------------------------------------------------------------
      -- The following constant represents a one-dimensional
      -- interpolatation based on the values at (X,Y+1) and (X+1,Y+1)
      High_X_Component : constant Float :=
        ( Normalized_X * Table.Slope_X (X_Start)(Y_Start + 1) ) +
        Table.Dependent(X_Start)(Y_Start + 1);


   begin

      ---------------------------------------------------------------------
      -- Return a one-dimensional interpolation in the Y direction between
      -- the two X components.
      return ( (High_X_Component - Low_X_Component) * Normalized_Y ) + Low_X_Component;

   end Interpolate;

   -----------------------------------------------------------------------------
   -- Read in the double point ASCII table and store in the Table list.
   -----------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        :    out Table_List) is

      -- The CSV parser pacakge
      package CSV is new Interpolation_Table.CSV_Parsing
        (Max_Dependents   => Table(1).Dependent'length * Table(1).Dependent(1)'length,
         Max_Dimensions   => 2);

      -- The lines of the text file that were read in
      Source_Lines    : CSV.CSV_Line_List;
      Number_Of_Lines : Integer;

      -- The names of the dimension headers
      Header : CSV.Dimension_Strings := (others => Ada.Strings.Unbounded.Null_Unbounded_String);
      Number_Of_Headers : Integer;

      Total_Dependents : Integer := 0;

      Dimensions : Integer := 2;

      --------------------------------------------------------------------------------
      -- Load all the dependent values for all the independents starting at the given
      -- dimension. If there is an entry missing, a message indicating the missing
      -- entry will be tacked onto the end of Missing_Dependents.
      --
      -- The algorithm used to match up input rows with expected independent values
      -- depends on the Rows being sorted so that their independents are in
      -- lexigraphical order.
      --------------------------------------------------------------------------------
      procedure Load_CSV_Dependents (Table       : in out Table_List;
                                     Num_Rows    : in     Positive;
                                     Sorted_Rows : in     CSV.CSV_Line_List) is

         -- The current indices of the expected independent values
         X_Index : Integer := 1;
         Y_Index : Integer := 1;

         -- The current index of the dependent
         Dependent_Index : Integer := 1;

         Error_Message : Ada.Strings.Unbounded.Unbounded_String
           := Ada.Strings.Unbounded.Null_Unbounded_String;
         Error_Length : Integer := 0;

         Dependent : Csv.Dependent_List;

      begin

         -- This loop runs through every combination of independent indices in the sorted order
         -- and copies their dependent from the sorted rows into the table.
         while X_Index <= Table(1).X_Count loop

            -- Attempt to read a dependent from the row with the specified independents.
            CSV.Get_Expected_Dependents
              (Independent => (Table(1).Independent_X(X_Index), Table(1).Independent_Y(Y_Index)),
               Sorted_Row  => Sorted_Rows(Dependent_Index),
               Dependent   => Dependent,
               Error       => Error_Message
              );

            -- If there was no error, copy the dependents into their tables
            if Error_Length = Ada.Strings.Unbounded.Length(Error_Message) then
               Dependent_Index := Dependent_Index + 1;

               -- Copy the dependents into their tables
               for Table_Index in 1..Table'Length loop
                  Table(Table_Index).Dependent(X_Index)(Y_Index) := Dependent(Table_Index);
               end loop;

            end if;
            Error_Length := Ada.Strings.Unbounded.Length(Error_Message);

            -- Increment the independent indices by one
            if Y_Index = Table(1).Y_Count then
               X_Index := X_Index + 1;
               Y_Index := 1;
            else
               Y_Index := Y_Index + 1;
            end if;

         end loop;

         -- If there were missing rows, raise and exception and report the error.
         if Error_Message /= Ada.Strings.Unbounded.Null_Unbounded_String then
            Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
              Ada.Strings.Unbounded.To_String(Error_Message));
         end if;

      end Load_CSV_Dependents;


   begin

      -- Parse the table file into CSV lines
      CSV.Parse
        (File_Name       => File_Name,
         Name_List       => Header,
         Number_Of_Names => Number_Of_Headers,
         Dimensions      => Dimensions,
         Dependents      => Table'Length,
         Num_Rows        => Number_Of_Lines,
         CSV_Rows        => Source_Lines);

      -- Load in the independents for both the dimensions
      CSV.Load_Independents (File_Name        => File_Name,
                             Rows             => Source_Lines,
                             Num_Rows         => Number_Of_Lines,
                             Dimension        => 1,
                             Independents     => Table(1).Independent_X,
                             Num_Independents => Table(1).X_Count);

      CSV.Load_Independents (File_Name        => File_Name,
                             Rows             => Source_Lines,
                             Num_Rows         => Number_Of_Lines,
                             Dimension        => 2,
                             Independents     => Table(1).Independent_Y,
                             Num_Independents => Table(1).Y_Count);

      -- Copy over the independents to the other tables
      for Table_Index in 2..Table'Length loop

         Table(Table_Index).Independent_X (1..Table(1).X_Count) :=
           Table(1).Independent_X (1..Table(1).X_Count);
         Table(Table_Index).X_Count := Table(1).X_Count;

         Table(Table_Index).Independent_Y (1..Table(1).Y_Count) :=
           Table(1).Independent_Y (1..Table(1).Y_Count);
         Table(Table_Index).Y_Count := Table(1).Y_Count;

      end loop;

      -- Load all the dependents into the table
      Load_CSV_Dependents
        (Table              => Table,
         Num_Rows           => Number_Of_Lines,
         Sorted_Rows        => Source_Lines);

      -- Precalculate the slopes, etc.
      for Table_Index in Table'range loop
         Calculate_Slopes(Table(Table_Index));
      end loop;

   end Read_ASCII;

   -----------------------------------------------------------------------------
   -- Read in the double point ASCII table and store in the Table.
   -----------------------------------------------------------------------------
   procedure Read_ASCII ( File_Name    : in     String;
                          Table        : in out Instance) is
      Single_Table : Table_List (1..1);
   begin

      Read_ASCII ( File_Name => File_Name,
                   Table     => Single_Table
                   );

      Table := Single_Table(1);

   end Read_ASCII;

   -----------------------------------------------------------------------------
   -- Write out the ASCII version of the double point table to a file.
   --
   -- It should be written in the same format expected by the Read_ANN
   -- routine.
   -----------------------------------------------------------------------------
   procedure Write_ASCII( File_Name   : in     String;
                          Table       : in     Instance) is


      Table_File     : Ada.Text_IO.File_Type;

   begin
      Ada.Text_IO.Create(File => Table_File,
                         Name => File_Name);

      -- Start with a comment naming the file, and listing the number of independents
      Ada.Text_IO.Put_Line(File => Table_File, Item => "# double point lookup data table " & File_Name);
      Ada.Text_IO.Put(File => Table_File, Item => "# Number of X Independents =" &
        Integer'Image(Table.X_Count) & ", Number of Y Independents =" &
        Integer'Image(Table.Y_Count));
      Ada.Text_IO.New_Line(File => Table_File, Spacing => 2);


      -- Write out all the dependents with their independent values
      for X_Coordinate in 1..Table.X_Count loop
         for Y_Coordinate in 1..Table.Y_Count loop

            Ada.Text_IO.Put_Line
              (File => Table_File,
               Item => Float'Image (Table.Independent_X(X_Coordinate)) & ", " &
                       Float'Image (Table.Independent_X(X_Coordinate)) & ", " &
                       Float'Image (Table.Dependent(X_Coordinate)(Y_Coordinate))
              );

         end loop;
      end loop;

      Ada.Text_IO.Close(Table_File);

   end Write_ASCII;

   --------------------------------------------------------------------------------
   -- Compute the interpolated value for the given input value base on the
   -- values read into the table.
   --
   -- The actual algorithm used is a piecewise linear interpolation. There is an
   -- explanation of this algorithm using hypercubes in the SDF. Scaling factors
   -- were added to support the general case of non-equal sides.
   --
   -- This routine is built for speed. But I do see two things that could be done
   -- to further speed it up if nessecary.
   -- o  Change the Table parameter to an "in out". The compiler may be able to
   --    do a much better job optimizing the code when it is not dealing with
   --    an aliased object. This would require the function to be changed to a
   --    a procedure though. Yuk!
   -- o  Change the "zone search algorithm". Currently it uses a kind of linear
   --    (although the top level is binary). A full binary search could speed
   --    things up when the correct zone is far from the last zone. However, if the
   --    zones don't change much (which is probably the case) then a binary
   --    search could actually slow it down.
   --------------------------------------------------------------------------------
   function Interpolate( X,Y   : in     Float;
                         Table : access Instance )
      return  Float is

      X_Independent : Float := X;
      Y_Independent : Float := Y;

   begin

      -- Make sure the table has been initialized
      if (Table.X_Count = 0 or Table.Y_Count = 0) then
         raise Invalid_Table;
      end if;

      -- Find the lower points of the enclosing zones for the given coordinates
      Zone_Index (Independents => Table.Independent_X(1..Table.X_Count),
                  Value        => X_Independent,
                  Old_Zone     => Table.Start_X_Zone,
                  New_Zone     => Table.Start_X_Zone);

      Zone_Index (Independents => Table.Independent_Y(1..Table.Y_Count),
                  Value        => Y_Independent,
                  Old_Zone     => Table.Start_Y_Zone,
                  New_Zone     => Table.Start_Y_Zone);

      return Interpolate
        (Input_X => X_Independent,
         Input_Y => Y_Independent,
         X_Start => Table.Start_X_Zone,
         Y_Start => Table.Start_Y_Zone,
         Table   => Table.all
         );

   end Interpolate;

   --------------------------------------------------------------------------------
   -- Procedure to read a table list from a table object file.
   -- It raises CONSTRAINT_ERROR if the list saved in the object file is not of
   -- the same type as the given Table.
   --------------------------------------------------------------------------------
   procedure Read ( File_Name : in     String;
                    Table     :    out Table_List) is
      File : Ada.Streams.Stream_Io.File_Type;
   begin

--      Ada.Streams.Stream_Io.Open
      Open_Stream
        (File => File,
         Name => File_Name,
         Mode => Ada.Streams.Stream_Io.In_File
         );

      -- Verify that the tag matches Instance
      declare
         Tag : constant String := Read_String (Ada.Streams.Stream_Io.Stream(File));
      begin
         if Machine_Independent_Tag (Table(Table'First)) /= Tag then

            Ada.Streams.Stream_Io.Close (File);
            Ada.Exceptions.Raise_Exception
              (Constraint_Error'Identity, "Table tags do not match." &
               Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
               "Expected " & Machine_Independent_Tag (Table(Table'First)) &
               " but file " & File_Name & " contains " & Tag & "."
               );
         end if;
      end;

      -- Read in all the table objects
      for Index in Table'Range loop
         Instance'Read (Ada.Streams.Stream_Io.Stream(File), Table(Index));
         Ada.Streams.Stream_Io.Close (File);
      end loop;

   exception
   when Error : Ada.Streams.Stream_Io.Name_Error =>
      Ada.Exceptions.Raise_Exception
        (Ada.Exceptions.Exception_Identity(Error),
         Ada.Exceptions.Exception_Message(Error) &
         Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
         "File " & '"' & File_Name & '"' &
         " not found");

   end Read;

   --------------------------------------------------------------------------------
   -- Procedure to write a table list into a table object file.
   -- It saves the instance's tag into the file.
   --------------------------------------------------------------------------------
   procedure Write ( File_Name   : in     String;
                     Table       : in     Table_List) is
      File : Ada.Streams.Stream_Io.File_Type;
   begin

      Ada.Streams.Stream_Io.Create
        (File => File,
         Name => File_Name
         );

      -- Write out the tag of Instance
      Write_String
        (Value       => Machine_Independent_Tag (Table(Table'First)),
         Destination => Ada.Streams.Stream_Io.Stream(File)
         );

       -- Write out all the table objects
      for Index in Table'Range loop
         Instance'Write (Ada.Streams.Stream_Io.Stream(File), Table(Index));
      end loop;

      Ada.Streams.Stream_Io.Close (File);

   end Write;

   --------------------------------------------------------------------------------
   -- Abstract routine to retrieve a machine-independent tag for the given
   -- instance.
   -- The scheme used to prevent clashes is to return the fully-qualified name of
   -- the type.
   --------------------------------------------------------------------------------
   function Machine_Independent_Tag ( Table : in Instance) return String is
   begin
      return "interpolation_table.doubly_indexed.instance";
   end Machine_Independent_Tag;

   --------------------------------------------------------------------------------
   -- Primitive routine to write an instance to a stream. This routine should
   -- write in a somewhat portable and storage-efficent manner.
   --
   -- Note that we *should* be able to write out entire arrays at once instead of
   -- using loops like is done below. Unfonrtunately GreenHills 1.8.9B has a bug
   -- where it writes arrays backwards.
   --------------------------------------------------------------------------------
   procedure Write ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                     Item   : in     Instance) is
   begin
      -- Write the start zones and independent counts
      Independent_Index'Write(Stream,Item.Start_X_Zone);
      Independent_Index'Write(Stream,Item.Start_Y_Zone);
      Integer'Write(Stream, Item.X_Count);
      Integer'Write(Stream, Item.Y_Count);

      -- Write the independents, dependents, and slopes. Only write valid bytes (not the
      -- entire array). In the case of the slope, that's one less than for the other
      -- arrays.

      -- Write the independents
      for Index in Item.Independent_X'First..Item.Independent_X'First + Item.X_Count - 1
      loop
         Float'Write(Stream,Item.Independent_X(Index));
      end loop;
      for Index in Item.Independent_Y'First..Item.Independent_Y'First + Item.Y_Count - 1
      loop
         Float'Write(Stream,Item.Independent_Y(Index));
      end loop;

      -- Write the dependents
      for X_Index in Item.Dependent'First..Item.Dependent'First + Item.X_Count - 1
      loop
         for Y_Index in Item.Dependent(X_Index)'First..Item.Dependent(X_Index)'First +
           Item.Y_Count - 1
         loop
            Float'Write(Stream,Item.Dependent(X_Index)(Y_Index));
         end loop;
      end loop;

      -- Write the slopes
      for X_Index in Item.Slope_X'First..Item.Slope_X'First + Item.X_Count - 2
      loop
         for Y_Index in Item.Slope_X(X_Index)'First..Item.Slope_X(X_Index)'First +
           Item.Y_Count - 1
         loop
            Float'Write(Stream,Item.Slope_X(X_Index)(Y_Index));
         end loop;
      end loop;
      for X_Index in Item.Slope_Y'First..Item.Slope_Y'First + Item.X_Count - 1
      loop
         for Y_Index in Item.Slope_Y(X_Index)'First..Item.Slope_Y(X_Index)'First +
           Item.Y_Count - 2
         loop
            Float'Write(Stream,Item.Slope_Y(X_Index)(Y_Index));
         end loop;
      end loop;

      -- Write the deltas
      for Index in Item.Delta_X'First..Item.Delta_X'First + Item.X_Count - 2 loop
         Float'Write(Stream,Item.Delta_X(Index));
      end loop;
      for Index in Item.Delta_Y'First..Item.Delta_Y'First + Item.Y_Count - 2
      loop
         Float'Write(Stream,Item.Delta_Y(Index));
      end loop;

   end Write;

   --------------------------------------------------------------------------------
   -- Primitive routine to read an instance from a stream. This routine should
   -- be able to read the output of the Write command above.
   --------------------------------------------------------------------------------
   procedure Read ( Stream : access Ada.Streams.Root_Stream_Type'Class;
                    Item   :    out Instance) is
   begin

      -- Read the start zones and independent counts
      Independent_Index'Read(Stream,Item.Start_X_Zone);
      Independent_Index'Read(Stream,Item.Start_Y_Zone);
      Integer'Read(Stream, Item.X_Count);
      Integer'Read(Stream, Item.Y_Count);

      -- Read the independents, dependents, and slopes. Only read valid bytes (not the
      -- entire array). In the case of the slope, that's one less than for the other
      -- arrays.

      -- Read the independents
      for Index in Item.Independent_X'First..Item.Independent_X'First + Item.X_Count - 1
      loop
         Float'Read(Stream,Item.Independent_X(Index));
      end loop;
      for Index in Item.Independent_Y'First..Item.Independent_Y'First + Item.Y_Count - 1
      loop
         Float'Read(Stream,Item.Independent_Y(Index));
      end loop;

      -- Read the dependents
      for X_Index in Item.Dependent'First..Item.Dependent'First + Item.X_Count - 1
      loop
         for Y_Index in Item.Dependent(X_Index)'First..Item.Dependent(X_Index)'First +
           Item.Y_Count - 1
         loop
            Float'Read(Stream,Item.Dependent(X_Index)(Y_Index));
         end loop;
      end loop;

      -- Read the slopes
      for X_Index in Item.Slope_X'First..Item.Slope_X'First + Item.X_Count - 2
      loop
         for Y_Index in Item.Slope_X(X_Index)'First..Item.Slope_X(X_Index)'First +
           Item.Y_Count - 1
         loop
            Float'Read(Stream,Item.Slope_X(X_Index)(Y_Index));
         end loop;
      end loop;
      for X_Index in Item.Slope_Y'First..Item.Slope_Y'First + Item.X_Count - 1
      loop
         for Y_Index in Item.Slope_Y(X_Index)'First..Item.Slope_Y(X_Index)'First +
           Item.Y_Count - 2
         loop
            Float'Read(Stream,Item.Slope_Y(X_Index)(Y_Index));
         end loop;
      end loop;

      -- Read the deltas
      for Index in Item.Delta_X'First..Item.Delta_X'First + Item.X_Count - 2 loop
         Float'Read(Stream,Item.Delta_X(Index));
      end loop;
      for Index in Item.Delta_Y'First..Item.Delta_Y'First + Item.Y_Count - 2
      loop
         Float'Read(Stream,Item.Delta_Y(Index));
      end loop;

   end Read;

end Interpolation_Table.Doubly_Indexed;

