-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Strings.Unbounded;  -- Used only for fatal error messages
with Token.Analyzer;

--------------------------------------------------------------------------------
-- This package implements a some common comma separated value (CSV) parsing
-- utilities for interpolation tables. It is intended to be used only from the
-- bodies of the interpolation_table family of packages.
--
-- The main reason this package was made generic is so that the resultant parsing
-- code can be re-entrant. See the comment in the package body for further
-- information.
--------------------------------------------------------------------------------
generic
   Max_Dependents   : in Positive;
   Max_Dimensions   : in Positive;
package Interpolation_Table.CSV_Parsing is

   -- The names of the columns in the CSV file (used only for error reporting during reads).
   type Dimension_Strings is array (1..Max_Indices) of Ada.Strings.Unbounded.Unbounded_String;

   -- The different tokens in an ASCII table file
   -- The Identifier token will match the same string as the Real token, so it
   -- needs to be listed after the real token.
   type CSV_Token_ID is (Real, Comma, Comment, Identifier, Whitespace, EOF);

   -- Information on lines of data read in from a CSV text file
   subtype Independent_Coordinates is Index_Vector (1..Max_Dimensions);

   type Float_List is array (Positive range <>) of Float;
   subtype Dependent_List is Float_List (1..Max_Tables);

   type CSV_Line is record
      Independent : Independent_Coordinates;
      Dependent   : Dependent_List;
      Line        : Integer;
   end record;
   type CSV_Line_List is array (1..Max_Dependents) of CSV_Line;

   --------------------------------------------------------------------------------
   -- Return the string value of the independent values at the given indices, in
   -- comma separated form (with a trailing comma).
   --------------------------------------------------------------------------------
   function Index_Images (Indices : in Index_Vector) return String;

   -----------------------------------------------------------------------------
   -- Parse the given CSV file for the given index of dependents. (1 signifies
   -- the first dependents on the row, 2 the second, etc.) The names of the
   -- columns, the number of dimensions, then number of dependents, and all the
   -- CSV rows are returned.
   -----------------------------------------------------------------------------
   procedure Parse (File_Name       : in     String;
                    Name_List       :    out Dimension_Strings;
                    Number_Of_Names :    out Integer;
                    Dimensions      : in out Natural;
                    Dependents      : in     Positive := 1;
                    Num_Rows        :    out Positive;
                    CSV_Rows        :    out CSV_Line_List);

   -----------------------------------------------------------------------------
   -- Load the given independent list with values from the given dimension of
   -- all the given CSV Lines. The list returned will contain unique vaues in
   -- increasing order.
   -----------------------------------------------------------------------------
   procedure Load_Independents (File_Name        : in     String;
                                Rows             : in     CSV_Line_List;
                                Num_Rows         : in     Positive;
                                Dimension        : in     Positive;
                                Independents     :    out Independent_List;
                                Num_Independents :    out Natural);

   -----------------------------------------------------------------------------
   -- If the given CSV row contains the given independent values, get its
   -- its dependent value. If not, the error message is annotated with the expected
   -- values for the missing row.
   -- Once you have a list of unique sorted independents (via Load_Independents
   -- above), it is a simple matter to predict and pass in the expected
   -- independent value for each row.
   -----------------------------------------------------------------------------
   procedure Get_Expected_Dependents (Independent : in     Index_Vector;
                                      Sorted_Row  : in     CSV_Line;
                                      Dependent   :    out Dependent_List;
                                      Error       : in out Ada.Strings.Unbounded.Unbounded_String);



end Interpolation_Table.CSV_Parsing;



