-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with Ada.Exceptions;       -- Custom exception messages
with Ada.Text_IO;
with Token;                -- Token analysis parent
with Token.Line_Comment;   -- Recognizer for Line-oriented comments
with Token.End_Of_File;    -- Recognizer for the end of a file
with Token.Real;           -- Recognizer for fixed and floating point real numbers
with Token.Keyword;        -- Recognizer for specified space-delimited words
with Token.Character_Set;  -- Recognizer for sequences of specified characters.
with Token.CSV_Field;      -- Recognizer for untyped CSV fields
with Ada.Characters.Latin_1;     -- CR and LF
with Ada.Characters.Handling;    -- Character handling
with Ada.Strings.Fixed;          -- String handling
with Ada.Strings.Maps.Constants;
with Log;                        -- Printing of warnings

--------------------------------------------------------------------------------
-- This package implements a some common comma separated value (CSV) parsing
-- utilities for interpolation tables. It is intended to be used only from the
-- bodies of the interpolation_table family of packages.
--
-- The reason this package has to be generic has to do with the way the Token
-- analysis packages work. They are generic packages, which declare a pointer to
-- token type. In order to use pointers without resorting to unchecked
-- programming, Ada dictates that the pointed-to object must not have a deeper
-- dynamic scope than the access type. But in order to make the parsing routines
-- task-safe, they must declare their own token and token analyzer locally. Thus
-- the token-analysis package must be instantiated locally as well. Since that
-- is done is the spec of this package, this package needs to be a generic so
-- that it may be instantiated locally.
--------------------------------------------------------------------------------
package body Interpolation_Table.CSV_Parsing is

   use type Ada.Strings.Unbounded.Unbounded_String;

   Syntax        : exception;
   Invalid_Table : exception;

   -- The different errors that can occur.
   type Syntax_Error_ID is
      (Comma_Expected, Missing_Real, Missing_Comma_Or_Real, Bad_Real,
       Number_Of_Dimensions_Changed, Too_Many_Dependents, Too_Many_Dimensions,
       No_Starting_Comment, Too_Many_Independents, Too_Few_Independents,
       Too_Few_Dimensions, Identifier_Expected);

   -- Global text file for reading parse data
   File : Ada.Text_Io.File_Type;

   -- Instantiate a table file tokenizer
   package Table_Tokenizer is new Token.Analyzer(CSV_Token_ID);

   -- The tokens
   Comment_Token     : aliased Token.Class := Token.Line_Comment.Get("#", True);
   EOF_Token         : aliased Token.Class := Token.End_Of_File.Get;
   Real_Literal      : aliased Token.Class := Token.Real.Get;
   Comma_Token       : aliased Token.Class := Token.Keyword.Get(",");
   Whitespace_Token  : aliased Token.Class := Token.Character_Set.Get
     (Token.Character_Set.Standard_Whitespace);
   Identifier_Token  : aliased Token.Class := Token.CSV_Field.Get;



   -- The specification of the syntax of an ASCII table
   Table_Syntax : Table_Tokenizer.Syntax :=
     (Real        => Real_Literal'Access,
      Comma       => Comma_Token'Access,
      Identifier  => Identifier_Token'Access,
      Comment     => Comment_Token'Access,
      Whitespace  => Whitespace_Token'Access,
      EOF         => EOF_Token'Access);

   function Get_Text_Line return String;

   type Text_Feeder is access function return String;
   Feeder : constant Text_Feeder := Get_Text_Line'Access;

   Tokenizer : Table_Tokenizer.Instance := Table_Tokenizer.Initialize
     (Language_Syntax => Table_Syntax,
      Feeder          => Get_Text_Line'access);

   --------------------------------------------------------------------------------
   -- Open a text file. If there is no file with the given name, a
   -- case-insensitive open will be attempted. If this succeeds, a warning message
   -- will be printed.
   --------------------------------------------------------------------------------
   procedure Open_Text_File
     ( File : in out Ada.Text_Io.File_Type;
       Name : in     String;
       Mode : in     Ada.Text_IO.File_Mode
     ) is

      Working_Name : String := Ada.Strings.Fixed.Translate
        (Source  => Name,
         Mapping => Ada.Strings.Maps.Constants.Lower_Case_Map
         );

      -----------------------------------------------------------------------------
      -- Attempt to open the given file with every permuation of its name from the
      -- given character index to the end.
      -----------------------------------------------------------------------------
      function Open_File_Permutation (Index : Natural) return Boolean is
      begin

         -- If this isn't the last character, return true if either case of this
         -- character combined with all case permuations of the following characters
         -- can open the stream file.
         if Index < Working_Name'Last then

            if Open_File_Permutation (Index + 1) then
               return True;
            else
               Working_Name (Index) := Ada.Characters.Handling.To_Upper(Working_Name(Index));
               if Open_File_Permutation (Index + 1) then
                  return True;
               end if;
            end if;

         -- If this is the last character, return true if either case of this
         -- character allows us to successfully open a file.
         else
            begin
               Ada.Text_IO.Open
                 (File => File,
                  Name => Working_Name,
                  Mode => Ada.Text_IO.In_File
                  );
               return True;
            exception
               when Ada.Text_IO.Name_Error =>
                  null;
            end;
            Working_Name (Index) := Ada.Characters.Handling.To_Upper(Working_Name(Index));
            begin
               Ada.Text_IO.Open
                 (File => File,
                  Name => Working_Name,
                  Mode => Ada.Text_IO.In_File
                  );
               return True;
            exception
               when Ada.Text_IO.Name_Error =>
                  null;
            end;
         end if;

         -- Failed to open the file. Put the character back to lower case and report the
         -- failure.
         Working_Name (Index) := Ada.Characters.Handling.To_Lower(Working_Name(Index));
         return False;

      end Open_File_Permutation;
   begin

      -- Attempt an open with the given name. If it succeeds, we are done.
      begin
         Ada.Text_IO.Open
           (File => File,
            Name => Name,
            Mode => Ada.Text_IO.In_File
            );
         return;
      exception
         when Ada.Text_IO.Name_Error =>
            null;
      end;


      -- Go through every possible capitalization of the given file name looking for
      -- a match
      if Open_File_Permutation (Working_Name'First) then
         Log.Report
           (Event    => "Table file """ & Name & """" &
            Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
            "     does not exist, so file """ &
            Working_Name & """ was used instead. " &
            Ada.Characters.Latin_1.Cr & Ada.Characters.Latin_1.Lf &
            "     Please change one or the other to match.",
            Severity => Log.Warning
            );
      else
         raise Ada.Text_IO.Name_Error;
      end if;
   end Open_Text_File;

   ------------------------------------------------------------------------------------
   -- The routine used to feed data into the token analizer. This routine reads from
   -- the analizer file one line at a time until EOF is reached.
   ------------------------------------------------------------------------------------
   function Get_Text_Line return String is
      Buffer : String(1..512);
      Data_Length : Integer;
   begin

      Ada.Text_IO.Get_Line (File => File, Item => Buffer, Last => Data_Length);

      if Ada.Text_IO.End_Of_File(File) then
         return Buffer(1..Data_Length) & Token.EOF_Character;
      else
         return Buffer(1..Data_Length) & Token.EOL_Character;
      end if;
   exception
      when Ada.Text_IO.End_Error =>
         return (1 => Token.EOF_Character);

   end Get_Text_Line;

   ------------------------------------------------------------------------------------
   -- Raise the given exception id, with an appropriate message based on the Syntax
   -- error's ID. Calling this routine causes an exception to be raised, so control
   -- will *not* return directly to the caller.
   ------------------------------------------------------------------------------------
   procedure Report_Error (ID        : in Syntax_Error_ID;
                           File_Name : in String;
                           Line      : in Integer := 0;
                           Dimension : in Integer := 0;
                           Bound     : in Integer := 0) is
   begin

      case ID is
      when Comma_Expected =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". Comma expected");

      when Missing_Real =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in table file" &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". Expected Float value, got " & '"' &
           Table_Tokenizer.Lexeme (Tokenizer)
           & '"');

      when Bad_Real =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". Expected Float value, got " & '"' &
           Table_Tokenizer.Lexeme (Tokenizer)
         & '"');

      when Number_Of_Dimensions_Changed =>
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity, "in table file " &
           File_Name & " before line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". " & Integer'Image(Bound) & " dimensions expected, " &
           Integer'Image(Dimension) & " read");

      when Missing_Comma_Or_Real =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". Float value or ',' expected");

      when Too_Many_Dimensions =>
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
       Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
       ". Too many dimensions (max is" & Integer'Image(Max_Dimensions) & ")");

      when Too_Few_Dimensions =>
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
       Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
       ". Too few dimensions (at least one is required)");

      when Too_Many_Dependents =>
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". Too many dependents (max is" & Integer'Image(Max_Dependents) & ")");

      when No_Starting_Comment =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in table file" &
           File_Name & " first line must be a comment." &
           " Read a " & CSV_Token_ID'Image(Table_Tokenizer.Token(Tokenizer)) &
           "instead");

      when Too_Many_Independents =>
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity, "in table file " &
           File_Name & " for dimension" &
           Integer'Image (Dimension) & " at line" & Integer'Image(Line) &
           ". Too many independents (max is" & Integer'Image(Bound) & ")");

      when Too_Few_Independents =>
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity, "in table file " &
           File_Name &
           " too few independents for dimension" & Integer'Image (Dimension) &
           " (must have at least" & Integer'Image(Bound) &
           " independent values in each dimension)");

      when Identifier_Expected =>
         Ada.Exceptions.Raise_Exception (Syntax'Identity, "in table file " &
           File_Name & " at line" &
           Integer'Image(Table_Tokenizer.Line(Tokenizer)) & ", column" &
           Integer'Image (Table_Tokenizer.Column(Tokenizer)) &
           ". Identifier expected for table header");
      end case;

   end Report_Error;

   -----------------------------------------------------------------------------
   -- Verify that the current token is a real value, and return that value.
   -----------------------------------------------------------------------------
   procedure Parse_Real (File_Name : in     String;
                         Tokenizer : in out Table_Tokenizer.Instance;
                         Value     :    out Float) is
   begin

      -- If its not a real, raise an error
      if Table_Tokenizer.Token(Tokenizer) /= Real then
         Report_Error (File_Name => File_Name,
                       ID        => Missing_Real);
      end if;

      Value := Float'Value(Table_Tokenizer.Lexeme(Tokenizer));

   exception
      when Ada.Text_IO.Data_Error =>
         Report_Error(File_Name => File_Name,
                      ID        => Bad_Real);
   end Parse_Real;

   -----------------------------------------------------------------------------
   -- Attempt to read in and validate a CSV row. The values for the row are
   -- returned, along with a flag indicating if there are more rows to read.
   -----------------------------------------------------------------------------
   procedure Parse_CSV_Row (File_Name     : in     String;
                            Tokenizer     : in out Table_Tokenizer.Instance;
                            Row           :    out CSV_Line;
                            Independents  :    out Integer;
                            Dependents    : in     Positive;
                            No_More_Rows  :    out Boolean) is

      subtype Parse_Value_List is Float_List (1..Max_Dimensions + Max_Indices);
      Value_List : Parse_Value_List;
      Value_List_Length : Natural := 1;
   begin

      -- Save off the line number of this row
      Row.Line := Table_Tokenizer.Line(Tokenizer);

      -- Read the first value on the row
      Parse_Real (File_Name => File_Name,
                  Tokenizer => Tokenizer,
                  Value     => Value_List (Value_List_Length)
                 );

      loop

         Table_Tokenizer.Find_Next (Tokenizer);
         case Table_Tokenizer.Token(Tokenizer) is

            -- if its a comma, cool.
            when Comma =>
               null;

               -- if its a Real, we are done reading this row
            when Real =>
               No_More_Rows := False;
               exit;

               -- if its the end of the file, we are done for good
            when EOF =>
               No_More_Rows := True;
               exit;

               -- otherwise, its an error
            when others =>

               Report_Error (File_Name => File_Name,
                             ID        => Missing_Comma_Or_Real);
         end case;


         Value_List_Length := Value_List_Length + 1;

         if Value_List_Length > Value_List'Length then
            Report_Error (File_Name => File_Name,
                          ID        => Too_Many_Dimensions);
         end if;

         -- Get the next value
         Table_Tokenizer.Find_Next (Tokenizer);
         Parse_Real (File_Name => File_Name,
                     Tokenizer => Tokenizer,
                     Value     => Value_List(Value_List_Length)
                    );

      end loop;

      -- Verify that there are a vaild number of dimensions and dependents
      Independents := Value_List_Length - Dependents;
      if Independents > Max_Dimensions then
         Report_Error (File_Name => File_Name,
                       ID        => Too_Many_Dimensions);
         return;
      end if;

      if Independents < 1 then
         Report_Error (File_Name => File_Name,
                       ID        => Too_Few_Dimensions);
         return;
      end if;

      -- Copy over the independents
      Row.Independent(1..Independents) := Index_Vector(Value_List(1..Independents));

      -- Copy over the dependents
      Row.Dependent(1..Dependents) := Value_List(Independents + 1 .. Value_List_Length);

   end Parse_CSV_Row;

   -----------------------------------------------------------------------------
   -- Read out any header names that might exist in the file
   -----------------------------------------------------------------------------
   procedure Get_Column_Names (File_Name       : in     String;
                               Tokenizer       : in out Table_Tokenizer.Instance;
                               Name_List       :    out Dimension_Strings;
                               Number_Of_Names :    out Integer) is
   begin

      Number_Of_Names := 0;
      Table_Tokenizer.Find_Next (Tokenizer);

      -- If its not a header line, quit
      if Table_Tokenizer.Token(Tokenizer) /= Identifier then
         return;
      end if;

      loop

         -- Read in the new name
         Number_Of_Names := Number_Of_Names + 1;

         case Table_Tokenizer.Token(Tokenizer) is

         -- null header
         when Comma =>
            if Number_Of_Names <= Name_List'Length then

               Name_List(Number_Of_Names) := Ada.Strings.Unbounded.Null_Unbounded_String;

            end if;

         -- Valid header contents
         when Identifier | Real =>
            if Number_Of_Names <= Dimension_Strings'Length then

               Name_List(Number_Of_Names) := Ada.Strings.Unbounded.To_Unbounded_String
                 (Table_Tokenizer.Lexeme(Tokenizer));

            end if;

         -- Unexpected stuff. Only EOF should ever happen (and that's an error).
         when others =>
            if Number_Of_Names <= Name_List'Length then

               Name_List(Number_Of_Names) := Ada.Strings.Unbounded.Null_Unbounded_String;
               Report_Error (File_Name => File_Name,
                             ID        => Identifier_Expected);
            end if;
         end case;

         -- Get the comma
         Table_Tokenizer.Find_Next (Tokenizer);

         case Table_Tokenizer.Token(Tokenizer) is
         when Comma =>
            null;
         when Identifier =>
            Report_Error (File_Name => File_Name,
                          ID        => Comma_Expected);
         when others =>
            exit;
         end case;

         -- Get the next header identifier
         Table_Tokenizer.Find_Next (Tokenizer);

      end loop;

      if Number_Of_Names > Dimension_Strings'Length then
         Number_Of_Names := 0;
         Report_Error (File_Name => File_Name,
                       Id        => Too_Many_Dimensions);
      end if;

   end Get_Column_Names;

   -----------------------------------------------------------------------------
   -- Parse all the CSV rows in the file
   -----------------------------------------------------------------------------
   procedure Parse_CSV_Rows (File_Name  : in     String;
                             Tokenizer  : in out Table_Tokenizer.Instance;
                             Dimensions : in out Natural;
                             Dependents : in     Natural;
                             Num_Rows   :    out Positive;
                             CSV_Rows   :    out CSV_Line_List) is

      Expected_Dimensions : Natural := Dimensions;
      Done_Reading        : Boolean;
   begin

      Num_Rows := 1;

      loop

         -- Process the next row
         Parse_CSV_Row (File_Name    => File_Name,
                        Tokenizer    => Tokenizer,
                        Row          => CSV_Rows(Num_Rows),
                        Independents => Dimensions,
                        Dependents   => Dependents,
                        No_More_Rows => Done_Reading);

         -- Verify the number of dimensions
         if Expected_Dimensions = 0 then

            Expected_Dimensions := Dimensions;

         elsif Dimensions /= Expected_Dimensions then

            Report_Error (File_Name => File_Name,
                          ID        => Number_Of_Dimensions_Changed,
                          Dimension => Dimensions,
                          Bound     => Expected_Dimensions);
         end if;

         exit when Done_Reading;

         Num_Rows := Num_Rows + 1;

         -- Make sure there is room for the next row.
         if Num_Rows > CSV_Rows'Last then
            Report_Error (File_Name => File_Name,
                          ID        => Too_Many_Dependents);
         end if;

      end loop;

   end Parse_CSV_Rows;


   -----------------------------------------------------------------------------
   -- Sort the given CSV rows in order of increasing values of the independent
   -- dimensions.
   -----------------------------------------------------------------------------
   procedure Sort (Rows       : in out CSV_Line_List;
                   Num_Rows   : in     Positive;
                   Dimensions : in     Positive) is

      Error_Message : Ada.Strings.Unbounded.Unbounded_String :=
        Ada.Strings.Unbounded.Null_Unbounded_String;

      -----------------------------------------------------------------------------
      -- Equality routine for CSV_Line's.
      -----------------------------------------------------------------------------
      function "=" (Left, Right : in CSV_Line) return Boolean is
      begin
         for Dimension in 1..Dimensions loop
            if Left.Independent(Dimension) /= Right.Independent(Dimension) then
               return False;
            end if;
         end loop;

         -- if we get here, they are equal
         return True;
      end "=";

      -----------------------------------------------------------------------------
      -- Less-than routine for CSV_Line's.
      -----------------------------------------------------------------------------
      function "<" (Left, Right : in CSV_Line) return Boolean is
      begin
         for Dimension in 1..Dimensions loop
            if Left.Independent(Dimension) /= Right.Independent(Dimension) then
               return Left.Independent(Dimension) < Right.Independent(Dimension);
            end if;
         end loop;

         -- if we get here, they are equal
         return False;
      end "<";

   begin -- Sort

      -- Sort the rows in increasing order
      for Sortee in 1..Num_Rows loop
         for Sorted in 1..Sortee-1 loop

            -- If there are duplicate entries, save them in the error message and look for more
            if Rows(Sortee) = Rows(Sorted) then

               Ada.Strings.Unbounded.Append
                 (Source   => Error_Message,
                  New_Item => Integer'Image(Rows(Sortee).Line) & " and" &
                              Integer'Image(Rows(Sorted).Line) &
                              Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF);

            -- If the sortee line is less than the sorted line, insert it in front of the
            -- sorted line, and move to the next sortee.
            elsif Rows(Sortee) < Rows(Sorted) then
               Rows(1..Sortee) := Rows(1..Sorted-1) & Rows(Sortee) & Rows(Sorted..Sortee-1);
               exit;
            end if;

         end loop;
      end loop;


      if Error_Message /= Ada.Strings.Unbounded.Null_Unbounded_String then
         Ada.Exceptions.Raise_Exception (Invalid_Table'Identity,
           "there are duplicate entires at the following lines:" &
           Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
           Ada.Strings.Unbounded.To_String(Error_Message));
      end if;
   end Sort;

   -----------------------------------------------------------------------------
   -- Parse the given CSV file for the given index of dependents. (1 signifies
   -- the first dependents on the row, 2 the second, etc.) The names of the
   -- columns, the number of dimensions, then number of dependents, and all the
   -- CSV rows are returned.
   -----------------------------------------------------------------------------
   procedure Parse (File_Name       : in     String;
                    Name_List       :    out Dimension_Strings;
                    Number_Of_Names :    out Integer;
                    Dimensions      : in out Natural;
                    Dependents      : in     Positive := 1;
                    Num_Rows        :    out Positive;
                    CSV_Rows        :    out CSV_Line_List) is
   begin

--      Ada.Text_IO.Open (Name => File_Name,
      Open_Text_File (Name => File_Name,
                      Mode => Ada.Text_IO.In_File,
                      File => File);

      -- Enforce that the first line must be a comment.
      -- This is done to "encourage" the developer to put some kind of helpful
      -- identifing information in the file.
      Table_Tokenizer.Find_Next (Tokenizer);
      if Table_Tokenizer.Token (Tokenizer) /= Comment then
         Report_Error (File_Name => File_Name,
                       ID        => No_Starting_Comment);
      end if;

      -- Turn off comment reporting, and parse the rest of the table file
      Comment_Token.Report := False;

      -- Get the names of each of the columns (if any).
      Get_Column_Names (File_Name       => File_Name,
                        Tokenizer       => Tokenizer,
                        Name_List       => Name_List,
                        Number_Of_Names => Number_Of_Names);

      Dimensions := Integer'Max(Dimensions, Number_Of_Names - Dependents);

      -- Get all the rows of dependents from the file.
      Parse_CSV_Rows (File_Name   => File_Name,
                      Tokenizer   => Tokenizer,
                      Dimensions  => Dimensions,
                      Dependents  => Dependents,
                      Num_Rows    => Num_Rows,
                      CSV_Rows    => CSV_Rows);

      Ada.Text_IO.Close( File );

      -- Sort the rows by their independents
      Sort
        (Rows       => CSV_Rows,
         Num_Rows   => Num_Rows,
         Dimensions => Dimensions
        );


   exception
   when Ada.Text_Io.Name_Error =>
      Ada.Exceptions.Raise_Exception(Syntax'Identity, "File " & '"' & File_Name & '"' &
        " not found");

   when Error : Table_Tokenizer.Syntax_Error =>
      Ada.Exceptions.Raise_Exception(Syntax'Identity,
        "SYNTAX ERROR in table file " & File_Name &
        Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF &
        Ada.Exceptions.Exception_Message(Error) & " at line" &
                            Integer'Image(Table_Tokenizer.Line(Tokenizer)) &
                            ", column" &
                            Integer'Image(Table_Tokenizer.Column(Tokenizer)));

   end Parse;

   --------------------------------------------------------------------------------
   -- Return the string value of the independent values at the given indices, in
   -- comma separated form (with a trailing comma).
   --------------------------------------------------------------------------------
   function Index_Images (Indices    : in Index_Vector) return String is

      Image : Ada.Strings.Unbounded.Unbounded_String :=
        Ada.Strings.Unbounded.Null_Unbounded_String;
   begin

      for Dimension in 1..Indices'length loop
         Ada.Strings.Unbounded.Append
           (Source   => Image,
            New_Item => Float'Image(Indices(Dimension)) & ", "
           );
      end loop;

      return Ada.Strings.Unbounded.To_String(Image);
   end Index_Images;

   -----------------------------------------------------------------------------
   -- If the given CSV row contains the given independent values, get its
   -- dependent value. If not, the error message is annotated with the expected
   -- values for the missing row.
   -- Once the rows are sorted (with Sort above), and you have a list of unique
   -- sorted independents (via Load_Independents above), it is a simple matter
   -- to predict and pass in the expected independent value for each row.
   -----------------------------------------------------------------------------
   procedure Get_Expected_Dependents (Independent : in     Index_Vector;
                                      Sorted_Row  : in     CSV_Line;
                                      Dependent   :    out Dependent_List;
                                      Error       : in out Ada.Strings.Unbounded.Unbounded_String) is
   begin

      -- If the independents don't match, that means that some entries were left
      -- out of the table file. Add the values of the independents for the missing
      -- entry to the error message we are building.
      for Dimension in 1..Independent'length loop
         if Independent(Dimension) /= Sorted_Row.Independent(Dimension) then

            if Error = Ada.Strings.Unbounded.Null_Unbounded_String then
               Error := Ada.Strings.Unbounded.To_Unbounded_String(
                 "The following rows of data are missing from the input table:" &
                 Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF);
            end if;

            Ada.Strings.Unbounded.Append
              (Source   => Error,
               New_Item => Index_Images(Independent) &
                           Ada.Characters.Latin_1.CR & Ada.Characters.Latin_1.LF);
            exit;
         end if;
      end loop;

      -- Copy the dependent value into the table
      Dependent := Sorted_Row.Dependent;
   end Get_Expected_Dependents;

   --------------------------------------------------------------------------------
   -- Insert the given value into the independent value list for the given
   -- dimension. The value will be inserted so that the list remains sorted in
   -- increasing order. If the value already exists it is not added again. If the
   -- value would add more than the list can hold, list length is updated but
   -- the value is not added and no exception is raised. The caller must check
   -- List_Length to make sure it doesn't surpass Independent'last.
   --------------------------------------------------------------------------------
   procedure Insert_Independent (Value       : in     Float;
                                 Text_Line   : in     Positive;
                                 List_Length : in out Natural;
                                 Independent : in out Independent_List) is
      Element : Positive := 1;
   begin

      -- Find the first element larger or equal to the given value
      while Element <= List_Length and then Value > Independent(Element) loop
         Element := Element + 1;
      end loop;

      -- If it won't fit, just increment the list length and return
      if Element > Independent'Last then

        List_Length := List_Length + 1;

      -- If the value was the biggest, add it to the end
      elsif Element > List_Length then

         if List_Length <= Independent'Last then
            Independent(Element) := Value;
         end if;
         List_Length := Element;

      -- If the value didn't match the element, insert the value before the element
      elsif Independent(Element) /= Value then
         if List_Length <= Independent'Last then
            Independent(1..List_Length + 1) :=
              Independent(1..Element-1) & Value &
              Independent(Element..List_Length);
         end if;
         List_Length := List_Length + 1;
      end if;

   end Insert_Independent;

   -----------------------------------------------------------------------------
   -- Load the given independent list with values from the given dimension of
   -- all the given CSV Lines. The list returned will contain unique values in
   -- increasing order.
   -----------------------------------------------------------------------------
   procedure Load_Independents (File_Name        : in     String;
                                Rows             : in     CSV_Line_List;
                                Num_Rows         : in     Positive;
                                Dimension        : in     Positive;
                                Independents     :    out Independent_List;
                                Num_Independents :    out Natural) is
   begin
      Num_Independents := 0;

      -- Load the independents into the table
      for Row in 1..Num_Rows loop

         Insert_Independent (Value        => Rows(Row).Independent(Dimension),
                             Text_Line    => Rows(Row).Line,
                             List_Length  => Num_Independents,
                             Independent  => Independents);

         -- If we tried to add too many, raise an error
         if Num_Independents > Independents'Last then
            Report_Error (File_Name => File_Name,
                          Line      => Rows(Row).Line,
                          Dimension => Dimension,
                          Bound     => Independents'Last,
                          ID        => Too_Many_Independents);
         end if;

      end loop;

      -- Make sure there are at least 2 values (it's tough to interpolate with just one!)
      if Num_Independents < 2 then
         Report_Error (File_Name => File_Name,
                       Dimension => Dimension,
                       Bound     => 2,
                       ID        => Too_Few_Independents);
      end if;

   end Load_Independents;


end Interpolation_Table.CSV_Parsing;








