-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Real_Time;
use type Ada.Real_Time.Time_Span;

-------------------------------------------------------------------------------
-- This package provides high-resolution timing under vxWorks. Even
-- Ada.Real_Time can only provide a resolution equal to the clock frequency.
-------------------------------------------------------------------------------
package body High_Resolution_Timer is

   -- vxWorks imported routines
   function Systimestampfreq return Time_Stamp;
   function Systimestamplock return Time_Stamp;
   function Sysclkrateget return Time_Stamp;

   pragma Interface (C, Systimestampfreq, "sysTimestampFreq");
   pragma Interface (C, Systimestamplock, "sysTimestampLock");
   pragma Interface (C, Sysclkrateget, "sysClkRateGet");

   Stamp_Rate : constant Float := 1.0 / Float(Systimestampfreq);

   ----------------------------------------------------------------------------
   -- Start a timer.
   ----------------------------------------------------------------------------
   procedure Start (Timer : out Instance) is
   begin
      Timer.Start := (Time  => Ada.Real_Time.Clock,
                      Stamp => Systimestamplock
                      );
   end Start;

   ----------------------------------------------------------------------------
   -- Stop a previously started timer.
   ----------------------------------------------------------------------------
   procedure Stop (Timer : in out Instance) is
   begin
      Timer.Stop := (Time  => Ada.Real_Time.Clock,
                     Stamp => Systimestamplock
                     );
   end Stop;

   ----------------------------------------------------------------------------
   -- Return the amount of milliseconds that elapsed while the timer was
   -- running.
   --
   -- The algorithm used accounts for (and depends on) the high-res counter
   -- getting reset to 0 every clock tick.
   ----------------------------------------------------------------------------
   function Milliseconds (Timer : Instance) return Float is
      Rt_Delta : constant Float :=
        Float (Ada.Real_Time.To_Duration (Timer.Stop.Time - Timer.Start.Time));
   begin
      return (Rt_Delta + ((Float(Timer.Stop.Stamp) - Float(Timer.Start.Stamp)) * Stamp_Rate)) * 1_000.0;
   end Milliseconds;

end High_Resolution_Timer;

