-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Real_Time;
with Ada.Streams;
with Lock;
with Perforated_Circular_Buffer;

-------------------------------------------------------------------------------
-- This package provides an abstract child class of stream to allow buffering
-- of data. To be usable as a buffer, the implementation of this class must
-- make sure that the implementations of Read and Write are atomic.
--
-- A buffer stream is created with a certian max size of data it may hold. When
-- a write is attempted and there isn't enough room, data up to a "perforation"
-- will be deleted to make room, if possible. If this occurs, the deletion will
-- start with the oldest perforation and move forward.
-------------------------------------------------------------------------------
package Buffer_Stream is

   type Instance is new Ada.Streams.Root_Stream_Type with private;

   type Handle is access all Instance'Class;

   -------------------------------------------------------------------------------
   -- Create a buffer stream of the given size. Note that there will be a bit of
   -- overhead (on the order of a several bytes) per open/close pair, so the size
   -- will need to include a bit of pad to compensate. Also, the behaviour of
   -- these streams is such that old writes will get overwritten by newer ones
   -- when the buffer fills.
   -------------------------------------------------------------------------------
   procedure Create
     (Stream   : in out Instance;
      Max_Size : in     Natural
     );

   -------------------------------------------------------------------------------
   -- Return True if data is available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Boolean;

   -------------------------------------------------------------------------------
   -- Return the amount of data available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Natural;

   -------------------------------------------------------------------------------
   -- Implementation of the inherited Read routine from Root_Stream_Type.
   -------------------------------------------------------------------------------
   procedure Read
     (Stream : in out Instance;
      Item   :    out Ada.Streams.Stream_Element_Array;
      Last   :    out Ada.Streams.Stream_Element_Offset
     );

   -------------------------------------------------------------------------------
   -- Implementation of the inherited Write routine from Root_Stream_Type.
   -------------------------------------------------------------------------------
   procedure Write
     (Stream : in out Instance;
      Item   : in     Ada.Streams.Stream_Element_Array
     );

   -------------------------------------------------------------------------------
   -- Prepare for a series of writes to the stream. This routine should be called
   -- before a stream is written to.
   -- The default routine just locks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Open_Writes (Stream : in out Instance);

   -------------------------------------------------------------------------------
   -- Complete a series of writes to the stream. This routine will be called
   -- after a series of writes has completed.
   -- The default routine just unlocks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Close_Writes (Stream : in out Instance);

   -------------------------------------------------------------------------------
   -- Prepare for a series of reads from the stream. This routine should be called
   -- before a stream is read from.
   -- The default routine just locks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Open_Reads (Stream : in out Instance);

   -------------------------------------------------------------------------------
   -- Complete a series of reads to the stream. This routine will be called
   -- after a series of reads has completed.
   -- The default routine just unlocks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Close_Reads (Stream : in out Instance);

   -------------------------------------------------------------------------------
   -- Hold the calling task until there is data in the stream.
   -------------------------------------------------------------------------------
   procedure Wait_For_Data (Stream : in out Instance);

private

   Bytes_Per_Element : constant := (Ada.Streams.Stream_Element'Size + 7) / 8;

   package Stream_Buffer is new Perforated_Circular_Buffer
     (Element => Ada.Streams.Stream_Element,
      Index   => Ada.Streams.Stream_Element_Offset,
      Data    => Ada.Streams.Stream_Element_Array
     );
   type Stream_Buffer_Ptr is access Stream_Buffer.Instance;

   type Instance is new Ada.Streams.Root_Stream_Type with record
      Use_Lock : Lock.Instance;
      Buffer   : Stream_Buffer_Ptr;
   end record;

end Buffer_Stream;
