-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Perforated_Circular_Buffer;
with Ada.Streams;
with Ada.Exceptions;

use type Ada.Streams.Stream_Element_Offset;

-------------------------------------------------------------------------------
-- This package provides an abstract child class of stream to allow buffering
-- of data. To be usable as a buffer, the implementation of this class must
-- make sure that the implementations of Read and Write are atomic.
-------------------------------------------------------------------------------
package body Buffer_Stream is

   -------------------------------------------------------------------------------
   -- Return True if data is available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Boolean is
   begin
      return Stream_Buffer.Size (Stream.Buffer.all) > 0;
   end Data_available;

   -------------------------------------------------------------------------------
   -- Return the amount of data available in the stream.
   -------------------------------------------------------------------------------
   function Data_available (Stream : access Instance) return Natural is
   begin
      return Stream_Buffer.Size (Stream.Buffer.all) * Bytes_Per_Element;
   end Data_available;

  -------------------------------------------------------------------------------
   -- Create a buffer stream of the given size. Note that there will be a bit of
   -- overhead (on the order of a several bytes) per open/close pair, so the size
   -- will need to include a bit of pad to compensate. Also, the behaviour of
   -- these streams is such that old writes will get overwritten by newer ones
   -- when the buffer fills.
   -------------------------------------------------------------------------------
   procedure Create
     (Stream   : in out Instance;
      Max_Size : in     Natural
     ) is
   begin
      Stream.Buffer := new Stream_Buffer.Instance
        (Ada.Streams.Stream_Element_Offset(Max_Size / Bytes_Per_Element));

      -- Initialize the buffer
      Stream_Buffer.Flush (Stream.Buffer.all);
   exception
      when Error : Storage_Error =>
         Ada.Exceptions.Raise_Exception
           (Ada.Exceptions.Exception_Identity(Error), "Unable to allocate a buffer stream of " &
            Integer'Image(Max_Size) &
            " bytes. Use a smaller value or add more memory.");
   end Create;

   -------------------------------------------------------------------------------
   -- Implementation of the inherited Read routine from Root_Stream_Type.
   -------------------------------------------------------------------------------
   procedure Read
     (Stream : in out Instance;
      Item   :    out Ada.Streams.Stream_Element_Array;
      Last   :    out Ada.Streams.Stream_Element_Offset
     )
   is
   begin -- Read
      Stream_Buffer.Read
        (Buffer   => Stream.Buffer.all,
         New_Data => Item
         );
      Last := Item'Last;
   end Read;


   -------------------------------------------------------------------------------
   -- Implementation of the inherited Write routine from Root_Stream_Type.
   -------------------------------------------------------------------------------
   procedure Write
     (Stream : in out Instance;
      Item   : in     Ada.Streams.Stream_Element_Array
     )
   is
   begin -- Write
      Stream_Buffer.Write
        (Buffer   => Stream.Buffer.all,
         New_Data => Item
        );
   end Write;

   -------------------------------------------------------------------------------
   -- Prepare for a series of writes to the stream. This routine should be called
   -- before a stream is written to.
   -- The default routine just locks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Open_Writes (Stream : in out Instance) is
   begin
      Stream.Use_Lock.Acquire;
   end Open_Writes;

   -------------------------------------------------------------------------------
   -- Complete a senries of writes to the stream. This routine will be called
   -- after a series of writes has completed.
   -- The default routine just unlocks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Close_Writes (Stream : in out Instance) is
   begin
      Stream.Use_Lock.Release;
   end Close_Writes;

   -------------------------------------------------------------------------------
   -- Prepare for a series of reads from the stream. This routine should be called
   -- before a stream is read from.
   -- The default routine just locks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Open_Reads (Stream : in out Instance) is
   begin
      Stream.Use_Lock.Acquire;
   end Open_Reads;

   -------------------------------------------------------------------------------
   -- Complete a series of reads to the stream. This routine will be called
   -- after a series of reads has completed.
   -- The default routine just unlocks the stream and quits.
   -------------------------------------------------------------------------------
   procedure Close_Reads (Stream : in out Instance) is
   begin
      Stream.Use_Lock.Release;
   end Close_Reads;

   -------------------------------------------------------------------------------
   -- Hold the calling task until there is data in the stream.
   -------------------------------------------------------------------------------
   procedure Wait_For_Data (Stream : in out Instance) is
   begin
      Stream_Buffer.Wait_For_Data (Stream.Buffer.all);
   end Wait_For_Data;

end Buffer_Stream;
