-- Copyright (C) 1994-1999 Grady Booch and Simon Wright.
-- All Rights Reserved.
--
--      This program is free software; you can redistribute it
--      and/or modify it under the terms of the Ada Community
--      License which comes with this Library.
--
--      This program is distributed in the hope that it will be
--      useful, but WITHOUT ANY WARRANTY; without even the implied
--      warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
--      PURPOSE. See the Ada Community License for more details.
--      You should have received a copy of the Ada Community
--      License with this library, in the file named "Ada Community
--      License" or "ACL". If not, contact the author of this library
--      for a copy.
--

-- $Id: bc-containers-rings-unbounded-synchronized.ads,v 1.1.2.3 1999/05/03 16:04:33 simon Exp $

with BC.Support.Synchronization;

generic
  type Monitor is new BC.Support.Synchronization.Monitor_Base with private;
package BC.Containers.Rings.Unbounded.Synchronized is

  type Synchronized_Unbounded_Ring is new Unbounded_Ring with private;

  function "=" (Left, Right : in Synchronized_Unbounded_Ring) return Boolean;

  procedure Clear (R : in out Synchronized_Unbounded_Ring);
  -- Empty the ring of all items. The mark is cleared.

  procedure Insert (R : in out Synchronized_Unbounded_Ring; Elem : Item);
  -- If the ring was empty, set the ring's mark and top to designate
  -- this item.
  -- Otherwise,
  --   this item becomes the new top;
  --   the previous top is located one place forward of the new top;
  --   the mark remains on the previously marked item.

  procedure Pop (R : in out Synchronized_Unbounded_Ring);
  -- Remove the top item from the ring.
  -- If the ring is still not empty, the new top is the item that was
  -- previously one place forward from the top.
  -- If the removed item was the marked item, the mark now designates
  -- the new top.

  procedure Rotate (R : in out Synchronized_Unbounded_Ring;
                    Dir : Direction := Forward);
  -- Rotate the top of the ring in the given direction. The ring's
  -- mark is unaffected. If there is exactly one item in the ring,
  -- rotating either direction always returns to the same item.

  procedure Mark (R : in out Synchronized_Unbounded_Ring);
  -- Designate the item at the top of the ring (if not empty) as
  -- marked.

  procedure Rotate_To_Mark (R : in out Synchronized_Unbounded_Ring);
  -- Rotate the ring so that the ring's mark is at the top.

  function Extent (R : Synchronized_Unbounded_Ring) return Natural;
  -- Return the number of items in the ring.

  function Is_Empty (R : Synchronized_Unbounded_Ring) return Boolean;
  -- Return True if and only if there are no items in the ring.

  function Top (R : Synchronized_Unbounded_Ring) return Item;
  -- Return a copy of the item at the top of the ring.

  function At_Mark (R : Synchronized_Unbounded_Ring) return Boolean;
  -- Return True if and only if the item at the top of the ring is
  -- marked; otherwise, return False. By implication, this member function
  -- will return True if the ring is empty, since the ring's top and mark
  -- both do not designate any item.
  -- XXX hmm, odd logic there!

private

  procedure Lock (R : in out Synchronized_Unbounded_Ring);
  procedure Unlock (R : in out Synchronized_Unbounded_Ring);

  type Synchronized_Unbounded_Ring is new Unbounded_Ring with record
    The_Monitor : BC.Support.Synchronization.Monitor_P;
  end record;

  procedure Initialize (R : in out Synchronized_Unbounded_Ring);
  procedure Adjust (R : in out Synchronized_Unbounded_Ring);
  procedure Finalize (R : in out Synchronized_Unbounded_Ring);

end BC.Containers.Rings.Unbounded.Synchronized;
