-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions;
with Radio_Utilities;
with Jpats_Radio_Db_If;
with Jpats_Radio_Db_If_Types;
with Log;
with Simulation_Dictionary;
with Ada.Unchecked_Deallocation;
with Ada.Text_Io;
use Ada.Text_Io;
with Ada.Float_Text_Io;
use Ada.Float_Text_Io;
use ada.Numerics.Elementary_Functions;

package body Reposition is

   package Ru renames Radio_Utilities;

   RAMP_FILENAME      : constant String := "ramp.dat";
   HOLDSHORT_FILENAME : constant String := "holdshort.dat";

   Local_Gsa : Float;

   Ft_To_Deg : constant :=  1.0/6076.1 * 1.0/60.0;

   Use_Air_Force_Ramp : Boolean;

   procedure Delete_Ramp is new Ada.Unchecked_Deallocation
     (Ramp_Type, Ramp_Access);

   procedure Delete_Holdshort is new Ada.Unchecked_Deallocation
     (Holdshort_Type, Holdshort_Access);

   type Repo_K is
      record
         X         : Float;
         Y         : Float;
         Z         : Float;
         Hdg       : Float;
         Kts       : Float;
         Gear      : Float;
         Flaps     : Float;
         Torq      : Float;
         Brake     : Boolean;
      end record;

   type Repo_Constants_Array is array(Jrt.Reposition_Select_Type'Range) of
     Repo_K;

   Repo_Constants : constant Repo_Constants_Array :=
     (Jrt.Takeoff  =>                (X    =>     50.0,
                                      Y    =>  00000.0,
                                      Z    =>   0004.5,
                                      Hdg  =>    000.0,
                                      Kts  =>    000.0,
                                      Gear =>      1.0,
                                      Flaps=>     23.0,
                                      Torq =>      4.0,
                                      Brake=>      False),
      Jrt.L_Localizer_Intercept  =>  (X    => -68000.0,
                                      Y    =>  -8600.0,
                                      Z    =>   2000.0,
                                      Hdg  =>   +030.0,
                                      Kts  =>    160.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      Jrt.R_Localizer_Intercept  =>  (X    => -68000.0,
                                      Y    =>  +8600.0,
                                      Z    =>   2000.0,
                                      Hdg  =>   -030.0,
                                      Kts  =>    160.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      Jrt.Ils_8nm                =>  (X    => -48640.0,
                                      Y    =>  00000.0,
                                      Z    =>   2000.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    160.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      Jrt.Faf_5nm                =>  (X    => -30400.0,
                                      Y    =>  00000.0,
                                      Z    =>    500.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    160.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     38.0,
                                      Brake=>      False),
      Jrt.Vfr_3nm                =>  (X    => -18240.0,
                                      Y    =>  00000.0,
                                      Z    =>    500.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    100.0,
                                      Gear =>      1.0,
                                      Flaps=>     50.0,
                                      Torq =>     40.0,
                                      Brake=>      False),
      Jrt.Flaps_Up_2nm           =>  (X    => -12160.0,
                                      Y    =>  00000.0,
                                      Z    =>    500.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    112.0,
                                      Gear =>      1.0,
                                      Flaps=>      0.0,
                                      Torq =>     29.0,
                                      Brake=>      False),
      Jrt.Flaps_To_2nm           =>  (X    => -12160.0,
                                      Y    =>  00000.0,
                                      Z    =>    500.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    107.0,
                                      Gear =>      1.0,
                                      Flaps=>     23.0,
                                      Torq =>     36.0,
                                      Brake=>      False),
      Jrt.Flaps_Ldg_2nm          =>  (X    => -12160.0,
                                      Y    =>  00000.0,
                                      Z    =>    500.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    102.0,
                                      Gear =>      1.0,
                                      Flaps=>     50.0,
                                      Torq =>     40.0,
                                      Brake=>      False),
      Jrt.Initial_2nm            =>  (X    => -12160.0,
                                      Y    =>  00000.0,
                                      Z    =>   1000.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    202.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     53.0,
                                      Brake=>      False),
      Jrt.Vfr_0p5nm              =>  (X    =>  -3040.0,
                                      Y    =>  00000.0,
                                      Z    =>    300.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    100.0,
                                      Gear =>      1.0,
                                      Flaps=>     50.0,
                                      Torq =>     40.0,
                                      Brake=>      False),
      Jrt.High_Key               =>  (X    =>  -1000.0,
                                      Y    =>  00000.0,
                                      Z    =>   3300.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    125.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>      5.0,
                                      Brake=>      False),
      Jrt.R_Outside_Downwind     =>  (X    =>  +3040.0,
                                      Y    => +12080.0,
                                      Z    =>   1000.0,
                                      Hdg  =>   -180.0,
                                      Kts  =>    200.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     53.0,
                                      Brake=>      False),
      Jrt.R_Downwind             =>  (X    =>  +3040.0,
                                      Y    =>  +5771.0,
                                      Z    =>    800.0,
                                      Hdg  =>   -180.0,
                                      Kts  =>    120.0,
                                      Gear =>      1.0,
                                      Flaps=>      0.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      Jrt.R_Inside_Downwind      =>  (X    =>  +3040.0,
                                      Y    =>  +5900.0,
                                      Z    =>   1000.0,
                                      Hdg  =>   -180.0,
                                      Kts  =>    120.0,
                                      Gear =>      1.0,
                                      Flaps=>     23.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      Jrt.L_Outside_Downwind     =>  (X    =>  +3040.0,
                                      Y    => -12080.0,
                                      Z    =>   1000.0,
                                      Hdg  =>   -180.0,
                                      Kts  =>    200.0,
                                      Gear =>      0.0,
                                      Flaps=>      0.0,
                                      Torq =>     53.0,
                                      Brake=>      False),
      Jrt.L_Downwind             =>  (X    =>  +3040.0,
                                      Y    =>  -5771.0,
                                      Z    =>    800.0,
                                      Hdg  =>   -180.0,
                                      Kts  =>    120.0,
                                      Gear =>      1.0,
                                      Flaps=>      0.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      Jrt.L_Inside_Downwind      =>  (X    =>  +3040.0,
                                      Y    =>  -5900.0,
                                      Z    =>   1000.0,
                                      Hdg  =>   -180.0,
                                      Kts  =>    120.0,
                                      Gear =>      1.0,
                                      Flaps=>     23.0,
                                      Torq =>     35.0,
                                      Brake=>      False),
      others                     =>  (X    =>  00000.0,
                                      Y    =>  00000.0,
                                      Z    =>   0000.0,
                                      Hdg  =>    000.0,
                                      Kts  =>    000.0,
                                      Gear =>      1.0,
                                      Flaps=>      0.0,
                                      Torq =>      0.0,
                                      Brake=>      False));

   -----------------------------------------------------------------------
   -- Ramp and holdshort reposition data are now stored in text files
   -- and loaded into doubly linked lists upon initialization.
   -- The following procedures manipulate those lists.
   -- At this time, no order is maintained since manual text file editing is 
   -- possible and likely and manual ordering cannot be guaranteed.
   -- List order could be added if desired.
   -----------------------------------------------------------------------

   ----------------------------------------------------------
   -- Deallocate memory associated with the holdshort list.
   --
   procedure Clear_Holdshort_List is
   
      Current : Holdshort_Access := null;
      Temp    : Holdshort_Access := null;

   begin

      Current := Holdshort_List;

      -- Dellocate each node in list
      While (Current /= null) loop
         Temp := Current.Next;
         Delete_Holdshort(Current);
         Current := Temp;
      end loop;

      Holdshort_List := null;

      exception

         When others =>
            Log.Report("Reposition.Clear_Holdshort_List()");
            raise;

   end Clear_Holdshort_List;

   ----------------------------------------------------------
   -- Deallocate memory associated with the ramp list.
   --
   procedure Clear_Ramp_List is
   
      Current : Ramp_Access := null;
      Temp    : Ramp_Access := null;

   begin

      Current := Ramp_List;

      -- Dellocate each node in list
      While (Current /= null) loop
         Temp := Current.Next;
         Delete_Ramp(Current);
         Current := Temp;
      end loop;

      Ramp_List := null;

      exception

         When others =>
            Log.Report("Reposition.Clear_Ramp_List()");
            raise;

   end Clear_Ramp_List;

   ----------------------------------------------------------
   -- Add a new holdshort to the list of holdshort positions.
   -- New holdshort is prepended to holdshort list
   --
   procedure Add_Holdshort(
         Airport  : in Jpats_Radio_Db_If_Types.Output_Ident;
         Runway   : in Jrt.String4;
         Position : in Jrt.Position_Type;
         Heading  : in Jrt.Bearing_Type ) is

      New_Holdshort : Holdshort_Access := null;

   begin

      New_Holdshort := new Holdshort_Type;
      New_Holdshort.Airport := Airport;
      New_Holdshort.Runway := Runway;
      New_Holdshort.Position.Latitude := Position.Latitude;
      New_Holdshort.Position.Longitude := Position.Longitude;
      New_Holdshort.Position.Altitude := Position.Altitude;
      New_Holdshort.Heading := Heading;
      New_Holdshort.Next := Holdshort_List;
      New_Holdshort.Next.Prev := New_Holdshort;
      Holdshort_List := New_Holdshort;

      exception

         When others =>
            Log.Report("Reposition.Add_Holdshort()");
            raise;

   end Add_Holdshort;

   ------------------------------------------------------
   -- Add a new ramp to the list of ramp positions.
   -- New ramp is prepended to ramp list.
   --
   procedure Add_Ramp(
         Airport  : in Jpats_Radio_Db_If_Types.Output_Ident;
         Position : in Jrt.Position_Type;
         Heading  : in Jrt.Bearing_Type ) is

      New_Ramp : Ramp_Access := null;

   begin

      New_Ramp := new Ramp_Type;
      New_Ramp.Airport := Airport;
      New_Ramp.Position.Latitude := Position.Latitude;
      New_Ramp.Position.Longitude := Position.Longitude;
      New_Ramp.Position.Altitude := Position.Altitude;
      New_Ramp.Heading := Heading;
      New_Ramp.Next := Ramp_List;
      New_Ramp.Next.Prev := New_Ramp;
      Ramp_List := New_Ramp;

      exception

         When others =>
            Log.Report("Reposition.Add_Ramp()");
            raise;

   end Add_Ramp;

   ------------------------------------------------
   -- Read ramp data from file and build ramp list
   --
   procedure Initialize_Ramp_List is

      File : Ada.Text_Io.File_Type;
      Current : Ramp_Access := null;
      New_Ramp : Ramp_Access := null;
      I : Integer := 0;
      C : Character := ' ';
      Latitude_In : Float := 0.0;
      Longitude_In : Float := 0.0;

   begin

      Use_Air_Force_Ramp := Simulation_Dictionary.Lookup("Visual_Models") = "USAF";

      -- Clear any existing list and deallocate memory
      Clear_Ramp_List;

      Current := null;
      New_Ramp := null;

      declare

         File_Name : String := Simulation_Dictionary.Lookup("Visual_Airport_Dir") & RAMP_FILENAME;

      begin

         Open (File, Mode=>Ada.Text_Io.In_File,
               Name=>File_Name);

         begin

            while not End_Of_File(File) loop

               New_Ramp := new Ramp_Type;
               New_Ramp.Next := null;
               New_Ramp.Prev := Current;
               I := 0;
               Get(File, C);
               while C = ' ' and not End_Of_File(File) loop
                  Get(File, C);
               end loop;
               if End_Of_File(File) then
                  exit;
               end if;
               while I < 4 and then C /= ' ' loop
                  if C /= ' ' then
                     I := I + 1;
                     New_Ramp.Airport(I) := C;
                  end if;
                  Get(File, C);
               end loop;
               while I < 4 loop
                  I := I + 1;
                  New_Ramp.Airport(I) := ' ';
               end loop;

               Get(File, Latitude_In);
               New_Ramp.Position.Latitude := Long_Float(Latitude_In);
               Get(File, Longitude_In);
               New_Ramp.Position.Longitude :=
                 Long_Float(Longitude_In);
               Get(File, New_Ramp.Position.Altitude);
               Get(File, New_Ramp.Heading);

               if (Current = null) then
                  Current := New_Ramp;
                  Ramp_List := New_Ramp;
               else
                  Current.Next := New_Ramp;
                  Current := New_Ramp;
               end if;

            end loop;

         exception

            when End_Error | Data_Error =>

               Log.Report("Error Reading From File");
               Current := Ramp_List.Next.Next;
               while Current /= null loop
                  New_Ramp := Current.Next;
                  Delete_Ramp(Current);
                  Current := New_Ramp;
               end loop;

         end;

         Close(File);

      end;

      exception

         when others =>

            Log.Report("Reposition.Initialize_Ramp_List()");
            raise;

   end Initialize_Ramp_List;

  -----------------------------------------------------------
  -- Read hold short data from file and build hold short list
  --
  procedure Initialize_Holdshort_List is

      File : Ada.Text_Io.File_Type;
      Current : Holdshort_Access := null;
      New_Holdshort : Holdshort_Access := null;
      I : Integer := 0;
      C : Character := ' ';
      Latitude_In : Float := 0.0;
      Longitude_In : Float := 0.0;

   begin

      -- Clear any existing list and deallocate memory
      Clear_Holdshort_List;

      Current := null;
      New_Holdshort := null;

      declare

         File_Name : String := Simulation_Dictionary.Lookup("Visual_Airport_Dir") & HOLDSHORT_FILENAME;

      begin

         Open (File, Mode=>Ada.Text_Io.In_File,
               Name=>File_Name);

         begin

            while not End_Of_File(File) loop

               New_Holdshort := new Holdshort_Type;
               New_Holdshort.Next := null;
               New_Holdshort.Prev := Current;
               I := 0;
               Get(File, C);
               while C = ' ' and not End_Of_File(File) loop
                  Get(File, C);
               end loop;
               if End_Of_File(File) then
                  exit;
               end if;
               while I < 4 and then C /= ' ' loop
                  if C /= ' ' then
                     I := I + 1;
                     New_Holdshort.Airport(I) := C;
                  end if;
                  Get(File, C);
               end loop;
               while I < 4 loop
                  I := I + 1;
                  New_Holdshort.Airport(I) := ' ';
               end loop;
               loop
                  Get(File, C);
                  exit when C /= ' ';
               end loop;

               I := 1;
               New_Holdshort.Runway(I) := C;
               while I < 3 and then C /= ' ' loop
                  Get(File, C);
                  if C /= ' ' then
                     I := I + 1;
                     New_Holdshort.Runway(I) := C;
                  end if;
               end loop;
               while I < 4 loop
                  I := I + 1;
                  New_Holdshort.Runway(I) := ' ';
               end loop;

               Get(File, Latitude_In);
               New_Holdshort.Position.Latitude := Long_Float(Latitude_In);
               Get(File, Longitude_In);
               New_Holdshort.Position.Longitude :=
                 Long_Float(Longitude_In);
               Get(File, New_Holdshort.Position.Altitude);
               Get(File, New_Holdshort.Heading);

               if (Current = null) then
                  Current := New_Holdshort;
                  Holdshort_List := New_Holdshort;
               else
                  Current.Next := New_Holdshort;
                  Current := New_Holdshort;
               end if;

            end loop;

         exception

            when End_Error | Data_Error =>

               Log.Report("Error Reading From File");
               Current := Holdshort_List.Next.Next;
               while Current /= null loop
                  New_Holdshort := Current.Next;
                  Delete_Holdshort(Current);
                  Current := New_Holdshort;
               end loop;

         end;

         Close(File);

      end;

      exception

         when others =>

            Log.Report("Reposition.Initialize_Holdshort_List()");
            raise;


   end Initialize_Holdshort_List;

   ------------------------------------------------
   -- Save ramp list to file
   --
   procedure Save_Ramps is

      File       : Ada.Text_Io.File_Type;
      Current    : Ramp_Access := null;

   begin

      declare

         File_Name : String := Simulation_Dictionary.Lookup("Visual_Airport_Dir") & RAMP_FILENAME;

      begin

         Open (File,Ada.Text_Io.Out_File,File_Name);

         Current := Ramp_List;

         while Current /= null loop

            Put(File,Current.Airport);
            Put(File => File, Fore => 3, Aft => 6, Exp => 0, Item => Float(Current.Position.Latitude));
            Put(File => File, Fore => 5, Aft => 6, Exp => 0, Item => Float(Current.Position.Longitude));
            Put(File => File, Fore => 6, Aft => 1, Exp => 0, Item => Float(Current.Position.Altitude));
            Put(File => File, Fore => 4, Aft => 1, Exp => 0, Item => Current.Heading);
            Current := Current.Next;
            if Current /= null then
               New_Line(File);
            end if;
            
         end loop;

         Close(File);

         exception
            when others =>
               Log.Report("Reposition.Save_Ramps()");
               raise;
      end;

   end Save_Ramps;

   --------------------------------------------------
   -- Save holdshort list to file
   --
   procedure Save_Holdshorts is

      File       : Ada.Text_Io.File_Type;
      Current    : Holdshort_Access := null;

   begin

      declare

         File_Name : String := Simulation_Dictionary.Lookup("Visual_Airport_Dir") & HOLDSHORT_FILENAME;

      begin 
 
         Open (File,Ada.Text_Io.Out_File,File_Name);

         Current := Holdshort_List;

         while Current /= null loop
           
            Put(File,Current.Airport);
            Put(File,' ');
            Put(File,Current.Runway);
            Put(File => File, Fore => 3, Aft => 6, Exp => 0, Item => Float(Current.Position.Latitude));
            Put(File => File, Fore => 5, Aft => 6, Exp => 0, Item => Float(Current.Position.Longitude));
            Put(File => File, Fore => 6, Aft => 1, Exp => 0, Item => Float(Current.Position.Altitude));
            Put(File => File, Fore => 4, Aft => 1, Exp => 0, Item => Current.Heading);
            Current := Current.Next;
            if Current /= null then
               New_Line(File);
            end if;
            
         end loop;

         Close(File);

         exception
            when others =>
               Log.Report("Reposition.Save_Holdshorts()");
               raise;
      end;

   end Save_Holdshorts;

   --------------------------------------------------------------------
   -- Set current aircraft position as the new ramp position. Add a new
   -- ramp if one is not defined already.
   --
   procedure Capture_Ramp(
         Airport  : in Jpats_Radio_Db_If_Types.Output_Ident;
         Position : in Jrt.Position_Type;
         Heading  : in Jrt.Bearing_Type ) is

      Current      : Ramp_Access := null;
      New_Ramp     : Ramp_Access := null;
      Temp_Airport : Jpats_Radio_Db_If_Types.Output_Ident;

   begin

      -- Ramp location at NAS Pensacola is different for USN and USAF
      if Use_Air_Force_Ramp and Airport = "KNPA" then
         Temp_Airport := "KNPX";
      else
         Temp_Airport := Airport;
      end if;

      Current := Ramp_List;
      while ((Current /= null) and then
              (Temp_Airport /= Current.Airport)) loop
         Current := Current.Next;
      end loop;

      if Current /= null then
         Current.Airport := Temp_Airport;
         Current.Position.Latitude := Position.Latitude;
         Current.Position.Longitude := Position.Longitude;
         Current.Position.Altitude := Position.Altitude;
         Current.Heading := Heading;
      else
         Add_Ramp( Airport  => Temp_Airport,
                   Position => Position,
                   Heading  => Heading );
      end if;

      exception

         When others =>
            Log.Report("Reposition.Capture_Ramp()");
            raise;

   end Capture_Ramp;

   --------------------------------------------------------------------------
   -- Set current aircraft position as the new hold short position. Add a new
   -- hold short if one is not defined already.
   --
   procedure Capture_Holdshort(
         Airport  : in Jpats_Radio_Db_If_Types.Output_Ident;
         Runway   : in Jrt.String4;
         Position : in Jrt.Position_Type;
         Heading  : in Jrt.Bearing_Type ) is

      Current : Holdshort_Access := null;

   begin

      Current := Holdshort_List;
      while ( (Current /= null) and then
              (not ((Airport = Current.Airport) and
               (Runway = Current.Runway)) ) ) loop
         Current := Current.Next;
      end loop;

      if Current /= null then
         Current.Airport := Airport;
         Current.Runway := Runway;
         Current.Position.Latitude := Position.Latitude;
         Current.Position.Longitude := Position.Longitude;
         Current.Position.Altitude := Position.Altitude;
         Current.Heading := Heading;
      else
         Add_Holdshort( Airport  => Airport,
                        Runway   => Runway,
                        Position => Position,
                        Heading  => Heading );
      end if;

      exception

         When others =>
            Log.Report("Reposition.Capture_Holdshort()");
            raise;

   end Capture_Holdshort;

   ------------------------------------------------------------
   -- for ramp repositions based upon active airport
   --
   procedure Ramp
     (Ref_Apt : in     Jpats_Radio_Db_If_Types.Output_Ident;
      Command : in out Jrt.Reposition_Command_Type) is

      Current        : Ramp_Access;
      Position       : Jpats_Radio_Db_If_Types.Position_Type;
      Temp_Apt       : Jpats_Radio_Db_If_Types.Output_Ident;

   begin

      -- Ramp location at NAS Pensacola is different for USN and USAF
      if Use_Air_Force_Ramp and Ref_Apt = "KNPA" then
         Temp_Apt := "KNPX";
      else
         Temp_Apt := Ref_Apt;
      end if;

      Current := Ramp_List;
      while (Current /= null) loop

         if Temp_Apt = Current.Airport then
            Position := (Current.Position.Latitude,
                         Current.Position.Longitude,
                         Current.Position.Altitude);
            Command.Active   := True;
            Command.Position := Current.Position;
            Command.Heading  := Current.Heading + JPATS_Radio_DB_IF.Magnetic_Variation(Position);
            Command.Airspeed := 0.0;
            Command.Gear_Position := 1.0;
            Command.Flap_Position := 0.0;
            Command.Torque := 4.0;
            Command.Parking_Brake := True;
            exit;
         end if;
         Current := Current.Next;
      end loop;

      exception

         When others =>
            Log.Report("Reposition.Ramp()");
            raise;

   end Ramp;

   -------------------------------------------------------------
   -- for hold short repositions based upon active airport/rwy
   --
   procedure Holdshort
     (Ref_Apt : in     Jpats_Radio_Db_If_Types.Output_Ident;
      Ref_Rwy : in     Jrt.String4;
      Command : in out Jrt.Reposition_Command_Type) is

      Current        : Holdshort_Access;
      Position       : Jpats_Radio_Db_If_Types.Position_Type;

   begin

     Current := Holdshort_List;
      while (Current /= null) loop
         if Ref_Apt = Current.Airport and Ref_Rwy = Current.Runway then
            Position := (Current.Position.Latitude,
                         Current.Position.Longitude,
                         Current.Position.Altitude);
            Command.Active   := True;
            Command.Position := Current.Position;
            Command.Heading  := Current.Heading + JPATS_Radio_DB_IF.Magnetic_Variation(Position);
            Command.Airspeed := 0.0;
            Command.Gear_Position := 1.0;
            Command.Flap_Position := 0.0;
            Command.Torque := 4.0;
            Command.Parking_Brake := True;
            exit;
         end if;
         Current := Current.Next;
      end loop;

      exception

         When others =>
            Log.Report("Reposition.Holdshort()");
            raise;

   end Holdshort;

   --
   -- End of Ramp/Holdshort List procedures section
   ----------------------------------------------------------------------

   --for relative repositions using ILS information:
   --either L_Localizer_Intercept,R_Localizer_Intercept,Ils_8nm,
   --Faf_5nm,Vfr_3nm,Straight_2nm,Initial_2nm,or Vfr_0p5nm are
   --selected and a valid ils has been detected for the active runway
   procedure Relative
     (Ref_Lat : in     Jrt.Lat_Type;
      Ref_Lon : in     Jrt.Lon_Type;
      Ref_Lgd : in     Float;
      Ref_Hdg : in     Jrt.Bearing_Type;
      Ref_Gsa : in     Float;
      Ref_Alt : in     Float;
      Ref_Sel : in     Jrt.Reposition_Select_Type;
      Command : in out Jrt.Reposition_Command_Type) is
   begin

      Command.Position.Latitude :=
        Ref_Lat + Long_Float
        (((Repo_Constants(Ref_Sel).X - Ref_Lgd) * Cos(Ref_Hdg,360.0) -
          Repo_Constants(Ref_Sel).Y * Sin(Ref_Hdg,360.0)) * Ft_To_Deg);

      Command.Position.Longitude :=
        Ref_Lon + Long_Float
        (((Repo_Constants(Ref_Sel).X - Ref_Lgd) * Sin(Ref_Hdg,360.0) +
          Repo_Constants(Ref_Sel).Y * cos(Ref_Hdg,360.0)) * Ft_To_Deg /
         Cos(Float(Ref_Lat),360.0));

      if Ref_Gsa <= 0.0 then
         Local_Gsa := 3.0;
      else
         Local_Gsa := Ref_Gsa;
      end if;

      Command.Position.Altitude := Ref_Alt +
        (abs(Repo_Constants(Ref_Sel).X) *
         Sin(Local_Gsa,360.0) / Cos(Local_Gsa,360.0));

      Command.Heading := Ru.Xn180(Ref_Hdg + Repo_Constants(Ref_Sel).Hdg);

      Command.Airspeed := Repo_Constants(Ref_Sel).Kts;

      Command.Gear_Position := Repo_Constants(Ref_Sel).Gear;

      Command.Flap_Position := Repo_Constants(Ref_Sel).Flaps;

      Command.Torque := Repo_Constants(Ref_Sel).Torq;

      Command.Parking_Brake := Repo_Constants(Ref_Sel).Brake;

      Command.Active := True;

   end;


   --for relative repositions using runway information:
   --either Takeoff,High_Key,R_Outside_Downwind,R_Downwind,R_Inside_Downwind,
   --L_Outside_Downwind,L_Downwind,or L_Inside_Downwind are selected and
   --valid runway data is available
   procedure Relative
     (Ref_Lat : in     Jrt.Lat_Type;
      Ref_Lon : in     Jrt.Lon_Type;
      Ref_Hdg : in     Jrt.Bearing_Type;
      Ref_Alt : in     Float;
      Ref_Sel : in     Jrt.Reposition_Select_Type;
      Command : in out Jrt.Reposition_Command_Type) is

      X_Distance : Float;
      Temp_Alt_A : Integer;
      Temp_Alt_B : Integer;

   begin

      if Ref_Sel in  Jrt.Ils_8nm .. Jrt.Vfr_0p5nm  then

         --use point 750 ft beyond dtr
         X_Distance := Repo_Constants(Ref_Sel).X + 750.0;

      else
         X_Distance := Repo_Constants(Ref_Sel).X;
      end if;

      Command.Position.Latitude :=
        Ref_Lat + Long_Float
        ((X_Distance * Cos(Ref_Hdg,360.0) -
          Repo_Constants(Ref_Sel).Y * Sin(Ref_Hdg,360.0)) * Ft_To_Deg);

      Command.Position.Longitude :=
        Ref_Lon + Long_Float
        ((X_Distance * Sin(Ref_Hdg,360.0) +
          Repo_Constants(Ref_Sel).Y * cos(Ref_Hdg,360.0)) * Ft_To_Deg /
         Cos(Float(Ref_Lat),360.0));

      --Round Altitude to nearest 100, then add Z
      Temp_Alt_A := integer(Ref_Alt / 100.0) * 100;
      Temp_Alt_B := integer(Ref_Alt) - Temp_Alt_A;

      if Temp_Alt_B > 49 then
        Command.Position.Altitude := Float(Temp_Alt_A + 100) + Repo_Constants(Ref_Sel).Z;
      else
        Command.Position.Altitude := Float(Temp_Alt_A) + Repo_Constants(Ref_Sel).Z;
      end if;

      Command.Heading := Ru.Xn180(Ref_Hdg + Repo_Constants(Ref_Sel).Hdg);

      Command.Airspeed := Repo_Constants(Ref_Sel).Kts;

      Command.Gear_Position := Repo_Constants(Ref_Sel).Gear;

      Command.Flap_Position := Repo_Constants(Ref_Sel).Flaps;

      Command.Torque := Repo_Constants(Ref_Sel).Torq;

      Command.Parking_Brake := Repo_Constants(Ref_Sel).Brake;

      Command.Active := True;

   end Relative;

   --for formation repositions based upon formation position, etc.
   procedure Formation
     (Ref_Lat     : in     Jrt.Lat_Type;
      Ref_Lon     : in     Jrt.Lon_Type;
      Ref_Alt     : in     Float;
      Ref_Hdg     : in     Jrt.Bearing_Type;
      Ref_Ias     : in     Float;
      Fore_Aft    : in     Float;
      Left_Right  : in     Float;
      Above_Below : in     Float;
      Command     : in out Jrt.Reposition_Command_Type) is

   begin

      Command.Position.Latitude :=
        Ref_Lat + Long_Float
        ((Fore_Aft * Cos(Ref_Hdg,360.0) -
          Left_Right * Sin(Ref_Hdg,360.0)) * Ft_To_Deg);

      Command.Position.Longitude :=
        Ref_Lon + Long_Float
        ((Fore_Aft * Sin(Ref_Hdg,360.0) +
          Left_Right * cos(Ref_Hdg,360.0)) * Ft_To_Deg /
         Cos(Float(Ref_Lat),360.0));

      Command.Position.Altitude := Ref_Alt + Above_Below;

      Command.Heading := Ref_Hdg;

      Command.Airspeed := Ref_Ias;

      Command.Active := True;

   end Formation;

end Reposition;