-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-- DATA SOURCE: IOS Sub-IPT Meeting Minutes
-------------------------------------------------------------------------------

with Jpats_Simulated_Aircraft,
  Jpats_Simulated_Aircraft.Ios_Interface,
  Jpats_Formation_Manager,
  Jpats_Radio_Db_If,
  Jpats_Radio_Db_If_Types,
  Jpats_Reposition_Types,
  Jpats_Visual_Buffer,
  Jpats_Landing_Gear,
  Jpats_Landing_Gear_Types,
  Jpats_Secondary_Flight_Controls,
  Jpats_Secondary_Flight_Controls_Types,
  JPATS_Powerplant,
  Radio_Utilities,
  Ada.Numerics.Elementary_Functions,
  JPATS_Reposition.Ios_If,
  JPATS_IOS_Interface,
  Log,
  Save_Restore;

with Jpats_Io_Types, Jpats_Io, Io_Types;
with Simulation_Dictionary;

with JPATS_Reposition.Container,
  Reposition;
with Ada.Text_Io,Log;
with Vector_Types;
with Vector_Types.Cross_Products;
use  Vector_Types.Cross_Products;

use Jpats_Reposition_Types,
  ada.Numerics.Elementary_Functions;
use Ada.Text_Io,Log;

package body JPATS_Reposition.Controller is

   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;
   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;
   package Jiot renames Jpats_Io_Types;
   package Iot renames Io_Types;

   package Formation renames Jpats_Formation_Manager;
   package Sim_Ac renames Jpats_Simulated_Aircraft;
   package Sim_Ac_Ios renames Sim_Ac.Ios_Interface;
   package Jrt renames Jpats_Reposition_Types;
   package Gear renames Jpats_Landing_Gear;
   package Gear_Types renames Jpats_Landing_Gear_Types;
   package Flight_Controls renames Jpats_Secondary_Flight_Controls;
   package Flight_Controls_Types renames Jpats_Secondary_Flight_Controls_Types;
   package Ru renames Radio_Utilities;

   Apt : Jpats_Radio_Db_If_Types.Apt_Struct_Type renames
     Container.This_Subsystem.The_Reference_Airfield;
   Rwy : Jpats_Radio_Db_If_Types.Rwy_Struct_Type renames
     Container.This_Subsystem.The_Reference_Runway;
   Apt_Sel : Jpats_Radio_Db_If_Types.Apt_Struct_Type renames
     Container.This_Subsystem.The_Selected_Airfield;
   Rwy_Sel : Jpats_Radio_Db_If_Types.Rwy_Struct_Type renames
     Container.This_Subsystem.The_Selected_Runway;
   Apt_Act : Jpats_Radio_Db_If_Types.Apt_Struct_Type renames
     Container.This_Subsystem.The_Active_Airfield;
   Rwy_Act : Jpats_Radio_Db_If_Types.Rwy_Struct_Type renames
     Container.This_Subsystem.The_Active_Runway;
   Repo_Command : Jrt.Reposition_Command_Type renames
     Container.This_Subsystem.The_Reposition_Command;
   Windshear : Jpats_Reposition_Types.Windshear_Type renames
     Container.This_Subsystem.The_Windshear_Command;
   Deltas    : Jpats_Reposition_Types.Formation_Delta_Type renames
     Container.This_Subsystem.The_Formation_Delta;

   Aptrwy : Jpats_Radio_Db_If_Types.Aptrwy_Request_Type;
   Ip_Request : Jpats_Radio_Db_If_Types.Ip_Request_Type;
   Ip_Struct : Jpats_Radio_Db_If_Types.Ip_Struct_Type;
   Slat_New  : Jpats_Radio_Db_If_Types.Lat_Type;
   Slon_New  : Jpats_Radio_Db_If_Types.Lon_Type;
   Coslat    : Float := 0.0;
   Nm_To_Dg  : constant := 1.0/60.0;
   To_Switch_N1 : Boolean;
   Approach_Switch_N1 : Boolean;
   Environ_Change_N1 : Boolean := False;
   Active_N1 : Boolean;
   Repo_Timer : Float;
   Visual_Terrain_Height_N1 : Float;
   Formation_Heading : Float;
   Magvar : Float;
   Snapshot : Boolean;
   Snapshot_Previous_Pass : Boolean;

   Deg_To_Feet : constant := 60.0 * 6076.1;
   Local_Flap_Handle : Integer := 0;
   Cnia_Page_Cleared : Boolean := True;
   Update_Torque     : Boolean := False;
   Airportn1 : Jpats_Radio_Db_If_Types.Aptrwy_Request_Type;
   Selectn1 : Jrt.Reposition_Select_Type := Jrt.None;
   Replay_In_Progress : Boolean := False;
   Replay_In_Progress_N1 : Boolean := False;
   Replay_Done        : Boolean := False;

   --initialize to close trail values
   Fore_Aft : Float := -52.0;
   Left_Right : Float := 0.0;
   Above_Below : Float := -10.0;

   procedure Register_Io_Variables is
   begin
      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "APPROACH_IAF_ON",
         A_Direction     => Iot.Input,
         An_Address      => Io.Approach_Iaf_On'Address,
         A_Size          => Io.Approach_Iaf_On'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "APPROACH_IAF_OFF",
         A_Direction     => Iot.Input,
         An_Address      => Io.Approach_Iaf_Off'Address,
         A_Size          => Io.Approach_Iaf_Off'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "T_O_POINT_ON",
         A_Direction     => Iot.Input,
         An_Address      => Io.To_Point_On'Address,
         A_Size          => Io.To_Point_On'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "T_O_POINT_OFF",
         A_Direction     => Iot.Input,
         An_Address      => Io.To_Point_Off'Address,
         A_Size          => Io.To_Point_Off'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "T_O_POINT_GREEN",
         A_Direction     => Iot.Output,
         An_Address      => Io.To_Point_Green'Address,
         A_Size          => Io.To_Point_Green'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "T_O_POINT_AMBER",
         A_Direction     => Iot.Output,
         An_Address      => Io.To_Point_Amber'Address,
         A_Size          => Io.To_Point_Amber'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "APPROACH_IAF_AMBER",
         A_Direction     => Iot.Output,
         An_Address      => Io.Approach_Iaf_Amber'Address,
         A_Size          => Io.Approach_Iaf_Amber'Size);

      Jpats_Io.Map_To_Icd_Mnemonic
        (An_Interface    => Jiot.Flight_Deck,
         An_Icd_Mnemonic => "APPROACH_IAF_GREEN",
         A_Direction     => Iot.Output,
         An_Address      => Io.Approach_Iaf_Green'Address,
         A_Size          => Io.Approach_Iaf_Green'Size);

   end;

   procedure Initialize is

      Init_Apt : Jrt.Ident_4_Type := "    ";
      Init_Rwy : Jrt.Ident_4_Type := "    ";

   begin

      --Find inital airport and runway
      Init_Apt := Simulation_Dictionary.Lookup("Initial_Airport");
      Init_Rwy := Simulation_Dictionary.Lookup("Initial_Runway");

      Ios.Airport := Init_Apt;
      Ios.Runway := Init_Rwy;
      Ios.Active_Airport := Init_Apt;
      Ios.Active_Runway := Init_Rwy;

      Ios.Reposition_Select := Jrt.Takeoff;
      Ios.Airport_Change := True;

      --Register IO Variables
      Register_Io_Variables;

      --Register IOS variables:
      JPATS_Reposition.Ios_If.Register_Ios_Variables;

      --Register array variables:
      JPATS_IOS_Interface.Register
        (Name               => "Reposition_Airport",
         Var_Array          => Container.This_Ios_Interface.Airport'address,
         Set_Routine        => JPATS_Reposition.Ios_If.Set_Airport'Access,
         Dimension_1_Length => 4
         );

      JPATS_IOS_Interface.Register
        (Name               => "Reposition_Runway",
         Var_Array          => Container.This_Ios_Interface.Runway'address,
         Set_Routine        => JPATS_Reposition.Ios_If.Set_Runway'Access,
         Dimension_1_Length => 4
         );

      JPATS_IOS_Interface.Register
        (Name               => "Active_Airport",
         Var_Array => Container.This_Ios_Interface.Active_Airport'address,
         Dimension_1_Length => 4
         );

      JPATS_IOS_Interface.Register
        (Name               => "Active_Runway",
         Var_Array => Container.This_Ios_Interface.Active_Runway'address,
         Dimension_1_Length => 4
         );

      JPATS_IOS_Interface.Register
        (Name               => "Reposition_Ident",
         Var_Array          => Container.This_Ios_Interface.Ident'address,
         Set_Routine        => JPATS_Reposition.Ios_If.Set_Ident'Access,
         Dimension_1_Length => 4
         );

      --Initialize reference airport information
      Apt.Idnt   := "    ";
      Apt.Icao   := "    ";
      Apt.Ata    := "    ";
      for I in Jpats_Radio_Db_If_Types.Bitfield'Range loop
         Apt.Typ1(I) := False;
         Apt.Typ2(I) := False;
      end loop;
      Apt.Lat    := 0.0;
      Apt.Lon    := 0.0;
      Apt.Var    := 0.0;
      Apt.Elv    := 0.0;
      Apt.Name   := "                                ";

      Apt_Sel := Apt;

      --Initialize reference runway information
      Rwy.Idnt     := "    ";
      Rwy.Icao     := "    ";
      for I in Jpats_Radio_Db_If_Types.Bitfield'Range loop
         Rwy.Typ1(I)     := False;
      end loop;
      Rwy.Typ2     := 0;
      Rwy.Hdg      := 0.0;
      Rwy.Lat      := 0.0;
      Rwy.Lon      := 0.0;
      Rwy.Len      := 0.0;
      Rwy.Dtr      := 0.0;
      Rwy.Wid      := 0.0;
      Rwy.Lid      := "    ";
      Rwy.Aid      := "    ";
      Rwy.Ifrq     := 0;
      Rwy.Afrq     := (0,0,0);

      Rwy_Sel := Rwy;

      --Initialize reposition command
      Repo_Command.Active := False;
      Repo_Command.Position.Latitude  := 0.0;
      Repo_Command.Position.Longitude := 0.0;
      Repo_Command.Position.Altitude  := 0.0;
      Repo_Command.Heading            := 0.0;
      Repo_Command.Airspeed           := 0.0;
      Repo_Command.Flap_Position      := 0.0;
      Repo_Command.Gear_Position      := 0.0;
      Repo_Command.On_Ground          := False;
      Repo_Command.Torque             := 0.0;
      Repo_Command.Parking_Brake      := False;
      Repo_Command.Roll_Angle         := 0.0;
      Repo_Command.Pitch_Angle        := 0.0;

      --Initialize windshear reference
      Windshear.Reference_Hdg := 0.0;
      Windshear.Reference_Lat := 0.0;
      Windshear.Reference_Lon := 0.0;

      --Initialize Ramps and Hold Shorts
      Reposition.Initialize_Ramp_List;
      Reposition.Initialize_Holdshort_List;

   end Initialize;

   procedure Update
     (Integration_Constant : in Float) is

      Xgas : Float;
      Ygas : Float;
      Sinhdg : Float;
      Coshdg : Float;

      -- This procedure is used to rotate the defined 3D
      -- wing position for each standard formation position
      -- whenever the lead roll and/or pitch angle changes.
      procedure Rotate_3D
        (Fore_Aft    : in Float;
         Left_Right  : in Float;
         Above_Below : in Float) is

         -- Fore_Aft is X axis (Rotate around X axis for Roll change)
         -- Left_Right is Y axis (Rotate around Y axis for Pitch change)
         -- Above_Below is Z axis (Rotate around Z axis for Yaw change. N/A here)

         Temp_Above_Below : Float;
     
      begin
           
         -- First, rotate defined wings level 3D position
         -- about the X axis (Roll)
         -- to arrive at intermediate 3D position. 
         Temp_Above_Below := 
           (Left_Right * sin(-(Formation.Lead_Bank_Angle * Jrt.Deg_To_Rad))) +
            (Above_Below * cos(-(Formation.Lead_Bank_Angle * Jrt.Deg_To_Rad)));
         Ios.Left_Right :=
           (Left_Right * cos(-(Formation.Lead_Bank_Angle * Jrt.Deg_To_Rad))) -
            (Above_Below * sin(-(Formation.Lead_Bank_Angle * Jrt.Deg_To_Rad)));

         -- Now rotate intermediate 3D position about the
         -- Y axis (Pitch)
         Ios.Above_Below :=
           (Temp_Above_Below * cos(-(Formation.Lead_Pitch_Angle * Jrt.Deg_To_Rad))) -
            (Fore_Aft * sin(-(Formation.Lead_Pitch_Angle * Jrt.Deg_To_Rad)));
         Ios.Fore_Aft :=
           (Temp_Above_Below * sin(-(Formation.Lead_Pitch_Angle * Jrt.Deg_To_Rad))) +
            (Fore_Aft * cos(-(Formation.Lead_Pitch_Angle * Jrt.Deg_To_Rad)));

      end Rotate_3D;

   begin

      --Replay in Progress
      Replay_In_Progress := Save_Restore.Replay_Trimming
        or Save_Restore.Replay_In_Progress;

      Replay_Done := not Replay_In_Progress
        and Replay_In_Progress_N1;

      Replay_In_Progress_N1 := Replay_In_Progress;

      --Magnetic Variation
      Magvar := JPATS_Radio_DB_IF.Magnetic_Variation;

      --calculate deltas between formation and simulated aircraft
      --

      --xgas=(formation lat - a/c lat) * 60.0  (nm/deg) * 6076.1 (ft/nm)
      Xgas :=
        Float((Formation.Lead_Latitude - Sim_Ac.Get_North) * Deg_To_Feet);

     --ygas = (formation lon - a/c lon) * deg_to_feet * cosine(a/c lat)
      Ygas :=
        float(RU.Xn180(Formation.Lead_Longitude -
                       Sim_Ac.Get_East))* Deg_To_Feet *
        cos(float(Sim_Ac.Get_North),360.0);

      --above_below = a/c geometric altitude - station elevation
      Deltas.Above_Below := Sim_Ac.Get_Aircraft_Geometric_Altitude -
        Formation.Lead_Geometric_Altitude;

      --sin,cos of formation heading
      Formation_Heading := Ru.Xn180(Formation.Lead_Heading * 57.3);

      Sinhdg := Sin(Formation_Heading,360.0);
      Coshdg := Cos(Formation_Heading,360.0);

      --fore/aft distance in feet
      Deltas.Fore_Aft := Ygas * Sinhdg + Xgas * Coshdg;

      --left/right distance in feet
      Deltas.Left_Right := Xgas * Sinhdg - Ygas * Coshdg;

--*********************************************************************


      --handle hard button inputs from IOS
      if Io.To_Point_On and not To_Switch_N1 then
         Io.To_Point_Amber := True;
         Io.To_Point_Green := False;
         Ios.Reposition_Select := Jpats_Reposition_Types.Takeoff;
      elsif Io.Approach_Iaf_On and not Approach_Switch_N1 then
         Io.Approach_Iaf_Amber := True;
         Io.Approach_Iaf_Green := False;
         Ios.Reposition_Select := Jpats_Reposition_Types.Ils_8nm;
      end if;
      To_Switch_N1              := Io.To_Point_On;
      Approach_Switch_N1        := Io.Approach_Iaf_On;

      --process active airport requests from ios
      if Ios.Airport_Change or Replay_Done then

         --call airport/runway procedure
         Aptrwy.Apt_Idnt      := Ios.Airport & ' ';
         Aptrwy.Rwy_Idnt      := Ios.Runway  & ' ';
         Aptrwy.Rwy_Sort_Mode := 0;
         Jpats_Radio_Db_If.N_Aptrwy_Srch(Aptrwy,Apt_Sel,Rwy_Sel);

         --clear airport change label
         Ios.Airport_Change := False;

         --Send the active airport and runway idents to IOS
         Ios.Active_Airport := Apt_Sel.Idnt;
         Ios.Active_Runway := Rwy_Sel.Idnt;

      end if;

      -- Process Ramp or Holdshort requests from ios
      --
      if Ios.Capture_Ramp then
         Ios.Ramp_Changes := True;
         Ios.Capture_Ramp := False;
         -- Get current data
         Reposition.Capture_Ramp(
                     Airport  => Ios.Active_Airport,
                     Position => ( Latitude  => Sim_Ac.Get_Aircraft_Position.Latitude,
                                   Longitude => Sim_Ac.Get_Aircraft_Position.Longitude,
                                   Altitude  => Sim_Ac.Get_Aircraft_Position.Altitude),
                     Heading  => Sim_Ac_Ios.This_IOS_Interface.Mag_Hdg );
      end if;

      if Ios.Capture_Holdshort then
         Ios.Holdshort_Changes := True;
         Ios.Capture_Holdshort := False;
         -- Get current data
         Reposition.Capture_Holdshort(
                     Airport  => Ios.Active_Airport,
                     Runway   => Ios.Active_Runway,
                     Position => ( Latitude  => Sim_Ac.Get_Aircraft_Position.Latitude,
                                   Longitude => Sim_Ac.Get_Aircraft_Position.Longitude,
                                   Altitude  => Sim_Ac.Get_Aircraft_Position.Altitude),
                     Heading  => Sim_Ac_Ios.This_IOS_Interface.Mag_Hdg );
      end if;

      if Ios.Save_Ramps then
         Ios.Save_Ramps := False;
         Ios.Ramp_Changes := False;
         Reposition.Save_Ramps;
      end if;

      if Ios.Save_Holdshorts then
         Ios.Save_Holdshorts := False;
         Ios.Holdshort_Changes := False;
         Reposition.Save_Holdshorts;
      end if;

      if Ios.Refresh_Ramps then
         Ios.Refresh_Ramps := False;
         Reposition.Initialize_Ramp_List;
      end if;

      if Ios.Refresh_Holdshorts then
         Ios.Refresh_Holdshorts := False;
         Reposition.Initialize_Holdshort_List;
      end if;

      if Ios.Environment_Change then

         --call airport/runway procedure
         Aptrwy.Apt_Idnt      := Ios.Airport & ' ';
         Aptrwy.Rwy_Idnt      := Ios.Runway  & ' ';
         Aptrwy.Rwy_Sort_Mode := 0;
         Jpats_Radio_Db_If.N_Aptrwy_Srch(Aptrwy,Apt_Sel,Rwy_Sel);

         --clear airport change label
         Ios.Environment_Change := False;
         Environ_Change_N1 := True;

         --Send the active airport and runway idents to IOS
         Ios.Active_Airport := Apt_Sel.Idnt;
         Ios.Active_Runway := Rwy_Sel.Idnt;

      end if;

      if Environ_Change_N1 and
        Apt_Sel.Idnt = Ios.Airport and
        Rwy_Sel.Idnt = Ios.Runway then

         Environ_Change_N1 := False;

         Apt := Apt_Sel;
         Rwy := Rwy_Sel;

      end if;

      if ((Apt_Sel.Idnt = Ios.Airport) and
          (Rwy_Sel.Idnt  = Ios.Runway))  then
         Apt_Act := Apt_Sel;
         Rwy_Act := Rwy_Sel;
      end if;

      case Ios.Reposition_Select is

         --no reposition selected.
         when Jrt.None =>
            null;

         --Repositions relative to runway information
         when Jrt.Faf_5nm | Jrt.Initial_2nm | Jrt.Takeoff .. Jrt.L_Inside_Downwind =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            --check if selected airport/runway same as active airport/runway
            if Apt_Sel.Idnt = Ios.Airport and
              Rwy_Sel.Idnt  = Ios.Runway  then

               --call reposition if runway/airport data valid
               Reposition.Relative(Ref_Lat => Rwy_Sel.Lat,
                                   Ref_Lon => Rwy_Sel.Lon,
                                   Ref_Hdg => Rwy_Sel.Hdg,
                                   Ref_Alt => Apt_Sel.Elv,
                                   Ref_Sel => Ios.Reposition_Select,
                                   Command => Repo_Command);


               if Ios.Reposition_Select = Jrt.Takeoff then
                  Repo_Command.On_Ground     := True;
               else
                  Repo_Command.On_Ground := False;
               end if;

               Apt := Apt_Sel;
               Rwy := Rwy_Sel;

            end if;

            --Reset reposition select.
            Ios.Reposition_Select := Jrt.None;

         --Repositions relative to ILS if available.  otherwise, relative
         --to runway.
         when Jrt.Ils_8nm | Jrt.Vfr_3nm .. Flaps_Ldg_2nm | Jrt.Vfr_0p5nm =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            --check if selected airport/runway same as active airport/runway
            if Apt_Sel.Idnt = Ios.Airport and
              Rwy_Sel.Idnt  = Ios.Runway  then

               --call ip search to obtain ils information if available.
               if Rwy_Sel.Lid /= "    " then
                  Ip_Request.Lat      := JPATS_Simulated_Aircraft.Get_North;
                  Ip_Request.Lon      := JPATS_Simulated_Aircraft.Get_East;
                  Ip_Request.Ip_Idnt  := Rwy_Sel.Lid & ' ';
                  Ip_Request.Ip_Optn  := "ils  ";
                  Ip_Request.Apt_Idnt :=
                    Container.This_Subsystem.The_Reference_Airfield.Idnt & ' ';
                  Jpats_Radio_Db_If.N_Ip_Srch(Ip_Request,Ip_Struct);

                  --reposition relative to ils information valid
                  if Rwy_Sel.Lid = Ip_Struct.Idnt then
                     Reposition.Relative(Ref_Lat => Ip_Struct.Slat,
                                         Ref_Lon => Ip_Struct.Slon,
                                         Ref_Lgd => Ip_Struct.Lgd,
                                         Ref_Hdg => Ip_Struct.Hdg,
                                         Ref_Gsa => Ip_Struct.Gsa,
                                         Ref_Alt => Apt_Sel.Elv,
                                         Ref_Sel => Ios.Reposition_Select,
                                         Command => Repo_Command);
                  end if;

               --ils not available, reposition using runway information
               else
                  Reposition.Relative(Ref_Lat => Rwy_Sel.Lat,
                                      Ref_Lon => Rwy_Sel.Lon,
                                      Ref_Hdg => Rwy_Sel.Hdg,
                                      Ref_Alt => Apt_Sel.Elv,
                                      Ref_Sel => Ios.Reposition_Select,
                                      Command => Repo_Command);

               end if;

               --Don't force on-ground reposition.
               Repo_Command.On_Ground := False;

               Apt := Apt_Sel;
               Rwy := Rwy_Sel;

            end if;

            --Reset reposition select.
            Ios.Reposition_Select := Jrt.None;

         when Jrt.Ramp =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            --reposition based on active airport if valid
            if Apt_Sel.Idnt = Ios.Airport then

               Reposition.Ramp(Ref_Apt => Apt_Sel.Idnt,
                               Command => Repo_Command);
               Repo_Command.On_Ground := True;

               Apt := Apt_Sel;
               Rwy := Rwy_Sel;

            end if;

            --Reset reposition select.
            if Repo_Command.Active then          --if a ramp was found
               Ios.Reposition_Select := Jrt.None;
            else                                 --else takeoff position
               Ios.Reposition_Select := Jrt.Takeoff;
            end if;

         -- Lat/Lon Reposition selected on IOS
         when Jrt.Lat_Lon_Reposition =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            --only latitude and longitude change.
            Repo_Command.Active := True;
            Repo_Command.Position.Latitude  := Ios.Latitude;
            Repo_Command.Position.Longitude := Ios.Longitude;

            --The IOS has the capability to set altitude, airspeed
            --and heading which is used during sortie file repositions.
            if Ios.Altitude_Was_Set then
               Repo_Command.Position.Altitude := Ios.Altitude;
               Ios.Altitude_Was_Set := False;
               Repo_Command.On_Ground := False;
            else
               Repo_Command.Position.Altitude :=
                 Sim_Ac.Get_Aircraft_Geometric_Altitude;
               Repo_Command.On_Ground :=
                 Gear.Is_Closed(Gear_Types.Left_Gear_Wow_Switch);
            end if;

            if Ios.Heading_Was_Set then
               Repo_Command.Heading :=
                 Ru.Xn180(Ios.Heading + Magvar);
               Ios.Heading_Was_Set := False;
            else
               Repo_Command.Heading :=
                 Ru.Xn180(Sim_Ac.Get_Hdg_Angle * 57.3);
            end if;

            if Ios.Airspeed_Was_Set then
               Repo_Command.Airspeed := Ios.Airspeed;
               Ios.Airspeed_Was_Set := False;
            else
               Repo_Command.Airspeed := Sim_Ac.Get_Calibrated_Airspeed;
            end if;

            Repo_Command.Flap_Position :=
              Flight_Controls.Mean_Flap_Position;
            Repo_Command.Gear_Position := Gear.Mean_Gear_Position;

            Apt := Apt_Sel;
            Rwy := Rwy_Sel;

            --clear IOS reposition select
            Ios.Reposition_Select := Jrt.None;

         when Jrt.Ident_Reposition =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            --first call the IP Search procedure:
            Ip_Request.Lat      := JPATS_Simulated_Aircraft.Get_North;
            Ip_Request.Lon      := JPATS_Simulated_Aircraft.Get_East;
            Ip_Request.Ip_Idnt  := Ios.Ident & ' ';
            Ip_Request.Ip_Optn  := "     ";
            Ip_Request.Apt_Idnt :=
              Container.This_Subsystem.The_Reference_Airfield.Idnt & ' ';
            Jpats_Radio_Db_If.N_Ip_Srch(Ip_Request,Ip_Struct);

            --is search return valid?
            if Ip_Struct.Slat /= 0.0 and
              Ip_Struct.Slon /= 0.0 then

               Slat_New := (Ip_Struct.Slat
                            +Long_Float((Ios.Distance*Nm_To_Dg)
                            *cos(Float(Ios.Radial+Ip_Struct.Hdg),360.0)));

               Coslat := cos(Float(Ip_Struct.Slat),360.0);

               if Coslat = 0.0 then Coslat := 0.001; end if;

               Slon_New := (Ip_Struct.Slon
                            +Long_Float(((Ios.Distance*Nm_To_Dg)
                            *Sin(float(Ios.Radial+Ip_Struct.Hdg),360.0))
                            /Coslat));

               --identical to lat/lon reposition at this point.
               Repo_Command.Active := True;
               Repo_Command.Position.Latitude  := Slat_New;
               Repo_Command.Position.Longitude := Slon_New;

               --The IOS has the capability to set altitude, airspeed
               --and heading which is used during sortie file repositions.
               if Ios.Altitude_Was_Set then
                  Repo_Command.Position.Altitude := Ios.Altitude;
                  Ios.Altitude_Was_Set := False;
                  Repo_Command.On_Ground := False;
               else
                  Repo_Command.Position.Altitude  :=
                    Sim_Ac.Get_Aircraft_Geometric_Altitude;
                  Repo_Command.On_Ground :=
                    Gear.Is_Closed(Gear_Types.Left_Gear_Wow_Switch);
               end if;

               if Ios.Heading_Was_Set then
                  Repo_Command.Heading :=
                    Ru.Xn180(Ios.Heading+Ip_Struct.Var);
                  Ios.Heading_Was_Set := False;
               else
                  Repo_Command.Heading :=
                    Ru.Xn180(Sim_Ac.Get_Hdg_Angle * 57.3);
               end if;

               if Ios.Airspeed_Was_Set then
                  Repo_Command.Airspeed := Ios.Airspeed;
                  Ios.Airspeed_Was_Set := False;
               else
                  Repo_Command.Airspeed := Sim_Ac.Get_Calibrated_Airspeed;
               end if;

               Repo_Command.Flap_Position :=
                 Flight_Controls.Mean_Flap_Position;
               Repo_Command.Gear_Position := Gear.Mean_Gear_Position;

            else
               Ios.Ident := "n/a ";
            end if;

            Apt := Apt_Sel;
            Rwy := Rwy_Sel;

            --clear IOS reposition select
            Ios.Reposition_Select := Jrt.None;

         when Jrt.Formation =>

            -- Do not allow formation reposition when the lead is at
            -- zero latitude and longitude. This prevents the strange
            -- checkerboard visual scene from appearing

            if (Formation.Lead_Latitude > 0.1 or
                Formation.Lead_Latitude < -0.1) and
               (Formation.Lead_Longitude > 0.1 or
                Formation.Lead_Longitude < -0.1) then

               Repo_Command.Roll_Angle := Formation.Lead_Bank_Angle * Jrt.Deg_To_Rad;
               Repo_Command.Pitch_Angle := Formation.Lead_Pitch_Angle * Jrt.Deg_To_Rad;
               Repo_Command.Heading := Formation.Lead_Heading;
               case Ios.Reposition_Formation_Id is  --drop down box
                  when 0 => --touch reposition
                     null;
                  when 1 => --Fingertip Left
                     Rotate_3D(
                       Fore_Aft    => -24.0,
                       Left_Right  => -43.0,
                       Above_Below => -1.0);
                  when 2 => --Fingertip Right
                     Rotate_3D(
                       Fore_Aft    => -24.0,
                       Left_Right  => 43.0,
                       Above_Below => -1.0);
                  when 3 => --Closed Trail
                     Rotate_3D(
                       Fore_Aft    => Fore_Aft,
                       Left_Right  => Left_Right,
                       Above_Below => Above_Below);
                  when 4 => --100 ft Route Left
                     Rotate_3D(
                       Fore_Aft    => -64.25,
                       Left_Right  => -132.5,
                       Above_Below => -3.0);
                  when 5 => --100 ft Route Right
                     Rotate_3D(
                       Fore_Aft    => -64.25,
                       Left_Right  => 132.5,
                       Above_Below => -3.0);
                  when 6 => -- Starboard Parade
                     Rotate_3D(
                       Fore_Aft    => -40.0,
                       Left_Right  => 45.0,
                       Above_Below => -12.0);
                  when 7 => -- Starboard Turns Away (VMC)
                     Ios.Fore_Aft    := -54.0;
                     Ios.Left_Right  := 66.0;
                     Ios.Above_Below := -2.0;
                  when 8 => -- Port Parade
                     Rotate_3D(
                       Fore_Aft    => -40.0,
                       Left_Right  => -45.0,
                       Above_Below => -12.0);
                  when 9 => -- Port Turns Away (VMC)
                     Ios.Fore_Aft    := -54.0;
                     Ios.Left_Right  := -66.0;
                     Ios.Above_Below := -2.0;
                  when 10 => -- B/R 60 Degree Bearing
                     Ios.Fore_Aft    := -260.0;
                     Ios.Left_Right  := -215.0;
                     Ios.Above_Below := 0.0;
                  when 11 => -- B/R Joinup
                     Rotate_3D(
                        Fore_Aft    => -118.0,
                        Left_Right  => 0.0,
                        Above_Below => -40.0);
                  when 12 => -- Lead
                     Rotate_3D(
                       Fore_Aft    => 40.0,
                       Left_Right  => -45.0,
                       Above_Below => 12.0);
                  when others => null;
               end case;

               --call formation reposition.
               Reposition.Formation
                 (Ref_Lat => Formation.Lead_Latitude,
                  Ref_Lon => Formation.Lead_Longitude,
                  Ref_Alt => Formation.Lead_Geometric_Altitude,
                  Ref_Hdg => Ru.Xn180(Formation.Lead_Heading),
                  Ref_Ias => Formation.Lead_Indicated_Airspeed,
                  Fore_Aft    => Ios.Fore_Aft,
                  Left_Right  => Ios.Left_Right,
                  Above_Below => Ios.Above_Below,
                  Command     => Repo_Command);

               --keep flap/gear positions unchanged, don't force
               --on ground reposition.

               Repo_Command.Flap_Position := Flight_Controls.Mean_Flap_Position;
               Repo_Command.Gear_Position := Gear.Mean_Gear_Position;
               Repo_Command.On_Ground     := False;

               Apt := Apt_Sel;
               Rwy := Rwy_Sel;

            end if;

            --clear IOS reposition select
            Ios.Reposition_Select := Jrt.None;

         when Jrt.Snapshot =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            Snapshot := True;

            --Reset reposition select.
            Ios.Reposition_Select := Jrt.None;

         when Jrt.Holdshort =>

            Repo_Command.Roll_Angle := 0.0;
            Repo_Command.Pitch_Angle := 0.0;

            --reposition based on active airport if valid
            if Apt_Sel.Idnt = Ios.Airport then

               Reposition.Holdshort(Ref_Apt => Apt_Sel.Idnt,
                                    Ref_Rwy => Rwy_Sel.Idnt,
                                    Command => Repo_Command);
               Repo_Command.On_Ground := True;

               Apt := Apt_Sel;
               Rwy := Rwy_Sel;

            end if;

            --Reset reposition select.
            if Repo_Command.Active then          --if a hold short position was found
               Ios.Reposition_Select := Jrt.None;
            else                                 --else takeoff position
               Ios.Reposition_Select := Jrt.Takeoff;
            end if;

      end case;

      --will hold this command for 3 seconds.  done this way because
      --I was unable to convince Yogi to supply a "reposition in progress"
      --notification.
      if (Repo_Command.Active and not Active_N1) or (Snapshot and not Snapshot_Previous_Pass) then
         Repo_Timer := 3.0;
         Ios.Cnia_Display_Page := False;
         Cnia_Page_Cleared := True;

         if (Snapshot and not Snapshot_Previous_Pass) then
            Repo_Command.Flap_Position :=
              Flight_Controls.Mean_Flap_Position;
            Repo_Command.Gear_Position := Gear.Mean_Gear_Position;
            Repo_Command.Torque := JPATS_Powerplant.Engine_Torque_Pct;
         end if;

      elsif ((Repo_Command.Active or Snapshot) and Repo_Timer > 0.0) then
         -- If I receive consecutive altitude values from Visual
         -- that agree for three seconds, I'll allow the reposition
         -- to continue, otherwise, wait for a valid altitude.
         if Repo_Command.On_Ground
           and Repo_Command.Position.Altitude /=
           (JPATS_Visual_Buffer.Height_Of_Terrain(1) + 4.25) then

            -- set the altitude
            Repo_Command.Position.Altitude :=
              JPATS_Visual_Buffer.Height_Of_Terrain(1) + 4.25;
            -- reset reposition timer
            Repo_Timer := 3.0;

         --check for underground reposition in case previous
         --altitude is under new terrain height.
         elsif Repo_Command.Position.Altitude <
           JPATS_Visual_Buffer.Height_Of_Terrain(1) + 4.25 then

            -- set the altitude
            Repo_Command.Position.Altitude :=
              JPATS_Visual_Buffer.Height_Of_Terrain(1) + 4.25;

            -- reset reposition timer
            Repo_Timer := 3.0;

            -- notify cockpit that gear should be down
            -- in case we were previously in air.
            Repo_Command.Gear_Position := 1.0;

         -- check for changes in ground height from visual for
         -- any other reposition
         elsif JPATS_Visual_Buffer.Height_Of_Terrain(1) /=
              Visual_Terrain_Height_N1 then

               Visual_Terrain_Height_N1 :=
                 JPATS_Visual_Buffer.Height_Of_Terrain(1);
               Repo_Timer := 3.0;

         end if;

         Repo_Timer := Repo_Timer - Integration_Constant;

         -- Check for controls not in agreement
         -- Set Required Gear Position (>0 is gear down)
         if Repo_Command.Gear_Position > 0.0 then
            Ios.Cnia_Required_Gear := False; --False is gear down
         else
            Ios.Cnia_Required_Gear := True; --True is gear up
         end if;

         -- Set Required Flap Position
         if Repo_Command.Flap_Position < 23.0 then
            Ios.Cnia_Required_Flaps := 0;
         elsif Repo_Command.Flap_Position < 50.0 then
            Ios.Cnia_Required_Flaps := 1;
         else
            Ios.Cnia_Required_Flaps := 2;
         end if;

         -- Set Required Torque
         if Repo_Command.Torque > 0.0 then
            Ios.Cnia_Required_Torque := Repo_Command.Torque;
            Update_Torque := False;
         else
            Ios.Cnia_Required_Torque := JPATS_Powerplant.Engine_Torque_Pct;
            Update_Torque := True;
         end if;

         -- Set Required Brake
         Ios.Cnia_Required_Parking_Brake :=  Repo_Command.Parking_Brake;

      elsif ((Repo_Command.Active or Snapshot)and Repo_Timer <= 0.0) then
         Repo_Command.Active := False;
         Snapshot := False;

         --Show CNIA page if required
         Cnia_Page_Cleared := False;

         --adjust IOS to/approach lights
         Io.To_Point_Amber := False;
         Io.To_Point_Green := True;
         Io.Approach_Iaf_Amber := False;
         Io.Approach_Iaf_Green := True;

      end if;

      ----------------------------------------------------------------------
      -- Set The Controls not in agreement page if comparison disagrees.  --
      -- Display the page IF the required flaps and the flap handle are   --
      -- not set the same, OR IF the required gear position and the gear  --
      -- handle are not the same, OR IF the required Engine Torque and the--
      -- Torque available are not within 3% AND the required Engine       --
      -- torque value is greater than zero, OR IF the required parking    --
      -- brake setting and the parking brake handle disagree.  On top of  --
      -- All that, ONLY set the Controls not in agreement page if flight  --
      -- freeze is still on.  Releasing the sim from freeze will clear    --
      -- the page even if the controls are not in agreement.              --
      ----------------------------------------------------------------------

      -- Set a local flap handle variable to compare to required
      if JPATS_Secondary_Flight_Controls.Flap_Handle_Up then
         Local_Flap_Handle := 0;
      elsif JPATS_Secondary_Flight_Controls.Flap_Handle_Landing then
         Local_Flap_Handle := 2;
      else
         Local_Flap_Handle := 1;
      end if;

      if Update_Torque then
         Ios.Cnia_Required_Torque := JPATS_Powerplant.Engine_Torque_Pct;
      end if;

      Ios.Cnia_Display_Page :=
        ((Ios.Cnia_Required_Flaps /= Local_Flap_Handle)
        or (Ios.Cnia_Required_Gear /= JPATS_Landing_Gear.Gear_Handle_Is_Up)
        or (abs(Ios.Cnia_Required_Torque
                - JPATS_Powerplant.Engine_Torque_Pct) > 10.0)
        or (Ios.Cnia_Required_Parking_Brake
            /= JPATS_Landing_Gear.Parking_Brake_Set))
        and Sim_Ac.Get_Flight_Freeze
        and not Cnia_Page_Cleared;

      Cnia_Page_Cleared := not Ios.Cnia_Display_Page;

      Active_N1 := Repo_Command.Active;
      Snapshot_Previous_Pass := Snapshot;
   end Update;

end JPATS_Reposition.Controller;



