-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Unchecked_Conversion;

-------------------------------------------------------------------------------
-- This package defines a header class for data saved on a buffer stream
-------------------------------------------------------------------------------
package body Saved_Data_Header is

   Bytes_Per_Element  : constant := (Ada.Streams.Stream_Element'Size + 7) / 8;
   Bytes_Per_Instance : constant Natural := (Instance'Size + 7) / 8;

   Elements_Per_Header : constant Ada.Streams.Stream_Element_Offset :=
     Ada.Streams.Stream_Element_Offset(
       (Bytes_Per_Instance + (Bytes_Per_Element - 1)) / Bytes_Per_Element);
   subtype Header_Array is Ada.Streams.Stream_Element_Array (1 .. Elements_Per_Header);

   function From_Instance is new Unchecked_Conversion
     (Source => Instance,
      Target => Header_Array);

   function To_Instance is new Unchecked_Conversion
     (Source => Header_Array,
      Target => Instance);

   -------------------------------------------------------------------------------
   -- Simple routines to convert to and from a Data_Header.
   -------------------------------------------------------------------------------
   function To_Stream_Element_Array (Source : in Instance) return
     Ada.Streams.Stream_Element_Array is
   begin
      return From_Instance(Source);
   end To_Stream_Element_Array;

   function From_Stream_Element_Array (Source : in Ada.Streams.Stream_Element_Array) return
     Instance is
   begin
      return To_Instance (Source);
   end From_Stream_Element_Array;

   -------------------------------------------------------------------------------
   -- Returns the number of stream elements in an instance of a saved data header.
   -- This number does not change, and may be treated as a constant.
   -------------------------------------------------------------------------------
   function Elements return Ada.Streams.Stream_Element_Offset is
   begin
      return Elements_Per_Header;
   end Elements;

   -------------------------------------------------------------------------------
   -- Returns the number of bytes in an instance of a saved data header.
   -- This number does not change, and may be treated as a constant.
   -------------------------------------------------------------------------------
   function Bytes return Natural is
   begin
      return Bytes_Per_Instance;
   end Bytes;

   -------------------------------------------------------------------------------
   -- Accessor methods to get the size from a header
   -------------------------------------------------------------------------------
   function Size (Header : in Instance) return Natural is
   begin
      return Header.Size;
   end Size;

   function Size
     (Header : in Ada.Streams.Stream_Element_Array)
      return Natural
   is
   begin
      return To_Instance(Header).Size;
   end Size;

   -------------------------------------------------------------------------------
   -- Accessor methods to get the time from a header
   -------------------------------------------------------------------------------
   function Time (Header : in Instance) return Ada.Real_Time.Time is
   begin
      return Header.Time;
   end Time;

   function Time
     (Header : in Ada.Streams.Stream_Element_Array)
      return Ada.Real_Time.Time
   is
   begin
      return To_Instance(Header).Time;
   end Time;

   -------------------------------------------------------------------------------
   -- Constructor methods
   -------------------------------------------------------------------------------
   function Create (Size : in Natural;
                    Time : in Ada.Real_Time.Time
                   ) return Instance is
   begin
      return (Time => Time, Size => Size);
   end Create;

   function Create (Size : in Natural;
                    Time : in Ada.Real_Time.Time
                   ) return Ada.Streams.Stream_Element_Array is
   begin
      return From_Instance((Time => Time, Size => Size));
   end Create;


   --
   -- Stream I/O methods
   --
   -------------------------------------------------------------------------------
   -- Write method.
   -- This method is guaranteed to have the same effect as calling
   -- Write (Stream, To_Stream_Element_Array(Item)) on the given stream
   -------------------------------------------------------------------------------
   procedure Write
     (Stream : access Ada.Streams.Root_Stream_Type'Class;
      Item   : in     Instance
     ) is
   begin

      Ada.Streams.Write
        (Stream => Stream.all,
         Item   => From_Instance(Item)
        );

   end Write;

   -------------------------------------------------------------------------------
   -- Read method.
   -- This method is guaranteed to have the same effect as calling
   -- Read (Stream, To_Stream_Element_Array(Item), Trash) on the given stream
   -- (except that the To_Stream_Element_Array part isn't quite legal).
   -------------------------------------------------------------------------------
   procedure Read
     (Stream : access Ada.Streams.Root_Stream_Type'Class;
      Item   : out    Instance
     ) is

      Trash : Ada.Streams.Stream_Element_Offset;
      Item_Elements : Header_Array;
   begin

      Ada.Streams.Read
        (Stream => Stream.all,
         Item   => Item_Elements,
         Last   => Trash
         );

      Item := To_Instance (Item_Elements);

   end Read;


end Saved_Data_Header;

