-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Real_Time;

with Scheduler.Snapshot;

-------------------------------------------------------------------------------
-- This class compiles and buffers save/restore data. Methods are provided for
-- saving the last 10 minutes or so of data to disk, for saving a snapshot to
-- disk, and for restoring either kind of save from disk.
-------------------------------------------------------------------------------
package Save_Restore is

   -------------------------------------------------------------------------------
   -- Save all of the last 10 minutes of the simulation past the given start time
   -- to the given file.
   -------------------------------------------------------------------------------
   procedure Save (File_name  : in String;
                   Start_Time : in Ada.Real_Time.Time);

   -------------------------------------------------------------------------------
   -- Replay the simulation from the given file at the given offset.
   -------------------------------------------------------------------------------
   procedure Replay (File_name : in String; Offset : in Duration);

   -------------------------------------------------------------------------------
   -- Replay the last Offset seconds of the simulation
   -------------------------------------------------------------------------------
   procedure Replay (Offset : in Duration);

   -------------------------------------------------------------------------------
   -- Save a snapshot of the simulation.
   -------------------------------------------------------------------------------
   procedure Snapshot;

   -------------------------------------------------------------------------------
   -- Restore a snapshot of the simulation from the given index.
   -------------------------------------------------------------------------------
   procedure Restore (Index : in Scheduler.Snapshot.Snapshot_Index);

   -------------------------------------------------------------------------------
   -- Temporarily pause the replay. As a side-effect, the simulation will be
   -- put into freeze. If False is passed in, a paused replay will continue.
   -------------------------------------------------------------------------------
   procedure Pause_Replay (Pause : in Boolean := True);

   -------------------------------------------------------------------------------
   -- Halt the replay and resume recording of the simulation. The recording of
   -- the previous 10 minutes will be lost.
   -------------------------------------------------------------------------------
   procedure Stop_Replay;

   -------------------------------------------------------------------------------
   -- Notify the Data Dispatcher that new save data has arrived on the Channel.
   -------------------------------------------------------------------------------
   procedure Save_Data_Arrived (Subsystem : in Scheduler.Handle);

   -------------------------------------------------------------------------------
   -- Notify the Data Dispatcher that the requested amount of restore space has
   -- arrived on the requested restore stream.
   -------------------------------------------------------------------------------
   procedure Restore_Space_Arrived;

   -------------------------------------------------------------------------------
   -- This routine returns True when a replay (from a file or memory) is in
   -- progress. A Restore does not affect matters.
   -------------------------------------------------------------------------------
   function Replay_In_Progress return Boolean;

   -------------------------------------------------------------------------------
   -- This routine returns True when a replay (from a file or memory) is trimming.
   -------------------------------------------------------------------------------
   function Replay_Trimming return Boolean;

private

   -------------------------------------------------------------------------------
   -- This routine returns the length of the named replay file.
   -------------------------------------------------------------------------------
   function Demo_Length (File_Name : in String) return Duration;


end Save_Restore;
