-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Strings; use Ada.Strings;
with Ada.Strings.Fixed; use Ada.Strings.Fixed;
with Ada.Strings.Unbounded;
with Ada.Streams;
with Ada.Streams.Stream_Io;
with Ada.Real_Time;

with Buffer_Stream;
with Buffer_Stream.Save;
with Buffer_Stream.Restore;
with Buffer_Stream.Simulation;
with JPATS_DCLS;
with Jpats_Formation_Profile_Playback;
with Jpats_Formation_Types;
with Jpats_Formation_Manager;
with Log;
with Restore_Stream_Notifier;
with Scheduler.Registrar;
with Scheduler.Executive;
with Scheduler.Configuration;
with Scheduler.IO;
with Saved_Data_Header;
with Save_Restore.IOS_Interface;
with Simulation_Dictionary;

use type Scheduler.Handle;

pragma Elaborate_All (Saved_Data_Header, Simulation_Dictionary);
-------------------------------------------------------------------------------
-- This class compiles and buffers save/restore data. Methods are provided for
-- saving the last 10 minutes or so of data to disk, for saving a snapshot to
-- disk, and for restoring either kind of save from disk.
-------------------------------------------------------------------------------
package body Save_Restore is

   package Formation renames Jpats_Formation_Profile_Playback;
   package Formation_Mgr renames Jpats_Formation_Manager;
   package Formation_Types renames Jpats_Formation_Types;

   subtype Header_Element_Array is Ada.Streams.Stream_Element_Array
     (1..Saved_Data_Header.Elements);

   -- This is declared here so that we can safely 'access it.
   Buffer : aliased Buffer_Stream.Simulation.Instance;

   -- The file used for any file-based replays
   Replay_File : Ada.Streams.Stream_IO.File_Type;

   -- Pointer to the streams used for replays
   Replay_Source : Ada.Streams.Stream_IO.Stream_Access;
   Replay_Target : Buffer_Stream.Restore.Handle;

   Replaying : Boolean := False;
   Trimming  : Boolean := False;
   Max_Offset : Duration := 0.0;

   Minutes_Per_Second : constant := 1.0/60.0; 

   Corrupt_Data : exception;

   -----------------------
   -- Internal Routines --
   -----------------------

   -------------------------------------------------------------------------------
   -- The save/restore data dispatcher task
   -------------------------------------------------------------------------------
   task Data_dispatcher is
      entry Save   (File_name : in String; Start_Time : in Ada.Real_Time.Time);
      entry Replay (File_name : in String; Offset : in Duration);
      entry Replay (Offset : in Duration);
      entry Pause;
      entry Start;
      entry Stop_Replay;
      entry Snapshot;
      entry Restore (Index : in Scheduler.Snapshot.Snapshot_Index);
      entry Save_Data_Arrived (Subsystem : in Scheduler.Handle);
      entry Restore_Space_Arrived;
   end Data_dispatcher;


   -------------------------------------
   -- Data Dispatcher helper routines --
   -- The following routines are used --
   -- inside the Data_Dispatcher to   --
   -- perform requested actions.      --
   -------------------------------------

   ----------------------------------------------------------------------------
   -- Find the time span until the given cycle occurs next.
   ----------------------------------------------------------------------------
   function Time_To_Cycle (Target : in Natural) return Ada.Real_Time.Time_Span is
      use type Ada.Real_Time.Time;

      Target_Cycle : Natural := Target;
   begin
      -- Make sure the result isn't negative or very close to 0
      if Scheduler.Executive.Last_Cycle > Target_Cycle then
         Target_Cycle := Target_Cycle + Scheduler.Configuration.Cycles_Per_Frame;
      end if;

      Log.Report ("Cycle is " & Integer'Image (Scheduler.Executive.Last_Cycle));
      Log.Report ("Time to target cycle is" &
                  Float'Image
                  (Float (Target_Cycle - Scheduler.Executive.Last_Cycle) *
                   Scheduler.Configuration.Cycle_Length
                   )
                  );

      return
        Ada.Real_Time.To_Time_Span
        (Duration (Float (Target_Cycle - Scheduler.Executive.Last_Cycle) *
                   Scheduler.Configuration.Cycle_Length
                   )
         );

   end Time_To_Cycle;

   ----------------------------------------------------------------------------
   -- Find the time to use for entities that will need to be available at the
   -- start of the next frame.
   ----------------------------------------------------------------------------
   function Just_Before_Next_Frame return Ada.Real_Time.Time is
      use type Ada.Real_Time.Time;

      -- Items need to occur this much earlier in the frame, to ensure that they are available
      -- for the model at the right time.
      Clock_Fudge : constant Float := 1.5 * Scheduler.Configuration.Cycle_Length;
   begin

      Log.Report ("Cycle is " & Integer'Image (Scheduler.Executive.Last_Cycle));
      Log.Report ("Time to target cycle is" &
                  Float'Image
                  (Float (Scheduler.Configuration.Cycles_Per_Frame - Scheduler.Executive.Last_Cycle) *
                   Scheduler.Configuration.Cycle_Length
                   - Clock_Fudge
                   )
                  );
      return
        Ada.Real_Time.Clock +
        Ada.Real_Time.To_Time_Span
        (Duration
         (Float (Scheduler.Configuration.Cycles_Per_Frame - Scheduler.Executive.Last_Cycle) *
          Scheduler.Configuration.Cycle_Length
          - Clock_Fudge
          )
         );

   end Just_Before_Next_Frame;


   ----------------------------------------------------------------------------
   -- Check the stream to see if it conatins any data to read.
   -- If its not a buffer stream, then it is assumed to be a file-based stream
   -- associated with Replay_File (which must be open).
   ----------------------------------------------------------------------------
   function Data_Available (From : access Ada.Streams.Root_Stream_Type'Class) return Boolean is
   begin
      if From.all in Buffer_Stream.Simulation.Instance'Class then
         return Buffer_Stream.Data_Available (Buffer_Stream.Handle(From));
      else
         return not Ada.Streams.Stream_IO.End_Of_File (Replay_File);
      end if;
   end Data_Available;

   ----------------------------------------------------------------------------
   -- Close the given stream source.
   -- If its not a buffer stream, then it is assumed to be a file-based stream
   -- associated with Replay_File (which must be open).
   ----------------------------------------------------------------------------
   procedure Close_Source (Source : access Ada.Streams.Root_Stream_Type'Class) is
   begin
      if Source.all in Buffer_Stream.Simulation.Instance'Class then
          Buffer_Stream.Simulation.Close_Reads (Buffer_Stream.Simulation.Handle(Source).all);
      else
         Ada.Streams.Stream_IO.Close (Replay_File);
      end if;

      -- Flush the replay buffer.
      -- We do this because if we didn't there'd be a gap in the recording that
      -- would be hard to account for during a later replay. It might be possible
      -- to recode things to put some kind of delay event in the replay buffer
      -- that would allow the clocks to be resynched.
      Buffer_Stream.Simulation.Open_Writes (Buffer);
      Buffer_Stream.Simulation.Flush (Buffer);
      Buffer_Stream.Simulation.Close_Writes (Buffer);

   end Close_Source;

   -------------------------------------------------------------------------------
   -- Copy a simulation entry from one stream to another
   -------------------------------------------------------------------------------
   procedure Copy_Simulation_Stream_Entry
     (From_Stream : access Ada.Streams.Root_Stream_Type'Class;
      To_Stream   : access Ada.Streams.Root_Stream_Type'Class;
      Entry_Time  : out    Ada.Real_Time.Time
     ) is

      Header : Ada.Streams.Stream_Element_Array (1..Saved_Data_Header.Elements);
      Last_Element : Ada.Streams.Stream_Element_Offset;
   begin

      -- Read in the header
      Ada.Streams.Read
        (Stream => From_Stream.all,
         Item   => Header,
         Last   => Last_Element
         );

      Ada.Streams.Write
        (Stream => To_Stream.all,
         Item   => Header
         );

      Entry_Time := Saved_Data_Header.Time (Saved_Data_Header.From_Stream_Element_Array (Header));

      -- Copy the rest of the data
      declare
         Load : Ada.Streams.Stream_Element_Array (1..
                Ada.Streams.Stream_Element_Offset (Saved_Data_Header.Size (Header)));
      begin

         Ada.Streams.Read
           (Stream => From_Stream.all,
            Item   => Load,
            Last   => Last_Element
           );

         Ada.Streams.Write
           (Stream => To_Stream.all,
            Item   => Load
           );

      end;

   end Copy_Simulation_Stream_Entry;

   -------------------------------------------------------------------------------
   -- Store save data from the given stream into the 10 minute buffer.
   -------------------------------------------------------------------------------
   procedure Save_New_Data (Subsystem : in     Scheduler.Handle;
                            Buffer    : access Buffer_Stream.Simulation.Instance;
                            Time      :    out Ada.Real_Time.Time) is

      Subsystem_Name : constant String := Scheduler.Registrar.Retrieve_Name (Subsystem);

   begin
      Buffer_Stream.Simulation.Open_Writes (Buffer.all);

      -- Write a header into the 10-minute stream noting the source
      String'Output (Buffer, Subsystem_Name);

      -- Copy the data from one stream into the other

      Buffer_Stream.Save.Open_Reads(Scheduler.Save_Stream(Subsystem.all).all);

      Copy_Simulation_Stream_Entry
        (From_Stream => Scheduler.Save_Stream(Subsystem.all),
         To_Stream   => Buffer,
         Entry_Time  => Time
         );

      Buffer_Stream.Save.Close_Reads(Scheduler.Save_Stream(Subsystem.all).all);

      Buffer_Stream.Simulation.Close_Writes (Buffer.all);

   end Save_New_Data;

   -------------------------------------------------------------------------------
   -- Read off any leftover save data from the given subsystem.
   -------------------------------------------------------------------------------
   procedure Discard_Save_Data (Subsystem : in Scheduler.Handle) is

      Stream : constant Buffer_Stream.Save.Handle := Scheduler.Save_Stream(Subsystem.all);

      Read_Buffer : Ada.Streams.Stream_Element_Array
        (1..Ada.Streams.Stream_Element_Offset
         (Natural'(Buffer_Stream.Save.Data_Available(Stream))));

      Last_Element : Ada.Streams.Stream_Element_Offset;
   begin
      Buffer_Stream.Save.Open_Reads(Stream.all);
      Buffer_Stream.Save.Read
        (Stream => Stream.all,
         Item   => Read_Buffer,
         Last   => Last_Element
         );
      Buffer_Stream.Save.Close_Reads(Stream.all);
   end Discard_Save_Data;

   -------------------------------------------------------------------------------
   -- Skip over the first Offset seconds of entries in the input file. The actual
   -- time stamp of the next entry in the stream will be returned.
   --
   -- For this routine to operate correctly, the data in the file has to conform
   -- to a certain expected format. In particular, it should have been initially
   -- written to a simulation stream.
   -------------------------------------------------------------------------------
   procedure Skip (File         : in out Ada.Streams.Stream_IO.File_Type;
                   Offset       : in     Ada.Real_Time.Time_Span;
                   Time         :    out Ada.Real_Time.Time;
                   Initial_Time :    out Ada.Real_Time.Time
                  ) is

      use type Ada.Real_Time.Time;
      use type Ada.Real_Time.Time_Span;
      use type Ada.Streams.Stream_Io.Count;

      File_Stream : constant Ada.Streams.Stream_Io.Stream_Access :=
        Ada.Streams.Stream_Io.Stream (File);

      -- Entry skipping variables
      Entry_Start  : Ada.Streams.Stream_Io.Positive_Count;
      Header       : Ada.Streams.Stream_Element_Array(1..Saved_Data_Header.Elements);

      Last : Ada.Streams.Stream_Element_Offset;
   begin
      Entry_Start := Ada.Streams.Stream_Io.Index(File);

      -- Read in the source (which we don't care about)
      declare
         Trash : constant String := String'Input (File_Stream);
      begin
         null;
      end;

      -- Read in the header.
      Ada.Streams.Read
        (Stream => File_Stream.all,
         Item   => Header,
         Last   => Last
         );

       -- Save off the first time
      Initial_Time := Saved_Data_Header.Time(Header);

      -- Quit looping when we find an entry later than the given delta
      while Saved_Data_Header.Time(Header) < Initial_Time + Offset loop

         -- Skip over the rest of the data in the entry
         if Saved_Data_Header.Size (Header) > 0 then

            Ada.Streams.Stream_IO.Set_Index
              (File => File,
               To   => Ada.Streams.Stream_IO.Index (File) +
               Ada.Streams.Stream_IO.Count(Saved_Data_Header.Size (Header))
               );

         end if;

         -- If we have read all the data, quit.
         if Ada.Streams.Stream_Io.Size(File) = 0 then
            return;
         end if;

         Entry_Start := Ada.Streams.Stream_Io.Index(File);

         -- Read in the source (which we don't care about)
         declare
            Trash : constant String := String'Input (File_Stream);
         begin
            null;
         end;

         -- Read in the header.
         Ada.Streams.Read
           (Stream => File_Stream.all,
            Item   => Header,
            Last   => Last
           );

      end loop;

      Ada.Streams.Stream_IO.Set_Index
        (File => File,
         To   => Entry_Start
         );

      Time := Saved_Data_Header.Time(Header);
   end Skip;

   -------------------------------------------------------------------------------
   -- Get the next restore header and targetted stream from the given stream.
   -- This routine automaticly discards any data with a bad source. When there was
   -- no more (valid) data to read from the input source, null is returned for the
   -- destination.
   -------------------------------------------------------------------------------
   procedure Get_Next_Restore_Header
     (Source      : access Ada.Streams.Root_Stream_Type'Class;
      Header      :    out Header_Element_Array;
      Destination :    out Scheduler.Handle
     ) is

      Last_Element : Ada.Streams.Stream_Element_Offset;
   begin
      -- Get the next subsystem to restore to
      loop

         -- If we have run out of data, set the subsystem to null and quit
         if not Data_Available (Source) then
            Destination := null;
            return;
         end if;

         begin
         declare
            Source_Name : constant String := String'Input(Source);
         begin
            Destination := Scheduler.Registrar.Retrieve_Handle (Source_Name);
            exit when Destination /= null;

            -- If there is no such subsystem, spit out a warning and continue
            Log.Report (Event    => "Unexpected data source """ & Source_Name &
                        """. Replay data may be corrupt.",
                        Severity => Log.Warning);

            -- Read in the header and its data, just to get rid of it.
            Ada.Streams.Read
              (Stream => Source.all,
               Item   => Header,
               Last   => Last_Element
               );


            if Saved_Data_Header.Size (Header) > 0 then
               declare
                  Load : Ada.Streams.Stream_Element_Array
                    (1..Ada.Streams.Stream_Element_Offset (Saved_Data_Header.Size (Header)));
               begin

                  Ada.Streams.Read
                    (Stream => Source.all,
                     Item   => Load,
                     Last   => Last_Element
                     );
               exception
                  when Constraint_Error =>
                     Log.Report
                       (Event    => "Corrupt data found in file. Bad saved data header read.",
                        Severity => Log.Error);
                     raise Corrupt_Data;
               end;
            end if;
         end;
         exception
            when Constraint_Error =>
               Log.Report
                 (Event    => "Corrupt data found in file. Bad saved data destination read.",
                  Severity => Log.Error);
               raise Corrupt_Data;
         end;
      end loop;

      -- Read in the next header
      Ada.Streams.Read
        (Stream => Source.all,
         Item   => Header,
         Last   => Last_Element
         );

   end Get_Next_Restore_Header;

   ----------------------------------------------------------------------------
   -- Copy as much data as possible from the source stream into the appropriate
   -- restore stream without having to wait for buffer space to free up.
   --
   -- Last_Header should be the last header that was read from the buffer.
   ----------------------------------------------------------------------------
   procedure Copy_Restore_Data_Batch
     (Source : access Ada.Streams.Root_Stream_Type'Class;
      Header : in out Header_Element_Array;
      Target : in out Buffer_Stream.Restore.Handle
     ) is

      Last_Element : Ada.Streams.Stream_Element_Offset;
      use type Buffer_Stream.Restore.Handle;

      Subsystem : Scheduler.Handle;
   begin
      loop

         -- Verify that there's enough space in the restore stream for the data.
         if
           Buffer_Stream.Restore.Space_Available (Target) <
           Header'Length + Saved_Data_Header.Size(Header)
         then
            Restore_Stream_Notifier.Agent.Look_For
              (Stream => Target,
               Amount => Header'Length + Saved_Data_Header.Size(Header)
               );
            return;
         end if;

         -- Copy the header and its data into the restore stream
         Buffer_Stream.Restore.Open_Writes (Target.all);

         Buffer_Stream.Restore.Write
           (Stream => Target.all,
            Item   => Header
            );

         -- The if part should not be needed, strictly speaking. However,
         -- GreenHills AdaMulti 1.8.9b has a bug with reading 0 length
         -- arrays from files.
         if Saved_Data_Header.Size (Header) > 0 then

            declare
               Load : Ada.Streams.Stream_Element_Array
                 (1..Ada.Streams.Stream_Element_Offset (Saved_Data_Header.Size (Header)));
            begin

               Ada.Streams.Read
                 (Stream => Source.all,
                  Item   => Load,
                  Last   => Last_Element
                  );

               Buffer_Stream.Restore.Write
                 (Stream => Target.all,
                  Item   => Load
                  );

            end;
         end if;

         Buffer_Stream.Restore.Close_Writes (Target.all);

         Get_Next_Restore_Header
           (Source      => Source,
            Header      => Header,
            Destination => Subsystem
            );
         if Subsystem = null then
            Target := null;
            return;
         end if;

         Target := Scheduler.Restore_Stream (Subsystem.all);

      end loop;
   end Copy_Restore_Data_Batch;

   -------------------------------------------------------------------------------
   -- Find the next record in the given replay stream from which it would be safe
   -- to start a replay.
   --
   -- This should be the first IO record in the series of IO records before a
   -- series of snapshots (non-IO records).
   -------------------------------------------------------------------------------
   procedure Find_Snapshot_Series (Source : access Ada.Streams.Root_Stream_Type'Class) is

      use type Ada.Real_Time.Time;

      -- The time stamp of the most recently encountered non-I/O entry.
      Last_Snapshot_Time : Ada.Real_Time.Time;
      Time_Updated       : Boolean;

      -- The location for the first entry after the previous non-I/O entry. This
      -- will probably be an I/O entry, but it could be the next non-I/O entry.
      Next_Entry : Buffer_Stream.Simulation.Read_Location;
      Next_Index : Ada.Streams.Stream_IO.Positive_Count := 1;

      Simulation_Header : Header_Element_Array;
      Subsystem : Scheduler.Handle;
   begin

      -- Mark the location of the first entry, read its header in, and save its time.
      if Source.all in Buffer_Stream.Simulation.Instance'Class then
         Next_Entry := Buffer_Stream.Simulation.Location
           (Buffer_Stream.Simulation.Instance(Source.all));
      else
         Next_Index := Ada.Streams.Stream_IO.Index (Replay_File);
      end if;

      Get_Next_Restore_Header
        (Source      => Source,
         Header      => Simulation_Header,
         Destination => Subsystem
         );
      Last_Snapshot_Time := Saved_Data_Header.Time(Simulation_Header);
      Time_Updated       := True;

      -- Find the start of the next gap between (non-IO) snapshots.
      loop

         -- Ditch the data
         declare
            Load : Ada.Streams.Stream_Element_Array
              (1..Ada.Streams.Stream_Element_Offset (Saved_Data_Header.Size (Simulation_Header)));
            Last_Element : Ada.Streams.Stream_Element_Offset;
         begin

            Ada.Streams.Read
              (Stream => Source.all,
               Item   => Load,
               Last   => Last_Element
               );
         end;

         -- If we just updated the last snapshot time, save this location
         if Time_Updated then
            if Source.all in Buffer_Stream.Simulation.Instance'Class then
               Next_Entry := Buffer_Stream.Simulation.Location
                 (Buffer_Stream.Simulation.Instance(Source.all));
            else
               Next_Index := Ada.Streams.Stream_IO.Index (Replay_File);
            end if;
            Time_Updated := False;
         end if;

         Get_Next_Restore_Header
           (Source      => Source,
            Header      => Simulation_Header,
            Destination => Subsystem
            );

         -- Exit if we haven't found any (non-IO) snapshots in the amount of time it
         -- would take for all schedulers to run once.
         exit when
           Saved_Data_Header.Time(Simulation_Header) >
           Last_Snapshot_Time + Scheduler.Configuration.Simulation_Frequency;

         -- If this is a (non-IO) snapshot, save its timestamp and prepare to mark
         -- the location of the next record.
         if Subsystem.all not in Scheduler.IO.Instance'Class then
            Last_Snapshot_Time := Saved_Data_Header.Time(Simulation_Header);
            Time_Updated := True;
         end if;

      end loop;

      -- Rewind the stream to the location where the gap in front of the next series
      -- starts.
      if Source.all in Buffer_Stream.Simulation.Instance'Class then
         Buffer_Stream.Simulation.Rewind
           (Stream   => Buffer_Stream.Simulation.Instance(Source.all),
            Location => Next_Entry
            );
      else
         Ada.Streams.Stream_IO.Set_Index
           (File => Replay_File,
            To   => Next_Index
            );
      end if;

   end Find_Snapshot_Series;

   -------------------------------------------------------------------------------
   -- Copy over all the IO snapshots in the given buffer until we encounter
   -- a non-IO snapshot. The header and destination of the first non-IO snapshot
   -- encountered will be read off and returned.
   -------------------------------------------------------------------------------
   procedure Copy_Io_Snapshots
     (Source : access Ada.Streams.Root_Stream_Type'Class;
      Header :    out Header_Element_Array;
      Target :    out Buffer_Stream.Restore.Handle
     ) is

      Last_Element : Ada.Streams.Stream_Element_Offset;
      use type Buffer_Stream.Restore.Handle;

      Subsystem : Scheduler.Handle;
   begin
      loop
         Get_Next_Restore_Header
           (Source      => Source,
            Header      => Header,
            Destination => Subsystem
            );
         Target := Scheduler.Restore_Stream (Subsystem.all);

         -- Return if we run out of data
         if Target = null then
            return;
         end if;

         -- Verify that the data target is an I/O scheduler
         exit when Subsystem.all not in Scheduler.IO.Instance'Class;

         -- Flush the restore stream and copy the header and its data into it.
         Buffer_Stream.Restore.Open_Writes (Target.all);

         Buffer_Stream.Restore.Flush (Target.all);

         Buffer_Stream.Restore.Write
           (Stream => Target.all,
            Item   => Header
            );

         declare
            Load : Ada.Streams.Stream_Element_Array
              (1..Ada.Streams.Stream_Element_Offset (Saved_Data_Header.Size (Header)));
         begin

            Ada.Streams.Read
              (Stream => Source.all,
               Item   => Load,
               Last   => Last_Element
               );

            Buffer_Stream.Restore.Write
              (Stream => Target.all,
               Item   => Load
               );

         end;
         Buffer_Stream.Restore.Close_Writes (Target.all);

      end loop;

   end Copy_Io_Snapshots;

   -------------------------------------------------------------------------------
   -- Initiate a replay of the last Offset seconds from the given buffer. There
   -- is not likely to be data exactly at that offset, to Offset will be modified
   -- to match the actual offset from which replay starts.
   --
   -- Find the sim time for the given offset. Then find the first entry in the
   -- buffer later than that time and start replay from there.
   -------------------------------------------------------------------------------
   procedure Start_Replay_From_Memory
     (Offset        : in out Ada.Real_Time.Time_Span;
      Buffer        : access Buffer_Stream.Simulation.Instance;
      Header        :    out Header_Element_Array;
      Destination   :    out Buffer_Stream.Restore.Handle
     ) is

      use type Ada.Real_Time.Time;
      use type Ada.Real_Time.Time_Span;

      Actual_Time : Ada.Real_Time.Time;

   begin

      Scheduler.Executive.Enable_Saving (False);

      Buffer_Stream.Simulation.Open_Reads (Buffer.all);

      -- Skip up to the next record after Offset seconds from now.
      Buffer_Stream.Simulation.Skip
        (Stream => Buffer.all,
         Past   => Ada.Real_Time.Clock + Offset,
         Time   => Actual_Time
        );

      -- Skip further up to the next gap between model snapshots.
      Find_Snapshot_Series (Buffer);

      -- Copy over the IO snapshots immediately preceeding the model snapshots
      Copy_Io_Snapshots
        (Source => Buffer,
         Header => Header,
         Target => Destination
         );

      -- Change Offset to the actual offset to the next major cycle
      Offset := Saved_Data_Header.Time(Header) - Just_Before_Next_Frame;
      Buffer_Stream.Restore.Set_Playback_Offset (Offset);

      -- Copy over the first batch of restore data
      Copy_Restore_Data_Batch
        (Source => Buffer,
         Header => Header,
         Target => Destination
         );

   end Start_Replay_From_Memory;

   -------------------------------------------------------------------------------
   -- Replay the last Offset seconds from the start of the given file.
   --
   -- Find the sim time for the given offset. Then find the first entry in the
   -- buffer later than that time and start replay from there.
   -------------------------------------------------------------------------------
   procedure Start_Replay_From_File
     (Offset      : in out Ada.Real_Time.Time_Span;
      File_Name   : in     String;
      Last_Time   :    out Ada.Real_Time.Time;
      File_Length :    out Duration;
      Header      :    out Header_Element_Array;
      Destination :    out Buffer_Stream.Restore.Handle
     ) is

      use type Ada.Real_Time.Time;
      use type Ada.Real_Time.Time_Span;
      use type Ada.Streams.Stream_IO.Count;

      File_Stream : Ada.Streams.Stream_Io.Stream_Access;

      Actual_Time  : Ada.Real_Time.Time;
      End_Time     : Ada.Real_Time.Time;

      Formation_Switch: Integer := 0;

      Subsystem   : Scheduler.Handle;

      Time_Zero  : constant Ada.Real_Time.Time :=
        Ada.Real_Time.Time_Of (0, Ada.Real_Time.To_Time_Span (0.0));
   begin

      Scheduler.Executive.Enable_Saving (False);

      -- Open the new file stream for reading
      Ada.Streams.Stream_IO.Open
        (File => Replay_File,
         Mode => Ada.Streams.Stream_IO.In_File,
         Name => File_Name
         );
      File_Stream := Ada.Streams.Stream_IO.Stream (Replay_File);

      Integer'Read(File_Stream, Formation_Switch);
      if (Formation_Switch = 1) then
         -- Read in the track # and filename (we don't care about them at this point)
         declare
            Trash_1 : constant Integer := Integer'Input(File_Stream);
            Trash_2 : constant String := String'Input (File_Stream);
        begin
           null;
        end; 
       end if;

      Ada.Real_Time.Time'Read(File_Stream, End_Time);

      -- Skip the first Offset records in the file
      Skip
        (File         => Replay_File,
         Offset       => Offset,
         Time         => Actual_Time,
         Initial_Time => Last_Time
         );

      -- Skip further up to the next gap between model snapshots.
      Find_Snapshot_Series (File_Stream);

      -- Copy over the IO snapshots immediately preceeding the model snapshots
      Copy_Io_Snapshots
        (Source => File_Stream,
         Header => Header,
         Target => Destination
         );

      -- Set the simulation's playback offset
      Offset := Saved_Data_Header.Time(Header) - Just_Before_Next_Frame;
      Buffer_Stream.Restore.Set_Playback_Offset (Offset);
      File_Length := Ada.Real_Time.To_Duration (End_Time - Last_Time);
      Log.Report ("Header time is " &
                  Duration'Image (Ada.Real_Time.To_Duration(Actual_Time - Time_Zero))
                  );

      -- Copy over the first batch of restore data
      Copy_Restore_Data_Batch
        (Source => File_Stream,
         Header => Header,
         Target => Destination
         );
   end Start_Replay_From_File;

   -------------------------------------------------------------------------------
   -- Complete the replay that was begun with Start_Replay_From_File or
   -- Start_Replay_From_Memory.
   -------------------------------------------------------------------------------
   procedure Stop_Replay (Source : access Ada.Streams.Root_Stream_Type'Class) is
   begin
      if Ios_Interface.Play_Formation_Demo then
         Ios_Interface.Play_Formation_Demo := False;
         Ios_Interface.New_Formation_Demo_Loaded := False;
         Ios_Interface.Formation_Demo_Leadship_Only := True;
      else
         Ios_Interface.Allow_Demo_Select := True;
      end if;
      Close_Source (Source);
      Scheduler.Executive.Enable_Saving;
   end Stop_Replay;

   -------------------------------------------------------------------------------
   -- Save the given replay buffer to the given disk file
   -------------------------------------------------------------------------------
   procedure Save_Replay
     (File_Name  : in     String;
      Start_Time : in     Ada.Real_Time.Time;
      Replay     : access Buffer_Stream.Simulation.Instance
     )
   is
      use type Ada.Real_Time.Time;
      Stream_File : Ada.Streams.Stream_Io.File_Type;

      Actual_Time : Ada.Real_Time.Time;

      -- Used to "shovel" data from one stream to the other
      Shovel_Capacity : constant := 1_000;
      Shovel : Ada.Streams.Stream_Element_Array (1..Shovel_Capacity);
      Shovel_Size : Ada.Streams.Stream_Element_Offset;
   begin

      -- Reset the replay stream for reading.
      Buffer_Stream.Simulation.Open_Reads (Replay.all);

      -- Open the new file stream for writing, and put the current time in it.
      Ada.Streams.Stream_Io.Create
        (File => Stream_File,
         Name => File_Name
         );

      if (Trim(IOS_Interface.Formation_Demo_Profile_Name, Both)'Length > 0) and (IOS_Interface.Record_Formation_Demo) then
         Integer'Write(Ada.Streams.Stream_IO.Stream (Stream_File), 1);
         Integer'Write(Ada.Streams.Stream_IO.Stream (Stream_File), Formation_Mgr.Get_Formation_Track_Number);
         String'Output(Ada.Streams.Stream_IO.Stream (Stream_File), Trim(IOS_Interface.Formation_Demo_Profile_Name, Both));
         IOS_Interface.Formation_Demo_Profile_Name := IOS_Interface.Bounded_64.to_string(IOS_Interface.Spaces);
      else
         Integer'Write(Ada.Streams.Stream_IO.Stream (Stream_File), 0);
      end if;

      Ada.Real_Time.Time'Write (Ada.Streams.Stream_IO.Stream (Stream_File), Ada.Real_Time.Clock);
      IOS_Interface.Set_Demo_File_Length
        (Ada.Real_Time.To_Duration (Ada.Real_Time.Clock - Start_Time));


      -- Skip forward past the given time
      Buffer_Stream.Simulation.Skip
        (Stream => Replay.all,
         Past   => Start_Time,
         Time   => Actual_Time
        );

      Log.Report ("Saving about " &
                  Integer'Image(Buffer_Stream.Simulation.Data_Available(Replay)) &
                  " bytes to demo file.");

      -- Shovel the data from the replay stream into the file stream
      while Buffer_Stream.Simulation.Data_Available(Replay) > Shovel'Length loop

         Buffer_Stream.Simulation.Read
           (Stream => Replay.all,
            Item   => Shovel,
            Last   => Shovel_Size
           );

         Ada.Streams.Stream_Io.Write
           (File => Stream_File,
            Item => Shovel
            );

      end loop;

      Buffer_Stream.Simulation.Read
        (Stream => Replay.all,
         Item   => Shovel(1..Ada.Streams.Stream_Element_Offset
                          (Natural'(Buffer_Stream.Simulation.Data_Available(Replay)))),
         Last   => Shovel_Size
         );

      Ada.Streams.Stream_Io.Write
        (File => Stream_File,
         Item => Shovel (1..Shovel_Size)
         );

      -- Close the file stream
      Ada.Streams.Stream_Io.Close (Stream_File);

      -- Reset the replay stream for writing again
      Buffer_Stream.Simulation.Close_Reads(Replay.all);

   end Save_Replay;

   task body Data_Dispatcher is

      use type Ada.Real_Time.Time;
      use type Ada.Real_Time.Time_Span;
      use type Ada.Streams.Stream_IO.Stream_Access;
      use type Buffer_Stream.Restore.Handle;

      -- This is going to have to be an estimate, I'm afraid.
      -- For that reason, it needs to be defined in the sim.cfg on the final product.
      Size_Of_Ten_Minutes : constant Integer :=
        Simulation_Dictionary.Lookup ("Recording_Buffer_Size");

      Not_Paused    : constant Ada.Real_Time.Time      := Ada.Real_Time.Time_First;
      Not_Trimming  : constant Ada.Real_Time.Time      := Ada.Real_Time.Time_First;
      Not_Replaying : constant Ada.Real_Time.Time_Span := Ada.Real_Time.Time_Span_First;
      Pause_Time    : Ada.Real_Time.Time      := Not_Paused;
      Replay_Delta  : Ada.Real_Time.Time_Span := Not_Replaying;
      Trim_Time     : Ada.Real_Time.Time      := Not_Trimming;

      Pause_Cycle   : Natural := 0;

      Last_Save_Time : Ada.Real_Time.Time := Ada.Real_Time.Clock;


      Replay_Start_Time : Ada.Real_Time.Time;
      Replay_File_Name  : Ada.Strings.Unbounded.Unbounded_String;

      Replay_Header : Header_Element_Array;

      --Snapshot_Index : Scheduler.Snapshot.Snapshot_Index;

      Flyout_Offset : Duration;

      Adjusted_Offset : Duration;

   begin

      Buffer_Stream.Simulation.Create
        (Stream   => Buffer,
         Max_Size => Size_Of_Ten_Minutes
        );

      loop

         select

            --
            -- New save data has arrived
            --
            accept Save_Data_Arrived (Subsystem : in Scheduler.Handle) do
               if Replay_Delta = Not_Replaying then
                  Save_New_Data (Subsystem => Subsystem,
                                 Buffer    => Buffer'Access,
                                 Time      => Last_Save_Time
                                 );
               else
                  Discard_Save_Data (Subsystem);
               end if;

            end Save_Data_Arrived;

         or

            --
            -- Operator has requested a replay from the last 10 minutes
            --
            accept Replay (Offset : in Duration) do
               if Replay_Delta /= Not_Replaying then

                  Buffer_Stream.Simulation.Close_Reads (Buffer);
                  Scheduler.Executive.Flush_All_Replay_Buffers;
                  if Replay_Target /= null then
                     accept Restore_Space_Arrived;
                  end if;

               end if;
               Replay_Delta := (Last_Save_Time - Ada.Real_Time.Clock) +
                 Ada.Real_Time.To_Time_Span(Offset);
            end Replay;

            -- Warn DCLS about possible discontinuities
            JPATS_DCLS.Set_Filter;

            -- Clear out any pause condition
            if Pause_Time /= Not_Paused then
               Pause_Time := Not_Paused;
               Scheduler.Executive.Freeze(False);
            end if;

            Log.Report ("Initiating replay from" &
                        Duration'Image(Ada.Real_Time.To_Duration(-Replay_Delta)) &
                        " seconds ago.");
            Replaying := True;
            Trimming  := True;
            Flyout_Offset := 0.0;
            Start_Replay_From_Memory (Offset      => Replay_Delta,
                                      Buffer      => Buffer'access,
                                      Header      => Replay_Header,
                                      Destination => Replay_Target
                                      );
            Trim_Time := Ada.Real_Time.Clock + Scheduler.Configuration.Simulation_Frequency +
              Scheduler.Configuration.Snapshot_Period;
            Replay_Source := Buffer'Access;
            Pause_Time    := Not_Paused;

            IOS_Interface.Update_Replay_Delta
              (Ada.Real_Time.To_Duration
               (Replay_Delta + Ada.Real_Time.Clock - Last_Save_Time)
               );

            Log.Report
              ("Trim will complete in" &
               Duration'Image (Ada.Real_Time.To_Duration (Trim_Time - Ada.Real_Time.Clock))
               & " seconds.");

         or
            --
            -- Operator has requested a replay from a file
            --
            accept Replay (File_name : in String; Offset : in Duration) do
               if Replay_Delta /= Not_Replaying then

                  Ada.Streams.Stream_IO.Close (Replay_File);
                  Scheduler.Executive.Flush_All_Replay_Buffers;
                  if Replay_Target /= null then
                     accept Restore_Space_Arrived;
                  end if;

               end if;

               Replay_Delta := Ada.Real_Time.To_Time_Span(Offset);
               Replay_File_Name := Ada.Strings.Unbounded.To_Unbounded_String (File_Name);
            end Replay;

            if Replay_Delta >= Ada.Real_Time.To_Time_Span(Max_Offset) then
                  Replay_Delta := Ada.Real_Time.To_Time_Span(0.0);
            end if;

            -- Warn DCLS about possible discontinuities
            JPATS_DCLS.Set_Filter;

            -- Clear out any pause condition
            if Pause_Time /= Not_Paused then
               Pause_Time := Not_Paused;
               Scheduler.Executive.Freeze(False);
            end if;

            Log.Report ("Initiating replay from " &
                        Duration'Image(Ada.Real_Time.To_Duration(Replay_Delta)) &
                        " seconds after the beginning of " &
                        Ada.Strings.Unbounded.To_String (Replay_File_Name) & ".");
            Replaying := True;
            Trimming  := True;
            begin
               Start_Replay_From_File
                 (Offset      => Replay_Delta,
                  File_Name   => Ada.Strings.Unbounded.To_String (Replay_File_Name),
                  Last_Time   => Last_Save_Time,
                  File_Length => Flyout_Offset,
                  Header      => Replay_Header,
                  Destination => Replay_Target
                  );
               Trim_Time := Ada.Real_Time.Clock + Scheduler.Configuration.Simulation_Frequency +
                 Scheduler.Configuration.Snapshot_Period;
               Replay_Source := Ada.Streams.Stream_Io.Stream(Replay_File);
               Pause_Time    := Not_Paused;

               IOS_Interface.Update_Replay_Delta
                 (Ada.Real_Time.To_Duration
                  (Replay_Delta + Ada.Real_Time.Clock - Last_Save_Time)
                  );

               Log.Report
                 ("Trim will complete in" &
                  Duration'Image (Ada.Real_Time.To_Duration (Trim_Time - Ada.Real_Time.Clock))
                  & " seconds.");

            exception
               when Ada.Streams.Stream_IO.End_Error =>
                  Log.Report (Event    => "There is no replay data """ &
                              Duration'Image(Ada.Real_Time.To_Duration(Replay_Delta)) &
                              " seconds after the beginning of " &
                              Ada.Strings.Unbounded.To_String (Replay_File_Name)
                              & """.",
                              Severity => Log.Error
                              );
                  Replaying := False;
                  Trimming  := False;
                  Replay_Delta := Not_Replaying;
                  Trim_Time    := Not_Trimming;
                  Scheduler.Executive.Freeze (False);
                  Pause_Time := Not_Paused;
                  IOS_Interface.Update_Replay_Delta (0.0);
                  Ada.Streams.Stream_IO.Close (Replay_File);
            end;

         or
            --
            -- Operator has requested a replay be started
            --
            accept Start;
            if Pause_Time = Not_Paused then
               Log.Report (Event    => "Replay start request ingored when replay not paused.",
                           Severity => Log.Warning);
            else

               -- Reset the playback offset for processing timestamped playback records
               Replay_Delta := Replay_Delta - (Ada.Real_Time.Clock - Pause_Time)
                 - Time_To_Cycle (Pause_Cycle);
               Buffer_Stream.Restore.Set_Playback_Offset (Replay_Delta);

               IOS_Interface.Update_Replay_Delta
                 (Ada.Real_Time.To_Duration
                  (Replay_Delta + Ada.Real_Time.Clock - Last_Save_Time)
                  );

               -- Reset the trim time offset
               if Trim_Time /= Not_Trimming then
                  Trim_Time := Trim_Time + (Ada.Real_Time.Clock - Pause_Time);
               end if;

               Pause_Time := Not_Paused;
               Scheduler.Executive.Thaw_On_Cycle (Pause_Cycle);
            end if;

         or
            accept Pause;

            if Replay_Delta /= Not_Replaying and Pause_Time = Not_Paused then

               -- Freeze the sim, Make note of the time, and go wait for another start
               Scheduler.Executive.Freeze;
               Pause_Time  := Ada.Real_Time.Clock;
               Pause_Cycle := Scheduler.Executive.Last_Cycle;
               Log.Report ("Simulation replay paused.");

            else
               Log.Report (Event    => "Replay pause request ingored when not replaying.",
                           Severity => Log.Warning);
            end if;

         or
            when Trim_Time = Not_Trimming or else Ada.Real_Time.Clock < Trim_Time =>
            -- We have space to write out another batch of restore data
            accept Restore_Space_Arrived;
            -- Copy over as much new data as we can without blocking
            Copy_Restore_Data_Batch
              (Source => Replay_Source,
               Header => Replay_Header,
               Target => Replay_Target
               );

            IOS_Interface.Update_Replay_Delta
              (Ada.Real_Time.To_Duration
               (Replay_Delta + Ada.Real_Time.Clock - Last_Save_Time)
               );

         or
            accept Stop_Replay;

            if Replay_Delta = Not_Replaying then
               Log.Report (Event    => "Replay start flyout request ignored when not replaying.",
                           Severity => Log.Warning);
            else
               Scheduler.Executive.Flush_All_Replay_Buffers;

               if Replay_Target /= null then
                  accept Restore_Space_Arrived;
               end if;

               Replay_Delta := Not_Replaying;
               Replaying    := False;

               IOS_Interface.Update_Replay_Delta (Flyout_Offset);

               Stop_Replay (Replay_Source);

               if Pause_Time /= Not_Paused then
                  Scheduler.Executive.Freeze (False);
                  Pause_Time := Not_Paused;
               end if;
               Log.Report ("Playback flyout initiated.");
            end if;

         or

            --
            -- Save the 10 minute replay buffer to a file
            --
            accept Save (File_Name  : in String;
                         Start_Time : in Ada.Real_Time.Time) do

               Replay_File_Name  := Ada.Strings.Unbounded.To_Unbounded_String (File_Name);
               Replay_Start_Time := Start_Time;
            end Save;

            Log.Report ("Saving demo file """ &
                        Ada.Strings.Unbounded.To_String (Replay_File_Name) & """...");
            Save_Replay
              (File_Name  => Ada.Strings.Unbounded.To_String (Replay_File_Name),
               Start_Time => Replay_Start_Time,
               Replay     => Buffer'Access
               );
            Log.Report ("...demo file saved.");
            IOS_Interface.Demo_File_Saved;

         or
            accept Snapshot;
            Scheduler.Snapshot.Create;
            Log.Report ("Snapshot created.");
         or
            accept Restore (Index : in Scheduler.Snapshot.Snapshot_Index) do
               if (not Scheduler.Snapshot.Snapshot_Index_Locked) then
                  Scheduler.Snapshot.Restore (Index);
                  Log.Report ("Snapshot" & Scheduler.Snapshot.Snapshot_Index'Image(Index) &
                        " restored.");
               end if;
            end Restore;

         or when Replay_Delta /= Not_Replaying and Pause_Time = Not_Paused and
           Replay_Target = null =>

            --
            -- If we are performing a replay and have read all the data, wait for the
            -- sim to catch up.
            --
            delay until Saved_Data_Header.Time (Replay_Header) - Replay_Delta;

            if Trim_Time /= Not_Trimming then
               Log.Report ("Read all data already during trim.");
            end if;
            -- Freeze the sim, Make note of the time, and go wait for another request
            Scheduler.Executive.Freeze;
            Pause_Time := Ada.Real_Time.Clock;
            IOS_Interface.Replay_Complete;
            Log.Report ("Replay finished. Simulation replay paused.");

         or when Trim_Time /= Not_Trimming and Pause_Time = Not_Paused =>

            --
            -- If we are still trimming, wait for the trim to complete
            --
            delay until Trim_Time;

            -- Freeze the sim, Make note of the time, and go wait for another start
            Scheduler.Executive.Freeze;
            Pause_Time  := Ada.Real_Time.Clock;
            Pause_Cycle := Scheduler.Executive.Last_Cycle;
            Log.Report ("Paused on cycle" & Integer'Image (Scheduler.Executive.Last_Cycle));
            Trim_Time   := Not_Trimming;
            Trimming    := False;
            IOS_Interface.Replay_Ready;
            Log.Report ("Replay trimmed and ready.");

         end select;
      end loop;

   exception
      when Error : others =>
         Log.Report (Event => "Save_Restore.Data_Dispatcher terminated due to " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);

   end Data_Dispatcher;

   -----------------------
   -- External Routines --
   -----------------------

   -------------------------------------------------------------------------------
   -- Replay the simulation from the given file at the given offset.
   -------------------------------------------------------------------------------
   procedure Replay (File_name : in String; Offset : in Duration) is
   begin
      Data_Dispatcher.Replay
        (File_Name => File_Name,
         Offset    => Offset);
   end Replay;

   -------------------------------------------------------------------------------
   -- Replay the last Offset seconds of the simulation
   -------------------------------------------------------------------------------
   procedure Replay (Offset : in Duration) is
   begin
      if Offset < 0.0 then
         Data_Dispatcher.Replay(Offset);
      else
         Log.Report
           (Event => "Request to perform instant replay starting at" &
            Duration'Image(Offset) & " seconds in the future ignored.",
            Severity => Log.Warning
            );
      end if;
   end Replay;

   -------------------------------------------------------------------------------
   -- Save the last 10 minutes of the simulation to the given file
   -------------------------------------------------------------------------------
   procedure Save (File_name  : in String;
                   Start_Time : in Ada.Real_Time.Time) is
   begin
      Data_Dispatcher.Save(File_Name  => File_Name,
                           Start_Time => Start_Time
                           );
   end Save;

   -------------------------------------------------------------------------------
   -- Save a snapshot of the simulation to the given index.
   -------------------------------------------------------------------------------
   procedure Snapshot is
   begin
      Data_Dispatcher.Snapshot;
   end Snapshot;

   -------------------------------------------------------------------------------
   -- Restore a snapshot of the simulation from the given index.
   -------------------------------------------------------------------------------
   procedure Restore (Index : in Scheduler.Snapshot.Snapshot_Index) is
   begin
      Data_Dispatcher.Restore (Index);
   end Restore;

   -------------------------------------------------------------------------------
   -- Temporarily pause the replay. As a side-effect, the simulation will be
   -- put into freeze. If False is passed in, a paused replay will continue.
   -------------------------------------------------------------------------------
   procedure Pause_Replay (Pause : in Boolean := True) is
   begin
      if Pause then
         Data_Dispatcher.Pause;
      else
         Data_Dispatcher.Start;
      end if;
   end Pause_Replay;

   -------------------------------------------------------------------------------
   -- Halt the replay and resume recording of the simulation. The recording of
   -- the previous 10 minutes will be lost.
   -------------------------------------------------------------------------------
   procedure Stop_Replay is
   begin
      Data_Dispatcher.Stop_Replay;
   end Stop_Replay;

   -------------------------------------------------------------------------------
   -- Notify the Data Dispatcher that new data has arrived on the Channel.
   -------------------------------------------------------------------------------
   procedure Save_Data_Arrived (Subsystem : in Scheduler.Handle) is
   begin
      Data_Dispatcher.Save_Data_Arrived (Subsystem);
   end Save_Data_Arrived;


   -------------------------------------------------------------------------------
   -- Notify the Data Dispatcher that the requested amount of restore space has
   -- arrived on the requested restore stream.
   -------------------------------------------------------------------------------
   procedure Restore_Space_Arrived is
   begin
      Data_Dispatcher.Restore_Space_Arrived;
   end Restore_Space_Arrived;

   -------------------------------------------------------------------------------
   -- This routine returns True when a replay (from a file or memory) is in
   -- progress. A Restore does not affect matters.
   -------------------------------------------------------------------------------
   function Replay_In_Progress return Boolean is
   begin
      return Replaying;
   end Replay_In_Progress;

   -------------------------------------------------------------------------------
   -- This routine returns True when a replay (from a file or memory) is trimming.
   -------------------------------------------------------------------------------
   function Replay_Trimming return Boolean is
   begin
      return Trimming;
   end Replay_Trimming;

   -------------------------------------------------------------------------------
   -- This routine returns the length of the named replay file.
   -------------------------------------------------------------------------------
   function Demo_Length (File_Name : in String) return Duration is

      use type Ada.Real_Time.Time;

      File_Stream : Ada.Streams.Stream_Io.Stream_Access;

      Header      : Header_Element_Array;
      Subsystem   : Scheduler.Handle;

      First_Time : Ada.Real_Time.Time;
      Last_Time  : Ada.Real_Time.Time;

      Formation_Switch     : Integer;

   begin
      if Ios_Interface.Allow_Demo_Select then
         Ios_Interface.Allow_Demo_Select := False;
      end if;

      -- Open the new file stream for reading
      Log.Report ("Opening the file stream for reading.");
      Ada.Streams.Stream_IO.Open
        (File => Replay_File,
         Mode => Ada.Streams.Stream_Io.In_File,
         Name => File_Name
         );
      File_Stream := Ada.Streams.Stream_Io.Stream (Replay_File);

      IOS_Interface.Formation_Demo_Profile_Name := IOS_Interface.Bounded_64.to_string(IOS_Interface.Spaces);
      Integer'Read(File_Stream, Formation_Switch);
      if (Formation_Switch = 1) then
         --Integer'Read(File_Stream, Formation_Mgr.Formation_Track_Number);
         declare
            Formation_Track_Number : Integer := Integer'Input(File_Stream);
            Formation_Profile_Name : constant String := String'Input (File_Stream);
         begin
            Log.Report ("Track read from file is: " & Integer'Image(Formation_Track_Number));
            Log.Report("Profile Name read is: " & Formation_Profile_Name);
            if not IOS_Interface.Play_Formation_Demo then
               IOS_Interface.Formation_Demo_Track_Number := Formation_Track_Number;
               IOS_Interface.Allow_Offset_Adjustment := False;
               IOS_Interface.Play_Formation_Demo := True;
               for i in 1..Formation_Profile_Name'Length loop
                  IOS_Interface.Formation_Demo_Profile_Name(i) := Formation_Profile_Name(i);
               end loop;

            end if;
         end;
      else
         IOS_Interface.Play_Formation_Demo := False;
         IOS_Interface.Allow_Offset_Adjustment := True;
         if Formation_Mgr.Profile_Driver and Ios_Interface.Formation_Demo_Leadship_Only then
            Ios_Interface.Formation_Demo_Leadship_Only := False;
            Formation_Mgr.Set_Profile_Driver(False);
            Formation_Mgr.Set_Init_Formation_Demo_Lp(False);
         end if;
      end if;
      
      Log.Report ("Retrieving the last time in the file.");
      Ada.Real_Time.Time'Read(File_Stream, Last_Time);

      -- Get the first header
      Log.Report ("Retrieving the header for the first entry.");
      Get_Next_Restore_Header
        (Source      => File_Stream,
         Header      => Header,
         Destination => Subsystem
         );

      Log.Report ("Retrieving the time of the first entry from the header.");
      First_Time :=
        Saved_Data_Header.Time (Saved_Data_Header.From_Stream_Element_Array (Header));
      Ada.Streams.Stream_IO.Close (Replay_File);
      Max_Offset := Ada.Real_Time.To_Duration (Last_Time - First_Time);
      Log.Report ("Returning the time delta between the last and first times.");
      return Ada.Real_Time.To_Duration (Last_Time - First_Time);
   end Demo_Length;

end Save_Restore;
