-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Real_Time;

with JPATS_IOS_Interface;
with Log;
with Simulation_Dictionary;
with Scheduler.Snapshot;

pragma Elaborate_All (Simulation_Dictionary);

-------------------------------------------------------------------------------
-- This package provides the IOS interface for the Save/Restore facility.
-------------------------------------------------------------------------------
package body Save_Restore.IOS_Interface is


   Minutes_Per_Second : constant := 1.0/60.0;

   Max_Demo_Minutes : constant := 10.0;

   Demo_File_Name : constant String := Simulation_Dictionary.Lookup ("Demo_File");

   -- The initialization allows the demo "rerun" tool to work
   Instant_Replay : Boolean := False;

   No_Demo_Request : constant := Max_Demo_Minutes;

   Formation_Demo_Initialized : Boolean := False;

   ----------------------------------------------------------------------------
   -- The IOS variables. These are read directly by the IOS interface at its
   -- own rate.
   --

   -- The current replay offset (in minutes)
   Replay_Delta : Float := 0.0;

   Ready  : Boolean := False;

   Flyout : Boolean := False;

   Play   : Boolean := False;

   Replaying : Boolean := False;

   Demo_Replay_Delta : Float := Max_Demo_Minutes;

   Demo_Record : Boolean := False;

   Creating_Demo_File : Boolean := False;

   Demo_File_Valid : Boolean := False;

   Demo_File_Minutes : Float := Max_Demo_Minutes;

   Create_Snapshot  : Boolean := False;
   Delete_Snapshot  : Integer := 0;
   Restore_Snapshot : Integer := 0;

   -- Export some of the variables for external replay control
   pragma Export (C, Demo_File_Valid, "Demo_File_Valid");
   pragma Export (C, Ready, "Replay_Ready");

   -- Variables that are updated by more than one task must be kept intact at all times.
   pragma Atomic (Ready);
   pragma Atomic (Replay_Delta);
   pragma Atomic (Replaying);
   pragma Atomic (Create_Snapshot);
   pragma Atomic (Delete_Snapshot);
   pragma Atomic (Restore_Snapshot);
   pragma Atomic (Creating_Demo_File);
   pragma Atomic (Demo_File_Valid);

   type Snapshot_Float_List is array (Scheduler.Snapshot.Snapshot_Index) of Float;
   type Snapshot_Long_Float_List is array (Scheduler.Snapshot.Snapshot_Index) of Long_Float;
   type Snapshot_Flag_List is array (Scheduler.Snapshot.Snapshot_Index) of Boolean;

   Snapshot_Time      : Snapshot_Float_List;
   Snapshot_Latitude  : Snapshot_Long_Float_List;
   Snapshot_Longitude : Snapshot_Long_Float_List;
   Snapshot_Altitude  : Snapshot_Float_List;
   Snapshot_Heading   : Snapshot_Float_List;
   Snapshot_Airspeed  : Snapshot_Float_List;
   Snapshot_Valid     : Snapshot_Flag_List := (others => False);

   ----------------------------------------------------------------------------
   -- A background task to forward save/restore requests to the Save/Restore
   -- package.
   -- They can't be processed directly, as they generally involve rendezvous
   -- with a low-priority task.

   -- Request variables
   Play_Request              : Boolean := Play;
   Demo_Record_Request       : Boolean := Demo_Record;
   Replay_Delta_Request      : Float   := Replay_Delta;
   Demo_Replay_Delta_Request : Float   := No_Demo_Request;
   Demo_File_Validity_Change : Boolean := Demo_File_Valid;

   -- Export some of the variables for external replay control
   pragma Export (C, Demo_Replay_Delta_Request, "Demo_Replay_Delta_Request");
   pragma Export (C, Play_Request, "Play_Request");

   pragma Atomic (Play_Request);
   pragma Atomic (Demo_Record_Request);
   pragma Atomic (Replay_Delta_Request);
   pragma Atomic (Demo_Replay_Delta_Request);
   pragma Atomic (Demo_File_Validity_Change);

   ----------------------------------------------------------------------------
   -- Routine used by the Agent to update all the IOS's snapshot display data
   -- from the descriptors.
   ----------------------------------------------------------------------------
   procedure Update_Descriptors is
      Descriptor : Scheduler.Snapshot.Snapshot_Descriptor;
   begin
      for Index in Scheduler.Snapshot.Snapshot_Index loop
         Descriptor := Scheduler.Snapshot.Get_Descriptor (Index);

         Snapshot_Valid (Index) := Descriptor.Valid;
         if Snapshot_Valid (Index) then
            Snapshot_Time      (Index) := Float(Descriptor.Mission_Time);
            Snapshot_Latitude  (Index) := Descriptor.Latitude;
            Snapshot_Longitude (Index) := Descriptor.Longitude;
            Snapshot_Altitude  (Index) := Descriptor.Altitude;
            Snapshot_Heading   (Index) := Descriptor.Heading;
            Snapshot_Airspeed  (Index) := Descriptor.Airspeed;
         end if;
      end loop;
   end Update_Descriptors;

   task Agent;
   task body Agent is
      use type Ada.Real_Time.Time;

      Agent_Rate : constant Ada.Real_Time.Time_Span := Ada.Real_Time.To_Time_Span (1.0/15.0);

      Next_Iteration : Ada.Real_Time.Time := Ada.Real_Time.Clock;

      Demo_Record_Start_Time : Ada.Real_Time.Time;

   begin
      loop

         -- See if we need to set a new replay delta
         if Replay_Delta_Request /= 0.0 then
            Replay (Duration(Replay_Delta_Request / Minutes_Per_Second));
            Replay_Delta         := Replay_Delta_Request;
            Replay_Delta_Request := 0.0;
            Ready                := False;
            Replaying            := True;
            Play                 := False;
            Play_Request         := False;
         end if;
         if Demo_Replay_Delta_Request /= No_Demo_Request then
            Replay (File_Name => Demo_File_Name,
                    Offset    => Duration(Demo_Replay_Delta_Request / Minutes_Per_Second)
                    );
            Demo_Replay_Delta         := Demo_Replay_Delta_Request;
            Demo_Replay_Delta_Request := No_Demo_Request;
            Ready                     := False;
            Replaying                 := True;
            Play                      := False;
            Play_Request              := False;
         end if;

         -- See if we need to perform a flyout
         if Flyout then
            Stop_Replay;

            Flyout       := False;
            Play_Request := False;
            Play         := False;
            Ready        := False;
            Replaying    := False;
            Formation_Demo_Initialized := False;
         end if;

         -- See if we need to Play/Pause the sim.
         if Play /= Play_Request then
            Pause_Replay (not Play_Request);
            Play := Play_Request;
         end if;

         -- See if we need to start or stop a demo recording.
         if Demo_Record /= Demo_Record_Request then
            if Demo_Record_Request then
               Demo_Record_Start_Time := Ada.Real_Time.Clock;
               Creating_Demo_File     := True;
            else
               Save
                 (File_Name  => Demo_File_Name,
                  Start_Time => Demo_Record_Start_Time
                  );
            end if;
            Demo_Record := Demo_Record_Request;
         end if;

         -- See if we need to create a snapshot
         if Create_Snapshot then
            Scheduler.Snapshot.Create;

            Update_Descriptors;
            Create_Snapshot := False;
         end if;

         -- See if we need to delete all snapshots
         if Delete_Snapshot = 21 then

            Delete_Snapshot := 20;

            While Delete_Snapshot > 0 Loop

                if Snapshot_Valid (Scheduler.Snapshot.Snapshot_Index(Delete_Snapshot)) then

                   Scheduler.Snapshot.Delete (Scheduler.Snapshot.Snapshot_Index(Delete_Snapshot));

                end if;

                Delete_Snapshot := Delete_Snapshot - 1;

            end loop;

            Update_Descriptors;
            Delete_Snapshot := 0;  -- For insurance purposes

         -- See if we need to delete a snapshot
         elsif Delete_Snapshot in
               Integer(Scheduler.Snapshot.Snapshot_Index'First) ..
               Integer(Scheduler.Snapshot.Snapshot_Index'Last) then
            Scheduler.Snapshot.Delete (Scheduler.Snapshot.Snapshot_Index(Delete_Snapshot));
            Update_Descriptors;
            Delete_Snapshot := 0;
         end if;

         -- See if we need to restore a snapshot
         if
           Restore_Snapshot in
           Integer(Scheduler.Snapshot.Snapshot_Index'First) ..
           Integer(Scheduler.Snapshot.Snapshot_Index'Last)
         then
            Scheduler.Snapshot.Restore (Scheduler.Snapshot.Snapshot_Index(Restore_Snapshot));

            Update_Descriptors;
            Restore_Snapshot := 0;
         end if;

         -- See if we need to update the file length field
         if Demo_File_Validity_Change /= Demo_File_Valid then
            Demo_File_Valid := Demo_File_Validity_Change;
            if Demo_File_Valid then
               Set_Demo_File_Length (Demo_Length (Demo_File_Name));
               if Play_Formation_Demo then
                  if not Formation_Demo_Initialized then
                     New_Formation_Demo_Loaded := True;
                  end if;
               else
                  Replay (File_Name => Demo_File_Name,
                          Offset    => 0.0);
                  New_Formation_Demo_Loaded := False;
                  Demo_Replay_Delta          := 0.0;
                  Demo_Replay_Delta_Request  := No_Demo_Request;
                  Ready                      := False;
                  Replaying                  := True;
                  Play                       := False;
                  Play_Request               := False;
               end if;
            else
               Demo_File_Minutes := 0.0;
               Demo_Replay_Delta := Max_Demo_Minutes;
               New_Formation_Demo_Loaded := False;
            end if;
         end if;

         Next_Iteration := Next_Iteration + Agent_Rate;
         delay until Next_Iteration;
      end loop;

   exception
      when Error : others =>
         Log.Report (Event => "Save_Restore.IOS_Interface.Agent terminated due to " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);
   end Agent;

   ----------------------------------------------------------------------------
   -- The IOS Set routines. These are called indirectly by the
   -- JPATS_Hostcomms_Scheduler.
   --

   ----------------------------------------------------------------------------
   -- The IOS requests that we prepare an instant replay from the given offset.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Replay_Delta (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Replay_Delta_Request := JPATS_IOS_Interface.Convert_To_Float(Value);
      Instant_Replay := True;
   end IOS_Set_Replay_Delta;

   ----------------------------------------------------------------------------
   -- The IOS requests that we prepare a demo replay from the given offset.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Demo_Replay_Delta (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Demo_Replay_Delta_Request := JPATS_IOS_Interface.Convert_To_Float(Value);
      Instant_Replay            := False;
   end IOS_Set_Demo_Replay_Delta;

   ----------------------------------------------------------------------------
   -- The IOS requests a flyout from a replay.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Flyout (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Flyout := True;
   end IOS_Set_Flyout;

   ----------------------------------------------------------------------------
   -- The IOS requests the replay be paused or unpaused.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Play (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Play_Request := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Play;

   ----------------------------------------------------------------------------
   -- The IOS requests the simulation be saved to a demo file.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Demo_Record (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Demo_Record_Request := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Demo_Record;

   ----------------------------------------------------------------------------
   -- The IOS is
   ----------------------------------------------------------------------------
   procedure IOS_Set_Demo_File_Valid (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Demo_File_Validity_Change := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Demo_File_Valid;

   ----------------------------------------------------------------------------
   -- The IOS requests that a snapshot of the simulation be created.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Create_Snapshot (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Create_Snapshot := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Create_Snapshot;

   ----------------------------------------------------------------------------
   -- The IOS requests that a snapshot be deleted.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Delete_Snapshot (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Delete_Snapshot := JPATS_IOS_Interface.Convert_To_Integer(Value);
   end IOS_Set_Delete_Snapshot;

   ----------------------------------------------------------------------------
   -- The IOS requests that a snapshot be restored.
   ----------------------------------------------------------------------------
   procedure IOS_Set_Restore_Snapshot (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Restore_Snapshot := JPATS_IOS_Interface.Convert_To_Integer(Value);
   end IOS_Set_Restore_Snapshot;

   procedure IOS_Set_Formation_Demo_Profile_Name
     (PData     : in    JPATS_IOS_Interface.Set_Value;
      Offset1d  : in    Integer;
      Offset2d  : in    Integer;
      Completed : in    Boolean )
   is
   begin
      if JPATS_IOS_Interface.Convert_To_Integer(Pdata) = 0 then
         Formation_Demo_Profile_Name(Offset1d+1) := ' ';
      else
         Formation_Demo_Profile_Name(Offset1d+1) :=
          JPATS_IOS_Interface.Convert_To_Character(Pdata);
      end if;
   exception
      when others =>
         Log.Report("Jpats_Restore.Ios_Interface.IOS_Set_Formation_Demo_Profile_Name()");
         raise;
   end IOS_Set_Formation_Demo_Profile_Name;

   procedure IOS_Set_Allow_Demo_Select (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Allow_Demo_Select := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Allow_Demo_Select;

   procedure IOS_Set_Allow_Offset_Adjustment (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Allow_Offset_Adjustment := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Allow_Offset_Adjustment;

   procedure IOS_Set_Play_Formation_Demo (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Play_Formation_Demo := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Play_Formation_Demo;

   procedure IOS_Set_Record_Formation_Demo (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Record_Formation_Demo := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Record_Formation_Demo;

   procedure IOS_Set_New_Formation_Demo_Loaded (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      New_Formation_Demo_Loaded := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_New_Formation_Demo_Loaded;

   procedure IOS_Set_Formation_Demo_Leadship_Loaded (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Formation_Demo_Leadship_Loaded := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Formation_Demo_Leadship_Loaded;

   procedure IOS_Set_Formation_Demo_Leadship_Only (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Formation_Demo_Leadship_Only := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Formation_Demo_Leadship_Only;

   procedure IOS_Set_Auto_Fly_Out (Value : in JPATS_IOS_Interface.Set_Value) is
   begin
      Auto_Fly_Out := JPATS_IOS_Interface.Convert_To_Bool(Value);
   end IOS_Set_Auto_Fly_Out;

   ----------------------------------------------------------------------------
   -- Externally visibile routines.
   --

   ----------------------------------------------------------------------------
   -- Register this package's IOS variables.
   -- This must be called after the JPATS_IOS_Interface package has been
   -- initialized, but before the JPATS_Hostcomms_Scheduler has been
   -- initialized.
   ----------------------------------------------------------------------------
   procedure Register_Variables is
   begin

      JPATS_IOS_Interface.Register
        (Name        => "Replay_Delta",
         Variable    => Replay_Delta'Address,
         Set_Routine => IOS_Set_Replay_Delta'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Replay_Ready",
         Variable    => Ready'Address
         );

      JPATS_IOS_Interface.Register
        (Name        => "Replaying",
         Variable    => Replaying'Address
         );

      JPATS_IOS_Interface.Register
        (Name        => "Replay_Flyout",
         Variable    => Flyout'Address,
         Set_Routine => IOS_Set_Flyout'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Replay_Pause_Play_Toggle",
         Variable    => Play'Address,
         Set_Routine => IOS_Set_Play'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Demo_Replay_Delta",
         Variable    => Demo_Replay_Delta'Address,
         Set_Routine => IOS_Set_Demo_Replay_Delta'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Demo_Record_Toggle",
         Variable    => Demo_Record'Address,
         Set_Routine => IOS_Set_Demo_Record'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Creating_Demo_File",
         Variable    => Creating_Demo_File'Address
         );

      JPATS_IOS_Interface.Register
        (Name        => "Demo_File_Valid",
         Variable    => Demo_File_Valid'Address,
         Set_Routine => IOS_Set_Demo_File_Valid'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Demo_File_Length",
         Variable    => Demo_File_Minutes'Address
         );

      JPATS_IOS_Interface.Register
        (Name        => "Create_Snapshot",
         Variable    => Create_Snapshot'Address,
         Set_Routine => IOS_Set_Create_Snapshot'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Delete_Snapshot",
         Variable    => Delete_Snapshot'Address,
         Set_Routine => IOS_Set_Delete_Snapshot'Access
         );

      JPATS_IOS_Interface.Register
        (Name        => "Restore_Snapshot",
         Variable    => Restore_Snapshot'Address,
         Set_Routine => IOS_Set_Restore_Snapshot'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Time",
         Var_Array          => Snapshot_Time(1)'Address,
         Dimension_1_Length => Snapshot_Time'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Latitude",
         Var_Array          => Snapshot_Latitude(1)'Address,
         Dimension_1_Length => Snapshot_Latitude'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Longitude",
         Var_Array          => Snapshot_Longitude(1)'Address,
         Dimension_1_Length => Snapshot_Longitude'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Altitude",
         Var_Array          => Snapshot_Altitude(1)'Address,
         Dimension_1_Length => Snapshot_Altitude'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Heading",
         Var_Array          => Snapshot_Heading(1)'Address,
         Dimension_1_Length => Snapshot_Heading'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Airspeed",
         Var_Array          => Snapshot_Airspeed(1)'Address,
         Dimension_1_Length => Snapshot_Airspeed'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Snapshot_Valid",
         Var_Array          => Snapshot_Valid(1)'Address,
         Dimension_1_Length => Snapshot_Valid'Length
         );

      JPATS_IOS_Interface.Register
        (Name               => "Formation_Demo_Profile_Name",
         Var_Array          => Formation_Demo_Profile_Name'Address,
         Set_Routine        => IOS_Set_Formation_Demo_Profile_Name'Access,
         Dimension_1_Length => 64
         );

      JPATS_IOS_Interface.Register
        (Name               => "Allow_Demo_Select",
         Variable           => Allow_Demo_Select'Address,
         Set_Routine        => IOS_Set_Allow_Demo_Select'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Allow_Offset_Adjustment",
         Variable           => Allow_Offset_Adjustment'Address,
         Set_Routine        => IOS_Set_Allow_Offset_Adjustment'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Play_Formation_Demo",
         Variable           => Play_Formation_Demo'Address,
         Set_Routine        => IOS_Set_Play_Formation_Demo'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Record_Formation_Demo",
         Variable           => Record_Formation_Demo'Address,
         Set_Routine        => IOS_Set_Record_Formation_Demo'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "New_Formation_Demo_Loaded",
         Variable           => New_Formation_Demo_Loaded'Address,
         Set_Routine        => IOS_Set_New_Formation_Demo_Loaded'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Formation_Demo_Leadship_Loaded",
         Variable           => Formation_Demo_Leadship_Loaded'Address,
         Set_Routine        => IOS_Set_Formation_Demo_Leadship_Loaded'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Formation_Demo_Leadship_Only",
         Variable           => Formation_Demo_Leadship_Only'Address,
         Set_Routine        => IOS_Set_Formation_Demo_Leadship_Only'Access
         );

      JPATS_IOS_Interface.Register
        (Name               => "Auto_Fly_Out",
         Variable           => Auto_Fly_Out'Address,
         Set_Routine        => IOS_Set_Auto_Fly_Out'Access
         );

   end Register_Variables;

   ----------------------------------------------------------------------------
   -- This routine changes the replay delta that the IOS displays to the
   -- operator.
   ----------------------------------------------------------------------------
   procedure Update_Replay_Delta (New_Delta : in Duration) is
   begin
      if Instant_Replay then
         Replay_Delta := Float(New_Delta) * Minutes_Per_Second;
      else
         Demo_Replay_Delta := Float(New_Delta) * Minutes_Per_Second;
      end if;
   end Update_Replay_Delta;

   ----------------------------------------------------------------------------
   -- This routine notifies the IOS that the a replay has been trimed.
   ----------------------------------------------------------------------------
   procedure Replay_Ready is
   begin
      Ready := True;
   end Replay_Ready;

   ----------------------------------------------------------------------------
   -- This routine notifies the IOS that the a replay has completed.
   ----------------------------------------------------------------------------
   procedure Replay_Complete is
   begin
      Ready        := False;
      Play         := False;
      Play_Request := False;
      if Auto_Fly_Out then
         Flyout := true;
      end if;
   end Replay_Complete;

   ----------------------------------------------------------------------------
   -- This routine notifies the IOS that the demo file has been saved.
   ----------------------------------------------------------------------------
   procedure Demo_File_Saved is
   begin
      Creating_Demo_File        := False;
      Demo_File_Valid           := True;
      Demo_File_Validity_Change := True;
   end Demo_File_Saved;

   ----------------------------------------------------------------------------
   -- This routine modifies the demo file length. This should be called after
   -- a demo replay is started before any call to Update_Replay_Delta.
   ----------------------------------------------------------------------------
   procedure Set_Demo_File_Length (Seconds : in Duration) is
   begin
      Log.Report("Set_Demo_File_Length");
      Demo_File_Minutes := Float(Seconds) * Minutes_Per_Second;
      Demo_Replay_Delta := Demo_File_Minutes;
   end Set_Demo_File_Length;

   procedure Init_Formation_Demo_Replay_Delta is
   begin
      Replay (File_Name => Demo_File_Name,
              Offset    => 0.0);
      Demo_Replay_Delta          := 0.0;
      Demo_Replay_Delta_Request  := No_Demo_Request;
      Ready                      := False;
      Replaying                  := True;
      Play                       := False;
      Play_Request               := False;
      Formation_Demo_Initialized := True;
   end Init_Formation_Demo_Replay_Delta;

end Save_Restore.IOS_Interface;
