-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Ted E. Dennison
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Ada.Real_Time;
with Buffer_Stream.Restore;
with Buffer_Stream.Save;

-- Allow builtin operators for the following types
use type Ada.Real_Time.Time;
use type Ada.Real_Time.Time_Span;

-------------------------------------------------------------------------------
-- This package provides control over the record/playback functionality.
-------------------------------------------------------------------------------
package body Record_Playback is

   -- The amount of time to wait between model snapshots
   Snapshot_Period : constant Ada.Real_Time.Time_Span := Ada.Real_Time.To_Time_Span(10.0);

   -- Indicates when the next snapshot should occur
   Next_Snap : Ada.Real_Time.Time := Ada.Real_Time.Clock + Snapshot_Period;

   -- Indicates whether saving of model data is enabled or disabled
   Saving : Boolean := True;

   ----------------------------------------------------------------------------
   -- Check to see if its time to snapshot all the models yet.
   -- The time is based on the time of previous calls to this routine.
   ----------------------------------------------------------------------------
   function Model_Snapshot_Time return Boolean is
   begin
      if Saving and Next_Snap < Ada.Real_Time.Clock then
         Next_Snap := Next_Snap + Snapshot_Period;

         return True;
      end if;

      return False;
   end Model_Snapshot_Time;

   -------------------------------------------------------------------------------
   -- Enable or disable saving of model data. An input value of False indicates
   -- saving is to be disabled.
   -------------------------------------------------------------------------------
   procedure Enable_Saving ( Enable : in Boolean := True) is
   begin
      Saving := Enable;
   end Enable_Saving;

   -------------------------------------------------------------------------------
   -- Take a snapshot of the state of the given module
   -------------------------------------------------------------------------------
   procedure Snapshot (Module : in out Subsystem_Scheduler.Instance'Class) is

      Save_Stream_Handle : constant Buffer_Stream.Save.Handle :=
        Subsystem_Scheduler.Save_Stream (Module) ;

      Open_Succeeded : Boolean;

   begin

      Buffer_Stream.Save.Open_Writes
        ( Stream  => Save_Stream_Handle.all,
          Success => Open_Succeeded);

      -- Make sure the stream isn't in use
      if Open_Succeeded then

         Subsystem_Scheduler.Save
           (An_Instance => Module,
            To_Stream   => Save_Stream_Handle.all
            );

         Buffer_Stream.Save.Close_Writes (Save_Stream_Handle.all);

      end if;

   end Snapshot;

   -------------------------------------------------------------------------------
   -- Attempt to restore the state of the given module, if need be.
   -------------------------------------------------------------------------------
   procedure Restore (Module : in out Subsystem_Scheduler.Instance'Class) is

      Restore_Stream_Handle : constant Buffer_Stream.Restore.Handle :=
        Subsystem_Scheduler.Restore_Stream (Module);

      Open_Succeeded : Boolean;

   begin

      -- If there is replay data, see if it needs to be restored
      if Buffer_Stream.Restore.Data_Available (Restore_Stream_Handle) then

         Buffer_Stream.Restore.Open_Reads
           (Stream  => Restore_Stream_Handle.all,
            Success => Open_Succeeded
            );

         if Open_Succeeded then

            Subsystem_Scheduler.Restore
              (An_Instance => Module,
               From_Stream => Restore_Stream_Handle.all
               );

            Buffer_Stream.Restore.Close_Reads (Restore_Stream_Handle.all);

         end if;
      end if;

   exception
      when Error : others =>
         Log.Report (Event => "Exception in Record_Playback.Restore: " &
                     "unhandled exception." & Ada.Characters.Latin_1.Cr &
                     Ada.Characters.Latin_1.Lf &
                     Ada.Exceptions.Exception_Information (Error),
                     Severity => Log.Error);

   end Restore;

end Record_Playback;
